<?php

namespace Resrequest\API\Authorization;

use DateTime;
use Zend\Http\Request;
use ZF\MvcAuth\MvcAuthEvent;
use ZF\MvcAuth\Authorization\AclAuthorizationFactory as AclFactory;
use Resrequest\Authentication\Service\Authenticate as Authenticate;

class AuthorizationListener
{
    private static $ALWAYS_ALLOW = [
        '^ZF\\\\OAuth2.*',
        '^MaglLegacyApplication',
        '^ZF\\\\Apigility\\\\Admin.*',
    ];

    private static $TRIGGERS = [
        "REFRESH_SESSION" => 1,
        "LOG_OUT" => 2
    ];

    protected $httpMethods = [
        Request::METHOD_DELETE => true,
        Request::METHOD_GET    => true,
        Request::METHOD_PATCH  => true,
        Request::METHOD_POST   => true,
        Request::METHOD_PUT    => true,
    ];

    protected $enterprise;
    protected $moduleConfig = array();
    protected $request = null;
    protected $method = "";

    public function __construct($enterprise, $moduleConfig, $request)
    {
        $this->enterprise = $enterprise;
        $this->moduleConfig = $moduleConfig;
        $this->request = $request;
        if ($request instanceof HttpRequest) {
            $this->method = $request->getMethod();
        } elseif($request instanceof ConsoleRequest ) {
            $this->method = "GET";
        }
    }

    public function __invoke(MvcAuthEvent $mvcAuthEvent)
    {
        $event = $mvcAuthEvent->getMvcEvent();
        $resource = $mvcAuthEvent->getResource();
        $identity = $mvcAuthEvent->getIdentity();

        if($identity instanceof \ZF\MvcAuth\Identity\AuthenticatedIdentity) {
            $username = $identity->getName();
            $this->enterprise->setUser($username);
        }
        
        $contentTypeHeaders = $this->request->getHeaders()->get('content-type');
        if (!empty($contentTypeHeaders) && $contentTypeHeaders->getFieldValue() == "application/json") {
            $content = $this->request->getContent();
            $content = !empty($content)?\Zend\Json\Json::decode($content, \Zend\Json\Json::TYPE_ARRAY):array();

            $trigger = 0;
            if (!empty($content)) {
                if (isset($content['grant_type']) && $content['grant_type'] == "refresh_token") {
                    $trigger = self::$TRIGGERS['REFRESH_SESSION'];
                } elseif (!empty($content) && isset($content['token_type_hint']) && $content['token_type_hint'] == "access_token" && strcasecmp($resource, "ZF\\\\OAuth2\\\\Controller\\\\Auth::revoke")) {
                    $trigger = self::$TRIGGERS['LOG_OUT'];
                }
            }
            $this->trigger($event, $trigger, $content);
        }

        return $this->checkAcl($mvcAuthEvent, $resource, $event);
    }

    /**
     * checkAcl based on acl
     *
     * @param ZF\MvcAuth\MvcAuthEvent $mvcAuthEvent
     * @param string $resource
     * @param array $content
     */
    protected function checkAcl($mvcAuthEvent, $resource, $event)
    {
        // Always allow certain resources
        foreach(self::$ALWAYS_ALLOW as $allow) {
            if(preg_match("/" . $allow . "/", $resource)) {
                return true;
            }
        }

        $username = false;
        $identity = $mvcAuthEvent->getIdentity();
        if($identity instanceof \ZF\MvcAuth\Identity\AuthenticatedIdentity) {
            $username = $identity->getAuthenticationIdentity()['user_id'];
        }

        /** @var \ZF\MvcAuth\Authorization\AclAuthorization $aclAuthorization */
        $aclAuthorization = $mvcAuthEvent->getAuthorizationService();

        $acl = $this->createAclFromConfig($this->moduleConfig['zf-mvc-auth']['authorization']);
        //This service requires authorization for this HTTP method
        $requiresLogin = isset($acl[$resource]['privileges'])&&in_array($this->method,$acl[$resource]['privileges'])?true:false;
        if ($requiresLogin)
        {
            $allow = false;
            $user = false;
            $em = $event->getApplication()->getServiceManager()->get("EnterpriseEntityManager");
            $conn = $em->getConnection();
            if(!empty($username)) {
                // If authenticated
                $user = $conn->fetchAssoc(
                    'SELECT * from '.$this->moduleConfig['zf-mvc-auth']['user_table'].' WHERE pr_user_name=?',
                    array($username)
                );
                if($user) {
                    $allow = true;

                }
            }

            // Deny everything else
            if(!$allow) {
                $aclAuthorization->deny(null, null);
            }
        } else {
            $allow = true;
        }
    }

    /**
     * Execute a specific action based on a predetermined trigger
     *
     * @param ZF\MvcAuth\MvcAuthEvent $event
     * @param int $trigger
     * @param array $content
     */
    protected function trigger($event, $trigger, $content)
    {
        $authorization = $event->getApplication()->getServiceManager()->get("Resrequest\Authentication\Service\Authenticate");
        switch ($trigger) {
            case self::$TRIGGERS['REFRESH_SESSION']:
                // refresh token request sent - refresh the session
                $userId = isset($_SESSION['userid'])?$_SESSION['userid']:"";
                $userDetails = $authorization->getUserById($userId);
                if ($userDetails) {
                    $authorization->setupContext($userDetails);
                }
                break;
            case self::$TRIGGERS['LOG_OUT']:
                // revoke token request; log out
                $userDetails = $authorization->getUserByToken($content['rrq_token'], Authenticate::TOKEN_TYPE_ACCESS);
                if ($userDetails) {
                    // set up context so session to log out from can be identified
                    $authorization->setupContext($userDetails);
                    $authorization->logout($userDetails['pr_user_name']);
                }
                break;
        }
    }

    /**
     * Generate the ACL array based on the zf-mvc-auth "authorization" configuration
     *
     * @param array $config
     * @return array
     */
    protected function createAclFromConfig(array $config)
    {
        $aclConfig = array();
        $denyByDefault = false;

        if (array_key_exists('deny_by_default', $config)) {
            $denyByDefault = $aclConfig['deny_by_default'] = (bool) $config['deny_by_default'];
            unset($config['deny_by_default']);
        }

        foreach ($config as $controllerService => $privileges) {
            $this->createAclConfigFromPrivileges($controllerService, $privileges, $aclConfig, $denyByDefault);
        }
        if (array_key_exists('deny_by_default', $aclConfig)) {
            unset($aclConfig['deny_by_default']);
        }
        return $aclConfig;
    }

    /**
     * Creates ACL configuration based on the privileges configured
     *
     * - Extracts a privilege per action
     * - Extracts privileges for each of "collection" and "entity" configured
     *
     * @param string $controllerService
     * @param array $privileges
     * @param array $aclConfig
     * @param bool $denyByDefault
     */
    protected function createAclConfigFromPrivileges($controllerService, array $privileges, &$aclConfig, $denyByDefault)
    {
        // Normalize the controller service name.
        // zend-mvc will always pass the name using namespace seprators, but
        // the admin may write the name using dash seprators.
        $controllerService = strtr($controllerService, '-', '\\');
        if (isset($privileges['actions'])) {
            foreach ($privileges['actions'] as $action => $methods) {
                $action = lcfirst($action);
                $aclConfig[sprintf('%s::%s', $controllerService, $action)] = [
                    'privileges' => $this->createPrivilegesFromMethods($methods, $denyByDefault),
                ];
            }
        }

        if (isset($privileges['collection'])) {
            $aclConfig[sprintf('%s::collection', $controllerService)] = [
                'privileges' => $this->createPrivilegesFromMethods($privileges['collection'], $denyByDefault),
            ];
        }

        if (isset($privileges['entity'])) {
            $aclConfig[sprintf('%s::entity', $controllerService)] = [
                'privileges' => $this->createPrivilegesFromMethods($privileges['entity'], $denyByDefault),
            ];
        }

        return $aclConfig;
    }

    /**
     * Create the list of HTTP methods defining privileges
     *
     * @param array $methods
     * @param bool $denyByDefault
     * @return array|null
     */
    protected function createPrivilegesFromMethods(array $methods, $denyByDefault)
    {
        $privileges = [];

        if (isset($methods['default']) && $methods['default']) {
            $privileges = $this->httpMethods;
            unset($methods['default']);
        }

        foreach ($methods as $method => $flag) {
            // If the flag evaluates true and we're denying by default, OR
            // if the flag evaluates false and we're allowing by default,
            // THEN no rule needs to be added
            if (( $denyByDefault && $flag)
                || (! $denyByDefault && ! $flag)
            ) {
                if (isset($privileges[$method])) {
                    unset($privileges[$method]);
                }
                continue;
            }

            // Otherwise, we need to add a rule
            $privileges[$method] = true;
        }

        if (empty($privileges)) {
            return null;
        }

        return array_keys($privileges);
    }
}

