<?php
namespace Resrequest\API\V1\Rpc\Chart;

use Resrequest\Application\Chart\ChartBuilder;
use Zend\Mvc\Controller\AbstractActionController;
use ZF\ApiProblem\ApiProblem;
use ZF\ApiProblem\ApiProblemResponse;
use ZF\ContentNegotiation\JsonModel;

class ChartController extends AbstractActionController
{
    private $accessGroupId;

    protected $em;
    protected $conn;
    protected $authentication;
    protected $authorisation;

    /**
     * ChartBuilder
     *
     * @var ChartBuilder
     */
    protected $chartBuilder;

    public function __construct($em, $enterprise, ChartBuilder $chartBuilder, $authentication, $authorisation)
    {
        $this->em = $em;
        $this->auth = $authentication;
        $this->conn = $em->getConnection();
        $this->chartBuilder = $chartBuilder;
        $this->accessGroupId = $enterprise->accessGroupId;
        $this->authorisation = $authorisation;
    }

    public function setupContext() {
        $identity = $this->getIdentity();
        if ($identity instanceof \ZF\MvcAuth\Identity\AuthenticatedIdentity) {
            $username = $identity->getAuthenticationIdentity()['user_id'];
        }

        if ($username === false) {
            return new ApiProblemResponse(
                new ApiProblem(403, "Unable to find user")
            );
        }

        $this->auth->setupContext($this->auth->getUserByName($username));
    }

    public function chartAction()
    {
        $this->setupContext();
        $params = $this->bodyParams();
        $chartRequest = $params['chartRequest'];

        $dateOptions = $this->buildDateOptions();

        $dateRangeOption = $this->getDateRangeOptions();

        $dialChartJson = '';
        $agentActivityChartJson = '';
        $bedNightActivityChartJson = '';
        $salesPerEmployeePerWeekdayChartJson = '';
        $bookingsByStatusChartJson = '';
        $bookingsByNationalityChartJson = '';
        $bookingRevenueYieldJson = '';
        $cardsJson = '';

        if (array_key_exists('dial', $chartRequest)) {
            $dateRange = $dateRangeOption[$chartRequest['dial']];
            $dialChartJson = $this->buildDialChart($dateRange);
        }
        if (array_key_exists('agentActivity', $chartRequest)) {
            $dateRange = $dateRangeOption[$chartRequest['agentActivity']];
            $agentActivityChartJson = $this->buildAgentActivityChart($dateRange);
        }
        if (array_key_exists('bedNightActivity', $chartRequest)) {
            $dateRange = $dateRangeOption[$chartRequest['bedNightActivity']];
            $bedNightActivityChartJson = $this->buildBedNightActivityChart($dateRange);
        }
        if (array_key_exists('salesPerEmployeePerWeekday', $chartRequest)) {
            $dateRange = $dateRangeOption[$chartRequest['salesPerEmployeePerWeekday']];
            $salesPerEmployeePerWeekdayChartJson = $this->salesPerEmployeePerWeekdayChart($dateRange);
        }
        if (array_key_exists('bookingsByStatus', $chartRequest)) {
            $dateRange = $dateRangeOption[$chartRequest['bookingsByStatus']];
            $bookingsByStatusChartJson = $this->buildBookingsByStatusChart($dateRange);
        }
        if (array_key_exists('bookingsByNationality', $chartRequest)) {
            $dateRange = $dateRangeOption[$chartRequest['bookingsByNationality']];
            $bookingsByNationalityChartJson = $this->bookingsByNationalityChart($dateRange);
        }
        if (array_key_exists('cards', $chartRequest)) {
            // Cards do not support this year and last year date range
            if ($chartRequest['cards'] == 'thisYearAndLastYear') {
                $chartRequest['cards'] = 'thisYear';
            }

            $dateRange = $dateRangeOption[$chartRequest['cards']];
            $cardsJson = $this->buildCards($dateRange, $chartRequest['cards']);
        }
        if (array_key_exists('bookingRevenueYield', $chartRequest)) {
            if ($this->authorisation->getFunctionAccessLevel(9000)[9000] === 15) { // Agent Overview management access
                $bookingRevenueYieldJson = $this->getChartById('11ea31004cdfaf038fc802366610d15e', [[
                    'source' => 'agent',
                    'option' => 'agent',
                    'value' => $params['contactId']
                ]]);
            }
        }

        return new JsonModel(
            [
                'dialChart' => $dialChartJson,
                'agentActivityChart' => $agentActivityChartJson,
                'bedNightActivityChart' => $bedNightActivityChartJson,
                'salesPerEmployeePerWeekdayChart' => $salesPerEmployeePerWeekdayChartJson,
                'bookingsByStatusChart' => $bookingsByStatusChartJson,
                'bookingsByNationalityChart' => $bookingsByNationalityChartJson,
                'cards' => $cardsJson,
                'bookingRevenueYieldChart' => $bookingRevenueYieldJson,
                'dateOptions' => $dateOptions
            ]
        );
    }

    public function getChartById($id, $optionValues = []) {
        $data = $this->chartBuilder->getChartAndData($id, $optionValues)['data'];
        return $data;
    }

    public function buildDateOptions() {
        $date = new \DateTime();
        $startDate = new \DateTime();
        $endDate = new \DateTime();

        $startDate->modify('monday this week');
        $endDate->modify('sunday this week');
        $dateOptions['thisWeek'] = [
            'startDate' => 'Monday',
            'endDate' => 'Sunday'
        ];

        $startDate = new \DateTime();
        $endDate = new \DateTime();

        $startDate->modify('first day of ' . $date->format('F') . ' ' . $date->format('Y'));
        $endDate->modify('last day of ' . $date->format('F') . ' ' . $date->format('Y'));
        $dateOptions['thisMonth'] = [
            'startDate' => $startDate->format("d M Y"),
            'endDate' => $endDate->format("d M Y")
        ];

        $quarter = $this->getQuarterDates();
        $dateOptions['thisQuarter'] = [
            'startDate' => $quarter['startDate']->format("d M Y"),
            'endDate' => $quarter['endDate']->format("d M Y")
        ];

        $startDate = new \DateTime();
        $endDate = new \DateTime();

        $dateOptions['thisYear'] = [
            'startDate' => $startDate->modify('first day of january this year')->format("d M Y"),
            'endDate' => $endDate->modify('last day of december this year')->format("d M Y")
        ];

        $startDate = new \DateTime();
        $endDate = new \DateTime();

        $dateOptions['thisYearAndLastYear'] = [
            'startDate' => $startDate->modify('first day of january last year')->format("d M Y"),
            'endDate' => $endDate->modify('last day of december this year')->format("d M Y")
        ];

        return $dateOptions;
    }

    public function getDateRangeOptions() {
        $date = new \DateTime();
        $startDate = new \DateTime();
        $endDate = new \DateTime();

        $startDate->modify('monday this week');
        $endDate->modify('sunday this week');
        $dateOptions['thisWeek'] = [
            'startDate' => $startDate,
            'endDate' => $endDate
        ];

        $startDate = new \DateTime();
        $endDate = new \DateTime();

        $startDate->modify('first day of ' . $date->format('F') . ' ' . $date->format('Y'));
        $endDate->modify('last day of ' . $date->format('F') . ' ' . $date->format('Y'));
        $dateOptions['thisMonth'] = [
            'startDate' => $startDate,
            'endDate' => $endDate
        ];

        $quarter = $this->getQuarterDates();
        $dateOptions['thisQuarter'] = [
            'startDate' => $quarter['startDate'],
            'endDate' => $quarter['endDate']
        ];

        $startDate = new \DateTime();
        $endDate = new \DateTime();

        $dateOptions['thisYear'] = [
            'startDate' => $startDate->modify('first day of january this year'),
            'endDate' => $endDate->modify('last day of december this year')
        ];

        $startDate = new \DateTime();
        $endDate = new \DateTime();

        $dateOptions['thisYearAndLastYear'] = [
            'startDate' => $startDate->modify('first day of january last year'),
            'endDate' => $endDate->modify('last day of december this year')
        ];
        
        return $dateOptions;
    }

    /** 
     * Returns the quarter period dates for the given date
     */
    public static function getQuarterDates($date = false) {
        if ($date === false) {
            $currentMonth = date('m');
        } else {
            $currentMonth = $date->format('m'); 
        }

        if($currentMonth >= 1 && $currentMonth <= 3) {
            $startDate = new \DateTime();
            $startDate->modify('first day of January');
            $endDate = new \DateTime();
            $endDate->modify('last day of March');
        } else if($currentMonth >= 4 && $currentMonth <= 6) {
            $startDate = new \DateTime();
            $startDate->modify('first day of April');
            $endDate = new \DateTime();
            $endDate->modify('last day of June');
        } else if($currentMonth >= 7 && $currentMonth <= 9) {
            $startDate = new \DateTime();
            $startDate->modify('first day of July');
            $endDate = new \DateTime();
            $endDate->modify('last day of October');
        } else if($currentMonth >= 10 && $currentMonth <= 12) {
            $startDate = new \DateTime();
            $startDate->modify('first day of October');
            $endDate = new \DateTime();
            $endDate->modify('last day of December');
        }

        return [
            'startDate' => $startDate,
            'endDate' => $endDate
        ];
    }

    public function buildBookingsByStatusChart($dateRange) {
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];

        $bookingsData = [];

        // Chart variables
        $chartData = [];
        $totalBookings = 0;

        $chartJson = [
            'chart' =>
            [
                'caption' => 'Bookings by status',
                'subcaption' => '',
                'showvalue' => '1',
                'showPercentValues' => '0',
                'theme' => 'fusion',
                'plottooltext' => '$label bookings: <b>$value</b>',
                'baseFont' => 'Helvetica',
                'outCnvBaseFont' => 'Helvetica',
                'decimals' => '0',
                'plottooltext' => '$label percentage: <b>$percentValue</b>',
                'centerLabel' => '$value',
            ],
            'data' => $chartData,
        ];

        $bookingsQuery = $this->em->createQueryBuilder();

        $params = $this->bodyParams();
        $contactId = $params['contactId']; // contactId parameter

        // Prepare date variables
        $startDateString = $startDate->format("Y-m-d"); // Convert to string and format for MariaDB
        
        $endDateString = $endDate->format("Y-m-d"); // Convert to string and format for MariaDB

        $reservationStatusFilter = [ // Whitelist of reservation status filters
            'Quotation',
            'Wait listed',
            // 'Allocation',
            'Provisional',
            // 'In progress',
            'Confirmed',
            // 'Invoiced',
            // 'Blocked',
            'Cancelled',
            // 'Deleted',
            // 'Deleted allocation',
        ];

        // Get data
        $bookings = $bookingsQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationId) AS reservationCount',
                    'reservationStatusTable.rfReservationStatusDesc AS reservationStatus',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->where('reservation.rvAgentId = :contactId')
            ->andWhere('reservation.rvDateRecorded >= :startDate')
            ->andWhere('reservation.rvDateRecorded <= :endDate')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->groupBy('reservationStatus')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'startDate' => $startDateString,
                    'endDate' => $endDateString,
                    'reservationStatusFilter' => $reservationStatusFilter
                ]
            )
            ->getQuery()
            ->getResult();

        // Process data

        foreach ($bookings as $booking) {
            $bookingCount = $booking['reservationCount'];
            $bookingStatus = $booking['reservationStatus'];

            $totalBookings = $totalBookings + $bookingCount;

            $bookingsData[$bookingStatus] = $bookingCount;
        }

        foreach ($bookingsData as $bookingStatus => $bookingCount) {
            array_push($chartData, [
                'label' => $bookingStatus,
                'value' => $bookingCount,
            ]);
        }

        $chartJson['chart']['subcaption'] = $startDate->format('d M Y') . " - " . $endDate->format('d M Y');
        
        if (empty($chartData)) {
            $chartJson['data'] = false;
        } else {
            $chartJson['data'] = $chartData;
        }

        return $chartJson;
    }

    public function bookingsByNationalityChart($dateRange) {
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];

        $bookingsData = [];
        $statusWhitelist = [ // Whitelist of reservation status filters
            'Quotation',
            'Wait listed',
            // 'Allocation',
            'Provisional',
            // 'In progress',
            'Confirmed',
            // 'Invoiced',
            // 'Blocked',
            'Cancelled',
            // 'Deleted',
            // 'Deleted allocation',
        ];

        // Chart variables
        $chartData = [];
        $totalBookings = 0;

        $chartJson = [
            'chart' =>
            [
                'caption' => 'Bookings by nationality (all statuses)',
                'subcaption' => '',
                'showvalue' => '1',
                'theme' => 'fusion',
                'plottooltext' => '$label percentage: <b>$percentValue</b>',
                'centerLabel' => '$value',
                'baseFont' => 'Helvetica',
                'outCnvBaseFont' => 'Helvetica',
                'legendIconScale' => '0.8',
                'decimals' => '0',
                'manageLabelOverflow' => '1',
                'showPercentValues' => '0',
                'minimiseWrappingInLegend' => '1',
            ],
            'data' => $chartData,
        ];

        $bookingsQuery = $this->em->createQueryBuilder();

        $params = $this->bodyParams();
        $contactId = $params['contactId']; // contactId parameter

        // Prepare date variables
        $startDateString = $startDate->format("Y-m-d"); // Convert to string and format for MariaDB
        
        $endDateString = $endDate->format("Y-m-d"); // Convert to string and format for MariaDB

        // Get data
        $bookings = $bookingsQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationId) AS reservationCount',
                    'reservationStatusTable.rfReservationStatusDesc AS reservationStatus',
                    'country.rfCountryName AS nationality',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'country', 'with', 'country.rfCountryIx = reservation.rfCountryId')
            ->where('reservation.rvAgentId = :contactId')
            ->andWhere('reservation.rvDateArrive >= :startDate')
            ->andWhere('reservation.rvDateDepart <= :endDate')
            ->groupBy('nationality')
            ->addGroupBy('reservationStatus')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'startDate' => $startDateString,
                    'endDate' => $endDateString,
                ]
            )
            ->getQuery()
            ->getResult();

        // Process data

        foreach ($bookings as $booking) {
            $bookingCount = $booking['reservationCount'];
            $bookingNationality = $booking['nationality'];
            $bookingStatus = $booking['reservationStatus'];

            if (!in_array($bookingStatus, $statusWhitelist)) {
                continue;
            }

            if (empty($bookingNationality)) {
                $bookingNationality = "No country";
            }

            $totalBookings = $totalBookings + $bookingCount;

            $bookingsData[$bookingNationality] = isset($bookingsData[$bookingNationality]) ? $bookingsData[$bookingNationality] + $bookingCount : $bookingCount;
        }

        foreach ($bookingsData as $bookingNationality => $bookingCount) {
            array_push($chartData, [
                'label' => $bookingNationality,
                'value' => $bookingCount,
            ]);
        }

        $chartJson['chart']['subcaption'] = $startDate->format('d M Y') . " - " . $endDate->format('d M Y');
        
        if (empty($chartData)) {
            $chartJson['data'] = false;
        } else {
            $chartJson['data'] = $chartData;
        }

        return $chartJson;
    }

    public function buildDialChart($dateRange) {
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];
        
        $bookingsData = [];
        $confirmedBookings = 0;

        // Chart variables
        $chartCategory = [];
        $chartDataset = [];
        $totalBookings = 0;

        $chartJson = [
            'chart' =>
            [
                'caption' => 'Confirmed ÷ All new bookings',
                'subcaption' => '',
                'lowerlimit' => '0',
                'upperlimit' => '100',
                'showvalue' => '1',
                'numbersuffix' => '%',
                'theme' => 'fusion',
                'showtooltip' => '1','baseFont' => 'Helvetica',
                'outCnvBaseFont' => 'Helvetica',
            ],
            'colorrange' =>
            [
                'color' =>
                [
                    [
                        'minvalue' => '0',
                        'maxvalue' => '10',
                        'code' => '#F2726F',
                    ],
                    [
                        'minvalue' => '10',
                        'maxvalue' => '25',
                        'code' => '#FFC533',
                    ],
                    [
                        'minvalue' => '25',
                        'maxvalue' => '50',
                        'code' => '#62B58F',
                    ],
                    [
                        'minvalue' => '50',
                        'maxvalue' => '75',
                        'code' => '#0080FF',
                    ],
                    [
                        'minvalue' => '75',
                        'maxvalue' => '100',
                        'code' => '#BF00E5',
                    ],
                ],
            ],
            'dials' =>
            [
                'dial' =>
                [
                    [
                        'value' => '0',
                    ],
                ],
            ],
        ];

        $bookingsQuery = $this->em->createQueryBuilder();

        $params = $this->bodyParams();
        $contactId = $params['contactId']; // contactId parameter

        // Prepare date variables
        $startDateString = $startDate->format("Y-m-d"); // Convert to string and format for MariaDB
        
        $endDateString = $endDate->format("Y-m-d"); // Convert to string and format for MariaDB

        // Get data
        $bookings = $bookingsQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationId) AS reservationCount',
                    'reservationStatusTable.rfReservationStatusDesc AS reservationStatus',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->where('reservation.rvAgentId = :contactId')
            ->andWhere('reservation.rvDateRecorded >= :startDate')
            ->andWhere('reservation.rvDateRecorded <= :endDate')
            ->groupBy('reservationStatus')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'startDate' => $startDateString,
                    'endDate' => $endDateString,
                ]
            )
            ->getQuery()
            ->getResult();

        // Process data

        foreach ($bookings as $booking) {
            $bookingCount = $booking['reservationCount'];
            $bookingStatus = $booking['reservationStatus'];

            if ($bookingStatus == "Confirmed") {
                $confirmedBookings = $confirmedBookings + $bookingCount;
            }

            $totalBookings = $totalBookings + $bookingCount;

            $bookingsData[$bookingStatus] = $bookingCount;
        }

        if (empty($confirmedBookings)) {
            $confirmedBookingsPercentage = 0;
        } else {
            $confirmedBookingsPercentage = $confirmedBookings / $totalBookings * 100;
        }

        $chartJson['chart']['subcaption'] = $startDate->format('d M Y') . " - " . $endDate->format('d M Y');
        $chartJson['dials']['dial'][0]['value'] = $confirmedBookingsPercentage;

        return $chartJson;
    }

    public function buildAgentActivityChart($dateRange) {
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];

        $months = []; // Contains months of the year in order
        $years = []; // Contains year daterange to get data for
        $dateRange = [];
        $properties = []; // List of properties
        $bookingsData = [];
        $eventsData = [];

        // Chart variables
        $chartCategory = [];
        $chartDataset = [];

        $chartJson = [ // Bookings and Events chart JSON
            'chart' => [
                'caption' => "Agent activity",
                'subcaption' => '',
                'yaxisname' => "New bookings",
                'syaxisname' => "CRM events",
                'showhovereffect' => "1",
                'drawcrossline' => "1",
                'flatscrollbars' => "1",
                'rotateLabels' => "1",
                'slantLabels' => "1",
                'theme' => "fusion",
                'sYAxisMaxValue' => 10,
                'yAxisMaxValue' => 10,
                'decimals' => 0,
                'baseFont' => 'Helvetica',
                'legendItemFont' => 'Helvetica',
                'legendItemFontSize' => '13',
                'outCnvBaseFontSize' => '12',
                'xAxisNameFontSize' => '12'
            ],
            'categories' => [
                [
                    'category' => $chartCategory,
                ],
            ],
            'dataset' => $chartDataset,
        ];

        $eventsQuery = $this->em->createQueryBuilder();
        $bookingsQuery = $this->em->createQueryBuilder();

        $params = $this->bodyParams();
        $contactId = $params['contactId']; // contactId parameter

        // Prepare date variables
        $beginningOfLastYear = new \DateTime(date('Y'));
        $beginningOfLastYear->modify("-1 year");
        $beginningOfLastYear->modify('last day of january ' . $beginningOfLastYear->format('Y'));
        array_push($years, $beginningOfLastYear->format("Y"));

        $beginningOfThisYear = new \DateTime(date('Y'));
        array_push($years, $beginningOfThisYear->format("Y"));

        $endOfNextYear = new \DateTime(date('Y'));
        $endOfNextYear->modify("+1 year");
        $endOfNextYear->modify('last day of december ' . $endOfNextYear->format('Y'));
        array_push($years, $endOfNextYear->format("Y"));

        for ($monthNumber = 1; $monthNumber <= 12; $monthNumber++) { // Generate months for chart data and labels
            $date = mktime(0, 0, 0, $monthNumber, 1);
            $month = date('M', $date); // Convert date to month name
            $months[$month] = []; // Set month name as index

        }

        foreach ($years as $year) {
            $dateRange[$year] = $months;
            foreach ($months as $monthName => $null) {
                array_push($chartCategory, ['label' => $monthName . ' ' . $year]); // Set label of chart to month and year
            }
        }

        // Get data
        $bookings = $bookingsQuery
            ->select(
                [
                    'COUNT(reservation.rvReservationId) AS reservationCount',
                    'persona.prNameLast AS property',
                    'reservation.rvDateRecorded AS date',
                    'MONTH(reservation.rvDateRecorded) AS month',
                    'YEAR(reservation.rvDateRecorded) AS year',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservationItem.prBusinessId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->where('reservation.rvAgentId = :contactId')
            ->andWhere('reservation.rvDateRecorded >= :beginningOfLastYear')
            ->andWhere('reservation.rvDateRecorded <= :endOfNextYear')
            ->groupBy('month')
            ->addGroupBy('year')
            ->addGroupBy('property')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'beginningOfLastYear' => $beginningOfLastYear,
                    'endOfNextYear' => $endOfNextYear,
                ]
            )
            ->getQuery()
            ->getResult();

        $events = $eventsQuery
            ->select(
                [
                    'COUNT(event.prEventId) AS eventCount',
                    'event.prEventDate AS date',
                    'MONTH(event.prEventDate) AS month',
                    'YEAR(event.prEventDate) AS year',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\PrEventPersona', 'eventPersona')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\PrEvent', 'event', 'WITH', 'event.prEventIx = eventPersona.prEventId')
            ->where('eventPersona.prPersonaId = :contactId')
            ->andWhere('event.prEventDate >= :beginningOfLastYear')
            ->andWhere('event.prEventDate <= :endOfNextYear')
            ->groupBy('year')
            ->addGroupBy('month')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'beginningOfLastYear' => $beginningOfLastYear,
                    'endOfNextYear' => $endOfNextYear,
                ]
            )
            ->getQuery()
            ->getResult();

        // Process data

        // ==Bookings==
        // Group bookings by year and month
        foreach ($bookings as $booking) {
            $bookingYear = $booking['date']->format('Y');
            $bookingMonth = $booking['date']->format('M');
            $bookingProperty = $booking['property'];
            $bookingCount = $booking['reservationCount'];

            if (!array_key_exists($bookingProperty, $bookingsData)) { // Add property name as index
                $bookingsData[$bookingProperty] = $dateRange;
            }

            $bookingsData[$bookingProperty][$bookingYear][$bookingMonth] = $bookingCount;
        }

        foreach ($bookingsData as $propertyName => $propertyDateRange) {
            $data = [];
            foreach ($propertyDateRange as $year => $month) {
                foreach ($month as $bookingCount) {
                    if (empty($bookingCount)) {
                        $bookingCount = 0;
                    } else {
                        $bookingCount = intval($bookingCount);
                    }
                    array_push($data, ['value' => $bookingCount]);
                }
            }
            array_push(
                $chartDataset,
                [
                    'seriesName' => $propertyName,
                    'renderAs' => 'line',
                    'data' => $data,
                    'plottooltext' => '<b>$dataValue</b> bookings at property $seriesName in $label',
                ]
            );
        }

        // ==Events==
        // Group events by year and month
        $eventsData = $dateRange;

        foreach ($events as $event) {
            $eventYear = $event['date']->format('Y');
            $eventMonth = $event['date']->format('M');
            $eventCount = $event['eventCount'];

            $eventsData[$eventYear][$eventMonth] = $eventCount;
        }

        $data = [];
        $eventDataEmpty = true;
        foreach ($eventsData as $year => $month) {
            foreach ($month as $eventCount) {
                if (empty($eventCount)) {
                    $eventCount = 0;
                } else {
                    $eventDataEmpty = false;
                    $eventCount = intval($eventCount);
                }
                array_push($data, ['value' => $eventCount]);
            }
        }
        array_push(
            $chartDataset,
            [
                'seriesName' => 'CRM events',
                'renderAs' => 'bar',
                'data' => $data,
                'parentyaxis' => 'S',
                'plottooltext' => '<b>$dataValue<b> CRM events',
            ]
        );

        $chartJson['categories'][0]['category'] = $chartCategory;
        
        if (empty($bookingsData) && $eventDataEmpty === true) {
            $chartJson['dataset'] = false;
        } else {
            $chartJson['dataset'] = $chartDataset;
        }

        $chartJson['chart']['subcaption'] = $beginningOfLastYear->format("d M Y") . ' - ' . $endOfNextYear->format("d M Y");

        return $chartJson;
    }

    public function buildBedNightActivityChart($dateRange)
    {
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];

        $months = []; // Contains months of the year in order
        $years = []; // Contains year daterange to get data for
        $dateRange = [];
        $bookingsData = [];
        $reservationStatusFilter = [ // List of reservation statuses to return bed nights for
            'Confirmed',
            'Provisional'
        ];

        // Chart variables
        $chartCategory = [];
        $chartDataset = [];

        $chartJson = [ // Bookings and Events chart JSON
            'chart' => [
                'caption' => "Bed night activity",
                'subcaption' => '',
                'yaxisname' => "Bed nights",
                'showhovereffect' => "1",
                'drawcrossline' => "1",
                'flatscrollbars' => "1",
                'numDivLines' => "4",
                'rotateLabels' => "1",
                'slantLabels' => "1",
                'theme' => "fusion",
                'baseFont' => 'Helvetica',
                'legendItemFont' => 'Helvetica',
                'legendItemFontSize' => '13',
                'outCnvBaseFontSize' => '12',
                'xAxisNameFontSize' => '12'
            ],
            'categories' => [
                [
                    'category' => $chartCategory,
                ],
            ],
            'dataset' => $chartDataset,
        ];

        $bookingsQuery = $this->em->createQueryBuilder();

        $params = $this->bodyParams();
        $contactId = $params['contactId']; // contactId parameter

        // Prepare date variables
        array_push($years, $startDate->format("Y"));
        if (!in_array($endDate->format("Y"), $years)) { // Don't add same year twice to list of years
            array_push($years, $endDate->format("Y"));
        }
            
        for ($monthNumber = 1; $monthNumber <= 12; $monthNumber++) { // Generate months for chart data and labels
            $date = mktime(0, 0, 0, $monthNumber, 1);
            $month = date('M', $date); // Convert date to month name
            $months[$month] = []; // Set month name as index

        }

        foreach ($years as $year) {
            $dateRange[$year] = $months;
            foreach ($months as $monthName => $null) {
                array_push($chartCategory, ['label' => $monthName . ' ' . $year]); // Set label of chart to month and year
            }
        }

        // Get data
        $bookings = $bookingsQuery
            ->select(
                [
                    'persona.prNameLast AS property',
                    'reservationItem.rvItemAdultCount + reservationItem.rvItemChildCount AS paxCount',
                    'reservationItem.rvItemAccommCount AS roomCount',
                    'reservationItem.rvItemDateArrive AS dateArrive',
                    'reservationItem.rvItemDateDepart AS dateDepart',
                    'reservationStatusTable.rfReservationStatusDesc AS bookingStatus',
                    'reservation.rvProvisionExpiryDate AS provisionalExpiryDate'
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservationItem.prBusinessId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->where('reservation.rvAgentId = :contactId')
            ->andWhere('reservationItem.rvItemDateArrive >= :startDate')
            ->andWhere('reservationItem.rvItemDateArrive <= :endDate')
            ->andWhere('reservationItem.rvItemDateDepart >= :startDate')
            ->andWhere('reservationItem.rvItemDateDepart <= :endDate')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'startDate' => $startDate,
                    'endDate' => $endDate,
                    'reservationStatusFilter' => $reservationStatusFilter,
                ]
            )
            ->getQuery()
            ->getResult();

        // Process data

        // ==Bookings==
        // Group bookings by year and month
        foreach ($bookings as $booking) {
            $bookingProperty = $booking['property'];
            $pax = $booking['paxCount'];
            $roomCount = $booking['roomCount'];
            $bookingStatus = $booking['bookingStatus'];
            $dateArrive = $booking['dateArrive'];
            $dateDepart = $booking['dateDepart'];
            $provisionalExpiryDate = $booking['provisionalExpiryDate'];

            // Skip expired provisional bookings
            if ($bookingStatus == "Provisional" && $provisionalExpiryDate <= new \DateTime()) {
                continue;
            }

            // Iterate through each night in the date range of the reservation item
            for($bookingNight = $dateArrive; $bookingNight < $dateDepart; $bookingNight->modify('+1 day')) {
                $bookingYear = $bookingNight->format('Y');
                $bookingMonth = $bookingNight->format('M');

                $bedNightCount = $pax *  $roomCount;

                if (!array_key_exists($bookingProperty, $bookingsData)) { // Add property name as index
                    $bookingsData[$bookingProperty] = $dateRange;
                }

                if (empty($bookingsData[$bookingProperty][$bookingYear][$bookingMonth])) {
                    $bookingsData[$bookingProperty][$bookingYear][$bookingMonth] = 0;
                }

                $bookingsData[$bookingProperty][$bookingYear][$bookingMonth] = $bookingsData[$bookingProperty][$bookingYear][$bookingMonth] + $bedNightCount;
            }
        }

        foreach ($bookingsData as $propertyName => $propertyDateRange) {
            $data = [];
            foreach ($propertyDateRange as $year => $month) {
                foreach ($month as $bookingCount) {
                    if (empty($bookingCount)) {
                        $bookingCount = 0;
                    } else {
                        $bookingCount = intval($bookingCount);
                    }
                    array_push($data, ['value' => $bookingCount]);
                }
            }
            array_push(
                $chartDataset,
                [
                    'seriesName' => $propertyName,
                    'renderAs' => 'line',
                    'data' => $data,
                    'plottooltext' => '<b>$dataValue</b> bed nights at property $seriesName in $label',
                ]
            );
        }

        $chartJson['categories'][0]['category'] = $chartCategory;

        if (empty($chartDataset)) {
            $chartJson['dataset'] = false;
        } else {
            $chartJson['dataset'] = $chartDataset;
        }
        
        $chartJson['chart']['subcaption'] = $startDate->format("d M Y") . ' - ' . $endDate->format("d M Y");

        return $chartJson;
    }

    public function salesPerEmployeePerWeekdayChart($dateRange)
    {
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];

        $weeks = []; // Weeks in the month
        $totalBookings = 0;
        $maxBookings = 0; // Highest bookings recorded in a day. Used for colour range
        $employeeSalesData = [];
        $employeeTotalSales = [];

        // Chart variables
        $chartRows = [];
        $chartColumns = [];
        $chartDataset = [];

        $chartJson = [
            'chart' => [
                'theme' => 'fusion',
                'caption' => 'Top 5 staff',
                'subcaption' => 'New bookings for this calendar month',
                'xaxisname' => 'Bookings',
                'showvalues' => '1',
                'valuefontcolor' => '#ffffff',
                'plottooltext' => '<b>$value</b> bookings created by $rowlabel on $columnlabel',
                'showtooltip' => '1',
                'chartleftmargin' => '0',
                'alignLegendWithCanvas' => '0',
                'alignCaptionWithCanvas' => '0',
                'baseFont' => 'Helvetica',
                'outCnvBaseFont' => 'Helvetica',
            ],
            'colorrange' => [
                'code' => '#FFFFFF', // Start color range
                'gradient' => '1',
                'minvalue' => '0',
                'color' => [
                    [
                        "code" => '#c9252f',
                        "maxValue" => '', // Max value is set to the highest amount of bookings
                    ],
                ],
            ],
            'dataset' => $chartDataset,
            'columns' => [
                'column' => $chartColumns,
            ],
            'rows' => [
                'row' => $chartRows,
            ],
        ];

        $employeeSalesQuery = $this->em->createQueryBuilder();

        $params = $this->bodyParams();
        $contactId = $params['contactId']; // contactId parameter

        // Prepare date variables
        $startDateString = $startDate->format("Y-m-d"); // Convert to string and format for MariaDB

        $endDateString = $endDate->format("Y-m-d");

        // Set columns as weeks in the month
        foreach ($this->getWeeksInMonth($startDate) as $week) {
            array_push($chartColumns, [
                'id' => (string)$week['number'],
                'label' => "Week $week[number]",
            ]);

            $weeks[$week['number']] = "0";
        }

        // Get data
        $employeeSales = $employeeSalesQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationId) AS reservationCount',
                    'persona.prNameFirst AS contactNameFirst',
                    'persona.prNameLast AS contactNameLast',
                    'reservation.rvDateRecorded AS date',
                    'reservation.rvBillingPersonaId AS contactId',
                    'MONTH(reservation.rvDateRecorded) AS month',
                    'YEAR(reservation.rvDateRecorded) AS year',
                    'DAY(reservation.rvDateRecorded) AS day',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->Join('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservation.rvBillingPersonaId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->where('reservation.rvAgentId = :contactId')
            ->andWhere('reservation.rvDateRecorded >= :startDate')
            ->andWhere('reservation.rvDateRecorded <= :endDate')
            ->groupBy('year')
            ->addGroupBy('contactNameFirst')
            ->addGroupBy('contactNameLast')
            ->addGroupBy('month')
            ->addGroupBy('day')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'startDate' => $startDateString,
                    'endDate' => $endDateString,
                ]
            )
            ->getQuery()
            ->getResult();

        // Process data

        // Group sales by day of the week
        foreach ($employeeSales as $employeeSale) {
            $employeeSaleWeek = $this->getWeekNumberInMonth($employeeSale['date']); // The week number in the month the sale was made on
            $employeeSaleContactName = $employeeSale['contactNameFirst'];
            $employeeSaleContactId = $employeeSale['contactId'];
            $employeeSaleCount = $employeeSale['reservationCount'];

            if (empty($employeeSaleContactName)) {
                $employeeSaleContactName = $employeeSale['contactNameLast'];
            }

            if (!array_key_exists($employeeSaleContactName, $employeeSalesData)) { // Add property name as index
                $employeeSalesData[$employeeSaleContactName] = $weeks;
                $employeeTotalSales[$employeeSaleContactName] = 0;
            }

            $employeeTotalSales[$employeeSaleContactName] = $employeeTotalSales[$employeeSaleContactName] + $employeeSaleCount;

            if (empty($employeeSalesData[$employeeSaleContactName][$employeeSaleWeek])) { // Add property name as index
                $employeeSalesData[$employeeSaleContactName][$employeeSaleWeek] = $employeeSaleCount;
            } else {
                $employeeSalesData[$employeeSaleContactName][$employeeSaleWeek] = $employeeSalesData[$employeeSaleContactName][$employeeSaleWeek] + $employeeSaleCount; 
            }

            if ($employeeSalesData[$employeeSaleContactName][$employeeSaleWeek] > $maxBookings) {
                $maxBookings = $employeeSalesData[$employeeSaleContactName][$employeeSaleWeek]; // Record the highest booking count for the week
            }
        }

        arsort($employeeTotalSales); // Sort names by most sales for period
        $employeeTotalSales = array_slice($employeeTotalSales, 0, 5); // Keep first 5 entries, remove the rest
        $employeeSalesData = array_intersect_key($employeeSalesData, $employeeTotalSales); // Remove data for employees not in the list

        if (empty($employeeSalesData)) {
            $chartJson['dataset'] = false;
        } else {
            $data = [];
            $rowCount = 0; // Used to generate column IDs
            foreach ($employeeSalesData as $employeeSaleContactName => $employeeSaleDateRange) {
                foreach ($employeeSaleDateRange as $employeeSaleWeek => $employeeSaleData) {
                    if (empty($employeeSaleData)) {
                        $employeeSaleCount = 0;
                    } else {
                        $employeeSaleCount = intval($employeeSaleData);
                    }
                    array_push($data, ['value' => $employeeSaleCount, 'columnid' => strval($employeeSaleWeek), 'rowid' => strval($rowCount)]);
                    $totalBookings = $totalBookings + $employeeSaleCount;
                }
                array_push(
                    $chartRows,
                    [
                        'id' => strval($rowCount),
                        'label' => $employeeSaleContactName,
                    ]
                );
                $rowCount++;
            }
            array_push(
                $chartDataset,
                [
                    'data' => $data,
                ]
            );
            $chartJson['dataset'] = $chartDataset;
        }

        $chartJson['colorrange']['color'][0]['maxValue'] = $maxBookings;
        $chartJson['rows']['row'] = $chartRows;
        $chartJson['columns']['column'] = $chartColumns;

        return $chartJson;
    }

    public function buildCards($dateRange, $dateOption) {
        $startDate = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];

        $bookingsData = [];
        $bookingsCompareData = [];

        $totalBookings = 0;
        $totalBookingsCompare = 0;
        $totalBookingsCompareDifference = 0;
        $totalBookingsComparePercentage = 0;

        $confirmedBookings = 0;
        $confirmedBookingsCompare = 0;
        $confirmedBookingsCompareDifference = 0;
        $confirmedBookingsComparePercentage = 0;

        $cancelledBookings = 0;
        $cancelledBookingsCompare = 0;
        $cancelledBookingsCompareDifference = 0;
        $cancelledBookingsComparePercentage = 0;

        $bookingsQuery = $this->em->createQueryBuilder();
        $bookingsCompareQuery = $this->em->createQueryBuilder();

        if ($dateOption == 'thisWeek') {
            $comparePeriod = "week";
            $compareStartDate = clone $startDate;
            $compareEndDate = clone $endDate;
            $compareStartDateString =  $compareStartDate->modify("-1 week")->format("Y-m-d");
            $compareEndDateString = $compareEndDate->modify("-1 week")->format("Y-m-d");
        } else if ($dateOption == 'thisMonth') {
            $comparePeriod = "month";
            $compareStartDate = clone $startDate;
            $compareEndDate = clone $endDate;
            $compareStartDateString =  $compareStartDate->modify("first day of last month")->format("Y-m-d");
            $compareEndDateString = $compareEndDate->modify("last day of last month")->format("Y-m-d");
        } else if ($dateOption == 'thisQuarter') {
            $comparePeriod = "quarter";

            // Get previous quarter date range
            $threeMonthsAgo = clone $startDate;
            $threeMonthsAgo->modify("-3 months");
            $previousQuarter = $this->getQuarterDates($threeMonthsAgo);

            if ($previousQuarter['startDate'] > $startDate) {
                $previousQuarter['startDate']->modify("first day of {$previousQuarter['startDate']->format('F')} last year");
                $previousQuarter['endDate']->modify("last day of {$previousQuarter['endDate']->format('F')} last year");
            }

            $compareStartDateString = $previousQuarter['startDate']->format("Y-m-d");
            $compareEndDateString = $previousQuarter['endDate']->format("Y-m-d");
        } else if ($dateOption == 'thisYear') {
            $comparePeriod = "year";

            $compareStartDate = clone $startDate;
            $compareEndDate = clone $endDate;
            $compareStartDateString =  $compareStartDate->modify('first day of january last year')->format("Y-m-d");
            $compareEndDateString = $compareEndDate->modify("last day of december last year")->format("Y-m-d");
        }

        $params = $this->bodyParams();
        $contactId = $params['contactId']; // contactId parameter

        // Prepare date variables
        $startDateString = $startDate->format("Y-m-d"); // Convert to string and format for MariaDB
        
        $endDateString = $endDate->format("Y-m-d"); // Convert to string and format for MariaDB

        $reservationStatusFilter = [ // Whitelist of reservation status filters
            'Quotation',
            'Wait listed',
            // 'Allocation',
            'Provisional',
            // 'In progress',
            'Confirmed',
            // 'Invoiced',
            // 'Blocked',
            'Cancelled',
            // 'Deleted',
            // 'Deleted allocation',
        ];

        $bookings = $bookingsQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationId) AS reservationCount',
                    'reservationStatusTable.rfReservationStatusDesc AS reservationStatus',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->where('reservation.rvAgentId = :contactId')
            ->andWhere('reservation.rvDateRecorded >= :startDate')
            ->andWhere('reservation.rvDateRecorded <= :endDate')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->groupBy('reservationStatus')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'startDate' => $startDateString,
                    'endDate' => $endDateString,
                    'reservationStatusFilter' => $reservationStatusFilter
                ]
            )
            ->getQuery()
            ->getResult();

        $bookingsCompare = $bookingsCompareQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationId) AS reservationCount',
                    'reservationStatusTable.rfReservationStatusDesc AS reservationStatus',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->where('reservation.rvAgentId = :contactId')
            ->andWhere('reservation.rvDateRecorded >= :compareStartDate')
            ->andWhere('reservation.rvDateRecorded <= :compareEndDate')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->groupBy('reservationStatus')
            ->setParameters(
                [
                    'contactId' => $contactId,
                    'compareStartDate' => $compareStartDateString,
                    'compareEndDate' => $compareEndDateString,
                    'reservationStatusFilter' => $reservationStatusFilter
                ]
            )
            ->getQuery()
            ->getResult();

        // Process data

        foreach ($bookings as $booking) {
            $bookingCount = $booking['reservationCount'];
            $bookingStatus = $booking['reservationStatus'];

            $bookingsData[$bookingStatus] = $bookingCount;
        }

        foreach ($bookingsData as $bookingStatus => $bookingCount) {
            $totalBookings = $totalBookings + $bookingCount;

            if ($bookingStatus == "Confirmed") {
                $confirmedBookings = $confirmedBookings + $bookingCount;
            } else if ($bookingStatus == "Cancelled") {
                $cancelledBookings = $cancelledBookings + $bookingCount;
            }
        }

        // Process comparative data

        foreach ($bookingsCompare as $booking) {
            $bookingCount = $booking['reservationCount'];
            $bookingStatus = $booking['reservationStatus'];

            $bookingsCompareData[$bookingStatus] = $bookingCount;
        }

        foreach ($bookingsCompareData as $bookingStatus => $bookingCount) {
            $totalBookingsCompare = $totalBookingsCompare + $bookingCount;

            if ($bookingStatus == "Confirmed") {
                $confirmedBookingsCompare = $confirmedBookingsCompare + $bookingCount;
            } else if ($bookingStatus == "Cancelled") {
                $cancelledBookingsCompare = $confirmedBookingsCompare + $bookingCount;
            }
        }

        // Compare data

        // Bookings created comparison
        $totalBookingsCompareDifference = $totalBookings - $totalBookingsCompare;
        if (empty($totalBookings) || empty($totalBookingsCompare)) {
            $totalBookingsComparePercentage = false; // Cannot calculate percentage
        } else {
        $totalBookingsCompareDifference = $totalBookings - $totalBookingsCompare;
            $totalBookingsComparePercentage = round($totalBookingsCompareDifference / $totalBookingsCompare * 100, 0);
        }

        // Confirmed bookings comparison
        $confirmedBookingsCompareDifference = $confirmedBookings - $confirmedBookingsCompare;
        if (empty($confirmedBookings) || empty($confirmedBookingsCompare)) {
            $confirmedBookingsComparePercentage = false; // Cannot calculate percentage
        } else {
            $confirmedBookingsComparePercentage = round($confirmedBookingsCompareDifference / $confirmedBookingsCompare  * 100, 0);
        }

        // Cancelled bookings comparison
        $cancelledBookingsCompareDifference = $cancelledBookings - $cancelledBookingsCompare;
        if (empty($cancelledBookings) || empty($cancelledBookingsCompare)) {
            $cancelledBookingsComparePercentage = false; // Cannot calculate percentage
        } else {
            $cancelledBookingsComparePercentage = round($cancelledBookingsCompareDifference / $cancelledBookingsCompare * 100, 0);
        }

        return [
            'bookingsCreatedCard' => [
                'value' => $totalBookings,
                'valueTitle' => 'This ' . $comparePeriod . ':',
                'compareValue' => $totalBookingsCompare,
                'compareValueTitle' => 'Last ' . $comparePeriod . ':',
                'compareValuePercentage' => $totalBookingsComparePercentage,
            ],
            'confirmedBookingsCard' => [
                'value' => $confirmedBookings,
                'valueTitle' => 'This ' . $comparePeriod . ':',
                'compareValue' => $confirmedBookingsCompare,
                'compareValueTitle' => 'Last ' . $comparePeriod . ':',
                'compareValuePercentage' => $confirmedBookingsComparePercentage,
            ],
            'cancelledBookingsCard' => [
                'value' => $cancelledBookings,
                'valueTitle' => 'This ' . $comparePeriod . ':',
                'compareValue' => $cancelledBookingsCompare,
                'compareValueTitle' => 'Last ' . $comparePeriod . ':',
                'compareValuePercentage' => $cancelledBookingsComparePercentage,
            ],
        ];
    }

    /**
     * Returns the week number in the month for the specified date
     *
     * @param \DateTime $date
     * @return int Week number in month. 
     */
    function getWeekNumberInMonth(\DateTime $date) {
        $year = $date->format("Y");
        
        $month = $date->format("M");
        
        $startMonth = clone $date;
        $startMonth->modify("first day of $month $year");
        
        $endMonth = clone $date;
        $endMonth->modify("last day of $month $year");
        
        $tempDate = new \DateTime();
        
        $number = 1;
        for ($weekOfYear = $startMonth->format("W"); $weekOfYear <= $endMonth->format("W"); $weekOfYear++) {
            $tempDate->setISODate($year, $weekOfYear);
            
            $weekStartDate = clone $tempDate;
            $weekEndDate = clone $tempDate->modify('+6 days');
        
            if ($weekStartDate < $startMonth) {
                $weekStartDate = $startMonth;
            }
        
            if ($weekEndDate > $endMonth) {
                $weekEndDate = $endMonth;
            }
            
            if ($date >= $weekStartDate && $date <= $weekEndDate) {
                return $number;
            }
    
            $number++;
        }
    }

    /**
     * Returns the week number in the month for the given date
     *
     * @param \DateTime $date
     * @return array Week beginning and end dates with the index as the week number
     */
    function getWeeksInMonth(\DateTime $date) {
        $year = $date->format("Y");
        $month = $date->format("M");
        
        $startMonth = clone $date;
        $startMonth->modify("first day of $month $year");
        
        $endMonth = clone $date;
        $endMonth->modify("last day of $month $year");
        
        $weeks = [];
        $number = 1;
        $tempDate = new \DateTime();
        for ($weekOfYear = $startMonth->format("W"); $weekOfYear <= $endMonth->format("W"); $weekOfYear++) {
            $tempDate->setISODate($year, $weekOfYear);
            
            $weekStartDate = clone $tempDate;
            $weekEndDate = clone $tempDate->modify('+6 days');
        
            if ($weekStartDate < $startMonth) {
                $weekStartDate = $startMonth;
            }
        
            if ($weekEndDate > $endMonth) {
                $weekEndDate = $endMonth;
            }

            $weeks[$number] = [
                'startDate' => $weekStartDate,
                'endDate' => $weekEndDate,
                'number' => $number
            ];

            $number++;
        }

        return $weeks;
    }

    public function error($message, $code = 403)
    {
        return new ApiProblemResponse(
            new ApiProblem($code, $message)
        );
    }
}
