<?php

namespace Resrequest\API\V1\Rpc\MinLosGetRule;

use Zend\Mvc\Controller\AbstractActionController;
use ZF\ApiProblem\ApiProblemResponse;
use ZF\ApiProblem\ApiProblem;
use ZF\ContentNegotiation\JsonModel;

class MinLosGetRuleController extends AbstractActionController
{
	private $em;
	private $db;
	private $authenticate;


	protected $days = [ // The data in this array must be protected at all costs.
		'Sunday',
		'Monday',
		'Tuesday',
		'Wednesday',
		'Thursday',
		'Friday',
		'Saturday'
	];

	public function __construct($em, $authenticate, $authorise)
	{
		$this->em = $em;
		$this->db = $em->getConnection();
		$this->authenticate = $authenticate;
		$this->authorise = $authorise;
	}

	public function minLosGetRuleAction()
	{
		$username = false;
        $identity = $this->getIdentity();
        if($identity instanceof \ZF\MvcAuth\Identity\AuthenticatedIdentity) {
            $username = $identity->getAuthenticationIdentity()['user_id'];
        }

        if($username === false) {
            return new ApiProblemResponse(
                new ApiProblem(403, "Unable to find user")
            );
        }

        $this->authenticate->setupContext($this->authenticate->getUserByName($username));

        $user = $this->db->fetchAssoc("
            SELECT
                pr_user.pr_user_id,
                sc_user.sc_group_id
            FROM
                sc_user
                INNER JOIN pr_user ON pr_user.pr_user_id = sc_user.pr_user_id
            WHERE
                pr_user.pr_user_name = ?
        ",[$username]);
        if($user === false) {
            return new ApiProblemResponse(
                new ApiProblem(403, "Unable to find user")
            );
        }
		
		$functionAccess = $this->authorise->getAccess([
			'functions'=>[68]
		], $user['sc_group_id']);

		if(!isset($functionAccess['functions'][68]) || $functionAccess['functions'][68] < 4) {
			return new ApiProblemResponse(
				new ApiProblem(403, "Access denied")
			);
		}

		$params = $this->bodyParams();

		if (empty($params['ids'])) {
			$rules = $this->db->fetchAll("
				SELECT
					ml_rule.ml_rule_ix AS id,
					ml_rule.ml_min_los AS minLos,
					ml_rule.ml_lapse_days AS lapseLeadNights,
					ml_rule.ml_rule_desc AS description,
					ml_rule.ml_allocations_yn AS applyToAllocations,
					ml_rule.ml_days_of_week AS minLosDays
				FROM
					ml_rule
			");
		} else {
			$rules = $this->db->fetchAll("
				SELECT
					ml_rule.ml_rule_ix AS id,
					ml_rule.ml_min_los AS minLos,
					ml_rule.ml_lapse_days AS lapseLeadNights,
					ml_rule.ml_rule_desc AS description,
					ml_rule.ml_allocations_yn AS applyToAllocations,
					ml_rule.ml_days_of_week AS minLosDays
				FROM
					ml_rule
				WHERE
					ml_rule.ml_rule_ix IN(?)
			", [
				join(',', $params['ids'])
			]);
		}

		if (empty($rules)) {
			$rules = [];
		}

		foreach ($rules as &$rule) {
			$ruleId = $rule['id'];

			$friendlyMinLosDays = [];
			$ruleDaysOfWeek = explode(',', $rule['minLosDays']);
			foreach ($ruleDaysOfWeek as &$dayOfWeek) {
				$day = $this->dayOfWeekToDay($dayOfWeek);
				$friendlyMinLosDays[] = $day;
			}
			unset($dayOfWeek);

			if (empty($friendlyMinLosDays)) {
				$friendlyMinLosDays = 'Everyday';
			}

			$minLosDays = [];
			foreach ($this->days as $day) {
				$minLosDays[$day] = in_array($day, $friendlyMinLosDays);
			}
			$rule['minLosDays'] = $minLosDays;
			$rule['friendlyMinLosDays'] = $friendlyMinLosDays;



			$rateTypes = $this->db->fetchAll("
				SELECT
					ml_rate_type.rt_rate_type_id AS id
				FROM
					ml_rate_type
				WHERE
					ml_rate_type.ml_rule_id = ?
			", [
				$ruleId
			]);

			if ($rateTypes === false) {
				$rateTypes = [];
			}
			$rule['rateTypes'] = array_column($rateTypes, 'id');

			$ratePeriods = $this->db->fetchAll("
				SELECT
					ml_period.rt_period_id AS id
				FROM
					ml_period
				WHERE
					ml_period.ml_rule_id = ?
			", [
				$ruleId
			]);

			if ($ratePeriods === false) {
				$ratePeriods = [];
			}
			$rule['ratePeriods'] = array_column($ratePeriods, 'id');

			$accommodationTypes = $this->db->fetchAll("
				SELECT
					ml_accomm.ac_accomm_type_id AS id
				FROM
					ml_accomm
				WHERE
					ml_accomm.ml_rule_id = ?
			", [
				$ruleId
			]);

			if ($accommodationTypes === false) {
				$accommodationTypes = [];
			}
			$rule['accommodationTypes'] = array_column($accommodationTypes, 'id');

			$agents = $this->db->fetchAll("
				SELECT
					ml_agent.pr_agent_id AS id
				FROM
					ml_agent
				WHERE
				ml_agent.ml_rule_id = ?
			", [
				$ruleId
			]);

			if ($agents === false) {
				$agents = [];
			}
			$rule['agents'] = array_column($agents, 'id');
		}


		$ratesQuery = $this->em->createQueryBuilder();
		$rates = $ratesQuery
			->select(
				[
					'rate.rtRateTypeIx as id',
					'rate.rtRateTypeDesc as description',
					'rate.rtRateTypeInactiveYn as inactive',
				]
			)
			->from('Resrequest\DB\Enterprise\Entity\RtRateType', 'rate')
			->addOrderBy('description', 'ASC')
			->getQuery()
			->getResult();


		$ratePeriodsQuery = $this->em->createQueryBuilder();
		$ratePeriods = $ratePeriodsQuery
			->select(
				[
					'period.rtPeriodIx as id',
					'period.rtPeriodDesc as description',
					'period.rtPeriodInactiveYn as inactive',
				]
			)
			->from('Resrequest\DB\Enterprise\Entity\RtPeriod', 'period')
			->where('period.rtPeriodDateTo > :today')
			->addOrderBy('description', 'ASC')
			->setparameter('today', new \DateTime())
			->getQuery()
			->getResult();


		$agentsQuery = $this->em->createQueryBuilder();
		$agents = $agentsQuery
			->select(
			  [
				'persona.prPersonaIx AS id',
				"TRIM(CONCAT(persona.prNameFirst, ' ', persona.prNameLast)) AS name"
			  ]
			)
			->distinct('guest.prPersonaIx')
			->from('Resrequest\DB\Enterprise\Entity\PrAgent', 'agent')
			->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = agent.prAgentId')
			->addOrderBy('name', 'ASC')
			->getQuery()
			->getResult();

		array_unshift($agents, ['id' => '0', 'name' => 'Direct booking']);
		

		return new JsonModel([
			'rules' => $rules,
			'rateTypes' => $rates,
			'ratePeriods' => $ratePeriods,
			'agents' => $agents
		]);
	}


	private function dayOfWeekToDay($dayOfWeek)
	{
		switch ($dayOfWeek) {
			case 0:
				return 'Sunday';
				break;

			case 1:
				return 'Monday';
				break;

			case 2:
				return 'Tuesday';
				break;

			case 3:
				return 'Wednesday';
				break;

			case 4:
				return 'Thursday';
				break;

			case 5:
				return 'Friday';
				break;

			case 6:
				return 'Saturday';
				break;

			default:
				return 'invalid';
				break;
		}
	}
}
