<?php

namespace Resrequest\API\V1\Rpc\ResVegaGetReports;

use Resrequest\Setup\Service\Enterprise;
use Zend\Mvc\Controller\AbstractActionController;
use ZF\ContentNegotiation\JsonModel;

class ResVegaGetReportsController extends AbstractActionController
{
    private $em;
    private $chartContainerService;
    private $auth;
    /**
     * Enterprise service
     *
     * @var Enterprise
     */
    private $enterprise;

    public function __construct($em, $chartContainerService, $auth, $enterprise)
    {
        $this->em = $em;
        $this->chartContainerService = $chartContainerService;
        $this->auth = $auth;
        $this->enterprise = $enterprise;
    }

    public function resVegaGetReportsAction()
    {
        $username = false;
        $identity = $this->getIdentity();
        if ($identity instanceof \ZF\MvcAuth\Identity\AuthenticatedIdentity) {
            $username = $identity->getAuthenticationIdentity()['user_id'];
            $this->auth->setupContext($this->auth->getUserByName($username));
        }

        $allowed = false;
        $access = $this->getResVegaAccess();
        if (!empty($access)) {
            $allowed = true;
        }
        if (empty($GLOBALS['pr_business_link'])) {
            $allowed = false;
        }


        $reports = [];
        $embedToken = '';
        if ($allowed) {
            $token = $this->getAccessToken();
            if (!empty($token)) {
                $reports = $this->getReports($token);
            }
            if (!empty($reports)) {
                $embedToken = $this->getEmbedToken($token, $reports);
            }
        }

        return new JsonModel(
            [
                'reports' => $reports,
                'embedToken' => $embedToken,
            ]
        );
    }

    private function getResVegaAccess()
    {
        $accessQuery = $this->em->createQueryBuilder();
        $accessIndicator = $accessQuery
            ->select('system.rfSysTourismTodayAccessInd')
            ->from('Resrequest\DB\Enterprise\Entity\RfSystem', 'system')
            ->getQuery()
            ->getSingleScalarResult();

        return intval($accessIndicator);
    }

    /**
     * Get the Azure AD access token
     *
     * @return string The Azure AD access token
     */
    private function getAccessToken()
    {
        $key = $this->enterprise->getTourismTodayKey();

        if (empty($key)) {
            return false;
        }

        // Authenticate with Azure AD
        $tenantId = '00f53dd7-728e-431d-a2ab-5a5b8c9979c5'; // Azure AD tenant ID
        $appId = 'ebdc49fb-0714-4202-9ea4-751a2a17bb84'; // Azure AD app ID
        $loginUrl = "https://login.microsoftonline.com/{$tenantId}/oauth2/v2.0/token";
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $loginUrl);
        curl_setopt($curl, CURLOPT_POST, true);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_POSTFIELDS, [
            'client_id' => $appId,
            'grant_type' => 'client_credentials',
            'client_secret' => $key,
            'scope' => 'https://analysis.windows.net/powerbi/api/.default'
        ]);

        $result = curl_exec($curl);
        curl_close($curl);

        if (!$result) {
            return false;
        }

        $result = json_decode($result, true);
        if (isset($result['access_token'])) {
            $accessToken = $result['access_token'];

            return $accessToken;
        } 
        else {
            return false;
        }
    }

    /**
     * Get report data
     *
     * @param string $token
     * @return array report data
     */
    public function getReports(string $token)
    {
        $groupId = 'd67af4a9-bb9d-4046-9bbf-2741dde166c5'; // Azure AD security group ID
        $reportsUrl = "https://api.powerbi.com/v1.0/myorg/groups/{$groupId}/reports";

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $reportsUrl);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array(
            "Authorization: Bearer {$token}",
        ));

        $result = curl_exec($curl);
        curl_close($curl);

        if (empty($result)) {
            return false;
        }

        $reportsData = json_decode($result, true)['value'];
        $reports = [];
        $resvegaContainers = \Resrequest\Application\Chart\ChartContainer::RESVEGA_CONTAINERS;

        foreach ($reportsData as $report) {
            if (in_array($report['id'], array_column($resvegaContainers, 'reportId'))) {
                $index = array_search($report['id'], array_column($resvegaContainers, 'reportId'));
                $containerId = array_keys($resvegaContainers)[$index]; // ID of the chart container

                $allowed = $this->chartContainerService->isChartContainerAllowed($containerId);

                if ($allowed) {
                    $reports[] = $report;
                }
            }
        }

        return $reports;
    }

    /**
     * Get the embed token for the reports
     *
     * @param string $token
     * @param array $reports
     * @return string embed token
     */
    public function getEmbedToken(string $token, array $reports)
    {
        $datasets = [];
        $reportRequest = [];
        foreach ($reports as $report) {
            $reportRequest[] = [
                'id' => $report['id'],
                'allowEdit' => false
            ];

            if (!in_array($report['datasetId'], array_column($datasets, 'id'))) {
                $datasets[] = [
                    'id' => $report['datasetId']
                ];
            }
        }

        $request = json_encode([
            'datasets' => $datasets,
            'reports' => $reportRequest
        ]);

        $generateTokenUrl = "https://api.powerbi.com/v1.0/myorg/GenerateToken";

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $generateTokenUrl);
        curl_setopt($curl, CURLOPT_POST, true);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $request);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array(
            'Content-Type: application/json',
            "Authorization: Bearer {$token}",
        ));

        $result = curl_exec($curl);
        curl_close($curl);

        if (empty($result)) {
            return false;
        }

        $embedToken = json_decode($result, true)['token'];

        return $embedToken;
    }
}
