<?php
namespace Resrequest\API\V1\Rpc\ReservationGetDetails;

use Exception;
use Zend\Mvc\Controller\AbstractActionController;
use Resrequest\API\AuthenticatedUser;
use ZF\ApiProblem\ApiProblem;
use ZF\ApiProblem\ApiProblemResponse;
use ZF\ContentNegotiation\JsonModel;

define('RESERVATION_GET_DETAILS_SERVER' , dirname(dirname(dirname(dirname(dirname(dirname(dirname(dirname(__FILE__))))))))); //Go back 5 directories
define('RESERVATION_GET_DETAILS_ROOT', RESERVATION_GET_DETAILS_SERVER. '/Application/src/Resrequest/legacy');

class ReservationGetDetailsController extends AbstractActionController
{
  use AuthenticatedUser;

  protected $em;
  protected $enterprise;
  protected $user;

  const BUSINESS_LINK = 'RS1';
  const AGENT_LINK = 'RS2';

  const RES_TABS_FUNCTION = 52;
  const RES_TABS_INVOICING_FUNCTION = 108;
  const RES_TABS_NOTES_FUNCTION = 123;
  const RES_TABS_ROOMING_FUNCTION = 124;
  const RES_TABS_FINANCIAL_FUNCTION = 125;

  public function __construct($em, $enterprise)
  {
    $this->em = $em;
    $this->enterprise = $enterprise;
  }

  public function reservationGetDetailsAction()
  {
    $this->checkUser();

    $userId = $this->user['prUserId'];
    $this->reservationAccess = $this->getReservationFunctionsAccess($userId);

    $params = $this->bodyParams();
    $reservations = $this->getReservations($params, $this->reservationAccess);

    return $this->response($reservations);
  }

  /**
   * Whether the reservation exists or not.
   *
   * @param string $reservationId
   * @return boolean
   */
  private function reservationExists($reservationId)
  {
    $builder = $this->em->createQueryBuilder();
    $reservation = $builder
      ->select(['rvReservation.rvReservationIx AS id'])
      ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
      ->where('rvReservation.rvReservationIx = :reservationId')
      ->setParameters(['reservationId' => $reservationId])
      ->getQuery()
      ->getResult();
    return !empty($reservation);
  }

  /**
   * Whether a user can view a reservation or not.
   *
   * @param string $userId
   * @param string $reservationId
   * @return boolean
   */
  private function canUserViewReservation($userId, $reservationId)
  {
    if ($this->isInternalUser($userId)) {
      return true;
    } else if ($this->isAgent($userId)) {
      return $this->isAgencyReservation($userId, $reservationId);
    } else {
      return false;
    }
  }

  /**
   * Whether or not the user is an internal user.
   *
   * @param string $userId
   * @return boolean
   */
  private function isInternalUser($userId)
  {
    $builder = $this->em->createQueryBuilder();
    $links = $builder
      ->select(['prLink.prLinkIx AS id'])
      ->from('Resrequest\DB\Enterprise\Entity\PrLink', 'prLink')
      ->where('prLink.prLinkFrom = :userId')
      ->andWhere('prLink.rfLinkTypeId = :linkType')
      ->setParameters([
        'userId' => $userId,
        'linkType' => self::BUSINESS_LINK
      ])
      ->getQuery()
      ->getResult();
    return !empty($links);
  }

  /**
   * Whether the user is an agent or not.
   *
   * @param string $userId
   * @return boolean
   */
  private function isAgent($userId)
  {
    $builder = $this->em->createQueryBuilder();
    $agents = $builder
      ->select(['prAgent.prAgentId AS id'])
      ->from('Resrequest\DB\Enterprise\Entity\PrAgent', 'prAgent')
      ->where('prAgent.prAgentId = :userId')
      ->setParameters(['userId' => $userId])
      ->getQuery()
      ->getResult();

    if (!empty($agents)) {
      return true;
    } else {
      $links = $builder
        ->select(['prLink.prLinkIx AS id'])
        ->from('Resrequest\DB\Enterprise\Entity\PrLink', 'prLink')
        ->where('prLink.prLinkFrom = :userId')
        ->andWhere('prLink.rfLinkTypeId = :linkType')
        ->setParameters([
          'userId' => $userId,
          'linkType' => self::AGENT_LINK
        ])
        ->getQuery()
        ->getResult();

      return !empty($links);
    }
  }

  /**
   * Retrieves the user's access to reservation tabs.
   *
   * @param string $userId
   * @return boolean
   */
  private function getReservationFunctionsAccess($userId)
  {
    $builder = $this->em->createQueryBuilder();
    $resFunctions = $builder
      ->select([
        'scFunGroup.scFunctionId as functionId',
        'scFunGroup.scFunGrpLevel AS accessLevel'
      ])
      ->from('Resrequest\DB\Enterprise\Entity\ScFunGroup', 'scFunGroup')
      ->innerJoin('Resrequest\DB\Enterprise\Entity\ScUser', 'scUser', 'with', 'scUser.scGroupId = scFunGroup.scGroupId')
      ->innerJoin('Resrequest\DB\Enterprise\Entity\ScFunction', 'scFunction', 'with', 'scFunction.scFunctionId = scFunGroup.scFunctionId')
      ->where('scUser.prUserId = :userId')
      ->andWhere('scFunction.scFunctionId IN (:ids)')
      ->setParameters([
        'userId' => $userId,
        'ids' => [
          self::RES_TABS_FUNCTION,
          self::RES_TABS_INVOICING_FUNCTION,
          self::RES_TABS_NOTES_FUNCTION,
          self::RES_TABS_ROOMING_FUNCTION,
          self::RES_TABS_FINANCIAL_FUNCTION
        ]
      ])
      ->getQuery()
      ->getResult();

    $resTabs = [];

    foreach ($resFunctions as $function) {
      $name = '';

      switch ($function['functionId']) {
      case self::RES_TABS_FUNCTION:
        $name = 'resTabs';
        break;
      case self::RES_TABS_INVOICING_FUNCTION:
        $name = 'resTabsInvoicing';
        break;
      case self::RES_TABS_NOTES_FUNCTION:
        $name = 'resTabsNotes';
        break;
      case self::RES_TABS_ROOMING_FUNCTION:
        $name = 'resTabsRooming';
        break;
      case self::RES_TABS_FINANCIAL_FUNCTION:
        $name = 'resTabsFinancial';
        break;
      default:
        $name = '';
      }

      $resTabs[$name] = $function['accessLevel'];
    }

    return $resTabs;
  }

  private function hasTabsAccess($functions)
  {
    return $functions['resTabs'] > 0;
  }

  private function hasNotesAcess($functions)
  {
    return $functions['resTabsNotes'] > 0;
  }

  private function hasReservationAccess($function)
  {
    return $functions['resTabsNotes'] > 0;
  }

  private function isAgencyReservation($userId, $reservationId)
  {
    $builder = $this->em->createQueryBuilder();
    $reservation = $builder
      ->select(['rvReservation.rvAgentId AS agentId'])
      ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
      ->where('rvReservation.rvReservationId = :reservationId')
      ->setParameters(['reservationId' => $reservationId])
      ->getQuery()
      ->getSingleResult();

    if (!$reservation['agentId']) {
      return false;
    }

    $reservationAgency = $this->getAgencyId($reservation['agentId']);
    $userAgency = $this->getAgencyId($userId);

    return $reservationAgency == $userAgency;
  }

  private function getAgencyId($userId)
  {
    $builder = $this->em->createQueryBuilder();

    try {
      $agent = $builder
        ->select(['prAgent.prAgentId AS id'])
        ->from('Resrequest\DB\Enterprise\Entity\PrAgent', 'prAgent')
        ->where('prAgent.prAgentId = :userId')
        ->setParameters(['userId' => $userId])
        ->getQuery()
        ->getSingleResult();
      return $agent['id'];
    } catch(Exception $e) {

    }

    try {
      $link = $builder
        ->select(['prLink.prLinkIx AS id'])
        ->from('Resrequest\DB\Enterprise\Entity\PrLink', 'prLink')
        ->where('prLink.prLinkFrom = :userId')
        ->andWhere('prLink.rfLinkTypeId = :linkType')
        ->setParameters([
          'userId' => $userId,
          'linkType' => self::AGENT_LINK
        ])
        ->getQuery()
        ->getSingleResult();
      return $link['id'];
    } catch(Exception $e) {

    }

    return '';
  }

  /**
   * Retrieve the reservation.
   *
   * @param string $reservationId
   * @param array $access
   * @return array
   */
  private function getReservations($params, $access)
  {
    if($access['resTabs'] == 0) {
      return [];
    }
    require_once(RESERVATION_GET_DETAILS_ROOT . '/inc.setup.php');

    $reservationParameters = [];
    $rvReservationIx = $params['rvReservationIx'] ?? false;
    $reservationParameters['rvReservationIx'] = $rvReservationIx;

    // toggles
    $contactDetailsYn = $params['contactDetailsYn'] ?? false;
    $itineraryYn = $params['itineraryYn'] ?? false;
    $notesYn = $params['notesYn'] ?? false;
    $extrasYn = $params['extrasYn'] ?? false;
    $roomingYn = $params['roomingYn'] ?? false;
    $financialsYn = $params['financialsYn'] ?? false;
    $communicationsYn = $params['communicationsYn'] ?? false;
    $auditTrailYn = $params['auditTrailYn'] ?? false;

    $reservationFields = [
      'rvReservationIx' => 'RvReservation.rvReservationIx',
      'rvResName' => 'RvReservation.rvResName',
      'rvAgentId' => 'RvReservation.rvAgentId',
      'rvAgentRef' => 'RvReservation.rvAgentRef',
      'rfReservationStatusId' => 'RvReservation.rfReservationStatusId',
      'rfReservationStatusDesc' => 'RfReservationStatus.rfReservationStatusDesc',
      'rvConsultantId' => 'RvReservation.rvConsultantId',
      'consultantName' => 'CONCAT(reservationConsultant.prNameFirst, \' \', reservationConsultant.prNameLast) AS consultantName',
      'rvCorrPersonaId' => 'RvReservation.rvCorrPersonaId'
    ];
    if($notesYn === true && $access['resTabsNotes'] > 0) {
      $reservationFields['rvNoteGeneral'] = 'RvReservation.rvNoteGeneral';
      $reservationFields['rvNoteGuests'] = 'RvReservation.rvNoteGuests';
      $reservationFields['rvNoteInternal'] = 'RvReservation.rvNoteInternal';
      $reservationFields['rfCountryName'] = 'RfCountry.rfCountryName';
    }
    if($financialsYn === true && $access['resTabsFinancial'] > 0) {
      $reservationFields['rvBillingPersonaId'] = 'RvReservation.rvBillingPersonaId';
      $reservationFields['billingContactName'] = 'CONCAT(billingContact.prNameFirst, \' \', billingContact.prNameLast) AS billingContactName';
      $reservationFields['rvInvoicePersonaId;'] = 'RvReservation.rvInvoicePersonaId';
      $reservationFields['invoiceContactName'] = 'CONCAT(invoiceContact.prNameFirst, \' \', invoiceContact.prNameLast) AS invoiceContactName';
      $reservationFields['rvFolioLevelInd'] = 'RvReservation.rvFolioLevelInd';
      $reservationFields['rvCommissionDeductYn'] = 'RvReservation.rvCommissionDeductYn';
      $reservationFields['rvCommissionPerc'] = 'RvReservation.rvCommissionPerc';
      $reservationFields['rtRateTypeId'] = 'RvReservation.rtRateTypeId';
      $reservationFields['rtRateTypeDesc'] = 'RtRateType.rtRateTypeDesc';
      $reservationFields['rvInvoiceCurrencyId'] = 'RvReservation.rvInvoiceCurrencyId';
      $reservationFields['invoiceCurrencyName'] = 'invoiceCurrency.rfCurrencyName';
      $reservationFields['rvExchgRate'] = 'RvReservation.rvExchgRate';
      $reservationFields['acPayPlanId'] = 'RvReservation.acPayPlanId';
      $reservationFields['acPayPlanDesc'] = 'AcPayPlan.acPayPlanDesc';
    }

    $reservationsQuery = $this->em->createQueryBuilder();
    $reservationsQuery
      ->select(array_values($reservationFields))
      ->distinct('RvReservation.rvReservationIx')
      ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'RvReservation')
      ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'RfReservationStatus', 'with', 'RfReservationStatus.rfReservationStatusId = RvReservation.rfReservationStatusId')
      ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'RfCountry', 'with', 'RfCountry.rfCountryIx = RvReservation.rfCountryId')
      ->leftJoin('Resrequest\DB\Enterprise\Entity\PrAgent', 'PrAgent', 'with', 'PrAgent.prAgentId = RvReservation.rvAgentId')
      ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'reservationAgent', 'with', 'reservationAgent.prPersonaIx = PrAgent.prAgentId')
      ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'reservationConsultant', 'with', 'reservationConsultant.prPersonaIx = RvReservation.rvConsultantId')
      ->where('RvReservation.rvReservationIx IN (:rvReservationIx)');
    if($financialsYn === true && $access['resTabsFinancial'] > 0) {
      $reservationsQuery
        ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'billingContact', 'with', 'billingContact.prPersonaIx = RvReservation.rvBillingPersonaId')
        ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'invoiceContact', 'with', 'invoiceContact.prPersonaIx = RvReservation.rvInvoicePersonaId')
        ->leftJoin('Resrequest\DB\Enterprise\Entity\RtRateType', 'RtRateType', 'with', 'RtRateType.rtRateTypeIx = RvReservation.rtRateTypeId')
        ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCurrency', 'invoiceCurrency', 'with', 'invoiceCurrency.rfCurrencyIx = RvReservation.rvInvoiceCurrencyId')
        ->leftJoin('Resrequest\DB\Enterprise\Entity\AcPayPlan', 'AcPayPlan', 'with', 'AcPayPlan.acPayPlanIx = RvReservation.acPayPlanId');
    }
    $reservations = $reservationsQuery
      ->setParameters($reservationParameters)
      ->getQuery()
      ->getResult();

    $filteredReservations = [];
    foreach ($reservations as $reservation) {
      require_once(RESERVATION_GET_DETAILS_ROOT . '/functions.persona.php');
      $filteredReservation['rvReservationIx'] = $reservation['rvReservationIx'];
      $filteredReservation['rvResName'] = $reservation['rvResName'];
      $filteredReservation['rvAgentRef'] = $reservation['rvAgentRef'];
      $filteredReservation['rvAgentId'] = $reservation['rvAgentId'];
      $filteredReservation['rfReservationStatusId'] = $reservation['rfReservationStatusId'];
      $filteredReservation['rfReservationStatusDesc'] = $reservation['rfReservationStatusDesc'];
      $filteredReservation['rvConsultantId'] = $reservation['rvConsultantId'];
      $filteredReservation['consultantName'] = trim($reservation['consultantName']);
      if($contactDetailsYn === true) {
        $billingContact = array_key_exists('rvBillingPersonaId', $reservation)
          ? getContactDetails($reservation['rvBillingPersonaId'])
          : null;

        $filteredReservation['contacts'] = [
          'agent' => getContactDetails($reservation['rvAgentId']),
          'contact' => getContactDetails($reservation['rvCorrPersonaId']),
          'billing' => $billingContact,
        ];
      }
      if($notesYn === true && $access['resTabsNotes'] > 0) {
        $filteredReservation['rvNoteGeneral'] = $reservation['rvNoteGeneral'];
        $filteredReservation['rvNoteGuests'] = $reservation['rvNoteGuests'];
        $filteredReservation['rvNoteInternal'] = $reservation['rvNoteInternal'];
        $filteredReservation['rfCountryName'] = $reservation['rfCountryName'];
      }

      $reservationItineraries = [];
      if($itineraryYn === true && $access['resTabs'] > 0) {
        $itinerariesQuery = $this->em->createQueryBuilder();
        $itineraries = $itinerariesQuery
          ->select(
            [
              'RvReservationItem.rvReservationItemIx',
              'RvReservationItem.rvItemDateArrive',
              'RvReservationItem.rvItemDateDepart',
              'RvReservationItem.rvItemNights',
              'property.prNameLast AS propertyName',
              'property.prPersonaIx AS propertyId',
              'AcAccommType.acAccommDesc',
              'AcAccommType.acAccommTypeIx AS acAccommId',
              'RvReservationItem.rvItemAdultCount',
              'RvReservationItem.rvItemChildCount',
              'RvReservationItem.rvItemAmtNett',
              'RvReservationItem.rvItemAccommCount',
              'rfCurrency.rfCurrencySymbol as currencySymbol'
            ]
          )
          ->from('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'RvReservationItem')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'AcAccommType', 'with', 'AcAccommType.acAccommTypeIx = RvReservationItem.acAccommTypeId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'property', 'with', 'property.prPersonaIx = AcAccommType.prBusinessId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\RtRateType', 'rtRateType', 'with', 'rtRateType.rtRateTypeIx = RvReservationItem.rtRateTypeId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\RfCurrency', 'rfCurrency', 'with', 'rfCurrency.rfCurrencyIx = rtRateType.rfCurrencyId')
          ->where('RvReservationItem.rvReservationId = :rvReservationId');
        $itineraryParameters['rvReservationId'] = $reservation['rvReservationIx'];
        $itineraries = $itinerariesQuery
          ->setParameters($itineraryParameters)
          ->getQuery()
          ->getResult();
        foreach($itineraries as $itinerary) {
          $reservationItineraries[] = [
            'rvReservationItemIx' => $itinerary['rvReservationItemIx'],
            'rvItemDateArrive' => $itinerary['rvItemDateArrive'],
            'rvItemDateDepart' => $itinerary['rvItemDateDepart'],
            'rvItemNights' => $itinerary['rvItemNights'],
            'propertyName' => $itinerary['propertyName'],
            'propertyId' => $itinerary['propertyId'],
            'acAccommDesc' => $itinerary['acAccommDesc'],
            'rvItemAdultCount' => $itinerary['rvItemAdultCount'],
            'rvItemChildCount' => $itinerary['rvItemChildCount'],
            'rvItemAmtNett' => $itinerary['rvItemAmtNett'],
            'rvItemAccommCount' => $itinerary['rvItemAccommCount'],
            'currencySymbol' => $itinerary['currencySymbol'],
            'acAccommId' => $itinerary['acAccommId']
          ];
        }
      }
      $filteredReservation['itinerary'] = $reservationItineraries;
      $reservationExtras = [];
      if($extrasYn === true) {
        $extrasQuery = $this->em->createQueryBuilder();
        $extras = $extrasQuery
          ->select(
            [
              'RvExtra.rvExtraIx',
              'RvExtra.rvExtraCharge',
              'RvExtra.rvExtraAmtNett',
              'RvExtra.rvExtraDateServ',
              'RvExtra.rvExtraUnits',
              'RvExtra.prBusinessId',
              'RvExtra.acExtraId',
              'AcExtra.acExtDesc',
              'property.prNameLast AS propertyName',
              'RvExtra.rfCurrencyId',
              'RfCurrency.rfCurrencySymbol',
              'supplier.prNameFirst AS supplierFirstName',
              'supplier.prNameLast AS supplierLastName',
              'RvExtra.rvExtraRef',
              'RvExtra.rvExtraTravelEta AS expectedArrivalTime',
              'RvExtra.rvExtraTravelEtd AS expectedDepartureTime',
              'RvExtra.rvExtraNote AS note',
              'RvExtra.rvExtraNoteInternal AS memo',
              'AcExtraCategory.acExtraCatInd AS categoryType'
            ]
          )
          ->from('Resrequest\DB\Enterprise\Entity\RvExtra', 'RvExtra')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\AcExtra', 'AcExtra', 'with', 'AcExtra.acExtraIx = RvExtra.acExtraId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\AcExtraCategory', 'AcExtraCategory', 'with', 'AcExtraCategory.acExtraCategoryIx = AcExtra.acExtraCategoryId')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\PrBusiness', 'PrBusiness', 'with', 'PrBusiness.prBusinessId = RvExtra.prBusinessId')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'property', 'with', 'property.prPersonaIx = PrBusiness.prBusinessId')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCurrency', 'RfCurrency', 'with', 'RfCurrency.rfCurrencyIx = RvExtra.rfCurrencyId')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'supplier', 'with', 'supplier.prPersonaIx = RvExtra.prSupplierId')
          ->where('RvExtra.rvReservationId = :rvReservationId')
          ->andWhere('RvExtra.rvExtraVoidInd = :rvExtraNotVoided')
          ->setParameters(['rvReservationId' => $reservation['rvReservationIx'], 'rvExtraNotVoided' => 0])
          ->getQuery()
          ->getResult();
        foreach($extras as $extra) {
          $reservationExtras[] = [
            'rvExtraIx' => $extra['rvExtraIx'],
            'rvExtraCharge' => $extra['rvExtraCharge'],
            'rvExtraAmtNett' => $extra['rvExtraAmtNett'],
            'rvExtraDateServ' => $extra['rvExtraDateServ'],
            'rvExtraUnits' => $extra['rvExtraUnits'],
            'acExtraId' => $extra['acExtraId'],
            'acExtDesc' => $extra['acExtDesc'],
            'prBusinessId' => $extra['prBusinessId'],
            'propertyName' => $extra['propertyName'],
            'rfCurrencyId' => $extra['rfCurrencyId'],
            'rfCurrencySymbol' => $extra['rfCurrencySymbol'],
            'rvExtraRef' => $extra['rvExtraRef'],
            'expectedArrivalTime' => $extra['expectedArrivalTime'],
            'expectedDepartureTime' => $extra['expectedDepartureTime'],
            'note' => $extra['note'],
            'memo' => $extra['memo'],
            'supplierFirstName' => $extra['supplierFirstName'] ? $extra['supplierFirstName'] : '',
            'supplierLastName' => $extra['supplierLastName'] ? $extra['supplierLastName'] : '',
            'categoryType' => $extra['categoryType'],
          ];
        }
      }
      $filteredReservation['extras'] = $reservationExtras;

      $reservationRooming = [];
      if($roomingYn === true && $access['resTabsRooming'] > 0) {
        $guestsQuery = $this->em->createQueryBuilder();
        $guests = $guestsQuery
          ->select(
            [
              'guest.prPersonaIx AS prGuestId',
              'guest.prNameFirst',
              'guest.prNameLast',
              'guest.prEmail',
              'PrPhone.prPhoneDetail'
            ]
          )
          ->distinct('guest.prPersonaIx')
          ->from('Resrequest\DB\Enterprise\Entity\RvReservationGuest', 'RvReservationGuest')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'guest', 'with', 'guest.prPersonaIx = RvReservationGuest.prGuestId')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPhone', 'PrPhone', 'with', 'PrPhone.prPersonaId = guest.prPersonaIx')
          ->where('RvReservationGuest.rvReservationId = :rvReservationId')
          ->setParameters(['rvReservationId' => $reservation['rvReservationIx']])
          ->getQuery()
          ->getResult();
        $reservationRooming['guests'] = $guests;

        $groupsQuery = $this->em->createQueryBuilder();
        $groups = $groupsQuery
          ->select(
            [
              'RvResItemGroup.rvResItemGroupIx',
              'RvResItemGroup.rvReservationItemId',
              'RvResItemGroup.rvGrpDateAllocated',
              'RvResItemGroup.rvGrpStatusInd',
              'CASE WHEN (RvResItemGroup.rvGrpStatusInd = 2) THEN \'Not checked in\' WHEN (RvResItemGroup.rvGrpStatusInd = 5) THEN \'Checked in\' WHEN (RvResItemGroup.rvGrpStatusInd = 8) THEN \'Checked out\' ELSE \'\' END AS groupStatus',
              'RvResItemGroup.acAccommTypeId',
              'AcAccommType.acAccommDesc',
              'AcAccommType.acAccommCapacity',
              'RvResItemGroup.acAccommRoomId',
              'AcAccommRoom.acDesc AS roomName'
            ]
          )
          ->from('Resrequest\DB\Enterprise\Entity\RvResItemGroup', 'RvResItemGroup')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'RvReservationItem', 'with', 'RvReservationItem.rvReservationItemIx = RvResItemGroup.rvReservationItemId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'AcAccommType', 'with', 'AcAccommType.acAccommTypeIx = RvResItemGroup.acAccommTypeId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommRoom', 'AcAccommRoom', 'with', 'AcAccommRoom.acAccommRoomIx = RvResItemGroup.acAccommRoomId')
          ->where('RvReservationItem.rvReservationId = :rvReservationId')
          ->setParameters(['rvReservationId' => $reservation['rvReservationIx']])
          ->getQuery()
          ->getResult();
        $reservationGroupGuests = [];
        foreach($groups as $group) {
          $groupGuestsQuery = $this->em->createQueryBuilder();
          $groupGuests = $groupGuestsQuery
            ->select(
              [
                'RvResItemGuest.prGuestId'
              ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvResItemGroup', 'RvResItemGroup')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvResItemGuest', 'RvResItemGuest', 'with', 'RvResItemGuest.rvResItemGroupId = RvResItemGroup.rvResItemGroupIx')
            ->where('RvResItemGroup.rvResItemGroupIx = :rvResItemGroupIx')
            ->setParameters(['rvResItemGroupIx' => $group['rvResItemGroupIx']])
            ->getQuery()
            ->getResult();
          $group['guests'] = $groupGuests;
          $reservationGroupGuests[] = $group;
        }
        $reservationRooming['groups'] = $reservationGroupGuests;
      }
      $filteredReservation['rooming'] = $reservationRooming;

      $reservationFinancials = [];
      if($financialsYn === true && $access['resTabsFinancial'] > 0) {
        require_once(RESERVATION_GET_DETAILS_ROOT . '/db.rv_reservation.php');
        require_once(RESERVATION_GET_DETAILS_ROOT . '/functions.financial.php');
        $reservationFinancials['rvInvoicePersonaId'] = $reservation['rvInvoicePersonaId'];
        $reservationFinancials['invoiceContactName'] = $reservation['invoiceContactName'];
        //db_rv_folio_level_desc: defined in db.rv_reservation.php
        $reservationFinancials['billingFolioLevel'] = db_rv_folio_level_desc($reservation['rvFolioLevelInd']);
        $reservationFinancials['billingCharge'] = $reservation['rvCommissionDeductYn'] == "0" ? "Gross" : "Nett";
        $reservationFinancials['rvCommissionPerc'] = $reservation['rvCommissionPerc'];
        $reservationFinancials['rtRateTypeId'] = $reservation['rtRateTypeId'];
        $reservationFinancials['rtRateTypeDesc'] = $reservation['rtRateTypeDesc'];
        $reservationFinancials['rvInvoiceCurrencyId'] = $reservation['rvInvoiceCurrencyId'];
        $reservationFinancials['invoiceCurrencyName'] = $reservation['rfCurrencyName'];
        $reservationFinancials['rvExchgRate'] = $reservation['rvExchgRate'];
        $reservationFinancials['acPayPlanId'] = $reservation['acPayPlanId'];
        $reservationFinancials['acPayPlanDesc'] = $reservation['acPayPlanDesc'];
        //getPayItems: defined in functions.financial.php
        $reservationFinancials['paymentPlan'] = getPayItems($reservation['rvReservationIx']);
        $foliosQuery = $this->em->createQueryBuilder();
        $folios = $foliosQuery
          ->select(
            [
              'FnFolio.fnFolioIx',
              'FnFolio.fnFolioFolioNum',
              'property.prPersonaIx AS propertyId',
              'property.prNameLast AS propertyName',
              'FnFolio.fnFolioToId',
              'folioPersona.prNameLast AS folioToName',
              'RtComponent.rtComponentDesc',
              'SUM(RvResItemComp.rvItemCompAmtComm) AS rvItemCompAmtComm',
              'SUM(RvResItemComp.rvItemCompAmtGross) AS rvItemCompAmtGross',
              'SUM(RvResItemComp.rvItemCompAmtNett) AS rvItemCompAmtNett',
              'SUM(RvResItemComp.rvItemCompAmtPayable) AS rvItemCompAmtPayable',
              'SUM(RvResItemComp.rvItemCompAmtTax) AS rvItemCompAmtTax',
              'FnFolio.fnFolioAmtPaid',
            ]
          )
          ->from('Resrequest\DB\Enterprise\Entity\FnFolio', 'FnFolio')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'RvReservationItem', 'with', 'RvReservationItem.fnFolioId = FnFolio.fnFolioIx')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'folioPersona', 'with', 'folioPersona.prPersonaIx = FnFolio.fnFolioToId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'property', 'with', 'property.prPersonaIx = FnFolio.prBusinessId')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\RvResItemComp', 'RvResItemComp', 'with', 'RvResItemComp.rvReservationItemId = RvReservationItem.rvReservationItemIx')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\RtComponent', 'RtComponent', 'with', 'RtComponent.rtComponentIx = RvResItemComp.rtComponentId')
          ->where('RvReservationItem.rvReservationId = :rvReservationId')
          ->andWhere('RvResItemComp.rvItemCompSeparateYn = 0 OR RtComponent.rtComponentId IS NULL')
          ->andWhere('FnFolio.fnInvoiceId IS NULL')
          ->groupBy('RvResItemComp.rtComponentId')
          ->addGroupBy('RvResItemComp.rtTaxId')
          ->setParameters(
            [
              'rvReservationId' => $reservation['rvReservationIx']
            ]
          )
          ->getQuery()
          ->getResult();
        $reservationFinancials['folios'] = $folios ?? [];
        $invoicesQuery = $this->em->createQueryBuilder();
        $invoices = $invoicesQuery
          ->select(
            [
              'FnInvoice.fnInvoiceIx',
              'FnInvoice.fnInvDate',
              'property.prPersonaIx AS propertyId',
              'property.prNameLast AS propertyName',
              'FnFolio.fnFolioToId',
              'folioPersona.prNameLast AS folioToName',
              'SUM(FnInvoiceItem.fnInvItemAmount) AS invoiceAmount',
              'SUM(FnInvoiceItem.fnInvItemCommPerc) AS invoiceCommissionPercentage',
              'SUM(FnInvoiceItem.fnInvItemAmtComm) AS invoiceCommission',
              'SUM(FnInvoiceItem.fnInvItemAmtTax) AS invoiceTax',
              'FnInvoice.fnInvAmtPaid'
            ]
          )
          ->from('Resrequest\DB\Enterprise\Entity\FnInvoice', 'FnInvoice')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\FnFolio', 'FnFolio', 'with', 'FnFolio.fnInvoiceId = FnInvoice.fnInvoiceIx')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\FnInvoiceItem', 'FnInvoiceItem', 'with', 'FnInvoiceItem.fnInvoiceId = FnInvoice.fnInvoiceIx')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'RvReservationItem', 'with', 'RvReservationItem.fnFolioId = FnFolio.fnFolioIx')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'folioPersona', 'with', 'folioPersona.prPersonaIx = FnFolio.fnFolioToId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'property', 'with', 'property.prPersonaIx = FnFolio.prBusinessId')
          ->where('RvReservationItem.rvReservationId = :rvReservationId')
          ->andWhere('FnInvoice.fnInvStatusInd = 2')
          ->setParameters(
            [
              'rvReservationId' => $reservation['rvReservationIx']
            ]
          )
          ->getQuery()
          ->getResult();
      }
      $reservationFinancials['invoices'] = $invoices ?? [];
      $filteredReservation['financials'] = $reservationFinancials;

      $reservationCommunications = [];
      if($communicationsYn === true) {
        $communicationsQuery = $this->em->createQueryBuilder();
        $communications = $communicationsQuery
          ->select(
            [
              'EmMail.emMailIx',
              'EmMail.emFrom',
              'EmMail.emTo',
              'EmMail.emSubject',
              'EmMail.emMessage',
              'EmMailLog.prUserId'
            ]
          )
          ->from('Resrequest\DB\Enterprise\Entity\EmMail', 'EmMail')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\EmMailLog', 'EmMailLog', 'with', 'EmMailLog.emMailId = EmMail.emMailIx')
          ->where('EmMail.rvReservationId = :rvReservationId')
          ->andWhere('EmMailLog.emAction = -1')
          ->setParameters(
            [
              'rvReservationId' => $reservation['rvReservationIx']
            ]
          )
          ->getQuery()
          ->getResult();
      }
      $filteredReservation['communications'] = $communications ?? [];

      $reservationAuditTrail = [];
      if($auditTrailYn === true) {
        $auditTrailQuery = $this->em->createQueryBuilder();
        $auditTrail = $auditTrailQuery
          ->select(
            [
              'AdModuleField.adModuleFieldDescription',
              'AdModuleField.adModuleFieldDisplayTypeInd',
              'RfAuditModule.rfAuditModuleName',
              'AdResDetail.adResDetailActionInd',
              'AdResDetail.adResDetailValueFrom',
              'AdResDetail.adResDetailValueTo',
              'AdResDetail.adResDetailValueFromDisplay',
              'AdResDetail.adResDetailValueToDisplay',
              'AdModuleField.adModuleFieldDisplay'
            ]
          )
          ->from('Resrequest\DB\Enterprise\Entity\AdReservation', 'AdReservation')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\AdResDetail', 'AdResDetail', 'with', 'AdResDetail.adReservationId = AdReservation.adReservationIx')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\AdModuleField', 'AdModuleField', 'with', 'AdModuleField.adModuleFieldId = AdResDetail.adModuleFieldId')
          ->innerJoin('Resrequest\DB\Enterprise\Entity\RfAuditModule', 'RfAuditModule', 'with', 'RfAuditModule.rfAuditModuleId = AdModuleField.rfAuditModuleId')
          ->leftJoin('Resrequest\DB\Enterprise\Entity\TcTable', 'TcTable', 'with', 'TcTable.tcTableId = AdModuleField.tcTableId')
          ->where('AdReservation.rvReservationId = :rvReservationId')
          ->orderBy('AdResDetail.adResDetailIx')
          ->setParameters(
            [
              'rvReservationId' => $reservation['rvReservationIx']
            ]
          )
          ->getQuery()
          ->getResult();
      }
      $filteredReservation['auditTrail'] = $auditTrail ?? [];

      $filteredReservations[] = $filteredReservation;
      unset($filteredReservation);
    }

    return $filteredReservations;
  }

  /**
   * Creates an error response to be returned to the client.
   *
   * @param string $message
   * @param int $code Defaults to 403 Access denied.
   * @return ApiProblemResponse
   */
  private function errorResponse($message, $code = 403)
  {
    return new ApiProblemResponse(
      new ApiProblem($code, $message)
    );
  }

  /**
   * Creates a response.
   *
   * @param array $data
   * @return JsonModel
   */
  private function response($data)
  {
    return new JsonModel($data);
  }

  private function checkUser()
  {
    $username = false;
    $identity = $this->getIdentity();
    if($identity instanceof \ZF\MvcAuth\Identity\AuthenticatedIdentity) {
      $username = $identity->getAuthenticationIdentity()['user_id'];
    }

    $userQuery = $this->em->createQueryBuilder();
    $this->user = $userQuery
         ->select(
           [
             'pr_user.prUserId',
             'sc_user.scGroupId'
           ]
         )
         ->from('Resrequest\DB\Enterprise\Entity\ScUser', 'sc_user')
         ->innerJoin('Resrequest\DB\Enterprise\Entity\PrUser', 'pr_user', 'with', 'pr_user.prUserId = sc_user.prUserId')
         ->where('pr_user.prUserName = :pr_user_name')
         ->setParameters(
           [ 'pr_user_name' => $username ]
         )
         ->setMaxResults(1)
         ->getQuery()
         ->getOneOrNullResult();
    if($username === false || $this->user === null) {
      $this->errorResponse("Unable to find user", 403);
    }
  }

}
