<?php
namespace Resrequest\API\V1\Rpc\RoomingSetItems;

use Zend\Mvc\Controller\AbstractActionController;
use ZF\ApiProblem\ApiProblemResponse;
use ZF\ApiProblem\ApiProblem;
use ZF\ContentNegotiation\JsonModel;

class RoomingSetItemsController extends AbstractActionController
{
    protected $em;
    protected $authenticate;
    protected $authorise;

    public function __construct($em, $authenticate, $authorise)
    {
        $this->em = $em;
        $this->authenticate = $authenticate;
        $this->authorise = $authorise;
        $this->legacy = __DIR__."/../../../../../../../Application/src/Resrequest/legacy/";
    }

    public function roomingSetItemsAction()
    {
        $params = $this->bodyParams();
        $db = $this->em->getConnection();

        $username = false;
        $identity = $this->getIdentity();
        if($identity instanceof \ZF\MvcAuth\Identity\AuthenticatedIdentity) {
            $username = $identity->getAuthenticationIdentity()['user_id'];
        }

        if($username === false) {
            return new ApiProblemResponse(
                new ApiProblem(403, "Unable to find user")
            );
        }

        $this->authenticate->setupContext($this->authenticate->getUserByName($username));

        $user = $db->fetchAssoc("
            SELECT
                pr_user.pr_user_id,
                sc_user.sc_group_id
            FROM
                sc_user
                INNER JOIN pr_user ON pr_user.pr_user_id = sc_user.pr_user_id
            WHERE
                pr_user.pr_user_name = ?
        ",[$username]);
        if($user === false) {
            return new ApiProblemResponse(
                new ApiProblem(403, "Unable to find user")
            );
        }

        $access = $this->authorise->getAccess([
            'functions'=>[81],
            'environment'=>["isMaster", "isProperty"]
        ]);
        if($access['functions'][81] < 10) {
            return new ApiProblemResponse(
                new ApiProblem(403, "Access denied")
            );
        }

        if(
            !array_key_exists("rooms",$params)
            || !is_array($params['rooms'])
            || sizeof($params['rooms']) == 0
        ) {
            return new ApiProblemResponse(
                new ApiProblem(400, "Invalid or missing room update data")
            );
        }

        require_once($this->legacy . "db.rv_res_item_group.php");

        $rooms = $params['rooms'];
        // Validate
        $errors = [];
        foreach($rooms as $id=>$room) {
            // Basic input validation
            if(
                !array_key_exists("index",$room)
                || (
                    !array_key_exists("room_id",$room)
                    && !array_key_exists("requested_yn",$room)
                    && !array_key_exists("status_ind",$room)
                )
                || (
                    (
                        !array_key_exists("reservation_item_id",$room)
                        || empty($room['reservation_item_id'])
                    ) && (
                        !array_key_exists("res_item_group_id",$room)
                        || empty($room['res_item_group_id'])
                    )
                )
            ) {
                return new JsonModel([
                    'status'=>false,
                    'errors'=>[
                        ['index'=>0, 'error'=>"Unable to process invalid room data"]
                    ]
                ]);
            }

            // Look up itinerary information
            if(empty($room['res_item_group_id'])) {
                $rooms[$id]['res_item_group_id'] = "";
            } else {
                $rooms[$id]['reservation_item_id'] = $db->fetchColumn("
                    SELECT
                        rv_res_item_group.rv_reservation_item_id
                    FROM
                        rv_res_item_group
                    WHERE
                        rv_res_item_group.rv_res_item_group_ix = ?
                ",[$room['res_item_group_id']]);
            }

            $rooms[$id]['checkingOutOnly'] = false;

            if (
                array_key_exists("status_ind",$room)
                && $room['status_ind'] == 8
            ) {
                $roomStatusInd = $db->fetchColumn("
                    SELECT
                        rv_res_item_group.rv_grp_status_ind
                    FROM
                        rv_res_item_group
                    WHERE
                        rv_res_item_group.rv_res_item_group_ix = ?
                ",[$room['res_item_group_id']]);

                if ($roomStatusInd == 5) {
                    if (!array_key_exists("room_id",$room)) {
                        // Item is being checked out only, so clashes and stuff
                        // should be ignored.
                        $rooms[$id]['checkingOutOnly'] = true;
                    }
                }
            }

            if(array_key_exists("room_id",$room)) {
                if(empty($room['room_id'])) {
                    $rooms[$id]['room_id'] = "0";
                }
            } else {
                $roomId = $db->fetchColumn("
                    SELECT
                        rv_res_item_group.ac_accomm_room_id
                    FROM
                        rv_res_item_group
                    WHERE
                        rv_res_item_group.rv_res_item_group_ix = ?
                ",[$rooms[$id]['res_item_group_id']]);
                if($roomId !== false) {
                    $rooms[$id]['room_id'] = $roomId;
                }
            }

            if(array_key_exists("requested_yn",$room)) {
                $rooms[$id]['requested_yn'] = (empty($room['requested_yn'])?"0":"1");
            } else {
                $rooms[$id]['requested_yn'] = "0";
                $rooms[$id]['requested_reason'] = "";
                $requested = $db->fetchAssoc("
                    SELECT
                        rv_res_item_group.rv_room_requested_yn,
                        rv_res_item_group.rv_room_request_reason
                    FROM
                        rv_res_item_group
                    WHERE
                        rv_res_item_group.rv_res_item_group_ix = ?
                ",[$rooms[$id]['res_item_group_id']]);
                if($requested !== false) {
                    $rooms[$id]['requested_yn'] = $requested['rv_room_requested_yn'];
                    $rooms[$id]['requested_reason'] = $requested['rv_room_request_reason'];
                }
            }
        }

        foreach($rooms as $id=>$room) {
            $rv_reservation_item_id = $room['reservation_item_id'];
            $rv_res_item_group_id = $room['res_item_group_id'];

            $itinerary = $db->fetchAssoc("
                SELECT
                    rv_reservation_item.rv_reservation_id,
                    rv_reservation_item.pr_business_id,
                    rv_reservation_item.rv_item_date_arrive,
                    rv_reservation_item.rv_item_date_depart
                FROM
                    rv_reservation_item
                WHERE
                    rv_reservation_item.rv_reservation_item_ix = ?
            ",[$rv_reservation_item_id]);
            if(empty($itinerary)) {
                $errors[] = [
                    'index'=>$room['index'],
                    'error'=>"Itinerary not found"
                ];
                continue;
            }
            $rooms[$id]['rv_reservation_id'] = $itinerary['rv_reservation_id'];

            if(!empty($room['room_id'])) {

                // Is the room allowed? (exists and is active for the itinerary period)
                $roomCheck = $db->fetchAssoc("
                    SELECT
                        IF(COUNT(*) > 0, true, false) AS allow,
                        ac_accomm_type.pr_business_id
                    FROM
                        ac_accomm_room
                        INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = ac_accomm_room.ac_accomm_type_id
                    WHERE
                        ac_accomm_room.ac_accomm_room_ix = ?
                        AND (
                            ac_accomm_type.ac_accomm_type_inactive_yn = '0'
                            OR (
                                ac_accomm_type.ac_accomm_type_inactive_yn != '0'
                                AND ac_accomm_type.ac_accomm_type_inactive_date >= ?
                            )
                        )
                        AND (
                            (
                                ac_accomm_room.ac_accomm_room_inactive_yn = '0'
                                AND ac_accomm_room_create_date <= ?
                            ) OR (
                                ac_accomm_room.ac_accomm_room_inactive_yn != '0'
                                AND ac_accomm_room_inactive_date > ?
                            )
                        )
                ",[
                    $room['room_id'],
                    $itinerary['rv_item_date_depart'],
                    $itinerary['rv_item_date_arrive'],
                    $itinerary['rv_item_date_depart']
                ]);

                if(!$roomCheck['allow']) {
                    $errors[] = [
                        'index'=>$room['index'],
                        'error'=>"Room or accommodation type inactive or invalid"
                    ];
                    continue;
                }

                if($itinerary['pr_business_id'] != $roomCheck['pr_business_id']) {
                    $errors[] = [
                        'index'=>$room['index'],
                        'error'=>"Itinerary property does not match selected room property"
                    ];
                    continue;
                }

                // Check availability
                $blockCheck = $db->fetchColumn("
                    SELECT
                        COUNT(*)
                    FROM
                        ac_accomm_room_block
                        INNER JOIN ac_accomm_block ON
                            ac_accomm_block.ac_accomm_block_ix = ac_accomm_room_block.ac_accomm_block_id
                    WHERE
                        ac_accomm_room_id = ?
                        AND (
                            (
                                ? >= ac_accomm_block.ac_start_date
                                AND ? < ac_accomm_block.ac_end_date
                            ) OR (
                                ? > ac_accomm_block.ac_start_date
                                AND ? <= ac_accomm_block.ac_end_date
                            ) OR (
                                ? < ac_accomm_block.ac_start_date
                                AND ? > ac_accomm_block.ac_end_date
                            )
                        )
                ",[
                    $room['room_id'],
                    $itinerary['rv_item_date_arrive'],
                    $itinerary['rv_item_date_arrive'],
                    $itinerary['rv_item_date_depart'],
                    $itinerary['rv_item_date_depart'],
                    $itinerary['rv_item_date_arrive'],
                    $itinerary['rv_item_date_depart']
                ]);
                if ($blockCheck > 0 && !$room['checkingOutOnly']) {
                    $errors[] = [
                        'index'=>$room['index'],
                        'error'=>"Selected room is blocked during the travel period of this itinerary"
                    ];
                    continue;
                }

                $availCheck = $db->fetchAll("
                    SELECT
                        rv_res_item_group.rv_res_item_group_ix
                    FROM
                        rv_res_item_group
                        INNER JOIN rv_reservation_item ON rv_reservation_item.rv_reservation_item_ix = rv_res_item_group.rv_reservation_item_id
                    WHERE
                        ac_accomm_room_id = ?
                        AND (
                            (
                                rv_reservation_item.rv_item_date_arrive >= ?
                                AND rv_reservation_item.rv_item_date_arrive < ?
                            ) OR (
                                rv_reservation_item.rv_item_date_arrive < ?
                                AND rv_reservation_item.rv_item_date_depart > ?
                                AND (
                                    rv_res_item_group.rv_grp_status_ind <> '8'
                                    OR (
                                        rv_res_item_group.rv_grp_status_ind = '8'
                                        AND rv_res_item_group.rv_grp_status_time > ?
                                    )
                                )
                            ) OR (
                                rv_res_item_group.rv_grp_status_ind = '5'
                                AND ? < ?
                                AND rv_reservation_item.rv_item_date_arrive < ?
                            )
                        )
                ",[
                    $room['room_id'],
                    $itinerary['rv_item_date_arrive'],
                    $itinerary['rv_item_date_depart'],
                    $itinerary['rv_item_date_arrive'],
                    $itinerary['rv_item_date_arrive'],
                    $itinerary['rv_item_date_arrive'] . " 23:59:59",
                    $itinerary['rv_item_date_arrive'],
                    date("Y-m-d"),
                    $itinerary['rv_item_date_depart'],
                ]);

                // Filter out items within the change set that are no longer a conflict
                $availCheck = array_filter($availCheck, function($group) use ($rooms, $room) {
                    $rv_res_item_group_id = $group['rv_res_item_group_ix'];
                    if($rv_res_item_group_id == $room['res_item_group_id']) {
                        return false;
                    }
                    foreach($rooms as $other) {
                        if($other['index'] == $room['index']) {
                            continue;
                        }
                        if(
                            $other['res_item_group_id'] == $rv_res_item_group_id
                            && $other['room_id'] != $room['room_id']
                        ) {
                            return false;
                        }
                    }
                    return true;
                });
                if (sizeof($availCheck) > 0 && !$room['checkingOutOnly']) {
                    $errors[] = [
                        'index'=>$room['index'],
                        'error'=>"Selected room is already used by another itinerary"
                    ];
                    continue;
                }

                $conflict = false;
                foreach($rooms as $other) {
                    if($other['index'] == $room['index']) {
                        continue;
                    }
                    // Look up itinerary information
                    if(empty($other['res_item_group_id'])) {
                        $other_reservation_item_id = $other['reservation_item_id'];
                    } else {
                        $other_reservation_item_id = $db->fetchColumn("
                            SELECT
                                rv_res_item_group.rv_reservation_item_id
                            FROM
                                rv_res_item_group
                            WHERE
                                rv_res_item_group.rv_res_item_group_ix = ?
                        ",[$other['res_item_group_id']]);
                    }

                    $otherItinerary = $db->fetchAssoc("
                        SELECT
                            rv_reservation_item.rv_item_date_arrive,
                            rv_reservation_item.rv_item_date_depart
                        FROM
                            rv_reservation_item
                        WHERE
                            rv_reservation_item.rv_reservation_item_ix = ?
                    ",[$other_reservation_item_id]);
                    if(
                        $other['room_id'] == $room['room_id']
                        && (
                            (
                                $otherItinerary['rv_item_date_arrive'] >= $itinerary['rv_item_date_arrive']
                                && $otherItinerary['rv_item_date_arrive'] < $itinerary['rv_item_date_depart']
                            ) || (
                                $otherItinerary['rv_item_date_depart'] > $itinerary['rv_item_date_arrive']
                                && $otherItinerary['rv_item_date_depart'] <= $itinerary['rv_item_date_depart']
                            ) || (
                                $otherItinerary['rv_item_date_arrive'] < $itinerary['rv_item_date_arrive']
                                && $otherItinerary['rv_item_date_depart'] > $itinerary['rv_item_date_depart']
                            )
                        )
                    ) {
                        $conflict = true;
                        break;
                    }
                }
                if ($conflict && !$room['checkingOutOnly']) {
                    $errors[] = [
                        'index'=>$room['index'],
                        'error'=>"Selected room conflicts with another selected room"
                    ];
                    continue;
                }
            }

            if(array_key_exists("status_ind",$room)) {
               if(!$access['environment']['isProperty']) {
                    return new ApiProblemResponse(
                        new ApiProblem(403, "Access denied")
                    );
                }

                if(
                    $room['status_ind'] != "2"
                    && $room['status_ind'] != "5"
                    && $room['status_ind'] != "8"
                ) {
                    return new JsonModel([
                        'status'=>false,
                        'errors'=>[
                            ['index'=>0, 'error'=>"Invalid check in/out status"]
                        ]
                    ]);
                }

                if(!array_key_exists("status_time",$room)) {
                    $rooms[$id]['status_time'] = false;
                }
            }
        }

        if(sizeof($errors) > 0) {
            return new JsonModel([
                'status'=>false,
                'errors'=>$errors
            ]);
        }

        require_once($this->legacy . "functions.rooming.php");
        require_once($this->legacy . "functions.reservation.php");
        // Update rooming
        foreach($rooms as $id=>$room) {
            if(array_key_exists("room_id",$room) && !$room['checkingOutOnly']) {
                if(empty($room['res_item_group_id'])) {
                    $rooms[$id]['res_item_group_id'] = db_rv_res_item_group_insert($room['reservation_item_id'], $room['room_id'], $room['requested_yn'], $room['requested_reason']);
                } else {
                    db_rv_res_item_group_set_room($room['res_item_group_id'], $room['room_id'], $room['requested_yn'], $room['requested_reason']);
                }
                if ($room['room_id'] == '0') {
                  ammendReservation($room['rv_reservation_id'], "Rooming (Deallocate)");	
                } else {
                  ammendReservation($room['rv_reservation_id'], "Rooming (Allocate [".getRoomDesc($room['room_id'])."])");	
                }
            }
        }

        foreach($rooms as $room) {
            if(array_key_exists("status_ind",$room)) {
                if(empty($room['res_item_group_id'])) {
                    return new JsonModel([
                        'status'=>false,
                        'errors'=>[
                            ['index'=>0, 'error'=>"Cannot check in/out itinerary without group"]
                        ]
                    ]);
                }
            }
        }

        foreach($rooms as $room) {
            if(array_key_exists("status_ind",$room)) {
                db_rv_res_item_group_set_status($room['res_item_group_id'],$room['status_ind'],$room['status_time']);
            }
        }

        return new JsonModel([
            'status'=>true
        ]);
    }
}
