<?php
namespace Resrequest\API\V1\Rpc\SystemInfo;

use Zend\Mvc\Controller\AbstractActionController;
use ZF\ContentNegotiation\ViewModel;

class SystemInfoController extends AbstractActionController
{
    protected $em, $password;
    private $params;

    public function __construct($em, $enterprise)
    {
        $this->em = $em;
        $this->password = $enterprise->passwords['systeminfo'];
        $this->shortName = $enterprise->config['shortName'];
        $this->principalId = $enterprise->config['principalId'];
        $this->databaseName = $enterprise->config['databaseName'];
        $this->environment = $enterprise->config['environment'];
        $this->archive = $enterprise->config['archive'];
        $this->codeVersion = $enterprise->config['codeVersion'];
    }

    private function calculateScore($months) {
        $score = 0;
        if ($months['month3'] > 0) {
            $score++;
        }
        if ($months['month3'] > $months['month6']) {
            $score++;
        }
        if (
            $months['month3'] > $months['month6'] && 
            $months['month3'] > $months['month9']
        ) {
            $score++;
        }
        return $score . "/3";
    }

    public function systemInfoAction()
    {
        $this->params = $this->bodyParams();

        $username = false;
        $identity = $this->getIdentity();
        if ($identity instanceof \ZF\MvcAuth\Identity\AuthenticatedIdentity) {
            $username = $identity->getAuthenticationIdentity()['user_id'];
        }

        if (
            $username === false &&
            (!isset($this->params['securityHash']) || $this->params['securityHash'] !== $this->password)
        ) {
            return new ViewModel(['data' => '', 'error' => 'Invalid security hash', 'code' => 401]); // Unauthorized
        }

        $query = $this->em->createQueryBuilder();
        $environmentListArray = $query
            ->select(
                [
                    "rfDatabase.rfDbCode AS environment"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfDatabase', 'rfDatabase')
            ->where('rfDatabase.rfDbInactiveYn = 0')
            ->andWhere('rfDatabase.rfDbCode != :wa')
            ->andWhere('rfDatabase.rfDbCode != :wb')
            ->setParameters(
                [
                    'wa' => "WA",
                    'wb' => "WB"
                ]
            )
            ->getQuery()
            ->getResult();
        
        $environmentList = array();
        foreach ($environmentListArray as $env) {
            $environmentList[] = $env['environment'];
        }
        $environmentList = join(',', $environmentList);

        $query = $this->em->createQueryBuilder();
        $environmentListArray = $query
            ->select(
                [
                    "rfDatabase.rfDbCode AS environment"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfDatabase', 'rfDatabase')
            ->where('rfDatabase.rfDbInactiveYn = 0')
            ->andWhere('rfDatabase.rfDbPosFolder IS NOT NULL')
            ->andWhere('rfDatabase.rfDbPosFolder != :empty')
            ->setParameters(
                [
                    'empty' => ""
                ]
            )
            ->getQuery()
            ->getResult();
        
        $posList = array();
        foreach ($environmentListArray as $env) {
            $posList[] = $env['environment'];
        }
        $posList = join(',', $posList);

        $query = $this->em->createQueryBuilder();
        $environmentListArray = $query
            ->select(
                [
                    "rfDatabase.rfDbCode AS environment"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfDatabase', 'rfDatabase')
            ->where('rfDatabase.rfDbInactiveYn = 0')
            ->andWhere('rfDatabase.rfDbPhoneFilename IS NOT NULL')
            ->andWhere('rfDatabase.rfDbPosFolder != :empty')
            ->setParameters(
                [
                    'empty' => ""
                ]
            )
            ->getQuery()
            ->getResult();
        
        $phoneList = array();
        foreach ($environmentListArray as $env) {
            $phoneList[] = $env['environment'];
        }
        $phoneList = join(',', $phoneList);

        $query = $this->em->createQueryBuilder();
        $fiListArray = $query
            ->select(
                [
                    "prPersona.prNameLast AS name",
                    "fnSystem.fnSystemDesc as fiDesc"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\PrBusiness', 'prBusiness')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\FnSystem', 'fnSystem', 'with', 'fnSystem.fnSystemIx = prBusiness.fnSystemId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona', 'with', 'prPersona.prPersonaIx = prBusiness.prBusinessId')
            ->getQuery()
            ->getResult();
        
        $fiList = array();
        $fiShortList = array();
        foreach ($fiListArray as $fi) {
            if (!empty($fi['fiDesc'])) {
                $fiList[] = $fi['name'] . ": " . $fi['fiDesc'];
                $fiShortList[] = $fi['name'];
            }
        }
        $fiList = join(',', $fiList);
        $fiShortList = join(',', $fiShortList);

        $query = $this->em->createQueryBuilder();
        $databaseVersion = $query
            ->select(
                [
                    "rfDatabase.rfDbVersionDb AS databaseVersion"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfDatabase', 'rfDatabase')
            ->where('rfDatabase.rfDbCode = :wb')
            ->setParameters(
                [
                    'wb' => "WB"
                ]
            )
            ->getQuery()
            ->getResult();
        $databaseVersion = $databaseVersion[0]['databaseVersion'];
        
        $query = $this->em->createQueryBuilder();
        $databaseSize = $query
            ->select(
                [
                    "rfDatabase.rfDbVersionDb AS databaseVersion"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfDatabase', 'rfDatabase')
            ->where('rfDatabase.rfDbCode = :wb')
            ->setParameters(
                [
                    'wb' => "WB"
                ]
            )
            ->getQuery()
            ->getResult();
        $databaseSize = $databaseSize[0]['databaseVersion'];
        
        $query = $this->em->createQueryBuilder();
        $databaseMaster = $query
            ->select(
                [
                    "rfSystem.rfSysDbMasterId AS system",
                    "rfDatabase.rfDatabaseId AS database"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfSystem', 'rfSystem')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfDatabase', 'rfDatabase', 'with', 'rfDatabase.rfDbCode = :thisEnv')
            ->setParameters(
                [
                    'thisEnv' => $this->environment
                ]
            )
            ->getQuery()
            ->getResult();
        $databaseMaster = $databaseMaster[0]['system'] == $databaseMaster[0]['database'] ? 1 : 0;

        $query = $this->em->createQueryBuilder();
        $principalName = $query
            ->select(
                [
                    "TRIM(CONCAT(prPersona.prNameFirst, ' ', prPersona.prNameLast)) AS name"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfDefault', 'rfDefault')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona', 'with', 'prPersona.prPersonaIx = rfDefault.rfPrincipalId')
            ->getQuery()
            ->getResult();
        $principalName = $principalName[0]['name'];

        $db = $this->em->getConnection();
        $databaseSize = $db->fetchColumn("
            SELECT
                SUM( data_length + index_length ) / 1024 / 1024 AS 'databaseSize'
            FROM
                information_schema.TABLES
            WHERE
                table_schema = 'cn_live_" . $this->principalId . "'
        ");

        $query = $this->em->createQueryBuilder();
        $propListArray = $query
            ->select(
                [
                    "prPersona.prNameLast AS name",
                    "prBusiness.prBusBillingYn as invoiceUnitRes",
                    "prBusiness.prBusBillingPropYn as invoiceUnitProp",
                    "prBusiness.prBusNameShort as nameShort",
                    "prBusiness.prBusInactiveYn as inactive",
                    "prBusiness.prBusPropLatitude as latitude",
                    "prBusiness.prBusPropLongitude as longitude"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\PrBusiness', 'prBusiness')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona', 'with', 'prPersona.prPersonaIx = prBusiness.prBusinessId')
            ->where('prBusiness.prBusPropertyYn = :yes')
            ->setParameters(
                    [
                        'yes' => 1
                    ]
                )
            ->getQuery()
            ->getResult();
        foreach ($propListArray AS $key => $prop) {
            $propListArray[$key]['latitude'] = is_numeric($prop['latitude']) ? $prop['latitude'] : "";
            $propListArray[$key]['longitude'] = is_numeric($prop['longitude']) ? $prop['longitude'] : "";
        }
        $propList = serialize($propListArray);

        $query = $this->em->createQueryBuilder();
        $rateListArray = $query
            ->select(
                [
                    "rtRateType.rtRateTypeDesc AS rate"
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RtRateType', 'rtRateType')
            ->where('rtRateType.rtRateTypeInactiveYn = :no')
            ->setParameters(
                    [
                        'no' => 0
                    ]
                )
            ->getQuery()
            ->getResult();
        $rateList = serialize($rateListArray);

        // Engagement Info

        $engagementInfo = array();
        $count = 0;
        if (isset($this->params['data']) && $this->params['data'] == 'engagementInfo') {

            $now = new \DateTime();
            $months3 = new \DateTime('-3 month');
            $months6 = new \DateTime('-6 month');
            $months9 = new \DateTime('-9 month');
            $months12 = new \DateTime('-12 month');
            $months15 = new \DateTime('-15 month');
            $months18 = new \DateTime('-18 month');
            $months21 = new \DateTime('-21 month');
            $months24 = new \DateTime('-24 month');

            $engagementInfo['dateRanges'] = array(
                "type" => "dateranges",
                "caption" => "Date Ranges",
                "ranges" => array(
                    'now' => $now->format('j M \'y'),
                    'months3' => $months3->format('j M \'y'),
                    'months6' => $months6->format('j M \'y'),
                    'months9' => $months9->format('j M \'y'),
                    'months12' => $months12->format('j M \'y'),
                    'months15' => $months15->format('j M \'y'),
                    'months18' => $months18->format('j M \'y'),
                    'months21' => $months21->format('j M \'y'),
                    'months24' => $months24->format('j M \'y')
                ),
                "order" => $count
            );
            $count++;

            // Extras Setup
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(acExtra.acExtraIx) AS extraCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\AcExtra', 'acExtra')
                ->getQuery()
                ->getResult();
            $engagementInfo['extrasSetup'] = array(
                "type" => "integer",
                "caption" => "Extras Setup",
                "value" => $result[0]['extraCount'],
                "score" => ($result[0]['extraCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Extras on the system.",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/extras/",
                "category" => "CRM"
            );
            $count++;

            // Extras Reservation
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvExtra.rvExtraIx) AS total"
                    ]
                )
                ->addSelect("SUM(CASE WHEN rvExtra.rvExtraTimestamp < :now AND rvExtra.rvExtraTimestamp > :months3 THEN 1 ELSE 0 END ) AS month3")
                ->addSelect("SUM( CASE WHEN rvExtra.rvExtraTimestamp < :months3 AND rvExtra.rvExtraTimestamp > :months6 THEN 1 ELSE 0 END ) AS month6")
                ->addSelect("SUM( CASE WHEN rvExtra.rvExtraTimestamp < :months6 AND rvExtra.rvExtraTimestamp > :months9 THEN 1 ELSE 0 END ) AS month9")
                ->addSelect("SUM( CASE WHEN rvExtra.rvExtraTimestamp < :months9 AND rvExtra.rvExtraTimestamp > :months12 THEN 1 ELSE 0 END ) AS month12")
                ->addSelect("SUM( CASE WHEN rvExtra.rvExtraTimestamp < :months12 AND rvExtra.rvExtraTimestamp > :months15 THEN 1 ELSE 0 END ) AS month15")
                ->addSelect("SUM( CASE WHEN rvExtra.rvExtraTimestamp < :months15 AND rvExtra.rvExtraTimestamp > :months18 THEN 1 ELSE 0 END ) AS month18")
                ->addSelect("SUM( CASE WHEN rvExtra.rvExtraTimestamp < :months18 AND rvExtra.rvExtraTimestamp > :months21 THEN 1 ELSE 0 END ) AS month21")
                ->addSelect("SUM( CASE WHEN rvExtra.rvExtraTimestamp < :months21 AND rvExtra.rvExtraTimestamp > :months24 THEN 1 ELSE 0 END ) AS month24")
                ->from('Resrequest\DB\Enterprise\Entity\RvExtra', 'rvExtra')
                ->where('rvExtra.rvExtraVoidInd = :unVoided')
                ->setParameters(
                    [
                        'now' => new \DateTime(),
                        'months3' => new \DateTime('-3 month'),
                        'months6' => new \DateTime('-6 month'),
                        'months9' => new \DateTime('-9 month'),
                        'months12' => new \DateTime('-12 month'),
                        'months15' => new \DateTime('-15 month'),
                        'months18' => new \DateTime('-18 month'),
                        'months21' => new \DateTime('-21 month'),
                        'months24' => new \DateTime('-24 month'),
                        'unVoided' => 0
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['extraReservations'] = array(
                "type" => "graph",
                "caption" => "Res Extras Created",
                "chartYAxis" => "Extras",
                "total" => array_shift($result[0]),
                "months" => $result[0],
                "score" => $this->calculateScore($result[0]),
                "order" => $count,
                "desc" => "Has Extras been used on reservations?",
                "learnOnline" => "https://support.resrequest.com/central-reservations/adding-extras/",
                "category" => "CRM"
            );
            $count++;

            // Reservations
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvReservation.rvReservationIx) AS total"
                    ]
                )
                ->addSelect("SUM(CASE WHEN rvReservation.rvDateRecorded < :now AND rvReservation.rvDateRecorded > :months3 THEN 1 ELSE 0 END ) AS month3")
                ->addSelect("SUM( CASE WHEN rvReservation.rvDateRecorded < :months3 AND rvReservation.rvDateRecorded > :months6 THEN 1 ELSE 0 END ) AS month6")
                ->addSelect("SUM( CASE WHEN rvReservation.rvDateRecorded < :months6 AND rvReservation.rvDateRecorded > :months9 THEN 1 ELSE 0 END ) AS month9")
                ->addSelect("SUM( CASE WHEN rvReservation.rvDateRecorded < :months9 AND rvReservation.rvDateRecorded > :months12 THEN 1 ELSE 0 END ) AS month12")
                ->addSelect("SUM( CASE WHEN rvReservation.rvDateRecorded < :months12 AND rvReservation.rvDateRecorded > :months15 THEN 1 ELSE 0 END ) AS month15")
                ->addSelect("SUM( CASE WHEN rvReservation.rvDateRecorded < :months15 AND rvReservation.rvDateRecorded > :months18 THEN 1 ELSE 0 END ) AS month18")
                ->addSelect("SUM( CASE WHEN rvReservation.rvDateRecorded < :months18 AND rvReservation.rvDateRecorded > :months21 THEN 1 ELSE 0 END ) AS month21")
                ->addSelect("SUM( CASE WHEN rvReservation.rvDateRecorded < :months21 AND rvReservation.rvDateRecorded > :months24 THEN 1 ELSE 0 END ) AS month24")
                ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
                ->setParameters(
                    [
                        'now' => new \DateTime(),
                        'months3' => new \DateTime('-3 month'),
                        'months6' => new \DateTime('-6 month'),
                        'months9' => new \DateTime('-9 month'),
                        'months12' => new \DateTime('-12 month'),
                        'months15' => new \DateTime('-15 month'),
                        'months18' => new \DateTime('-18 month'),
                        'months21' => new \DateTime('-21 month'),
                        'months24' => new \DateTime('-24 month')
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['reservations'] = array(
                "type" => "graph",
                "caption" => "Reservations Created",
                "chartYAxis" => "Reservations",
                "total" => array_shift($result[0]),
                "months" => $result[0],
                "score" => $this->calculateScore($result[0]),
                "order" => $count,
                "desc" => "Has Reservations been created on the system?",
                "learnOnline" => "https://support.resrequest.com/central-reservations/making-a-booking/",
                "category" => "CRM"
            );
            $count++;

            // Majority of Reservations by environment
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvReservation.rvReservationIx) AS webTotal"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
                ->where('rvReservation.rvReservationDb IN (:wa, :wb)')
                ->setParameters(
                    [
                        'wa' => "WA",
                        'wb' => "WB"
                    ]
                )
                ->getQuery()
                ->getResult();
            $web = $result[0]['webTotal'];

            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvReservation.rvReservationIx) AS nonWebTotal"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
                ->where('rvReservation.rvReservationDb NOT IN (:wa, :wb)')
                ->setParameters(
                    [
                        'wa' => "WA",
                        'wb' => "WB"
                    ]
                )
                ->getQuery()
                ->getResult();
            $nonWeb = $result[0]['nonWebTotal'];
            $engagementInfo['majorityWeb'] = array(
                "type" => "boolean",
                "caption" => "Majority res on Web",
                "value" => $web > $nonWeb ? true : false,
                "score" => "0/0",
                "order" => $count,
                "desc" => "Is the majority of reservations created on the Web?",
                "category" => "CRM"
            );
            $count++;

            // Emails
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(emMail.emMailIx) AS mailTotal"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\EmMail', 'emMail')
                ->getQuery()
                ->getResult();
            $engagementInfo['mailSending'] = array(
                "type" => "boolean",
                "caption" => "Mail Sending",
                "value" => $result[0]['mailTotal'] > 0 ? true : false,
                "score" => ($result[0]['mailTotal'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are emails being sent from the system?",
                "category" => "CRM"
            );
            $count++;

            // Standard notes
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rfNote.rfNoteIx AS noteSCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RfNote', 'rfNote')
                ->getQuery()
                ->getResult();
            $engagementInfo['notesStandard'] = array(
                "type" => "integer",
                "caption" => "Standard Notes",
                "value" => $result[0]['noteSCount'],
                "score" => ($result[0]['noteSCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Standard Notes on the system.",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/standard-notes/",
                "category" => "CRM"
            );
            $count++;

            // Custom notes
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rfNoteCustom.rfNoteCustomIx AS noteCCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RfNoteCustom', 'rfNoteCustom')
                ->getQuery()
                ->getResult();
            $engagementInfo['notesCustom'] = array(
                "type" => "integer",
                "caption" => "Custom Notes",
                "value" => $result[0]['noteCCount'],
                "score" => ($result[0]['noteCCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Custom Notes on the system.",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/custom-notes/",
                "category" => "CRM"
            );
            $count++;

            // Sources
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rfSource.rfSourceIx AS sourceCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RfSource', 'rfSource')
                ->where('rfSource.rfSourceInactiveYn = :notInactive')
                ->setParameters(
                    [
                        'notInactive' => 0
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['sources'] = array(
                "type" => "integer",
                "caption" => "Sources Setup",
                "value" => $result[0]['sourceCount'],
                "score" => ($result[0]['sourceCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Sources set up on the system.",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/source/",
                "category" => "CRM"
            );
            $count++;

            // Originator usage
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvReservation.rvReservationIx AS originatorCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
                ->where('rvReservation.rvOriginAgentId != :noOriginator')
                ->setParameters(
                    [
                        'noOriginator' => ''
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['originatorUsage'] = array(
                "type" => "boolean",
                "caption" => "Originator usage",
                "value" => $result[0]['originatorCount'] > 0 ? true : false,
                "score" => ($result[0]['originatorCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Is Originator being used on the system?",
                "category" => "CRM"
            );
            $count++;

            // Reservation Group usage
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvResItemGroup.rvResItemGroupIx AS groupCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvResItemGroup', 'rvResItemGroup')
                ->getQuery()
                ->getResult();
            $engagementInfo['reservationGroupUsage'] = array(
                "type" => "boolean",
                "caption" => "Reservation Group usage",
                "value" => $result[0]['groupCount'] > 0 ? true : false,
                "score" => ($result[0]['groupCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are Reservation Groups being used on the system?",
                "category" => "CRM"
            );
            $count++;

            // Rate Override Percentage
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvReservationItem.rvReservationItemIx AS nonOverrideCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'rvReservationItem')
                ->where('rvReservationItem.rvItemOverideLevelInd = :nonoverride')
                ->setParameters(
                    [
                        'nonoverride' => '0'
                    ]
                )
                ->getQuery()
                ->getResult();
            $nonOverriddenCount = $result[0]['nonOverrideCount'];

            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvReservationItem.rvReservationItemIx AS overrideCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'rvReservationItem')
                ->where('rvReservationItem.rvItemOverideLevelInd != :nonoverride')
                ->setParameters(
                    [
                        'nonoverride' => '0'
                    ]
                )
                ->getQuery()
                ->getResult();
            $overriddenCount = $result[0]['overrideCount'];
            $totalItins = $nonOverriddenCount + $overriddenCount;
            $overridePercentage = 100 / $totalItins * $overriddenCount;
            $engagementInfo['rateOverridePercentage'] = array(
                "type" => "string",
                "caption" => "Rate Override Percentage",
                "value" => ceil($overridePercentage) . "%",
                "score" => ($overridePercentage > 10 ? "0" : "1") . "/1",
                "order" => $count,
                "desc" => "Percentage of Reservation Rate Overrides.",
                "category" => "CRM"
            );
            $count++;

            // Wetu Setup
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "rfSystem.rfSysWetuUsername AS wetuName",
                        "rfSystem.rfSysWetuPassword AS wetuPass",
                        "rfSystem.rfSysWetuApiKey AS wetuKey"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RfSystem', 'rfSystem')
                ->getQuery()
                ->getResult();
            $credentialsPopulated = (
                !empty($result[0]['wetuName']) &&
                !empty($result[0]['wetuPass']) &&
                !empty($result[0]['wetuKey'])
            ) ? true : false;
            $engagementInfo['wetuSetup'] = array(
                "type" => "boolean",
                "caption" => "Wetu Setup",
                "value" => $credentialsPopulated,
                "score" => ($credentialsPopulated ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Has Wetu Username, Password and API Key been set up on system?",
                "learnOnline" => "https://support.resrequest.com/central-reservations/wetu-user-setup-guide/",
                "category" => "CM"
            );
            $count++;

            // Wetu Reservations
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvReservation.rvWetuId) AS total"
                    ]
                )
                ->addSelect("SUM(CASE WHEN rvReservation.rvWetuTimestamp < :now AND rvReservation.rvWetuTimestamp > :months3 THEN 1 ELSE 0 END ) AS month3")
                ->addSelect("SUM( CASE WHEN rvReservation.rvWetuTimestamp < :months3 AND rvReservation.rvWetuTimestamp > :months6 THEN 1 ELSE 0 END ) AS month6")
                ->addSelect("SUM( CASE WHEN rvReservation.rvWetuTimestamp < :months6 AND rvReservation.rvWetuTimestamp > :months9 THEN 1 ELSE 0 END ) AS month9")
                ->addSelect("SUM( CASE WHEN rvReservation.rvWetuTimestamp < :months9 AND rvReservation.rvWetuTimestamp > :months12 THEN 1 ELSE 0 END ) AS month12")
                ->addSelect("SUM( CASE WHEN rvReservation.rvWetuTimestamp < :months12 AND rvReservation.rvWetuTimestamp > :months15 THEN 1 ELSE 0 END ) AS month15")
                ->addSelect("SUM( CASE WHEN rvReservation.rvWetuTimestamp < :months15 AND rvReservation.rvWetuTimestamp > :months18 THEN 1 ELSE 0 END ) AS month18")
                ->addSelect("SUM( CASE WHEN rvReservation.rvWetuTimestamp < :months18 AND rvReservation.rvWetuTimestamp > :months21 THEN 1 ELSE 0 END ) AS month21")
                ->addSelect("SUM( CASE WHEN rvReservation.rvWetuTimestamp < :months21 AND rvReservation.rvWetuTimestamp > :months24 THEN 1 ELSE 0 END ) AS month24")
                ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
                ->where('rvReservation.rvWetuId IS NOT NULL')
                ->setParameters(
                    [
                        'now' => new \DateTime(),
                        'months3' => new \DateTime('-3 month'),
                        'months6' => new \DateTime('-6 month'),
                        'months9' => new \DateTime('-9 month'),
                        'months12' => new \DateTime('-12 month'),
                        'months15' => new \DateTime('-15 month'),
                        'months18' => new \DateTime('-18 month'),
                        'months21' => new \DateTime('-21 month'),
                        'months24' => new \DateTime('-24 month')
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['wetuReservations'] = array(
                "type" => "graph",
                "caption" => "Wetu Itins Generated",
                "chartYAxis" => "Wetu Itineraries",
                "total" => array_shift($result[0]),
                "months" => $result[0],
                "score" => $this->calculateScore($result[0]),
                "order" => $count,
                "desc" => "Has Wetu itineraries been generated from reservations?",
                "category" => "CM"
            );
            $count++;

            // Contacts
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(prPersona.prPersonaIx) AS personaCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona')
                ->where('prPersona.prOrgYn = :individual')
                ->setParameters(
                    [
                        'individual' => 0
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['contacts'] = array(
                "type" => "integer",
                "caption" => "Contacts",
                "value" => $result[0]['personaCount'],
                "score" => ($result[0]['personaCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Contacts on the system.",
                "learnOnline" => "https://support.resrequest.com/customer-relationship-management/resrequest-contact-components/",
                "category" => "CR"
            );
            $count++;

            // Guest Surveys Setup
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(zgAccount.zgAccountIx) AS surveyCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\ZgAccount', 'zgAccount')
                ->getQuery()
                ->getResult();
            $engagementInfo['guestSurveysSetup'] = array(
                "type" => "integer",
                "caption" => "Surveys Setup",
                "value" => $result[0]['surveyCount'],
                "score" => ($result[0]['surveyCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Guest Surveys on the system.",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/guest-survey/",
                "category" => "CR"
            );
            $count++;

            // Guest Surveys Sent
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(zgTransfer.zgTransferIx) AS total"
                    ]
                )
                ->addSelect("SUM(CASE WHEN zgTransfer.zgTrfDateOpen < :now AND zgTransfer.zgTrfDateOpen > :months3 THEN 1 ELSE 0 END ) AS month3")
                ->addSelect("SUM( CASE WHEN zgTransfer.zgTrfDateOpen < :months3 AND zgTransfer.zgTrfDateOpen > :months6 THEN 1 ELSE 0 END ) AS month6")
                ->addSelect("SUM( CASE WHEN zgTransfer.zgTrfDateOpen < :months6 AND zgTransfer.zgTrfDateOpen > :months9 THEN 1 ELSE 0 END ) AS month9")
                ->addSelect("SUM( CASE WHEN zgTransfer.zgTrfDateOpen < :months9 AND zgTransfer.zgTrfDateOpen > :months12 THEN 1 ELSE 0 END ) AS month12")
                ->addSelect("SUM( CASE WHEN zgTransfer.zgTrfDateOpen < :months12 AND zgTransfer.zgTrfDateOpen > :months15 THEN 1 ELSE 0 END ) AS month15")
                ->addSelect("SUM( CASE WHEN zgTransfer.zgTrfDateOpen < :months15 AND zgTransfer.zgTrfDateOpen > :months18 THEN 1 ELSE 0 END ) AS month18")
                ->addSelect("SUM( CASE WHEN zgTransfer.zgTrfDateOpen < :months18 AND zgTransfer.zgTrfDateOpen > :months21 THEN 1 ELSE 0 END ) AS month21")
                ->addSelect("SUM( CASE WHEN zgTransfer.zgTrfDateOpen < :months21 AND zgTransfer.zgTrfDateOpen > :months24 THEN 1 ELSE 0 END ) AS month24")
                ->from('Resrequest\DB\Enterprise\Entity\ZgTransfer', 'zgTransfer')
                ->setParameters(
                    [
                        'now' => new \DateTime(),
                        'months3' => new \DateTime('-3 month'),
                        'months6' => new \DateTime('-6 month'),
                        'months9' => new \DateTime('-9 month'),
                        'months12' => new \DateTime('-12 month'),
                        'months15' => new \DateTime('-15 month'),
                        'months18' => new \DateTime('-18 month'),
                        'months21' => new \DateTime('-21 month'),
                        'months24' => new \DateTime('-24 month')
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['guestSurveysSent'] = array(
                "type" => "graph",
                "caption" => "Surveys Sent",
                "chartYAxis" => "Guest Surveys",
                "total" => array_shift($result[0]),
                "months" => $result[0],
                "score" => $this->calculateScore($result[0]),
                "order" => $count,
                "desc" => "Has Guest Surveys been sent from the system?",
                "category" => "CR"
            );
            $count++;

            // Guest Nationality usage
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(prGuest.prGuestId AS nationalityCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PrGuest', 'prGuest')
                ->where('prGuest.rfCountryId != :noCountry')
                ->setParameters(
                    [
                        'noCountry' => '0'
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['guestNationalityUsage'] = array(
                "type" => "boolean",
                "caption" => "Guest Nationality usage",
                "value" => $result[0]['nationalityCount'] > 0 ? true : false,
                "score" => ($result[0]['nationalityCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Is Guest Nationality being used on the system?",
                "category" => "CR"
            );
            $count++;

            // Guest profile usage
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(prGuestObjData.prGuestObjDataIx AS guestCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PrGuestObjData', 'prGuestObjData')
                ->getQuery()
                ->getResult();
            $engagementInfo['reservationGroupUsage'] = array(
                "type" => "boolean",
                "caption" => "Guest Profile usage",
                "value" => $result[0]['guestCount'] > 0 ? true : false,
                "score" => ($result[0]['guestCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are Guest Profiles being used on the system?",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/guest-relationship-management/#guestprofile",
                "category" => "CR"
            );
            $count++;

            // Usage time, Years on system
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "rvReservation.rvDateRecorded AS resDate"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
                ->addOrderBy('rvReservation.rvDateRecorded', 'ASC')
                ->setMaxResults(1)
                ->getQuery()
                ->getResult();
            $date = $result[0]['resDate'];
            $now = new \DateTime();
            $interval = $date->diff($now);
            $score = 0;
            if ($interval->format('%y') == 1) { $score = 1; }
            if ($interval->format('%y') == 2) { $score = 2; }
            if ($interval->format('%y') > 2) { $score = 3; }

            $engagementInfo['usageTime'] = array(
                "type" => "string",
                "caption" => "Usage Time",
                "value" => $interval->format('%y years %m months'),
                "score" => $score . "/3",
                "order" => $count,
                "desc" => "How long has the system been in use, based on first reservation created?",
                "category" => "BA"
            );
            $count++;
            
            // Google Analytics
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "rfDefault.rfAnalyticsCode AS analyticsCode"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RfDefault', 'rfDefault')
                ->getQuery()
                ->getResult();
            $engagementInfo['googleAnalytics'] = array(
                "type" => "boolean",
                "caption" => "Google Analytics",
                "value" => (!empty($result[0]['analyticsCode'])) ? true : false,
                "score" => ((!empty($result[0]['analyticsCode'])) ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Has Google Analytics been set up on the system?",
                "learnOnline" => "https://support.resrequest.com/system-setup/system-configurations/internet-marketing/",
                "category" => "BA"
            );
            $count++;

            // Report Profiles
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "pfObject.pfObjectIx AS object"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PfObject', 'pfObject')
                ->where('pfObject.pfObjectName IN (:doc, :proforma, :invoice, :pref, :template)')
                ->setParameters(
                    [
                        'doc' => "email_document",
                        'proforma' => "email_proforma",
                        'invoice' => "email_invoice",
                        'pref' => 'user_preferences',
                        'template' => 'email_templates'
                    ]
                )
                ->getQuery()
                ->getResult();

            $objectIds = array();
            foreach ($result as $object) {
                $objectIds[] = $object['object'];
            }

            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(pfOption.pfOptionIx AS reportProfilesCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PfOption', 'pfOption')
                ->where('pfOption.pfObjectId NOT IN (:objectTypes)')
                ->setParameters(
                    [
                        'objectTypes' => $objectIds
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['reportProfiles'] = array(
                "type" => "integer",
                "caption" => "Report Profiles",
                "value" => $result[0]['reportProfilesCount'],
                "score" => ($result[0]['reportProfilesCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Report Profiles set up on the system.",
                "category" => "BA"
            );
            $count++;

            // FAR and Ops Chart Report Profiles
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "pfObject.pfObjectIx AS object"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PfObject', 'pfObject')
                ->where('pfObject.pfObjectName IN (:farday, :farfolio, :ops)')
                ->setParameters(
                    [
                        'farday' => "financial_analysis_day",
                        'farfolio' => "financial_analysis_folio",
                        'ops' => "operation_chart"
                    ]
                )
                ->getQuery()
                ->getResult();

            $objectIds = array();
            foreach ($result as $object) {
                $objectIds[] = $object['object'];
            }
            
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(pfOption.pfOptionIx AS reportProfilesCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PfOption', 'pfOption')
                ->where('pfOption.pfObjectId NOT IN (:objectTypes)')
                ->setParameters(
                    [
                        'objectTypes' => $objectIds
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['farAndOpsChartReportProfiles'] = array(
                "type" => "integer",
                "caption" => "FAR and Ops Chart Report Profiles",
                "value" => $result[0]['reportProfilesCount'],
                "score" => ($result[0]['reportProfilesCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Report Profiles set up on the system for the FAR and Ops Chart.",
                "category" => "BA"
            );
            $count++;

            // Specials Setup
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(spSpecial.spSpecialIx) AS specialCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\SpSpecial', 'spSpecial')
                ->getQuery()
                ->getResult();
            $engagementInfo['specialsSetup'] = array(
                "type" => "integer",
                "caption" => "Specials Setup",
                "value" => $result[0]['specialCount'],
                "score" => ($result[0]['specialCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "How many Specials have been set up on the system?",
                "learnOnline" => "https://support.resrequest.com/system-setup/rates/how-to-set-up-specials/",
                "category" => "CRM"
            );
            $count++;

            // Specials Application
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvSpecial.rvSpecialIx) AS total"
                    ]
                )
                ->addSelect("SUM(CASE WHEN rvSpecial.adCreateDate < :now AND rvSpecial.adCreateDate > :months3 THEN 1 ELSE 0 END ) AS month3")
                ->addSelect("SUM( CASE WHEN rvSpecial.adCreateDate < :months3 AND rvSpecial.adCreateDate > :months6 THEN 1 ELSE 0 END ) AS month6")
                ->addSelect("SUM( CASE WHEN rvSpecial.adCreateDate < :months6 AND rvSpecial.adCreateDate > :months9 THEN 1 ELSE 0 END ) AS month9")
                ->addSelect("SUM( CASE WHEN rvSpecial.adCreateDate < :months9 AND rvSpecial.adCreateDate > :months12 THEN 1 ELSE 0 END ) AS month12")
                ->addSelect("SUM( CASE WHEN rvSpecial.adCreateDate < :months12 AND rvSpecial.adCreateDate > :months15 THEN 1 ELSE 0 END ) AS month15")
                ->addSelect("SUM( CASE WHEN rvSpecial.adCreateDate < :months15 AND rvSpecial.adCreateDate > :months18 THEN 1 ELSE 0 END ) AS month18")
                ->addSelect("SUM( CASE WHEN rvSpecial.adCreateDate < :months18 AND rvSpecial.adCreateDate > :months21 THEN 1 ELSE 0 END ) AS month21")
                ->addSelect("SUM( CASE WHEN rvSpecial.adCreateDate < :months21 AND rvSpecial.adCreateDate > :months24 THEN 1 ELSE 0 END ) AS month24")
                ->from('Resrequest\DB\Enterprise\Entity\RvSpecial', 'rvSpecial')
                ->setParameters(
                    [
                        'now' => new \DateTime(),
                        'months3' => new \DateTime('-3 month'),
                        'months6' => new \DateTime('-6 month'),
                        'months9' => new \DateTime('-9 month'),
                        'months12' => new \DateTime('-12 month'),
                        'months15' => new \DateTime('-15 month'),
                        'months18' => new \DateTime('-18 month'),
                        'months21' => new \DateTime('-21 month'),
                        'months24' => new \DateTime('-24 month')
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['specialsApplication'] = array(
                "type" => "graph",
                "caption" => "Specials Applied",
                "chartYAxis" => "Specials",
                "total" => array_shift($result[0]),
                "months" => $result[0],
                "score" => $this->calculateScore($result[0]),
                "order" => $count,
                "desc" => "Has Specials been applied to reservations?",
                "category" => "CRM"
            );
            $count++;

            // Invoices
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(fnInvoice.fnInvoiceIx) AS total"
                    ]
                )
                ->addSelect("SUM(CASE WHEN fnInvoice.fnInvDate < :now AND fnInvoice.fnInvDate > :months3 THEN 1 ELSE 0 END ) AS month3")
                ->addSelect("SUM( CASE WHEN fnInvoice.fnInvDate < :months3 AND fnInvoice.fnInvDate > :months6 THEN 1 ELSE 0 END ) AS month6")
                ->addSelect("SUM( CASE WHEN fnInvoice.fnInvDate < :months6 AND fnInvoice.fnInvDate > :months9 THEN 1 ELSE 0 END ) AS month9")
                ->addSelect("SUM( CASE WHEN fnInvoice.fnInvDate < :months9 AND fnInvoice.fnInvDate > :months12 THEN 1 ELSE 0 END ) AS month12")
                ->addSelect("SUM( CASE WHEN fnInvoice.fnInvDate < :months12 AND fnInvoice.fnInvDate > :months15 THEN 1 ELSE 0 END ) AS month15")
                ->addSelect("SUM( CASE WHEN fnInvoice.fnInvDate < :months15 AND fnInvoice.fnInvDate > :months18 THEN 1 ELSE 0 END ) AS month18")
                ->addSelect("SUM( CASE WHEN fnInvoice.fnInvDate < :months18 AND fnInvoice.fnInvDate > :months21 THEN 1 ELSE 0 END ) AS month21")
                ->addSelect("SUM( CASE WHEN fnInvoice.fnInvDate < :months21 AND fnInvoice.fnInvDate > :months24 THEN 1 ELSE 0 END ) AS month24")
                ->from('Resrequest\DB\Enterprise\Entity\FnInvoice', 'fnInvoice')
                ->where('fnInvoice.fnInvVoidYn = :unVoided AND fnInvoice.fnInvReverseLinkId IS NULL')
                ->setParameters(
                    [
                        'now' => new \DateTime(),
                        'months3' => new \DateTime('-3 month'),
                        'months6' => new \DateTime('-6 month'),
                        'months9' => new \DateTime('-9 month'),
                        'months12' => new \DateTime('-12 month'),
                        'months15' => new \DateTime('-15 month'),
                        'months18' => new \DateTime('-18 month'),
                        'months21' => new \DateTime('-21 month'),
                        'months24' => new \DateTime('-24 month'),
                        'unVoided' => 0
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['invoices'] = array(
                "type" => "graph",
                "caption" => "Invoices Created",
                "chartYAxis" => "Invoices",
                "total" => array_shift($result[0]),
                "months" => $result[0],
                "score" => $this->calculateScore($result[0]),
                "order" => $count,
                "desc" => "Has Invoices been generated on the system?",
                "learnOnline" => "https://support.resrequest.com/central-reservations/invoicing/",
                "category" => "FM"
            );
            $count++;

            // Payments
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvPaymentItem.rvPaymentItemIx) AS total"
                    ]
                )
                ->addSelect("SUM(CASE WHEN rvPaymentItem.rvPayItemTimestamp < :now AND rvPaymentItem.rvPayItemTimestamp > :months3 THEN 1 ELSE 0 END ) AS month3")
                ->addSelect("SUM( CASE WHEN rvPaymentItem.rvPayItemTimestamp < :months3 AND rvPaymentItem.rvPayItemTimestamp > :months6 THEN 1 ELSE 0 END ) AS month6")
                ->addSelect("SUM( CASE WHEN rvPaymentItem.rvPayItemTimestamp < :months6 AND rvPaymentItem.rvPayItemTimestamp > :months9 THEN 1 ELSE 0 END ) AS month9")
                ->addSelect("SUM( CASE WHEN rvPaymentItem.rvPayItemTimestamp < :months9 AND rvPaymentItem.rvPayItemTimestamp > :months12 THEN 1 ELSE 0 END ) AS month12")
                ->addSelect("SUM( CASE WHEN rvPaymentItem.rvPayItemTimestamp < :months12 AND rvPaymentItem.rvPayItemTimestamp > :months15 THEN 1 ELSE 0 END ) AS month15")
                ->addSelect("SUM( CASE WHEN rvPaymentItem.rvPayItemTimestamp < :months15 AND rvPaymentItem.rvPayItemTimestamp > :months18 THEN 1 ELSE 0 END ) AS month18")
                ->addSelect("SUM( CASE WHEN rvPaymentItem.rvPayItemTimestamp < :months18 AND rvPaymentItem.rvPayItemTimestamp > :months21 THEN 1 ELSE 0 END ) AS month21")
                ->addSelect("SUM( CASE WHEN rvPaymentItem.rvPayItemTimestamp < :months21 AND rvPaymentItem.rvPayItemTimestamp > :months24 THEN 1 ELSE 0 END ) AS month24")
                ->from('Resrequest\DB\Enterprise\Entity\RvPaymentItem', 'rvPaymentItem')
                ->where('rvPaymentItem.rvPaymentContraId IS NULL')
                ->setParameters(
                    [
                        'now' => new \DateTime(),
                        'months3' => new \DateTime('-3 month'),
                        'months6' => new \DateTime('-6 month'),
                        'months9' => new \DateTime('-9 month'),
                        'months12' => new \DateTime('-12 month'),
                        'months15' => new \DateTime('-15 month'),
                        'months18' => new \DateTime('-18 month'),
                        'months21' => new \DateTime('-21 month'),
                        'months24' => new \DateTime('-24 month')
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['payments'] = array(
                "type" => "graph",
                "caption" => "Payments Captured",
                "chartYAxis" => "Payments",
                "total" => array_shift($result[0]),
                "months" => $result[0],
                "score" => $this->calculateScore($result[0]),
                "order" => $count,
                "desc" => "Has Payments been generated on the system?",
                "learnOnline" => "https://support.resrequest.com/central-reservations/making-payments/",
                "category" => "FM"
            );
            $count++;

            // Financial interfaces used
            $engagementInfo['fiUsage'] = array(
                "type" => "boolean",
                "caption" => "Fin Interface Setup",
                "value" => !empty($fiList) ? true : false,
                "score" => (!empty($fiList) ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are Financial interfaces used on the system.",
                "category" => "FM"
            );
            $count++;

            // Multiple invoicing units (res and property)
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(prBusiness.prBusinessId) AS invoiceTotal"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PrBusiness', 'prBusiness')
                ->where('prBusiness.prBusBillingYn = :isResInvUnit')
                ->andWhere('prBusiness.prBusBillingPropYn = :isPropInvUnit')
                ->andWhere('prBusiness.prBusInactiveYn = :notInactive')
                ->setParameters(
                    [
                        'isResInvUnit' => 1,
                        'isPropInvUnit' => 1,
                        'notInactive' => 0
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['multiInvoicing'] = array(
                "type" => "boolean",
                "caption" => "Multi Invoicing",
                "value" => $result[0]['invoiceTotal'] > 0 ? true : false,
                "score" => "0/0",
                "order" => $count,
                "desc" => "Are there multiple invoicing units on the system?",
                "category" => "FM"
            );
            $count++;

            // Multiple currencies (reservation currencies)
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvReservation.rvInvoiceCurrencyId) AS currencyTotal"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'rvReservation')
                ->groupBy('rvReservation.rvInvoiceCurrencyId')
                ->getQuery()
                ->getResult();
            $engagementInfo['multiCurrency'] = array(
                "type" => "boolean",
                "caption" => "Multi Currency",
                "value" => count($result) > 0 ? true : false,
                "score" => "0/0",
                "order" => $count,
                "desc" => "Are there multiple currencies on the system?",
                "category" => "FM"
            );
            $count++;

            // Financial Interfaces
            $engagementInfo['financialInterfaces'] = array(
                "type" => "csvlist",
                "caption" => "Financial Interfaces",
                "value" => !empty($fiList) ? $fiList : "",
                "score" => "0/0",
                "order" => $count,
                "desc" => "List of properties using Financial Interfaces.",
                "category" => "FM"
            );
            $count++;

            // Invoicing at the lodge
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(fnInvoice.fnInvoiceIx AS invoiceCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\FnInvoice', 'fnInvoice')
                ->where('fnInvoice.fnInvoiceDb != :wa AND fnInvoice.fnInvoiceDb != :wb')
                ->setParameters(
                    [
                        'wa' => 'WA',
                        'wb' => 'WB'
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['invoicingAtTheLodge'] = array(
                "type" => "boolean",
                "caption" => "Lodge Invoicing",
                "value" => $result[0]['invoiceCount'] > 0 ? true : false,
                "score" => ($result[0]['invoiceCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are Invoices being generated at the lodge (non-web environments)?",
                "category" => "PM"
            );
            $count++;

            // Payments at the lodge
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvPayment.rvPaymentIx AS paymentCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvPayment', 'rvPayment')
                ->where('rvPayment.rvPaymentDb != :wa AND rvPayment.rvPaymentDb != :wb')
                ->setParameters(
                    [
                        'wa' => 'WA',
                        'wb' => 'WB'
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['paymentsAtTheLodge'] = array(
                "type" => "boolean",
                "caption" => "Lodge Payments",
                "value" => $result[0]['paymentCount'] > 0 ? true : false,
                "score" => ($result[0]['paymentCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are Payments being captured at the lodge (non-web environments)?",
                "category" => "PM"
            );
            $count++;

            // POS interfaces used
            $engagementInfo['posUsage'] = array(
                "type" => "boolean",
                "caption" => "Point of Sale Setup",
                "value" => !empty($posList) ? true : false,
                "score" => (!empty($posList) ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Point Of Sale interfaces on the system.",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/pos/",
                "category" => "PM"
            );
            $count++;

            // Phone interfaces used
            $engagementInfo['phoneUsage'] = array(
                "type" => "boolean",
                "caption" => "Phone Setup",
                "value" => !empty($phoneList) ? true : false,
                "score" => "0/0",
                "order" => $count,
                "desc" => "Are Phone interfaces used on the system.",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/telephone/",
                "category" => "PM"
            );
            $count++;

            // Web only
            $engagementInfo['webOnly'] = array(
                "type" => "boolean",
                "caption" => "Web Only",
                "value" => empty($environmentList) ? true : false,
                "score" => "0/0",
                "order" => $count,
                "desc" => "Is the system Web-only (no offlines)?",
                "category" => "A"
            );
            $count++;

            // Number of offlines
            $environments = explode(",", $environmentList);
            $engagementInfo['numberOfOfflines'] = array(
                "type" => "integer",
                "caption" => "Number of Offlines",
                "value" => count($environments),
                "score" => "0/0",
                "order" => $count,
                "desc" => "Number of Offlines on the system.",
                "category" => "PM"
            );
            $count++;

            // Multiple properties
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(prBusiness.prBusinessId) AS propTotal"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PrBusiness', 'prBusiness')
                ->where('prBusiness.prBusPropertyYn = :isProperty')
                ->andWhere('prBusiness.prBusInactiveYn = :notInactive')
                ->setParameters(
                    [
                        'isProperty' => 1,
                        'notInactive' => 0
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['multiProperty'] = array(
                "type" => "boolean",
                "caption" => "Multi Property",
                "value" => $result[0]['propTotal'] > 0 ? true : false,
                "score" => "0/0",
                "order" => $count,
                "desc" => "Are there multiple properties on the system?",
                "category" => "PM"
            );
            $count++;

            // Room Allocation usage
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvResItemGroup.rvResItemGroupIx AS groupCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvResItemGroup', 'rvResItemGroup')
                ->where('rvResItemGroup.acAccommRoomId IS NOT NULL OR rvResItemGroup.acAccommRoomId != :empty')
                ->setParameters(
                    [
                        'empty' => ''
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['roomAllocationUsage'] = array(
                "type" => "boolean",
                "caption" => "Room Allocation usage",
                "value" => $result[0]['groupCount'] > 0 ? true : false,
                "score" => ($result[0]['groupCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Is Room Allocation (accomodation override) being used on the system?",
                "category" => "PM"
            );
            $count++;

            // Check-in and -out usage
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvResItemGroup.rvResItemGroupIx AS groupCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvResItemGroup', 'rvResItemGroup')
                ->where('rvResItemGroup.rvGrpStatusInd = :checkedin OR rvResItemGroup.rvGrpStatusInd = :checkedout')
                ->setParameters(
                    [
                        'checkedin' => '5',
                        'checkedout' => '8'
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['checkinoutUsage'] = array(
                "type" => "boolean",
                "caption" => "Check-in and -out usage",
                "value" => $result[0]['groupCount'] > 0 ? true : false,
                "score" => ($result[0]['groupCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are rooms being checked in and out?",
                "category" => "PM"
            );
            $count++;

            // Guest tripsheet usage
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rvTripObjData.rvTripObjDataIx AS tripCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvTripObjData', 'rvTripObjData')
                ->getQuery()
                ->getResult();
            $engagementInfo['guestTripsheetUsage'] = array(
                "type" => "boolean",
                "caption" => "Guest Tripsheet usage",
                "value" => $result[0]['tripCount'] > 0 ? true : false,
                "score" => ($result[0]['tripCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are Guest Tripsheets being used on the system?",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/guest-relationship-management/#guesttripsheet",
                "category" => "CRM"
            );
            $count++;

            // Cash Points
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rfBank.rfBankIx AS cashpointCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RfBank', 'rfBank')
                ->where('rfBank.rfBankInd = :cashpoint')
                ->setParameters(
                    [
                        'cashpoint' => '2'
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['cashPoints'] = array(
                "type" => "integer",
                "caption" => "Cash Points",
                "value" => $result[0]['cashpointCount'],
                "score" => ($result[0]['cashpointCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Cash Points set up on the system.",
                "learnOnline" => "https://support.resrequest.com/system-setup/code-tables/bank-cash-point/",
                "category" => "PM"
            );
            $count++;

            // Automated Data Transfers
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rfDatabase.rfDatabaseId) AS adtTotal"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RfDatabase', 'rfDatabase')
                ->where('rfDatabase.rfDbTrfAutoYn = :adtYes')
                ->andWhere('rfDatabase.rfDbInactiveYn = :notInactive')
                ->setParameters(
                    [
                        'adtYes' => 1,
                        'notInactive' => 0
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['automatedDataTransfers'] = array(
                "type" => "boolean",
                "caption" => "Auto Data Transfers",
                "value" => $result[0]['adtTotal'] > 0 ? true : false,
                "score" => ($result[0]['adtTotal'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are there offlines with Automated Data Transfers on the system?",
                "category" => "A"
            );
            $count++;

            // Auto CC
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "pfObject.pfObjectIx AS object"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PfObject', 'pfObject')
                ->where('pfObject.pfObjectName IN (:doc, :proforma, :invoice)')
                ->setParameters(
                    [
                        'doc' => "email_document",
                        'proforma' => "email_proforma",
                        'invoice' => "email_invoice"
                    ]
                )
                ->getQuery()
                ->getResult();

            $objectIds = array();
            foreach ($result as $object) {
                $objectIds[] = $object['object'];
            }

            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(pfOption.pfOptionIx AS optionCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PfOption', 'pfOption')
                ->where('pfOption.pfObjectId IN (:objectTypes)')
                ->setParameters(
                    [
                        'objectTypes' => $objectIds
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['autoCc'] = array(
                "type" => "integer",
                "caption" => "Auto CC Setup",
                "value" => $result[0]['optionCount'],
                "score" => ($result[0]['optionCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Auto CC email addresses on the system.",
                "category" => "A"
            );
            $count++;

            // Mail Templates
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(pfField.pfFieldIx AS templateCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\PfField', 'pfField')
                ->where('pfField.pfFieldName = :template')
                ->setParameters(
                    [
                        'template' => 'templateText'
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['mailTemplates'] = array(
                "type" => "integer",
                "caption" => "Mail Templates",
                "value" => $result[0]['templateCount'],
                "score" => ($result[0]['templateCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Mail Templates set up on the system.",
                "category" => "CRM"
            );
            $count++;

            // Mail attachments
            $attachmentDir = __DIR__ . "/../../../../../../../../../../images/" . $this->principalId . "/attachments";
            $hasAttachments = false;
            if (is_dir($attachmentDir)) {
                $files = scandir($attachmentDir);
                if (count($files) > 2) {
                    $hasAttachments = true;
                }
            }
            $engagementInfo['mailAttachments'] = array(
                "type" => "boolean",
                "caption" => "Mail Attachments",
                "value" => $hasAttachments,
                "score" => ($hasAttachments ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Are Mail attachments present on the system?",
                "category" => "CRM"
            );
            $count++;

            // Bank Accounts
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "COUNT(rfBank.rfBankIx AS bankaccountCount"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RfBank', 'rfBank')
                ->where('rfBank.rfBankInd = :bankaccount')
                ->setParameters(
                    [
                        'bankaccount' => '0'
                    ]
                )
                ->getQuery()
                ->getResult();
            $engagementInfo['bankAccounts'] = array(
                "type" => "integer",
                "caption" => "Bank Accounts",
                "value" => $result[0]['bankaccountCount'],
                "score" => ($result[0]['bankaccountCount'] > 0 ? "1" : "0") . "/1",
                "order" => $count,
                "desc" => "Number of Bank Accounts set up on the system.",
                "category" => "FM"
            );
            $count++;

            // API Connections
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select(
                    [
                        "MAX(adApi.adApiTime) AS Date_Time",
                        "TRIM(CONCAT(prPersona.prNameFirst, ' ', prPersona.prNameLast)) AS Name"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\AdApi', 'adApi')
                ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona', 'with', 'prPersona.prPersonaIx = adApi.prUserId')
                ->groupBy('adApi.prUserId')
                ->addOrderBy('adApi.adApiTime', 'ASC')
                ->getQuery()
                ->getResult();

            $connList = array();
            $connList[] = '"CONNECTION PERSONA NAME","LAST CONNECTION TIME"';
            foreach ($result as $conn) {
                $connList[] = '"' . $conn['Name'] . '","' . $conn['Date_Time'] . '"';
            }
            $connList = join('\n', $connList);

            $engagementInfo['apiConnections'] = array(
                "type" => "csvlist",
                "displayFormat" => "table",
                "caption" => "API Connections",
                "value" => !empty($connList) ? $connList : "",
                "score" => "0/0",
                "order" => $count,
                "desc" => "List of API connections.",
                "category" => "CM"
            );
            $count++;

            // Tourism.Today
            $query = $this->em->createQueryBuilder();
            $result = $query
                ->select('system.rfSysTourismTodayAccessInd')
                ->from('Resrequest\DB\Enterprise\Entity\RfSystem', 'system')
                ->getQuery()
                ->getSingleScalarResult();

            switch ($result) {
                case "1":
                    $value = "Free Access";
                    break;
                case "2":
                    $value = "Pro Access";
                    break;
                default:
                    $value = "No Access";
            }
            $engagementInfo['tourismToday'] = array(
                "type" => "string",
                "caption" => "Tourism.Today",
                "value" => $value,
                "score" => $result . "/2",
                "order" => $count,
                "desc" => "Has Tourism.Today access been activated on the system.",
                "category" => "BA"
            );
            $count++;

        }

        // API Connections Info

        $apiConnectionsInfo = array();
        if (isset($this->params['data']) && $this->params['data'] == 'apiConnectionsInfo') {
            $query = $this->em->createQueryBuilder();
            $apiConnectionsInfo = $query
                ->select(
                    [
                        "adApi.prUserId AS Id",
                        "MAX(adApi.adApiTime) AS Date_Time",
                        "TRIM(CONCAT(prPersona.prNameFirst, ' ', prPersona.prNameLast)) AS Name",
                        "prUser.prUserName AS Username",
                        "prUser.prUserInactiveYn AS Active",
                        "scGroup.scGrpDesc AS AccessGroup"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\AdApi', 'adApi')
                ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona', 'with', 'prPersona.prPersonaIx = adApi.prUserId')
                ->leftJoin('Resrequest\DB\Enterprise\Entity\PrUser', 'prUser', 'with', 'prUser.prUserId = adApi.prUserId')
                ->leftJoin('Resrequest\DB\Enterprise\Entity\ScUser', 'scUser', 'with', 'scUser.prUserId = adApi.prUserId')
                ->leftJoin('Resrequest\DB\Enterprise\Entity\ScGroup', 'scGroup', 'with', 'scGroup.scGroupId = scUser.scGroupId')
                ->groupBy('adApi.prUserId')
                ->addOrderBy('adApi.adApiTime', 'ASC')
                ->getQuery()
                ->getResult();

            foreach ($apiConnectionsInfo as $key => $connection) {
                $query = $this->em->createQueryBuilder();
                $apiConnectionsInfo[$key]['relationships'] = $query
                    ->select(
                        [
                            "TRIM(CONCAT(prPersona.prNameFirst, ' ', prPersona.prNameLast)) AS Name",
                            "CASE WHEN prLink.rfLinkTypeId = 'RS1' THEN 'Staff' ELSE 'Agent' END AS LinkType",
                        ]
                    )
                    ->from('Resrequest\DB\Enterprise\Entity\PrLink', 'prLink')
                    ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona', 'with', 'prPersona.prPersonaIx = prLink.prLinkTo')
                    ->where('prLink.prLinkFrom = :prUserId')
                    ->setParameters(
                        [
                            'prUserId' => $connection['Id']
                        ]
                    )
                    ->getQuery()
                    ->getResult();
                $staff = $agent = false;
                foreach ($apiConnectionsInfo[$key]['relationships'] as $relationship) {
                    $staff = $relationship['LinkType'] == 'Staff' ? true : $staff;
                    $agent = $relationship['LinkType'] == 'Agent' ? true : $agent;
                }
                if (!$staff && !$agent) {
                    $apiConnectionsInfo[$key]['relationshipType'] = 'None';
                } elseif ($staff && $agent) {
                    $apiConnectionsInfo[$key]['relationshipType'] = 'Staff / Agent';
                } elseif ($staff && !$agent) {
                    $apiConnectionsInfo[$key]['relationshipType'] = 'Staff';
                } elseif (!$staff && $agent) {
                    $apiConnectionsInfo[$key]['relationshipType'] = 'Agent';
                }

            }
        }

        // User Login Info

        $userLoginInfo = array();
        if (isset($this->params['data']) && $this->params['data'] == 'userLoginInfo') {
            $query = $this->em->createQueryBuilder();
            $userLogins = $query
                ->select(
                    [
                        "adUserLog.prUserId AS Id",
                        "MAX(adUserLog.adDateTime) AS Date_Time",
                        "TRIM(CONCAT(prPersona.prNameFirst, ' ', prPersona.prNameLast)) AS Name"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\AdUserLog', 'adUserLog')
                ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona', 'with', 'prPersona.prPersonaIx = adUserLog.prUserId')
                ->groupBy('adUserLog.prUserId')
                ->addOrderBy('adUserLog.adDateTime', 'DESC')
                ->getQuery()
                ->getResult();

            $query = $this->em->createQueryBuilder();
            $loggedIn = $query
                ->select(
                    [
                        "oauthAccessTokens.expires AS Date_Time",
                        "TRIM(CONCAT(prPersona.prNameFirst, ' ', prPersona.prNameLast)) AS Name"
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\OauthAccessTokens', 'oauthAccessTokens')
                ->leftJoin('Resrequest\DB\Enterprise\Entity\PrUser', 'prUser', 'with', 'prUser.prUserName = oauthAccessTokens.userId')
                ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'prPersona', 'with', 'prPersona.prPersonaIx = prUser.prUserId')
                ->getQuery()
                ->getResult();

            $userLoginInfo = array(
                "userLogins" => $userLogins,
                "loggedIn" => $loggedIn
            );
        }

        return new ViewModel([
            'principalId' => $this->principalId,
            'shortName' => $this->shortName,
            'principalName' => $principalName,
            'databaseName' => $this->databaseName,
            'environment' => $this->environment,
            'environmentList' => $environmentList,
            'posList' => $posList,
            'phoneList' => $phoneList,
            'fiList' => $fiList,
            'propList' => $propList,
            'rateList' => $rateList,
            'databaseVersion' => $databaseVersion,
            'databaseSize' => $databaseSize,
            'databaseMaster' => $databaseMaster,
            'codeVersion' => $this->codeVersion,
            'hostName' => gethostname(),
            'archive' => $this->archive,
            'engagementInfo' => $engagementInfo,
            'apiConnectionsInfo' => $apiConnectionsInfo,
            'userLoginInfo' => $userLoginInfo
        ]);
    }
}