<?php

namespace Resrequest\Application\Chart;

use MaglLegacyApplication\Application\MaglLegacy;

abstract class Chart
{
    protected $sm;
    protected $em;

    /**
     * The id of the chart.
     *
     * @var string
     */
    protected $id;

    /**
     * The name of the chart.
     *
     * @var string
     */
    protected $name;

    /**
     * The description of the chart.
     *
     * @var string
     */
    protected $description;

    /**
     * A description of how to achieve the same chart
     * data with a report.
     *
     * @var string
     */
    protected $reportInfo;

    /**
     * The type of chart.
     *
     * @var string
     */
    protected $type;

    /**
     * The library used to implement the chart.
     *
     * @var string
     */
    protected $library;
    /**
     * The chart's configuration.
     *
     * @var array
     */
    protected $config;

    /**
     * The datasets that the chart uses.
     *
     * @var array
     */
    protected $datasets;

    /**
     * The datastructure used while processing each dataset.
     *
     * @var array
     */
    protected $accumulator = [];

    /**
     * The data the chart was generated from.
     */
    protected $data = [];

    protected $debugData = [];

    protected function onInit()
    { }

    public function __construct($config)
    {
        // Refactor into class
        $this->sm = MaglLegacy::getServiceManager();
        $this->em = $this->sm->get('doctrine.entitymanager.orm_enterprise');

        $this->id = $config['id'];
        $this->name = $config['name'];
        $this->description = $config['description'];
        $this->reportInfo = $config['reportInfo'];
        $this->chartClass = $config['chartClass'];
        $this->library = $config['library'];
        $this->config = $config['config'];

        if (!isset($this->config['ui']['menu'])) {
            $this->config['ui'] = ['menu' => []];
        }

        $this->datasets = $this->getDatasets();
        $this->onInit();
    }

    protected function getDatasetConfigs()
    {
        $configs = [];

        foreach ($this->datasets as $dataset) {
            $configs[] = $dataset->toArray();
        }

        return $configs;
    }

    public function toArray()
    {
        $config = [
            'library' => $this->config['library'],
            'mappings' => $this->config['mappings'],
            'datasets' => $this->getDatasetConfigs(),
            'ui' => $this->config['ui']
        ];

        return [
            'id' => $this->id,
            'name' => $this->name,
            'description' => $this->description,
            'reportInfo' => $this->reportInfo,
            'type' => $this->type,
            'chartClass' => $this->chartClass,
            'library' => $this->library,
            'config' => $config
        ];
    }

    public function getData()
    {
        $processedData = $this->processDatasets();

        // If the data failed validation, set output to false
        if ($processedData === false) {
            return false;
        }

        $finalisedData = $this->finaliseData($processedData);

        $data = array_merge($this->config['library'], $finalisedData);

        if (!empty($this->data[0])) {
            $data = $this->template($data, $this->data[0]);
        }

        return $data;
    }

    protected function getDatasets()
    {
        $datasets = [];

        foreach ($this->config['datasets'] as $dataset) {
            $name = $dataset['name'];
            $config = $dataset['config'];

            if (!Registry::hasDataset($name)) {
                throw new \Exception("Dataset '$name' does not exist");
            }

            $datasetClass = Registry::getDatasetClass($name);
            $datasets[] = new $datasetClass($name, $config);
        }

        return $datasets;
    }

    protected function processDatasets()
    {
        $elements = $this->getDataAndMappings();

        $this->data = $elements[0]['data'];

        $accumulator = $this->accumulator;

        foreach ($elements as $element) {
            $accumulator = $this->processData($accumulator, $element['data'], $element['mapping']);
        }

        if ($this->validateData($accumulator) === false) {
            return false;
        }

        return $accumulator;
    }

    protected function getDataAndMappings()
    {
        $data = [];
        $this->debugData['datasets'] = [];

        foreach ($this->datasets as $key => $dataset) {
            $mappings = $this->config['mappings'][$key];
            $data[] = [
                'mapping' => $mappings,
                'data' => $dataset->getData(),
            ];
            $this->debugData['datasets'][] = [
                'name' => $dataset->getName(),
                'debugData' => $dataset->getDebugData(),
            ];
        }

        return $data;
    }

    /**
     * Used to process and map dataset data into an intermediate state
     * which will then be finalised using the finaliseData method
     *
     * @param array $accumulator
     * @param array $data
     * @param array $mapping
     * @return array
     */
    protected abstract function processData($accumulator, $data, $mapping);

    /**
     * Used to convert processed data into the structure the chart type requires
     *
     * @param array $processedData
     * @return array
     */
    protected abstract function finaliseData($processedData);


    /**
     * Called after dataset data has been built and processed.
     * Used to determine whether the data for the chart type is valid or empty.
     *
     * @param array $accumulator
     * @return boolean
     */
    protected function validateData($accumulator)
    {
        return true;
    }

    public function getDebugData() {
        return $this->debugData;
    }

    /**
     * Searches the final chart configuration for templates and replaces
     * placeholders with the appropriate data.
     *
     * @param array $data
     * @param array $dataset Key value array containing data for placeholders.
     * @return void
     */
    protected function template($data, $dataset) {
        foreach ($data as &$value) {
            if (is_string($value)) {
                if (preg_match('/{{[\S]+}}/', $value)) {
                    $value = preg_replace_callback('{{[\S]+}}', function($matches) use ($dataset) {
                        foreach ($matches as $match) {
                            $field = substr($match, 2, -2);
            
                            if (isset($dataset[$field])) {
                                return $dataset[$field];
                            } else {
                                return '';
                            }
                        }
                    }, $value);
                }
            } else if (is_array($value)) {
                $value = $this->template($value, $dataset);
            } else {
                continue;
            }
        }

        return $data;
    }

}
