<?php

namespace Resrequest\Application\Chart;

use Resrequest\Application\Chart\Store\ChartStore;

class ChartBuilder
{
    private $chartFactory;

    protected $enterprise;
    protected $em;

    private $chartStore;

    public function __construct($enterprise, $em, ChartFactory $chartFactory)
    {
        $this->enterprise = $enterprise;
        $this->em = $em;
        $this->chartFactory = $chartFactory;
        $this->chartStore = new ChartStore();
    }

    /**
     * Whether the chart exists or not.
     *
     * @param string $chartId The id of the chart.
     * @return boolean
     */
    public function chartExists($chartId)
    {
        return $this->chartStore->hasChart($chartId);
    }

    /**
     * Get an instance of the chart.
     *
     * @param string $id The id of the chart.
     * @return Chart
     */
    public function getChart($id, $optionValues = [])
    {
        $config = $this->getChartConfig($id);

        return $this->buildChart($config, $optionValues);
    }

    /**
     * Retrieves the configuration of the chart from the database.
     * 
     * @param string $chartId The id of the chart.
     * @return array
     */
    protected function getChartConfig($chartId)
    {
        if ($this->chartStore->hasChart($chartId)) {
            return $this->chartStore->getChartConfig($chartId);
        } else {
            throw new \Exception("Chart $chartId does not exist.");
        }
    }

    public function buildChart($config, $optionValues = [])
    {
        if (empty($config)) {
            throw new \Exception("Invalid chart configuration");
        }

        // Apply updated option values to chart config
        foreach ($optionValues as $option) {
            $id = $option['source'];
            $name = $option['option'];
            $value = $option['value'];

            foreach ($config['config']['datasets'] as $datasetIndex => $dataset) {
                $valueUpdated = false;

                // Apply filter and dataset option values
                foreach ($dataset['config']['options'] as $optionIndex => $option) {
                    $optionConfig = $option['config'];
                    if ($optionConfig['id'] == $id) {
                        foreach ($optionConfig['options'] as $subOptionIndex => $subOption) {
                            if ($subOption['name'] == $name) {
                                $config['config']['datasets'][$datasetIndex]['config']['options'][$optionIndex]['config']['options'][$subOptionIndex]['value'] = $value;

                                $valueUpdated = true;
                                break 2;
                            }
                        }
                    }
                }

                if ($valueUpdated === true) {
                    continue;
                }

                // Apply converter option values
                foreach ($dataset['config']['converters'] as $converterIndex => $converter) {
                    if ($converter['id'] == $id) {
                        foreach ($converter['options'] as $subOptionIndex => $subOption) {
                            if ($subOption['name'] == $name) {
                                $config['config']['datasets'][$datasetIndex]['config']['converters'][$converterIndex]['options'][$subOptionIndex]['value'] = $value;
                            }
                        }
                    }
                }
            }
        }

        // Create instance of chart
        $chartClass = $this->chartFactory->getChartTypeClass($config['library'], $config['chartClass']);
        if ($chartClass === false) {
            throw new \Exception("Invalid chart type $config[type]");
        }
        $chart = new $chartClass($config);

        return $chart;
    }

    /**
     * Gets the chart and the data.
     *
     * @param string|array $source The id of an existing chart or the configuration.
     * @return array
     */
    public function getChartAndData($source, $optionValues = [])
    {
        if (is_array($source)) {
            $config = $source;
            $chart = $this->buildChart($config, $optionValues);
        } else {
            $id = $source;
            $chart = $this->getChart($id, $optionValues);
        }

        $data = $chart->getData();

        return [
            'chart' => $chart->toArray(),
            'data' => $data,
            'debugData' => $chart->getDebugData(),
        ];
    }

    public function canAccessChart($chartId, $accessGroupId = false)
    {
        if ($accessGroupId === false) {
            $accessGroupId = $this->enterprise->accessGroupId;
        }

        $chartAccessQuery = $this->em->createQueryBuilder();

        $chartAccess = $chartAccessQuery
            ->select(
                [
                    'chartAccess.scGroupChartLevel AS accessLevel',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\ScGroupChart', 'chartAccess')
            ->where('chartAccess.scGroupId = :accessGroupId')
            ->andWhere('chartAccess.chChartId = :chartId')
            ->setParameters(
                [
                    'accessGroupId' => $accessGroupId,
                    'chartId' => $chartId,
                ]
            )
            ->getQuery()
            ->getResult();

        if (!empty($chartAccess)) {
            if ($chartAccess[0]['accessLevel'] >= 5) {
                return true;
            }
        }

        return false;
    }

    public function setChartAccessLevel($chartId, $accessGroupId, $accessLevel)
    {
        $chartAccess = $this->em->getRepository('Resrequest\DB\Enterprise\Entity\ScGroupChart')->findBy(array('scGroupId' => $accessGroupId, 'chChartId' => $chartId));

        if (empty($chartAccess)) {
            $chartAccess = new \Resrequest\DB\Enterprise\Entity\ScGroupChart;
        } else {
            $chartAccess = $chartAccess[0];
        }

        $chartAccess->setChChartId($chartId);
        $chartAccess->setScGroupId($accessGroupId);
        $chartAccess->setScGroupChartLevel($accessLevel);

        $this->em->persist($chartAccess);
        $this->em->flush();
        $this->em->clear();
    }

    public function getChartSummary($id)
    {
        $config = $this->chartStore->getChartSummary($id);

        return [
            'data' => false,
            'chart' => [
                'id' => $config['id'],
                'name' => $config['name'],
                'description' => $config['description'],
                'chartClass' => $config['chartClass'],
                'library' => $config['library'],
                'config' => [
                    'library' => [],
                    'mappings' => [],
                    'datasets' => []
                ]
            ]
        ];
    }

    /**
     * Get the summary of the charts
     *
     * @return array
     */
    public function getChartSummaries($chartIds = [])
    {
        $charts = [];
        foreach ($this->chartStore->getCharts() as $id) {
            if (!empty($chartIds) && !in_array($id, $chartIds)) {
                continue;
            }

            $charts[] = $this->getChartSummary($id)['chart'];
        }

        if (!empty($chartIds)) {
            $sortedCharts = [];

            foreach ($chartIds as $id) {
                foreach ($charts as $chart) {
                    if ($charts['id'] === $id) {
                        $sortedCharts[] = $chart;
                    }
                }
            }
        }

        return $charts;
    }
}
