<?php
namespace Resrequest\Application\Chart;

use Resrequest\Application\Chart\Store\ContainerStore;

class ChartContainer {
    const RESVEGA_CONTAINERS = [
        '11eac9c7a7a83245ab4102ae0e1a0d08' => [ // Sales & travel movements
            'version' => 'free',
            'reportId' => 'c9578477-d226-4626-9d83-608ede494345'
        ],
        '11eac9c7afb68b7bab4102ae0e1a0d08' => [ // Travel behaviour
            'version' => 'free',
            'reportId' => 'fcae124e-c518-46cf-afe2-381ca3bbce54'
        ],
        '11eac9c7b21d3cb8ab4102ae0e1a0d08' => [ // Booking tempo
            'version' => 'pro',
            'reportId' => 'b3959b2d-980f-4f37-aa41-0c7fc53030dc'
        ],
        '11eac9c7c2bf3186ab4102ae0e1a0d08' => [ // Travel routes
            'version' => 'pro',
            'reportId' => '120fa301-b704-450f-8fc1-168df0bcd7b3'
        ],
        '11ead7940a7d595da11002ae0e1a0d08' => [ // Occupancies
            'version' => 'pro',
            'reportId' => '6ef781b6-16b8-4ff7-8fb0-bc57635620d7'
        ],
        '11ead79410027fffa11002ae0e1a0d08' => [ // Yield, lead times & materialisation
            'version' => 'pro',
            'reportId' => 'e85c100b-3d35-402b-8b50-cfaa385a7a00'
        ],
    ];

    private $chartBuilder;
    private $containerStore;

    protected $em;
    protected $enterprise;

    public function __construct($enterprise, $em, ChartBuilder $chartBuilder)
    {
        $this->enterprise = $enterprise;
        $this->em = $em;
        $this->chartBuilder = $chartBuilder;
        $this->containerStore = new ContainerStore();
    }

    /**
     * Returns chart container data as stored in the database
     *
     * @param string $chartContainerId
     * @return array
     */
    public function storedChartContainer($chartContainerId, $preloadChartData = false, $processOptions = false)
    {
        if (!$this->containerStore->hasContainer($chartContainerId)) {
            throw new \Exception("Invalid chart container ID specified");
        }

        $container = $this->containerStore->getContainerConfig($chartContainerId, $processOptions);

        $charts = [];

        foreach ($container['config']['charts'] as $chartId) {
            if ($this->chartBuilder->canAccessChart($chartId)) {
                if ($preloadChartData === true) {
                    $charts[$chartId] = $this->chartBuilder->getChartAndData($chartId);
                } else {
                    $charts[$chartId] = ['chart' => $this->chartBuilder->getChart($chartId), 'data' => null];
                }
            }
        }

        $container['charts'] = $charts;
        return $container;
    }

    /**
     * Returns all stored chart containers
     *
     * @return array
     */
    public function storedChartContainers() {
        $containers = $this->containerStore->getContainerConfigs();
        $containers = array_map(function($container) {
            return $this->storedChartContainer($container['id']);
        }, $containers);

        return $containers;
    }

    // Get pre-configured chart container by ID
    public function buildChartContainer($id, $preloadChartData)
    {
        $container = $this->storedChartContainer($id, $preloadChartData, true);

        foreach ($container['config']['charts'] as $index => $chart) {
            $container['config']['dashboard']['items'][$index]['chartId'] = $chart;
        }

        if (!$this->isChartContainerAllowed($id)) {
            $container['config'] = false;
        }

        return $container;
    }

    /**
     * Determine if chart container is hidden from user access
     *
     * @param string $chartContainerId
     * @return boolean
     */
    public function isChartContainerHidden($chartContainerId) {
        if (array_key_exists($chartContainerId, $this::RESVEGA_CONTAINERS)) {
            return true;
        }

        return false;
    }

    public function isChartContainerDisabled($chartContainerId)
    {
        // ResVega dashboards
        if (array_key_exists($chartContainerId, $this::RESVEGA_CONTAINERS)) {
            $accessQuery = $this->em->createQueryBuilder();
            $accessIndicator = $accessQuery
                ->select('system.rfSysTourismTodayAccessInd')
                ->from('Resrequest\DB\Enterprise\Entity\RfSystem', 'system')
                ->getQuery()
                ->getSingleScalarResult();

            $accessIndicator = intval($accessIndicator);

            /**
             * Tourism Today access indicator
             * 
             * 0 = No access to free and pro tier
             * 1 = Access to free tier only
             * 2 - Access to free and pro tier
             */
            $allowedTiers = [];
            if ($accessIndicator === 2) {
                $allowedTiers[] = 'free';
                $allowedTiers[] = 'pro';
            } else if ($accessIndicator === 1) {
                $allowedTiers[] = 'free';
            }

            $containerTier = $this::RESVEGA_CONTAINERS[$chartContainerId]['version'];
            if (in_array($containerTier, $allowedTiers)) {
                return false;
            }

            return true;
        }


        return false;
    }

    public function isChartContainerAllowed($chartContainerId, $accessGroupId = false, $overrideDisabledContainers = false) {
        if (empty($GLOBALS['pr_business_link'])) {
            return false;
        }
        if ($accessGroupId === false) {
            $accessGroupId = $this->enterprise->accessGroupId;
        }

        $disabled = $this->isChartContainerDisabled($chartContainerId);

        if ($disabled && !$overrideDisabledContainers) {
            return false;
        }

        $chartAccessQuery = $this->em->createQueryBuilder();

        $chartAccess = $chartAccessQuery
            ->select(
                [
                    'chartContainerAccess.scGroupChartContainerLevel AS accessLevel',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\ScGroupChartContainer', 'chartContainerAccess')
            ->where('chartContainerAccess.scGroupId = :accessGroupId')
            ->andWhere('chartContainerAccess.chChartContainerId = :chartContainerId')
            ->setParameters(
                [
                    'accessGroupId' => $accessGroupId,
                    'chartContainerId' => $chartContainerId,
                ]
            )
            ->getQuery()
            ->getResult();

        if (!empty($chartAccess)) {
            if ($chartAccess[0]['accessLevel'] >= 5) {
                return true;
            }
        }

        return false;
    }

    public function setChartContainerAccessLevel($chartContainerId, $accessGroupId, $accessLevel) {
        $chartContainerAccess = $this->em->getRepository('Resrequest\DB\Enterprise\Entity\ScGroupChartContainer')->findBy(array('scGroupId' => $accessGroupId, 'chChartContainerId' => $chartContainerId));

        if (empty($chartContainerAccess)) {
            $chartContainerAccess = new \Resrequest\DB\Enterprise\Entity\ScGroupChartContainer;
        } else {
            $chartContainerAccess = $chartContainerAccess[0];
        }

        $chartContainerAccess->setChChartContainerId($chartContainerId); 
        $chartContainerAccess->setScGroupId($accessGroupId);
        $chartContainerAccess->setScGroupChartContainerLevel($accessLevel);

        $this->em->persist($chartContainerAccess);
        $this->em->flush();
        $this->em->clear();
    } 

    /**
     * Whether the specified container exists or not.
     *
     * @param string $id
     * @return boolean
     */
    public function containerExists($id)
    {
        return $this->containerStore->hasContainer($id);
    }

    /**
     * Whether a user group has access to dashboards.
     *
     * @param string $accessGroupId Optional user group id.
     * @return boolean
     */
    public function hasDashboardsAccess($accessGroupId = false)
    {
        $containers = $this->containerStore->getContainerConfigs();

        foreach ($containers as $container) {
            if ($this->isChartContainerAllowed($container['id'], $accessGroupId)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Get the summary of the containers and the charts in them.
     *
     * @return array
     */
    public function getContainerSummaries($containerIds = [])
    {
        $containers = $this->containerStore->getContainerConfigs();

        foreach ($containers as $index => &$container) {
            if (!empty($containerIds) && !in_array($container['id'], $containerIds)) {
                unset($containers[$index]);
                continue;
            }
            $charts = [];

            foreach ($container['config']['charts'] as $chartId) {
                $charts[] = $this->chartBuilder->getChartSummary($chartId)['chart'];
            }

            $container['charts'] = $charts;
        }
        unset($container);

        if (!empty($containerIds)) {
            $sortedContainers = [];

            foreach ($containerIds as $id) {
                foreach($containers as $container) {
                    if ($container['id'] === $id) {
                        $sortedContainers[] = $container;
                    }
                }
            }

            return $sortedContainers;
        }

        return $containers;
    }
}