<?php

namespace Resrequest\Application\Chart\Config;

class Chart
{
    /**
     * The id of the chart.
     *
     * @var string
     */
    private $id;

    /**
     * The library used for the chart.
     *
     * @var string
     */
    private $library;

    /**
     * The libary config for the chart.
     *
     * @var array
     */
    private $libraryConfig;

    /**
     * The chart type class
     *
     * @var string
     */
    private $chartClass;

    /**
     * The name of the chart.
     *
     * @var string
     */
    private $name;

    /**
     * The description of the chart.
     *
     * @var string
     */
    private $description;

    /**
     * The report info for the chart.
     *
     * @var string
     */
    private $reportInfo;

    /**
     * The mappings for the chart.
     *
     * @var array
     */
    private $mappings;

    /**
     * Adds datasets to the chart.
     *
     * @var array
     */
    private $datasets;

    /**
     * The UI config for the chart.
     *
     * @var array
     */
    private $uiConfig;

    public function __construct()
    {
        $this->id = '';
        $this->libary = '';
        $this->libraryConfig = [];
        $this->chartClass = '';
        $this->name = '';
        $this->description = '';
        $this->reportInfo = '';
        $this->mappings = [];
        $this->datasets = [];
        $this->uiConfig = [];
    }

    /**
     * Sets the id of the chart.
     *
     * @param string $id
     * @return Chart
     */
    public function setId($id)
    {
        $this->id = $id;
        return $this;
    }

    /**
     * The library used for the chart.
     *
     * @param string $library
     * @return Chart
     */
    public function setLibrary($library)
    {
        $this->library = $library;
        return $this;
    }

    /**
     * Sets the library configuration.
     *
     * @param array $config
     * @return Chart
     */
    public function setLibaryConfig($config)
    {
        $this->libraryConfig = $config;
        return $this;
    }

    /**
     * Sets the chart type class.
     *
     * @param string $chartClass
     * @return Chart
     */
    public function setChartClass($chartClass)
    {
        $this->chartClass = $chartClass;
        return $this;
    }

    /**
     * Sets the name of the chart.
     *
     * @param string $name
     * @return Chart
     */
    public function setName($name)
    {
        $this->name = $name;
        return $this;
    }

    /**
     * Sets the description of the chart.
     *
     * @param string $description
     * @return Chart
     */
    public function setDescription($description)
    {
        $this->description = $description;
        return $this;
    }

    /**
     * Sets the report info for the chart.
     *
     * @param string $reportInfo
     * @return Chart
     */
    public function setReportInfo($reportInfo)
    {
        $this->reportInfo = $reportInfo;
        return $this;
    }

    /**
     * Add mappings for the chart.
     *
     * @param array $mappings
     * @return Chart
     */
    public function addMappings($mappings)
    {
        $this->mappings = array_merge($this->mappings, $mappings);
        return $this;
    }

    /**
     * Adds a mapping for the chart.
     *
     * @param array $mapping
     * @return Chart
     */
    public function addMapping($mapping)
    {
        $this->addMappings([$mapping]);
        return $this;
    }

    /**
     * Adds datasets to the chart.
     *
     * @param array $datasets
     * @return Chart
     */
    public function addDatasets($datasets)
    {
        $this->datasets = array_merge($this->datasets, $datasets);
        return $this;
    }

    /**
     * Adds a dataset to the chart.
     *
     * @param Dataset $dataset
     * @return Chart
     */
    public function addDataset($dataset)
    {
        $this->addDatasets([$dataset]);
        return $this;
    }

    /**
     * Sets the UI config for the chart.
     *
     * @param UiConfig $config
     * @return Chart
     */
    public function setUiConfig($config)
    {
        $this->uiConfig = $config;
        return $this;
    }

    /**
     * Returns the arraay representation of the chart.
     *
     * @return array
     */
    public function toArray()
    {
        $datasets = array_map(function($dataset) {
            return $dataset->toArray();
        }, $this->datasets);

        return [
            'id' => $this->id,
            'chartClass' => $this->chartClass,
            'library' => $this->library,
            'name' => $this->name,
            'description' => $this->description,
            'reportInfo' => $this->reportInfo,
            'config' => [
                'library' => $this->libraryConfig,
                'mappings' => $this->mappings,
                'datasets' => $datasets,
                'ui' => $this->uiConfig->toArray()
            ]
        ];
    }
}