<?php

namespace Resrequest\Application\Chart\Config;

class Dataset
{
    /**
     * The name of the dataset.
     *
     * @var string
     */
    private $name;

    /**
     * The options for the dataset.
     *
     * @var array
     */
    private $options;

    /**
     * The converters to be used in the dataset.
     *
     * @var array
     */
    private $converters;

    public function __construct($name)
    {
        $this->name = $name;
        $this->options = [];
        $this->converters = [];
    }

    /**
     * Sets the name of the dataset.
     *
     * @param string $name
     * @return Dataset
     */
    public function setName($name)
    {
        $this->name = $name;
        return $this;
    }

    /**
     * Adds filters to the dataset.
     *
     * @param array $filters
     * @return Dataset
     */
    public function addFilters($filters)
    {
        $filters = array_map(function($filter) {
            return [
                'type' => 'filter',
                'config' => $filter
            ];
        }, $filters);
        $this->options = array_merge($this->options, $filters);
        return $this;
    }
    /**
     * Adds a filter to the dataset.
     *
     * @param Filter $filter
     * @return Dataset
     */
    public function addFilter($filter)
    {
        $this->addFilters([$filter]);
        return $this;
    }

    /**
     * Adds dataset options to the dataset.
     *
     * @param array $options
     * @return Dataset
     */
    public function addDatasetOptions($options)
    {
        $options = array_map(function($option) {
            return [
                'type' => 'data',
                'config' => $option
            ];
        }, $options);
        $this->options = array_merge($this->options, $options);
        return $this;
    }

    /**
     * Adds a dataset option to the dataset.
     *
     * @param DatasetOption $datasetOption
     * @return Dataset
     */
    public function addDatasetOption($option)
    {
        $this->addDatasetOptions([$option]);
        return $this;
    }

    /**
     * Adds converters to the dataset.
     *
     * @param array $converters
     * @return Dataset
     */
    public function addConverters($converters)
    {
        $this->converters = array_merge($this->converters, $converters);
        return $this;
    }

    /**
     * Adds a converter to the dataset.
     *
     * @param Converter $converter
     * @return Dataset
     */
    public function addConverter($converter)
    {
        $this->addConverters([$converter]);
        return $this;
    }

    /**
     * Returns the array representation of the dataset.
     *
     * @return array
     */
    public function toArray()
    {
        $options = array_map(function($option) {
            return [
                'type' => $option['type'],
                'config' => $option['config']->toArray()
            ];
        }, $this->options);

        $converters = array_map(function($converter) {
            return $converter->toArray();
        }, $this->converters);

        return [
            'name' => $this->name,
            'config' => [
                'options' => $options,
                'converters' => $converters
            ]
        ];
    }
}