<?php

namespace Resrequest\Application\Chart\Config;

class Process
{
    /**
     * The name that the process should map to.
     *
     * @var string
     */
    private $name;

    /**
     * The assigned id of the process.
     *
     * @var string
     */
    private $id;

    /**
     * The options to configure the process.
     *
     * @var array
     */
    private $options;

    /**
     * The input mappings for the process.
     *
     * @var array
     */
    private $inputMappings;

    /**
     * The output mappings for the process.
     *
     * @var array
     */
    private $outputMappings;

    /**
     * The subtype of the process.
     *
     * @var string
     */
    private $subType;

    /**
     * Create a new process builder instance.
     *
     * @param string $name
     * @param string $id
     */
    public function __construct($name, $id='')
    {
        $this->name = $name;
        $this->id = $id;
        $this->options = [];
        $this->inputMappings = [];
        $this->outputMappings = [];
        $this->subType = '';
    }

    /**
     * Sets the name of the process.
     *
     * @param string $name
     * @return Process
     */
    public function setName($name)
    {
        $this->name = $name;
        return $this;
    }

    /**
     * Sets the id of the process.
     *
     * @param string $id
     * @return Process
     */
    public function setId($id)
    {
        $this->id = $id;
        return $this;
    }

    /**
     * Adds options to the process.
     *
     * @param array $options The array of option config builders.
     * @return Process
     */
    public function addOptions($options)
    {
        $this->options = array_merge($this->options, $options);
        return $this;
    }

    /**
     * Adds an option to the process.
     *
     * @param array $option An option config builder.
     * @return Process
     */
    public function addOption($option)
    {
        $this->addOptions([$option]);
        return $this;
    }

    /**
     * Add input mappings to the process.
     *
     * @param array $mappings
     * @return Process
     */
    public function addInputMappings($mappings)
    {
        $this->inputMappings = array_merge($this->inputMappings, $mappings);
        return $this;
    }

    /**
     * Add an input mapping to the process.
     *
     * @param array $mapping
     * @return Process
     */
    public function addInputMapping($mapping)
    {
        $this->addInputMappings([$mapping]);
        return $this;
    }

    /**
     * Add output mappings to the process.
     *
     * @param array $mappings
     * @return Process
     */
    public function addOutputMappings($mappings)
    {
        $this->outputMappings = array_merge($this->outputMappings, $mappings);
        return $this;
    }

    /**
     * Add an output mapping to the process.
     *
     * @param array $mapping
     * @return Process
     */
    public function addOutputMapping($mapping)
    {
        $this->addOutputMappings([$mapping]);
        return $this;
    }

    /**
     * Set the subtype of the process.
     *
     * @param string $type
     * @return Process
     */
    public function setSubType($type)
    {
        $this->subType = $type;
        return $this;
    }

    public function toArray()
    {
        $options = array_map(function($option) {
            return $option->toArray();
        }, $this->options);

        $inputMappings = array_map(function($mapping) {
            return $mapping->toArray();
        }, $this->inputMappings);

        $outputMappings = array_map(function($mapping) {
            return $mapping->toArray();
        }, $this->outputMappings);

        return [
            'name' => $this->name,
            'type' => $this->subType,
            'id' => $this->id,
            'options' => $options,
            'mappings' => [
                'inputs' => $inputMappings,
                'outputs' => $outputMappings
            ]
        ];
    }
}