<?php

namespace Resrequest\Application\Chart\Converter;

use ArrayObject;
use Resrequest\Application\Chart\Converter;
use Resrequest\Application\Chart\Input\CurrencyConversion;
use Resrequest\Application\Chart\Option;

class Currency extends Converter
{
    protected $subType = 'currency';

    const INITIAL_ACCUMULATOR = [
        'results' => []
    ];

    protected $supportedInputs = [
        'currencyFrom',
        'amount'
    ];

    protected $supportedOutputs = [
        'currencyTo',
        'amount'
    ];

    public function isTargetCurrency($currencyFrom)
    {
        return $this->getCurrencyTo() == $currencyFrom;
    }

    public function generateOptions()
    {
        $currencies = $this->currencies();
        $defaultCurrencyQuery = $this->em->createQueryBuilder();
        $defaultCurrency = $defaultCurrencyQuery
            ->select(
                [
                    'currency.rfCurrencySymbol AS currencySymbol',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfDefault', 'default')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RtRateType', 'rateType', 'with', 'rateType.rtRateTypeIx = default.rtRateTypeId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfCurrency', 'currency', 'with', 'currency.rfCurrencyIx = rateType.rfCurrencyId')
            ->getQuery()
            ->getResult();

        $defaultCurrency = !empty($defaultCurrency) ? $defaultCurrency[0]['currencySymbol'] : false;
        $defaultCurrencyIndex = 0;
        foreach ($currencies as $index => $currency) {
            if ($defaultCurrency == $currency) {
                $defaultCurrencyIndex = $index;
                break;
            }
        }

        $default = [
            'currencyTo' => $currencies[$defaultCurrencyIndex],
            'rates' => new ArrayObject()
        ];

        $currencyUi = new CurrencyConversion($default);
        $currencyUi->addSupportedCurrencies($currencies);

        return [
            new Option('currency', $currencyUi),
        ];
    }

    private function getCurrencyTo()
    {
        $currencyInput = $this->optionByName('currency')->input;
        return $currencyInput->getValue()['currencyTo'];
    }

    private function hasExchangeRate($currencyFrom)
    {
        $currencyInput = $this->optionByName('currency')->input;
        $value = $currencyInput->getValue();
        $currencyTo = $value['currencyTo'];

        return !empty($value['rates'][$currencyTo][$currencyFrom]);
    }

    public function convert($amount, $currencyFrom)
    {
        $currencyInput = $this->optionByName('currency')->input;
        $value = $currencyInput->getValue();
        $currencyTo = $value['currencyTo'];

        if ($currencyFrom == $currencyTo) {
            return $amount;
        }

        $exchangeRate = trimExchangeRate($value['rates'][$currencyTo][$currencyFrom]['exchangeRate']);

        return $amount * $exchangeRate;
    }

    protected function processData($originalData, $extractedData, $accumulator)
    {
        $currencyFrom = $this->getSingleInput($extractedData['currencyFrom']);

        $isTargetCurrency = false;

        if ($this->isTargetCurrency($currencyFrom)) {
            $isTargetCurrency = true;
        } else if (!$this->hasExchangeRate($currencyFrom)) {
            return $accumulator;
        }

        $convertedAmounts = [];

        foreach ($extractedData['amount'] as $amountField => $amount) {
            if ($isTargetCurrency) {
                $convertedAmounts[$amountField] = $amount;
            } else {
                $convertedAmounts[$amountField] = $this->convert($amount, $currencyFrom);
            }
        }

        $currencyTo = $this->getCurrencyTo();

        $accumulator['results'][] = [
            'outputs' => [
                'currencyTo' => $currencyTo,
                'amount' => $convertedAmounts
            ],
            'data' => $originalData
        ];

        return $accumulator;
    }

    protected function finaliseData($accumulator)
    {
        return $accumulator['results'];
    }

    /**
     * Returns a list of currencies stored in the database
     *
     * @return 
     */
    protected function currencies()
    {
        $currenciesQuery = $this->em->createQueryBuilder();

        $storedCurrencies = $currenciesQuery
            ->select(
                [
                    'currency.rfCurrencySymbol AS currencySymbol',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RfCurrency', 'currency')
            ->getQuery()
            ->getResult();

        foreach ($storedCurrencies as $currency) {
            $currencies[] = $currency['currencySymbol'];
        }

        return $currencies;
    }
}
