<?php

namespace Resrequest\Application\Chart\Converter;

use Resrequest\Application\Chart\Converter;
use Resrequest\Application\Chart\Input;
use Resrequest\Application\Chart\Option;

class GroupDistributor extends Converter
{
    protected $subType = 'groupDistributor';

    const INITIAL_ACCUMULATOR = [
        'groups' => [],
        'data' => false,
        'index' => 0,
        'results' => []
    ];

    protected $supportedInputs = [
        'groupBy',
    ];

    protected $supportedOutputs = [];

    public function generateOptions() {
        return [
            /**
             * Array of arrays with key value pairs, with the
             * keys being the field name to add to the data group.
             * Each element is distributed to a unique group.
             * Instances of the same group will also have the
             * same data distributed
             * 
             * Example
             *  [
             *      [
             *          'key1' => value1,
             *          'key2' => value2
             *      ],
             *      [
             *          'key1' => value3,
             *          'key2' => value4
             *      ]
             *  ]
             */
            new Option('data', new Input([]))
        ];
    }

    protected function processData($originalData, $extractedData, $accumulator)
    {
        if ($accumulator['data'] === false) {
            // Add data to accumulator
            $accumulator['data'] = $this->optionByName('data')->input->getValue();
        }
        $data = &$accumulator['data'];
        $index = &$accumulator['index'];
        $groupBy = join('::', ($extractedData['groupBy']));

        if (!isset($accumulator['groups'][$groupBy])) {
            if (!isset($data[$index])) {
                // Reached the end of array, reset
                $index = 0;
            }
            $accumulator['groups'][$groupBy] = $data[$index];
            $index++;
        }

        $groupData = &$accumulator['groups'][$groupBy];

        $accumulator['results'][] = [
            'outputs' => $groupData,
            'data' => $originalData
        ];
        return $accumulator;
    }

    protected function finaliseData($accumulator)
    {
        $result = [];
        foreach ($accumulator['results'] as $data) {
            $result[] = [
                'outputs' => [],
                'data' => array_merge($data['data'], $data['outputs'])
            ];
        }

        return $result;
    }
}
