<?php

namespace Resrequest\Application\Chart\Converter;

use Resrequest\Application\Chart\Converter;
use Resrequest\Application\Chart\Input;
use Resrequest\Application\Chart\Option;

/**
 * Determines the nationality of reservations by the nationality field,
 * agent, originator or contact person.
 */
class ReservationCountry extends Converter
{
    protected $subType = 'reservationCountry';

    const INITIAL_ACCUMULATOR = [
        'reservationIds' => [],
        'data' => []
    ];

    protected $supportedInputs = [
        'reservationId',
    ];

    protected $supportedOutputs = [
        'originCountryName',
        'originCountryCode',

        'nationalityCountryName',
        'nationalityCountryCode',

        'agentPhysCountryName',
        'agentPhysCountryCode',
        'agentPostCountryName',
        'agentPostCountryCode',

        'originatorPhysCountryName',
        'originatorPhysCountryCode',
        'originatorPostCountryName',
        'originatorPostCountryCode',

        'contactPhysCountryName',
        'contactPhysCountryCode',
        'contactPostCountryName',
        'contactPostCountryCode',
    ];

    protected function generateOptions()
    {
        return [
            new Option('excludeReservationsWithoutCountry', new Input(false))
        ];
    }


    // Calculate the total value of fields across the entire dataset
    protected function processData($originalData, $extractedData, $accumulator)
    {
        $accumulator['reservationIds'][] = $this->getSingleInput($extractedData['reservationId']);
        $accumulator['data'][] = $originalData;
        return $accumulator;
    }

    protected function finaliseData($accumulator)
    {
        $excludeReservationsWithoutCountry = $this->optionByName('excludeReservationsWithoutCountry')->getValue();
        $reservationCountryData = $this->getNationalityData($accumulator['reservationIds'], $excludeReservationsWithoutCountry);

        $result = [];
        foreach ($accumulator['data'] as $data) {
            $reservationId = $data['reservationId'];
            if (isset($reservationCountryData[$reservationId])) {
                $result[] = [
                    'data' => $data,
                    'outputs' => $reservationCountryData[$reservationId]
                ];
            }
        }

        return $result;
    }

    protected function getNationalityData($reservationIds, $excludeReservationsWithoutCountry) {
        $reservationCountryDataQuery = $this->em->createQueryBuilder();
        $reservationCountryDataResult = $reservationCountryDataQuery
            ->select(
                [
                    'reservation.rvReservationIx as reservationId',
                    'IF(reservation.rvAgentId = \'0\', TRUE, FALSE) AS isDirect',
                    'IF(reservation.rvAgentId IS NULL OR reservation.rvAgentId = \'0\', FALSE, TRUE) AS hasAgent',
                    'IF(reservation.rvOriginAgentId IS NULL, FALSE, TRUE) AS hasOriginator',
                    'IF(reservation.rvBillingPersonaId IS NULL, FALSE, TRUE) AS hasContact',

                    'nationality.rfCountryName AS nationalityCountryName',
                    'nationality.rfCountryCode AS nationalityCountryCode',

                    'agentPhysCountry.rfCountryName AS agentPhysCountryName',
                    'agentPhysCountry.rfCountryCode AS agentPhysCountryCode',
                    'agentPostCountry.rfCountryName AS agentPostCountryName',
                    'agentPostCountry.rfCountryCode AS agentPostCountryCode',

                    'originatorPhysCountry.rfCountryName AS originatorPhysCountryName',
                    'originatorPhysCountry.rfCountryCode AS originatorPhysCountryCode',
                    'originatorPostCountry.rfCountryName AS originatorPostCountryName',
                    'originatorPostCountry.rfCountryCode AS originatorPostCountryCode',

                    'contactPhysCountry.rfCountryName AS contactPhysCountryName',
                    'contactPhysCountry.rfCountryCode AS contactPhysCountryCode',
                    'contactPostCountry.rfCountryName AS contactPostCountryName',
                    'contactPostCountry.rfCountryCode AS contactPostCountryCode',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'agent', 'with', 'agent.prPersonaIx = reservation.rvAgentId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'originator', 'with', 'originator.prPersonaIx = reservation.rvOriginAgentId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'contact', 'with', 'contact.prPersonaIx = reservation.rvBillingPersonaId')

            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'nationality', 'with', 'nationality.rfCountryIx = reservation.rfCountryId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'agentPhysCountry', 'with', 'agentPhysCountry.rfCountryIx = agent.prPhysCountryId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'agentPostCountry', 'with', 'agentPostCountry.rfCountryIx = agent.prCorrCountryId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'originatorPhysCountry', 'with', 'originatorPhysCountry.rfCountryIx = originator.prPhysCountryId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'originatorPostCountry', 'with', 'originatorPostCountry.rfCountryIx = originator.prCorrCountryId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'contactPhysCountry', 'with', 'contactPhysCountry.rfCountryIx = contact.prPhysCountryId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RfCountry', 'contactPostCountry', 'with', 'contactPostCountry.rfCountryIx = contact.prCorrCountryId')
            ->where('reservation.rvReservationIx IN(:reservationIds)')
            ->groupBy('reservation.rvReservationIx')
            ->setParameters(
                [
                    'reservationIds' => $reservationIds,
                ]
            )
            ->getQuery()
            ->getResult();
        
        $reservationCountryData = [];
        foreach ($reservationCountryDataResult as $element) {
            $reservationId = $element['reservationId'];
            if (isset($reservationCountryData[$reservationId])) {
                foreach ($element as $field => $value) {
                    if (is_null($value)) {
                        $reservationCountryData[$reservationId][$field] = '';
                    } else {
                        $reservationCountryData[$reservationId][$field] = $value;
                    }
                }
            } else {
                foreach ($element as $field => $value) {
                    if (is_null($value)) {
                        $element[$field] = '';
                    }
                }
                $reservationCountryData[$reservationId] = $element;
            }
        }

        // 1. nationality
        // 2. agent (no orig)
        // 3. orig (only orig or orig and agent)
        // 4. If booking is DIRECT and NO NATIONALITY, contact person's address

        foreach ($reservationCountryData as &$data) {
            $data['originCountryName'] = '';
            $data['originCountryCode'] = '';
            $originCountryCode = &$data['originCountryCode'];
            $originCountryName = &$data['originCountryName'];

            // Nationality is set
            if (!empty($data['nationalityCountryCode'])) {
                $originCountryCode = $data['nationalityCountryCode'];
                $originCountryName = $data['nationalityCountryName'];
                continue;
            }

            // Direct booking, use contact person address
            if ($data['isDirect'] == "1") {
                if (!empty($data['contactPhysCountryCode'])) {
                    $originCountryCode = $data['contactPhysCountryCode'];
                    $originCountryName = $data['contactPhysCountryName'];
                    continue;
                }
                if (!empty($data['contactPostCountryCode'])) {
                    $originCountryCode = $data['contactPostCountryCode'];
                    $originCountryName = $data['contactPostCountryName'];
                    continue;
                }

                if ($data['hasAgent'] == "1") {
                    if (!empty($data['agentPhysCountryCode'])) {
                        $originCountryCode = $data['agentPhysCountryCode'];
                        $originCountryName = $data['agentPhysCountryName'];
                        continue;
                    }
                    if (!empty($data['agentPostCountryCode'])) {
                        $originCountryCode = $data['agentPostCountryCode'];
                        $originCountryName = $data['agentPostCountryName'];
                        continue;
                    }
                }

                if ($data['hasOriginator'] == "1") {
                    if (!empty($data['originatorPhysCountryCode'])) {
                        $originCountryCode = $data['originatorPhysCountryCode'];
                        $originCountryName = $data['originatorPhysCountryName'];
                        continue;
                    }
                    if (!empty($data['originatorPostCountryCode'])) {
                        $originCountryCode = $data['agentPostCountryCode'];
                        $originCountryName = $data['originatorPostCountryName'];
                        continue;
                    }
                }
                continue;
            } else {
                if ($data['hasAgent'] == "1") {
                    if (!empty($data['agentPhysCountryCode'])) {
                        $originCountryCode = $data['agentPhysCountryCode'];
                        $originCountryName = $data['agentPhysCountryName'];
                        continue;
                    }
                    if (!empty($data['agentPostCountryCode'])) {
                        $originCountryCode = $data['agentPostCountryCode'];
                        $originCountryName = $data['agentPostCountryName'];
                        continue;
                    }
                }

                if ($data['hasOriginator'] == "1") {
                    if (!empty($data['originatorPhysCountryCode'])) {
                        $originCountryCode = $data['originatorPhysCountryCode'];
                        $originCountryName = $data['originatorPhysCountryName'];
                        continue;
                    }
                    if (!empty($data['originatorPostCountryCode'])) {
                        $originCountryCode = $data['agentPostCountryCode'];
                        $originCountryName = $data['originatorPostCountryName'];
                        continue;
                    }
                }
            }
        }
        unset($data);
        
        if ($excludeReservationsWithoutCountry === true) {
            foreach ($reservationCountryData as $key => $data) {
                if (empty($data['originCountryCode'])) {
                    unset($reservationCountryData[$key]);
                }
            }
        }

        return $reservationCountryData;
    }
}
