<?php

namespace Resrequest\Application\Chart;

use MaglLegacyApplication\Application\MaglLegacy;

abstract class Dataset
{
    protected $em;
    protected $name;
    protected $options = [];
    protected $converters = [];
    protected $debugData = [];

    public function __construct(string $name, array $config)
    {
        $this->em = MaglLegacy::getServiceManager()->get('doctrine.entitymanager.orm_enterprise');
        $this->name = $name;

        foreach ($this->generateOptions() as $option) {
            $name = $option->getName();
            $this->options[$name] = $option;
        }

        // Update supported options with the provided config
        if (!empty($config['options'])) {
            foreach ($config['options'] as $option) {
                if (empty($option['config']['name'])) {
                    throw new \Exception('Invalid option config');
                }

                $optionConfig = $option['config'];
                $name = $optionConfig['name'];
                if ($this->isOptionSupported($name)) {
                    $option = $this->getOptionByName($name);
                    // Apply option config
                    $option->update($optionConfig);
                }
            }
        }

        if (!empty($config['converters'])) {
            $convertersConfig = $config['converters'];

            foreach ($convertersConfig as $converterConfig) {
                $name = $converterConfig['name'];
                $type = $converterConfig['type'];
                if (Registry::hasConverter($type)) {
                    $converterClass = Registry::getConverterClass($type);
                    $config = $converterConfig;
                    $this->converters[] = new $converterClass($name, $config);
                } else {
                    throw new \Exception("Invalid converter '$type'");
                }
            }
        }
    }

    protected function isOptionSupported($name)
    {
        return array_key_exists($name, $this->options);
    }

    protected function getOptionByName($name)
    {
        if (array_key_exists($name, $this->options)) {
            return $this->options[$name];
        } else {
            throw new \Exception("Option '$name' does not exist");
        }
    }

    public function getData()
    {
        $this->debugData = [];
        $startTime = microtime(true); 
        $initialMemory = memory_get_usage();
        $initialPeakMemory = memory_get_peak_usage();

        $data = $this->buildData();

        $endTime = microtime(true);
        $executionTime = ($endTime - $startTime);
        $elementCount = count($data);
        $averageProcessTimerPerElement = empty($elementCount) ? 0 : ($executionTime / $elementCount);

        $this->debugData['buildData'] = [
            'startTime' => $startTime,
            'endTime' => $endTime,
            'initialMemory' => $initialMemory,
            'memory' => memory_get_usage(),
            'initialPeakMemory' => $initialPeakMemory,
            'peakMemory' => memory_get_peak_usage(),
            'executionTime' => $executionTime,
            'elementCount' => $elementCount,
            'averageProcessTimerPerElement' => $averageProcessTimerPerElement
        ];

        $convertStartTime = microtime(true); 
        $this->debugData['convertData'] = [
            'startTime' => $convertStartTime,
            'endTime' => null,
            'initialMemory' => memory_get_usage(),
            'memory' => null,
            'peakMemory' => null,
            'executionTime' => null,
            'initialElementCount' => $elementCount,
            'elementCount' => null,
            'averageProcessTimerPerElement' => null,
            'converters' => []
        ];


        // Break here to debug converter input and output
        foreach ($this->converters as $converter) {
            $startTime = microtime(true); 
            $initialElementCount = count($data);
            $initialMemory = memory_get_usage();

            $data = $converter->run($data);
            
            $endTime = microtime(true);
            $executionTime = ($endTime - $startTime);

            $elementCount = count($data);

            $averageProcessTimerPerElement = empty($elementCount) ? 0 : ($executionTime / $elementCount);

            $this->debugData['convertData']['converters'][] = [
                'subType' => $converter->getSubType(),
                'startTime' => $startTime,
                'endTime' => $endTime,
                'initialMemory' => $initialMemory,
                'memory' => memory_get_usage(),
                'peakMemory' => memory_get_peak_usage(),
                'executionTime' => $executionTime,
                'initialElementCount' => $initialElementCount,
                'elementCount' => $elementCount,
                'averageProcessTimerPerElement' => $averageProcessTimerPerElement
            ];
        }

        $convertEndTime = microtime(true);
        $executionTime = ($convertEndTime - $convertStartTime);

        $elementCount = count($data);
        $averageProcessTimerPerElement = empty($elementCount) ? 0 : ($executionTime / $elementCount);


        $this->debugData['convertData'] = array_merge($this->debugData['convertData'], [
            'endTime' => $convertEndTime,
            'memory' => memory_get_usage(),
            'peakMemory' => memory_get_peak_usage(),
            'executionTime' => $executionTime,
            'elementCount' => $elementCount,
            'averageProcessTimerPerElement' => $averageProcessTimerPerElement,
        ]);

        return $data;
    }

    /**
     * Implement in dataset child class.
     * Dataset logic goes here
     *
     * @return void
     */
    protected abstract function buildData();

    /**
     * Returns dataset name, which is the class name
     * in lower camel case
     *
     * @return string $datasetName
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Override in child class
     *
     * @return array Array of options
     */
    protected function generateOptions()
    {
        return [];
    }

    protected function valueForOption($name)
    {
        $option = $this->getOptionByName($name);
        $value = $option->run();

        return $value[0];
    }

    public function getDebugData() {
        return $this->debugData;
    }

    function toArray()
    {
        $optionConfigs = [];

        foreach ($this->options as $option) {
            $optionConfigs[] = [
                'type' => $option->getType(),
                'config' => $option->toArray()
            ];
        }

        $converterConfigs = [];

        foreach ($this->converters as $converter) {
            $converterConfigs[] = $converter->toArray();
        }

        return [
            'name' => $this->name,
            'config' => [
                'options' => $optionConfigs,
                'converters' => $converterConfigs
            ]
        ];
    }
}
