<?php

namespace Resrequest\Application\Chart\Dataset;

use Resrequest\Application\Chart\Dataset;
use Resrequest\Application\Chart\Filter\AccommodationAccess;
use Resrequest\Application\Chart\Filter\DateRange;

class ArrivalsWithoutFullPayment extends Dataset
{
    protected function generateOptions()
    {
        return [
            new AccommodationAccess('accommodationAccess'),
            new DateRange('arrivalPeriod'),
        ];
    }

    public function buildData()
    {
        $reservationsQuery = $this->em->createQueryBuilder();

        $reservationStatusFilter = [
            'Confirmed',
            'Provisional'
        ];

        $dateFilter = $this->valueForOption('arrivalPeriod');
        $accommodationAccess = $this->valueForOption('accommodationAccess');
        $today = new \DateTime('today');
        $today = $today->format('Y-m-d');

        $reservations = $reservationsQuery
            ->select(
                [
                    'DISTINCT reservation.rvReservationIx AS reservationId',
                    'reservation.rvAmtAccommPayable AS accommAmount',
                    'reservation.rvAmtExtraPayable AS extraAmount',
                    'reservation.rvAmtTravelPayable AS travelAmount',
                    'reservation.rvAmtPaid AS paidAmount',
                    'persona.prNameFirst as consultantFirstName',
                    'persona.prNameLast as consultantLastName',
                    'reservation.rvConsultantId  AS consultantId',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservation.rvConsultantId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->where('reservation.rvDateArrive >= :arrivalStartDate')
            ->andWhere('reservation.rvDateArrive <= :arrivalEndDate')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->andWhere('(reservation.rvProvisionExpiryDate >= :today AND reservation.rfReservationStatusId = 20) OR reservation.rfReservationStatusId != 20')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->setParameters(
                [
                    'reservationStatusFilter' => $reservationStatusFilter,
                    'arrivalStartDate' => $dateFilter['startDate'],
                    'arrivalEndDate' => $dateFilter['endDate'],
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations'],
                    'today' => $today
                ]
            )
            ->getQuery()
            ->getResult();

        $consultants = [];
        $totalReservations = 0;
        $bookingsWithoutFullPayment = [];

        foreach ($reservations as $reservation) {
            $consultantId = $reservation['consultantId'];
            if (!isset($bookingsWithoutFullPayment[$consultantId])) {
                $bookingsWithoutFullPayment[$consultantId] = 0;
            }

            $accommAmount = $reservation['accommAmount'];
            $travelAmount = $reservation['travelAmount'];
            $extraAmount = $reservation['extraAmount'];
            $paidAmount = $reservation['paidAmount'];

            if (!$this->isPaid($accommAmount, $extraAmount, $travelAmount, $paidAmount)) {
                $bookingsWithoutFullPayment[$consultantId]++;
            }

            if (!isset($consultants[$reservation['consultantId']])) {
                $firstName = $reservation['consultantFirstName'];
                $lastName = $reservation['consultantLastName'];

                if (is_numeric($consultantId) && $consultantId == 0) {
                    $firstName = "No consultant";
                }

                $consultants[$consultantId] = trim("$firstName $lastName");
            }
            $totalReservations++;
        }

        $dataset = [];

        foreach ($consultants as $consultantId => $consultantName) {
            array_push($dataset, [
                'consultantId' => $consultantId,
                'consultantName' => $consultantName,
                'bookingsWithoutFullPayment' => $bookingsWithoutFullPayment[$consultantId],
                'totalBookings' => $totalReservations
            ]);
        }

        return $dataset;
    }

    // Adapted from getPaymentStatus() in init.form376.func.php
    public function isPaid($accommAmount, $extraAmount, $travelAmount, $paidAmount)
    {
        if (is_null($accommAmount)) {
            // TBA is unpaid
            return false;
        }

        $total = $accommAmount + $extraAmount + $travelAmount;
        $outstanding = $total - $paidAmount;
        if ($outstanding == 0) {
            // Paid
            return true;
        } else if ($outstanding > 0) {
            if ($paidAmount == 0) {
                // No payment
                return false;
            } else {
                // Part paid
                return false;
            }
        } else {
            // Over paid
            return true;
        }
    }
}
