<?php

namespace Resrequest\Application\Chart\Dataset;

require_once(__DIR__ . '../../../../legacy/class.itinerary.php');

use Resrequest\Application\Chart\Dataset;
use Resrequest\Application\Chart\DatasetOption;
use Resrequest\Application\Chart\Filter\AccommodationAccess;
use Resrequest\Application\Chart\Filter\DateRange;
use Resrequest\Application\Chart\Filter\RateComponent;
use Resrequest\Application\Chart\Filter\RateType;
use Resrequest\Application\Chart\Filter\ReservationStatus;
use Resrequest\Application\Chart\Input;
use Resrequest\Application\Chart\Input\ButtonToggle;
use Resrequest\Application\Chart\Option;
use Resrequest\Application\Chart\Input\RadioGroup;
use Resrequest\Application\Chart\Input\SliderToggle;

class BookingRevenue extends Dataset
{
    protected function generateOptions()
    {
        $radioGroup = new RadioGroup();
        $radioGroup->setValue('nett');
        $radioGroup->addOption('Nett', 'nett');
        $radioGroup->addOption('Gross', 'gross');
        $radioGroup->addOption('Payable', 'payable');

        $accommTaxToggle = new ButtonToggle();
        $accommTaxToggle->setValue(true);
        $accommTaxToggle->addOption('Inclusive', true);
        $accommTaxToggle->addOption('Exclusive', false);

        $farByToggle = new ButtonToggle();
        $farByToggle->setValue('day');
        $farByToggle->addOption('Day', 'day');
        $farByToggle->addOption('Folio', 'folio');

        return [
            new RateType('rateType'),
            new RateComponent('rateComponent'),
            new AccommodationAccess('accommodationAccess'),
            new ReservationStatus('reservationStatus'),
            new DateRange('dateRange'),
            new DateRange('compare1DateRange', [
                'startDate' => '1 January this year',
                'endDate' => '1 January this year'
            ]),
            new DateRange('compare2DateRange', [
                'startDate' => '1 January this year',
                'endDate' => '1 January this year'
            ]),
            new DatasetOption('farByFolio', [new Option('farByFolio', $farByToggle)]),
            new DatasetOption('excludeTba', [new Option('excludeTba', new SliderToggle(false, 'Exclude TBAs'))]),
            new DatasetOption('commission', [new Option('commission', $radioGroup)]),
            new DatasetOption('includeAccommTax', [new Option('includeAccommTax', $accommTaxToggle)]),
            new DatasetOption('agent', [new Option('agent', new Input(false))]),
            new DatasetOption('calculateLostGainedRevenue', [new Option('calculateLostGainedRevenue', new Input(false))]),
            new DateRange('saleLostGainedDateRange', [
                'startDate' => 'first day of January this year',
                'endDate' => 'last day of January this year'
            ]),
            new DateRange('compare1SaleLostGainedDateRange', [
                'startDate' => 'first day of January this year',
                'endDate' => 'last day of January this year'
            ]),
            new DateRange('compare2SaleLostGainedDateRange', [
                'startDate' => 'first day of January this year',
                'endDate' => 'last day of January this year'
            ]),
        ];
    }

    public function buildData()
    {
        $farByFolio = $this->valueForOption('farByFolio')['farByFolio'];
        $commission = $this->valueForOption('commission')['commission'];
        $includeAccommTax = $this->valueForOption('includeAccommTax')['includeAccommTax'];
        $excludeTba = $this->valueForOption('excludeTba')['excludeTba'];
        $reservationStatus = $this->valueForOption('reservationStatus');
        $components = $this->valueForOption('rateComponent')['selectedComponents'];
        $rateTypes = $this->valueForOption('rateType')['selectedRateTypes'];
        $dateFilter = $this->valueForOption('dateRange');
        $compare1DateFilter = $this->valueForOption('compare1DateRange');
        $compare2DateFilter = $this->valueForOption('compare2DateRange');
        $agent = $this->valueForOption('agent')['agent'];
        $calculateLostGainedRevenue = $this->valueForOption('calculateLostGainedRevenue')['calculateLostGainedRevenue'];
        $saleLostGainedDateRange = $this->valueForOption('saleLostGainedDateRange');
        $c1SaleLostGainedDateRange = $this->valueForOption('compare1SaleLostGainedDateRange');
        $c2SaleLostGainedDateRange = $this->valueForOption('compare2SaleLostGainedDateRange');

        $startDate = $dateFilter['startDate'];
        $endDate = $dateFilter['endDate'];

        $c1StartDate = $compare1DateFilter['startDate'];
        $c1EndDate = $compare1DateFilter['endDate'];

        $c2StartDate = $compare2DateFilter['startDate'];
        $c2EndDate = $compare2DateFilter['endDate'];

        if ($farByFolio === 'folio') {
            $data = $this->folio($startDate, $endDate, $commission, $includeAccommTax, $excludeTba, $reservationStatus, $components, $rateTypes, $agent, $calculateLostGainedRevenue, $saleLostGainedDateRange, 0);
            $compare1Data = $this->folio($c1StartDate, $c1EndDate, $commission, $includeAccommTax, $excludeTba, $reservationStatus, $components, $rateTypes, $agent, $calculateLostGainedRevenue, $c1SaleLostGainedDateRange, 1);
            $compare2Data = $this->folio($c2StartDate, $c2EndDate, $commission, $includeAccommTax, $excludeTba, $reservationStatus, $components, $rateTypes, $agent, $calculateLostGainedRevenue, $c2SaleLostGainedDateRange, 2);
        } else {
            $data = $this->day($startDate, $endDate, $commission, $includeAccommTax, $excludeTba, $reservationStatus, $components, $rateTypes, $agent, $calculateLostGainedRevenue, $saleLostGainedDateRange, 0);
            $compare1Data = $this->day($c1StartDate, $c1EndDate, $commission, $includeAccommTax, $excludeTba, $reservationStatus, $components, $rateTypes, $agent, $calculateLostGainedRevenue, $c1SaleLostGainedDateRange, 1);
            $compare2Data = $this->day($c2StartDate, $c2EndDate, $commission, $includeAccommTax, $excludeTba, $reservationStatus, $components, $rateTypes, $agent, $calculateLostGainedRevenue, $c2SaleLostGainedDateRange, 2);
        }

        $data = array_merge($data, $compare1Data, $compare2Data);

        // Add date ranges to final output
        foreach ($data as &$element) {
            $element['travelStartDate'] = $startDate;
            $element['travelEndDate'] = $endDate;

            $element['c1TravelStartDate'] = $c1StartDate;
            $element['c1TravelEndDate'] = $c1EndDate;

            $element['c2TravelStartDate'] = $c2StartDate;
            $element['c2TravelEndDate'] = $c2EndDate;
        }

        return $data;
    }


    /**
     * Calculate revenue for each day of stay
     *
     * @return array
     */
    function day($startDate, $endDate, $commission, $includeAccommTax, $excludeTba, $reservationStatus, $components, $rateTypes, $agent = false, $calculateLostGainedRevenue = false, $saleLostGainedDate = null, $comparison = 0)
    {
        $reservationItemQuery = $this->em->createQueryBuilder();
        $accommodationAccess = $this->valueForOption('accommodationAccess');

        $reservationItemQuery = $reservationItemQuery
            ->select(
                [
                    'reservation.rvReservationIx as reservationId',
                    'currency.rfCurrencySymbol as invoiceCurrency',
                    'IF(reservation.rfReservationStatusId = 20, IF(reservation.rvProvisionExpiryDate >= :today, TRUE, FALSE), NULL) AS valid',
                    'reservation.rvProvisionExpiryDate AS expiryDate',
                    'reservationStatusTable.rfReservationStatusDesc AS status',

                    'reservationItem.rvReservationItemIx as reservationItemId',

                    'reservationItem.rvItemNights as nights',
                    'reservationItem.rvItemAccommCount as rooms',
                    'reservationItem.rvItemDateArrive as arrivalDate',
                    'reservationItem.rvItemDateDepart as departureDate',
                    'reservationItem.rvItemAdultCount as adultCount',
                    'reservationItem.rvItemChildCount as childCount',

                    'SUM(resComp.rvItemCompAmtPayable) as payableAmount',
                    'SUM(resComp.rvItemCompAmtGross) as grossAmount',
                    'SUM(resComp.rvItemCompAmtNett) as nettAmount',
                    'SUM(resComp.rvItemCompAmtTax) as taxAmount',
                    'SUM(resComp.rvItemCompAmtComm) as commissionAmount',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation', 'with', 'reservation.rvReservationIx = reservationItem.rvReservationId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvResItemComp', 'resComp', 'with', 'resComp.rvReservationItemId = reservationItem.rvReservationItemIx AND resComp.rtComponentId IN(:selectedComponents)')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RtComponent', 'comp', 'with', 'comp.rtComponentIx = resComp.rtComponentId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfCurrency', 'currency', 'with', 'currency.rfCurrencyIx = reservation.rvInvoiceCurrencyId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->where('(
                (
                    reservationItem.rvItemDateArrive >= :startDate
                    AND reservationItem.rvItemDateArrive <= :endDate
                ) OR (
                    reservationItem.rvItemDateDepart >= :startDate
                    AND reservationItem.rvItemDateDepart <= :endDate
                ) OR (
                    reservationItem.rvItemDateArrive < :startDate
                    AND reservationItem.rvItemDateDepart > :endDate
                )
            )')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->andWhere('reservationStatusTable.rfReservationStatusId IN(:reservationStatuses)')
            ->andWhere('reservationItem.rtRateTypeId IN(:selectedRateTypes)')
            ->having('valid IN(:provisionalFilter) OR valid IS NULL')
            ->groupBy('reservationItem.rvReservationItemIx')
            ->setParameters(
                [
                    'reservationStatuses' => $reservationStatus['selectedStatuses'],
                    'startDate' => $startDate,
                    'endDate' => $endDate,
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations'],
                    'selectedComponents' => $components,
                    'selectedRateTypes' => $rateTypes,
                    'provisionalFilter' => $reservationStatus['provisionalFilter'],
                    'today' => (new \DateTime())->format('Y-m-d')
                ]
            );

        $attachAuditTrail = $calculateLostGainedRevenue;

        if ($attachAuditTrail) {
            $reservationItemQuery
                ->addSelect(
                    'adReservation.adResTime AS time',
                    'adReservation.adResForm AS description'
                )
                ->leftJoin('Resrequest\DB\Enterprise\Entity\AdReservation', 'adReservation', 'with', 'adReservation.rvReservationId = reservation.rvReservationIx')
                ->addGroupBy('adReservation.adReservationIx');
        }

        // Exclude TBA reservations
        if ($excludeTba === true) {
            $reservationItemQuery->andWhere('reservation.rvAmtAccommGross IS NOT NULL');
        }

        if ($agent !== false) {
            $reservationItemQuery
                ->andWhere('reservation.rvAgentId = :agentId')
                ->setParameter('agentId', $agent);
        }

        $reservationItems = $reservationItemQuery->getQuery()->getResult();

        // Fetch the audit trail and attach it to each reservation item
        if ($attachAuditTrail) {
            $reservationItems = $this->groupData(
                $reservationItems,
                'reservationItemId',
                [
                    'tmpAuditTrail' => [
                        'time',
                        'description'
                    ]
                ],
                true
            );

            foreach ($reservationItems as $index => $reservationItem) {
                $reservationItems[$index]['auditTrail'] = $this->processAuditTrail($reservationItems[$index]['tmpAuditTrail']);
                unset($reservationItems[$index]['tmpAuditTrail']);
            }

            $reservationItems = array_values($reservationItems);
        }

        $output = [];
        foreach ($reservationItems as $reservationItem) {
            $arrivalDate = $reservationItem['arrivalDate'];
            $departureDate = $reservationItem['departureDate'];
            $reservationItemNights = $reservationItem['nights'];
            $rooms = $reservationItem['rooms'];

            $adultCount = $reservationItem['adultCount'];
            $childCount = $reservationItem['childCount'];

            $pax = $adultCount + $childCount;

            $bedNights = $rooms * $reservationItemNights * $pax;
            $roomNights = $rooms * $reservationItemNights;

            if ($calculateLostGainedRevenue) {
                $saleData = $this->lostOrGainedSale($reservationItem);
                $saleGained = $saleData['gained'];
                $saleTime = $saleData['time'];

                // Exclude bookings that never had a stock holding status
                if ($saleTime === false) {
                    continue;
                } else {
                    // Remove timestamp for comparison
                    $saleTime = $saleTime->setTime(0, 0);
                }

                $tmpStartDate = new \DateTime($saleLostGainedDate['startDate']);
                $tmpEndDate = new \DateTime($saleLostGainedDate['endDate']);

                // Set lost/gain sale date filter to the same year as the travel period
                $tmpStartDate->modify((new \DateTime($startDate))->format('Y') . '-' . $tmpStartDate->format('m-d'));
                $tmpEndDate->modify((new \DateTime($endDate))->format('Y') . '-' . $tmpEndDate->format('m-d'));

                // Ignore bookings that don't fall in the sale lost/gained period
                if (
                    $saleTime < $tmpStartDate || $saleTime > $tmpEndDate
                ) {
                    continue;
                }
            }

            $interval = \DateInterval::createFromDateString('1 day');
            $reservationItemPeriod = new \DatePeriod($arrivalDate, $interval, $departureDate);

            // Calculate revenue for each day of reservation item
            foreach ($reservationItemPeriod as $date) {
                $data = [
                        'commissionAmount' => 0,
                        'nettAmount' => 0,
                        'payableAmount' => 0,
                        'grossAmount' => 0,
                        'taxAmount' => 0,
                        'bedNights' => 0,
                        'roomNights' => 0,
                        'revenue' => 0,
                        'revenueGained' => 0,
                        'revenueLost' => 0,

                        'reservationId' => '',
                        'reservationItemId' => '',
                        'invoiceCurrency' => '',
                        'comparisonData' => "$comparison",
                        'date' => $date->format('Y-m-d')
                    ];

                $data['reservationId'] = $reservationItem['reservationId'];
                $data['reservationItemId'] = $reservationItem['reservationItemId'];
                $data['invoiceCurrency'] = $reservationItem['invoiceCurrency'];
                $data['commissionAmount'] = $reservationItem['commissionAmount'] / $reservationItemNights;
                $data['nettAmount'] = $reservationItem['nettAmount'] / $reservationItemNights;
                $data['payableAmount'] = $reservationItem['payableAmount'] / $reservationItemNights;
                $data['grossAmount'] = $reservationItem['grossAmount'] / $reservationItemNights;
                $data['taxAmount'] = $reservationItem['taxAmount'] / $reservationItemNights;
                $data['bedNights'] = $bedNights / $reservationItemNights;
                $data['roomNights'] = $roomNights / $reservationItemNights;

                // Set revenue to the gross/nett/payable amount
                switch ($commission) {
                    case ('nett'):
                        $revenue = $reservationItem['nettAmount'];
                        break;
                    case ('gross'):
                        $revenue = $reservationItem['grossAmount'];
                        break;
                    case ('payable'):
                        $revenue = $reservationItem['payableAmount'];
                        break;
                }

                $revenue = $revenue / $reservationItemNights;

                if ($includeAccommTax === false) {
                    $accommodationTax = $reservationItem['taxAmount'] / $reservationItemNights;
                    $revenue = $revenue - $accommodationTax;
                }

                $data['revenue'] = $revenue;

                if ($calculateLostGainedRevenue) {
                    if ($saleGained) {
                        $data['revenueGained'] = $revenue;
                    } else {
                        $data['revenueLost'] = $revenue;
                    }
                }

                // Skip items that fall out of the date range
                if (
                    $date->format('Y-m-d') < $startDate
                    || $date->format('Y-m-d') > $endDate
                ) {
                    continue;
                }

                $output[] = $data;
            }
        }

        return $output;
    }


    /**
     * Calculate revenue using the folio/invoice date
     *
     * @return array
     */
    function folio($startDate, $endDate, $commission, $includeAccommTax, $excludeTba, $reservationStatus, $components, $rateTypes, $agent = false, $calculateLostGainedRevenue = false, $saleLostGainedDate = null, $comparison = 0)
    {
        $reservationItemQuery = $this->em->createQueryBuilder();
        $accommodationAccess = $this->valueForOption('accommodationAccess');

        $reservationItemQuery = $reservationItemQuery
            ->select(
                [
                    'reservation.rvReservationIx as reservationId',
                    'currency.rfCurrencySymbol as invoiceCurrency',
                    'IF(reservation.rfReservationStatusId = 20, IF(reservation.rvProvisionExpiryDate >= :today, TRUE, FALSE), NULL) AS valid',
                    'reservation.rvProvisionExpiryDate AS expiryDate',
                    'reservationStatusTable.rfReservationStatusDesc AS status',

                    'SUM(resComp.rvItemCompAmtPayable) as payableAmount',
                    'SUM(resComp.rvItemCompAmtGross) as grossAmount',
                    'SUM(resComp.rvItemCompAmtNett) as nettAmount',
                    'SUM(resComp.rvItemCompAmtTax) as taxAmount',
                    'SUM(resComp.rvItemCompAmtComm) as commissionAmount',

                    'reservationItem.rvItemNights as nights',
                    'reservationItem.rvItemAccommCount as rooms',
                    'reservationItem.rvItemDateArrive as arrivalDate',
                    'reservationItem.rvItemDateDepart as departureDate',
                    'reservationItem.rvItemAdultCount as adultCount',
                    'reservationItem.rvItemChildCount as childCount',
                    'folio.fnFolioDate as folioDate',
                    'invoice.fnInvDate as invoiceDate',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation', 'with', 'reservation.rvReservationIx = reservationItem.rvReservationId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvResItemComp', 'resComp', 'with', 'resComp.rvReservationItemId = reservationItem.rvReservationItemIx AND resComp.rtComponentId IN(:selectedComponents)')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RtComponent', 'comp', 'with', 'comp.rtComponentIx = resComp.rtComponentId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\FnFolio', 'folio', 'with', 'folio.fnFolioIx = reservationItem.fnFolioId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\FnInvoice', 'invoice', 'with', 'invoice.fnInvoiceIx = folio.fnInvoiceId AND folio.fnFolioStatusInd = 2 AND invoice.fnInvStatusInd <> 8')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfCurrency', 'currency', 'with', 'currency.rfCurrencyIx = reservation.rvInvoiceCurrencyId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->where('(
                (
                    folio.fnFolioStatusInd <> 2 AND folio.fnFolioDate >= :startDate AND folio.fnFolioDate <= :endDate
                ) OR (
                    folio.fnFolioStatusInd = 2 AND invoice.fnInvDate >= :startDate AND invoice.fnInvDate <= :endDate
                )
            )')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->andWhere('reservation.rfReservationStatusId IN(:reservationStatuses)')
            ->andWhere('reservationItem.rtRateTypeId IN(:selectedRateTypes)')
            ->having('valid IN(:provisionalFilter) OR valid IS NULL')
            ->setParameters(
                [
                    'reservationStatuses' => $reservationStatus['selectedStatuses'],
                    'startDate' => $startDate,
                    'endDate' => $endDate,
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations'],
                    'selectedComponents' => $components,
                    'selectedRateTypes' => $rateTypes,
                    'provisionalFilter' => $reservationStatus['provisionalFilter'],
                    'today' => (new \DateTime())->format('Y-m-d')
                ]
            )
            ->groupBy(
                'reservationItem.rvReservationItemIx'
            );

        $attachAuditTrail = $calculateLostGainedRevenue;

        if ($attachAuditTrail) {
            $reservationItemQuery
                ->addSelect(
                    'adReservation.adResTime AS time',
                    'adReservation.adResForm AS description'
                )
                ->leftJoin('Resrequest\DB\Enterprise\Entity\AdReservation', 'adReservation', 'with', 'adReservation.rvReservationId = reservation.rvReservationIx')
                ->groupBy('reservationItem.rvReservationItemIx', 'adReservation.adReservationIx');
        } else {
            $reservationItemQuery
                ->groupBy('reservationItem.rvReservationItemIx');
        }

        // Exclude TBA reservations
        if ($excludeTba === true) {
            $reservationItemQuery->andWhere('reservation.rvAmtAccommGross IS NOT NULL');
        }

        if ($agent !== false) {
            $reservationItemQuery
                ->andWhere('reservation.rvAgentId = :agentId')
                ->setParameter('agentId', $agent);
        }

        $reservationItems = $reservationItemQuery->getQuery()->getResult();

         // Fetch the audit trail and attach it to each reservation item
         if ($attachAuditTrail) {
            $reservationItems = $this->groupData(
                $reservationItems,
                'reservationId',
                [
                    'tmpAuditTrail' => [
                        'time',
                        'description'
                    ]
                ],
                true
            );

            foreach ($reservationItems as $index => $reservationItem) {
                $reservationItems[$index]['auditTrail'] = $this->processAuditTrail($reservationItems[$index]['tmpAuditTrail']);
                unset($reservationItems[$index]['tmpAuditTrail']);
            }

            $reservationItems = array_values($reservationItems);
        }

        $output = [];
        foreach ($reservationItems as $reservationItem) {
            $invoiceDate = $reservationItem['invoiceDate'];
            $folioDate = $reservationItem['folioDate'];
            $reservationItemNights = $reservationItem['nights'];
            $rooms = $reservationItem['rooms'];
            $currency = $reservationItem['invoiceCurrency'];
            $accommodationTax = $reservationItem['taxAmount'];

            $adultCount = $reservationItem['adultCount'];
            $childCount = $reservationItem['childCount'];
            $pax = $adultCount + $childCount;

            $bedNights = $rooms * $reservationItemNights * $pax;
            $roomNights = $rooms * $reservationItemNights;

            if ($calculateLostGainedRevenue) {
                $saleData = $this->lostOrGainedSale($reservationItem);
                $saleGained = $saleData['gained'];
                $saleTime = $saleData['time'];

                // Exclude bookings that never had a stock holding status
                if ($saleTime === false) {
                    continue;
                } else {
                    // Remove timestamp for comparison
                    $saleTime = $saleTime->setTime(0, 0);
                }

                $tmpStartDate = new \DateTime($saleLostGainedDate['startDate']);
                $tmpEndDate = new \DateTime($saleLostGainedDate['endDate']);

                // Set lost/gain sale date filter to the same year as the travel period
                $tmpStartDate->modify((new \DateTime($startDate))->format('Y') . '-' . $tmpStartDate->format('m-d'));
                $tmpEndDate->modify((new \DateTime($endDate))->format('Y') . '-' . $tmpEndDate->format('m-d'));

                // Ignore bookings that don't fall in the sale lost/gained period
                if (
                    $saleTime < $tmpStartDate || $saleTime > $tmpEndDate
                ) {
                    continue;
                }
            }

            // If there is no invoice, use the folio date
            if (is_null($invoiceDate)) {
                $dayString = $folioDate->format('Y-m-d');
            } else {
                $dayString = $invoiceDate->format('Y-m-d');
            }

            $data = [
                'commissionAmount' => 0,
                'nettAmount' => 0,
                'payableAmount' => 0,
                'grossAmount' => 0,
                'taxAmount' => 0,
                'bedNights' => 0,
                'roomNights' => 0,
                'revenue' => 0,
                'revenueGained' => 0,
                'revenueLost' => 0,

                'reservationId' => '',
                'reservationItemId' => 'NA',
                'invoiceCurrency' => '',
                'comparisonData' => "$comparison",
                'date' => $dayString
            ];

            $data['reservationId'] = $reservationItem['reservationId'];
            $data['invoiceCurrency'] = $reservationItem['invoiceCurrency'];
            $data['commissionAmount'] = $reservationItem['commissionAmount'];
            $data['nettAmount'] = $reservationItem['nettAmount'];
            $data['payableAmount'] = $reservationItem['payableAmount'];
            $data['grossAmount'] = $reservationItem['grossAmount'];
            $data['taxAmount'] = $reservationItem['taxAmount'];
            $data['bedNights'] = $bedNights;
            $data['roomNights'] = $roomNights;

            // Set revenue to the gross/nett/payable amount
            switch ($commission) {
                case ('nett'):
                    $revenue = $reservationItem['nettAmount'];
                    break;
                case ('gross'):
                    $revenue = $reservationItem['grossAmount'];
                    break;
                case ('payable'):
                    $revenue = $reservationItem['payableAmount'];
                    break;
            }

            if ($includeAccommTax === false) {
                $revenue = $revenue - $accommodationTax;
            }

            $data['revenue'] = $revenue;

            if ($calculateLostGainedRevenue) {
                if ($saleGained) {
                    $data['revenueGained'] = $revenue;
                } else {
                    $data['revenueLost'] = $revenue;
                }
            }

            // Skip items that fall out of the date range
            if (
                $dayString < $startDate
                || $dayString > $endDate
            ) {
                continue;
            }

            $output[] = $data;
        }

        return $output;
    }

    function getReservationAuditTrail(array $reservations) {
        if (empty($reservations)) {
            return [];
        }

        $auditTrailQuery = $this->em->createQueryBuilder();

        $auditTrail = $auditTrailQuery
            ->select(
                [
                    'reservation.rvReservationIx as reservationId',
                    'adReservation.adResTime AS time',
                    'adReservation.adResForm AS description'
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\AdReservation', 'adReservation')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation', 'with', 'reservation.rvReservationIx = adReservation.rvReservationId')
            ->where('adReservation.rvReservationId IN(:reservations)')
            ->orderBy('adReservation.adResTime', 'ASC')
            ->setParameters(
                [
                    'reservations' => $reservations,
                ]
            )
            ->getQuery()
            ->getResult();

        $auditTrail = $this->groupData(
            $auditTrail,
            'reservationId',
            [
                'auditTrail' => [
                    'time',
                    'description'
                ]
            ],
            true
        );

        return $auditTrail;
    }

    private function groupData(array $data, string $groupByColumn, array $groups, $associative = false) {
        $groupedData = [];

        foreach ($data as $element) {
            $groupBy = $element[$groupByColumn];
            if (!array_key_exists($groupBy, $groupedData)) {
                $groupedData[$groupBy] = $element;
            }

            $tmpData = [];
            foreach ($groups as $key => $columns) {
                if (!array_key_exists($key, $groupedData[$groupBy])) {
                    $groupedData[$groupBy][$key] = [];
                }

                foreach ($columns as $column) {
                    $tmpData[$column] = $element[$column];
                    unset($groupedData[$groupBy][$column]);
                }
            }
            $groupedData[$groupBy][$key][] = $tmpData;
        }

        if ($associative) {
            return $groupedData;
        } else {
            return array_values($groupedData);
        }
    }

    private function processAuditTrail($auditTrail) {
        $processedTrail = [];
        foreach ($auditTrail as $change) {
            // Process reservation status changes
            if (preg_match("/^Reservation Status: ([\w\s]*) to ([\w\s]*)(?:, expires ([\w ]*))?/", $change['description'], $matches)) {
                $from = $matches[1];
                $to = $matches[2];
                $expires = false;
                if (isset($matches[3])) {
                    $expires = new \DateTime($matches[3]);
                }

                $processedTrail[] = [
                    'type' => 'statusChange',
                    'time' => $change['time'],
                    'change' => [
                        'from' => $from,
                        'to' => $to,
                        'expires' => $expires
                    ]
                ];
            }
        }

        return $processedTrail;
    }
    /**
     * Determine whether a reservation item revenue was gained or lost.
     * 
     * If the booking was stock holding at any point and now isn't
     * it's lost revenue.
     * 
     * If the booking was never stock holding
     * no profit was gained or lost.
     * 
     * If the booking is currently stock holding
     * it is gained profit.
     *
     * @return boolean False for last sale, true for gained sale
     */
    private function lostOrGainedSale($reservationItem) {
        $status = $reservationItem['status']; // Curreny reservation status
        $expired = !boolval($reservationItem['valid']); // Expired provisional
        $expiryDate = $reservationItem['expiryDate']; // Provisional expiry date
        $auditTrail = $reservationItem['auditTrail'];

        $stockHoldingStatuses = [
            'Confirmed',
            'Provisional'
        ];

        $gained = false; // Whether the sale was gained or lost
        $time = false; // The time of the gained or lost sale

        foreach ($auditTrail as $change) {
            if ($change['type'] !== 'statusChange') {
                continue;
            }

            $from = $change['change']['from'];
            $to = $change['change']['to'];

            if (
                in_array($from, $stockHoldingStatuses)
                && in_array($to, $stockHoldingStatuses)
            ) {
                // Changed from a stock holding status to a stock holding status

                if ($status === 'Provisional' && $expired === true) {
                    $gained = false;
                    $time = $expiryDate;
                } else {
                    $gained = true;
                    $time = $change['time'];
                }
            } else if (in_array($from, $stockHoldingStatuses)) {
                // Changed from a stock holding status to a non stock holding status

                $gained = false;
                $time = $change['time'];
            } else if (in_array($to, $stockHoldingStatuses)) {
                // Changed from a non stock holding status to a stock holding status
                if ($status === 'Provisional' && $expired === true) {
                    $gained = false;
                    $time = $expiryDate;
                } else {
                    $gained = true;
                    $time = $change['time'];
                }
            } else {
                // Changed from non stock holding status to a non stock holding status
            }

        }

        return ['gained' => $gained, 'time' => $time];
    }
}
