<?php

namespace Resrequest\Application\Chart\Dataset;

use Resrequest\Application\Chart\Dataset;
use Resrequest\Application\Chart\Filter\AccommodationAccess;
use Resrequest\Application\Chart\Filter\ReservationStatus;
use Resrequest\Application\Chart\Filter\DateRange;

class ConsultantBookingsByStatus extends Dataset
{
    protected function generateOptions()
    {
        return [
            new AccommodationAccess('accommodationAccess'),
            new DateRange('dateRecordedPeriod'),
            new ReservationStatus('reservationStatus')
        ];
    }

    public function buildData()
    {
        $reservationsQuery = $this->em->createQueryBuilder();

        $dateRecordedPeriod = $this->valueForOption('dateRecordedPeriod');
        $accommodationAccess = $this->valueForOption('accommodationAccess');
        $reservationStatuses = $this->valueForOption('reservationStatus')['selectedStatuses'];
        
        $today = new \DateTime('today');
        $today = $today->format('Y-m-d');

        $reservationsData  = $reservationsQuery
            ->select(
                [
                    'reservation.rvReservationIx',
                    'IF((reservation.rvProvisionExpiryDate >= :today AND reservation.rfReservationStatusId = 20), FALSE, TRUE) AS expired',
                    'COUNT(DISTINCT reservation.rvReservationIx) AS reservations',
                    'reservationStatusTable.rfReservationStatusDesc AS reservationStatus',
                    'reservation.rvConsultantId AS consultantId',
                    'persona.prNameFirst as consultantFirstName',
                    'persona.prNameLast as consultantLastName',
                    'scGroup.scGrpConsultantYn AS consultantAccess'
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservation.rvConsultantId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\ScUser', 'scUser', 'with', 'scUser.prUserId = persona.prPersonaIx')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\ScGroup', 'scGroup', 'with', 'scGroup.scGroupId = scUser.scGroupId')
            ->where('reservation.rvDateRecorded >= :startDate')
            ->andWhere('reservation.rvDateRecorded <= :endDate')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->andWhere('reservationStatusTable.rfReservationStatusId IN(:reservationStatuses)')
            ->groupBy('reservationStatusTable.rfReservationStatusId, expired, reservation.rvConsultantId')
            ->setParameters(
                [
                    'startDate' => $dateRecordedPeriod['startDate'],
                    'endDate' => $dateRecordedPeriod['endDate'],
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations'],
                    'reservationStatuses' => $reservationStatuses,
                    'today' => $today
                ]
            )
            ->getQuery()
            ->getResult();

        $reservations = [];
        $consultants = [];

        foreach ($reservationsData as $reservation) {
            $reservationCount = (int) $reservation['reservations'];
            $consultantId = $reservation['consultantId'];
            $consultantAccess = $reservation['consultantAccess'];
            $status = $reservation['reservationStatus'];
            $expired = boolval($reservation['expired']);

            // Do not include data for users that do not have consultant access
            if (!is_null($consultantAccess) && $consultantAccess !== true) {
                continue;
            }

            if ($status == 'Wait listed') {
                $status = 'Waitlisted';
            }

            if ($status == 'Provisional') {
                if ($expired) {
                    $status = 'expiredProvisional';
                } else {
                    $status = 'validProvisional';
                }
            }

            $status = lcfirst($status);

            // Add consultant
            if (!isset($reservations[$consultantId])) {
                $reservations[$consultantId] = [
                    'quotation' => 0,
                    'waitlisted' => 0,
                    'validProvisional' => 0,
                    'expiredProvisional' => 0,
                    'cancelled' => 0,
                    'confirmed' => 0,
                    'deleted' => 0,
                ];
            }

            $reservations[$consultantId][$status] = $reservationCount;

            if (!isset($consultants[$reservation['consultantId']])) {
                $firstName = $reservation['consultantFirstName'];
                $lastName = $reservation['consultantLastName'];

                if (is_numeric($consultantId) && $consultantId == 0) {
                    $firstName = "No consultant";
                }
                
                $consultants[$consultantId] = trim("$firstName $lastName");
            }
        }

        $dataset = [];

        foreach ($consultants as $consultantId => $consultantName) {
            $consultantReservations = $reservations[$consultantId];

            $consultantReservations['consultantId'] = $consultantId;
            $consultantReservations['consultantName'] = $consultantName;

            array_push($dataset, $consultantReservations);
        }


        return $dataset;
    }
}
