<?php

namespace Resrequest\Application\Chart\Dataset;

use Resrequest\Application\Chart\Dataset;
use Resrequest\Application\Chart\Filter\ReservationStatus;

class ConsultantSalesPerformance extends Dataset
{
    protected function generateOptions()
    {
        return [
            new AccommodationAccess('accommodationAccess'),
            new ReservationStatus('reservationStatus')
        ];
    }
    
    public function buildData()
    {
        $currentAndFutureBookingsQuery = $this->em->createQueryBuilder();

        $sixDaysAgo = new \DateTime('6 days ago');
        $sixDaysAgo = $sixDaysAgo->format('Y-m-d');

        $today = new \DateTime('today');
        $today = $today->format('Y-m-d');

        $accommodationAccess = $this->valueForOption('accommodationAccess');
        $reservationStatuses = $this->valueForOption('reservationStatus')['selectedStatuses'];

        $consultantSalesPerformance = $currentAndFutureBookingsQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationIx) AS reservations',
                    'persona.prNameFirst as consultantFirstName',
                    'persona.prNameLast as consultantLastName',
                    'reservation.rvConsultantId  AS consultantId',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservation.rvConsultantId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->where('reservation.rvDateRecorded >= :sixDaysAgo')
            ->andWhere('reservation.rvDateRecorded <= :today')
            ->andWhere('(reservation.rvProvisionExpiryDate >= :today AND reservation.rfReservationStatusId = 20) OR reservation.rfReservationStatusId != 20')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->andWhere('reservationStatusTable.rfReservationStatusId IN(:reservationStatuses)')
            ->groupBy('reservation.rvConsultantId')
            ->setParameters(
                [
                    'reservationStatuses' => $reservationStatuses,
                    'sixDaysAgo' => $sixDaysAgo,
                    'today' => $today,
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations']
                ]
            )
            ->getQuery()
            ->getResult();

        $consultants = [];
        $currentAndFutureBookingsData = [];

        foreach ($consultantSalesPerformance as $data) {
            $consultantId = $data['consultantId'];
            if (!isset($currentAndFutureBookingsData[$consultantId])) {
                $currentAndFutureBookingsData[$consultantId] = 0;
            }
            $currentAndFutureBookingsData[$consultantId] = $currentAndFutureBookingsData[$consultantId] + (int) $data['reservations'];

            if (!isset($consultants[$data['consultantId']])) {
                $firstName = $data['consultantFirstName'];
                $lastName = $data['consultantLastName'];

                if (is_numeric($consultantId) && $consultantId == 0) {
                    $firstName = "No consultant";
                }
                
                $consultants[$consultantId] = trim("$firstName $lastName");
            }
        }

        $dataset = [];

        foreach ($consultants as $consultantId => $consultantName) {
            $bookings = $currentAndFutureBookingsData[$consultantId];

            array_push($dataset, [
                'consultantId' => $consultantId,
                'consultantName' => $consultantName,
                'bookings' => $bookings
            ]);
        }

        return $dataset;
    }
}
