<?php

namespace Resrequest\Application\Chart\Dataset;

use Resrequest\Application\Chart\Dataset;
use Resrequest\Application\Chart\Filter\AccommodationAccess;

class ConsultantWorkload extends Dataset
{
    protected function generateOptions()
    {
        return [
            new AccommodationAccess('accommodationAccess'),
        ];
    }
    
    public function buildData()
    {
        $currentAndFutureBookingsQuery = $this->em->createQueryBuilder();
        $otherBookingsQuery = $this->em->createQueryBuilder();

        $reservationStatusFilter = [
            'Confirmed',
            'Provisional'
        ];

        $sixDaysFromNow = new \DateTime('+6 days');
        $sixDaysFromNow = $sixDaysFromNow->format('Y-m-d');

        $today = new \DateTime('today');
        $today = $today->format('Y-m-d');

        $accommodationAccess = $this->valueForOption('accommodationAccess');

        $currentAndFutureBookings = $currentAndFutureBookingsQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationIx) AS reservations',
                    'persona.prNameFirst as consultantFirstName',
                    'persona.prNameLast as consultantLastName',
                    'reservation.rvConsultantId  AS consultantId',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservation.rvConsultantId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->andWhere('reservation.rvDateArrive >= :today')
            ->andWhere('reservation.rvDateArrive <= :sixDaysFromNow')
            ->andWhere('(reservation.rvProvisionExpiryDate >= :today AND reservation.rfReservationStatusId = 20) OR reservation.rfReservationStatusId != 20')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->groupBy('reservation.rvConsultantId')
            ->setParameters(
                [
                    'reservationStatusFilter' => $reservationStatusFilter,
                    'sixDaysFromNow' => $sixDaysFromNow,
                    'today' => $today,
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations']
                ]
            )
            ->getQuery()
            ->getResult();


        $otherBookings = $otherBookingsQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationIx) AS reservations',
                    'persona.prNameFirst AS consultantFirstName',
                    'persona.prNameLast AS consultantLastName',
                    'reservation.rvConsultantId AS consultantId',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservation.rvConsultantId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->andWhere('reservation.rvDateArrive > :sixDaysFromNow')
            ->andWhere('(reservation.rvProvisionExpiryDate >= :today AND reservation.rfReservationStatusId = 20) OR reservation.rfReservationStatusId != 20')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->groupBy('reservation.rvConsultantId')
            ->setParameters(
                [
                    'reservationStatusFilter' => $reservationStatusFilter,
                    'sixDaysFromNow' => $sixDaysFromNow,
                    'today' => $today,
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations']
                ]
            )
            ->getQuery()
            ->getResult();

        $consultants = [];
        $currentAndFutureBookingsData = [];

        foreach ($currentAndFutureBookings as $data) {
            $consultantId = $data['consultantId'];
            if (!isset($currentAndFutureBookingsData[$consultantId])) {
                $currentAndFutureBookingsData[$consultantId] = 0;
            }
            $currentAndFutureBookingsData[$consultantId] = $currentAndFutureBookingsData[$consultantId] + (int) $data['reservations'];

            if (!isset($consultants[$data['consultantId']])) {
                $firstName = $data['consultantFirstName'];
                $lastName = $data['consultantLastName'];

                if (is_numeric($consultantId) && $consultantId == 0) {
                    $firstName = "No consultant";
                }
                
                $consultants[$consultantId] = trim("$firstName $lastName");
            }
        }

        $otherBookingsData = [];

        foreach ($otherBookings as $data) {
            $consultantId = $data['consultantId'];
            if (!isset($otherBookingsData[$data['consultantId']])) {
                $otherBookingsData[$data['consultantId']] = 0;
            }
            $otherBookingsData[$data['consultantId']] = $otherBookingsData[$data['consultantId']] + (int) $data['reservations'];

            if (!isset($consultants[$consultantId])) {
                $firstName = $data['consultantFirstName'];
                $lastName = $data['consultantLastName'];

                if (is_numeric($consultantId) && $consultantId == 0) {
                    $firstName = "No consultant";
                }
                
                $consultants[$consultantId] = trim("$firstName $lastName");
            }
        }

        $totalBookings = [];
        $dataset = [];

        foreach ($consultants as $consultantId => $consultantName) {
            $total = 0;
            if (isset($currentAndFutureBookingsData[$consultantId])) {
                $total = $total + $currentAndFutureBookingsData[$consultantId];
            } else {
                $currentAndFutureBookingsData[$consultantId] = 0;
            }
            if (isset($otherBookingsData[$consultantId])) {
                $total = $total + $otherBookingsData[$consultantId];
            } else {
                $otherBookingsData[$consultantId] = 0;
            }

            $totalBookings[$consultantId] = $total;

            array_push($dataset, [
                'consultantId' => $consultantId,
                'consultantName' => $consultantName,
                'currentAndFutureBookings' => $currentAndFutureBookingsData[$consultantId],
                'otherBookings' => $otherBookingsData[$consultantId],
                'totalBookings' => $totalBookings[$consultantId],
            ]);
        }

        return $dataset;
    }
}
