<?php

namespace Resrequest\Application\Chart\Dataset;

require_once(__DIR__ . '../../../../legacy/class.itinerary.php');

use Resrequest\Application\Chart\Dataset;
use Resrequest\Application\Chart\DatasetOption;
use Resrequest\Application\Chart\Filter\AccommodationAccess;
use Resrequest\Application\Chart\Filter\DateRange;
use Resrequest\Application\Chart\Filter\ReservationStatus;
use Resrequest\Application\Chart\Input\ButtonToggle;
use Resrequest\Application\Chart\Option;

class Occupancy extends Dataset
{
    protected function generateOptions()
    {
        $selectDataToggle = new ButtonToggle();
        $selectDataToggle->setValue('bedNights');
        $selectDataToggle->addOption('Bed Nights', 'bedNights');
        $selectDataToggle->addOption('Rooms', 'rooms');

        return [
            new AccommodationAccess('accommodationAccess'),
            (new ReservationStatus('reservationStatus'))->disableProvisionalExpired(),
            new DateRange('dateRange'),
            new DatasetOption('selectData', [new Option('selectData', $selectDataToggle)]),
        ];
    }

    public function buildData()
    {
        $selectedAccommodationTypes = $this->valueForOption('accommodationAccess')['selectedAccommodationTypes'];
        $dateFilter = $this->valueForOption('dateRange');
        $startDate = $dateFilter['startDate'];
        $endDate = $dateFilter['endDate'];
        $selectData = $this->valueForOption('selectData')['selectData'];
        $reservationStatus = $this->valueForOption('reservationStatus');

        $c1StartDate = (new \DateTime($startDate))->modify('-1 year')->modify('first day of January')->format('Y-m-d');
        $c1EndDate = (new \DateTime($endDate))->modify('-1 year')->modify('last day of December')->format('Y-m-d');

        if ($selectData == 'bedNights') {
            $calculateOccupiedBedNights = true;
        } else {
            $calculateOccupiedBedNights = false;
        }

        $occupancy = $this->getOccupancy($startDate, $endDate, $selectedAccommodationTypes, $reservationStatus, $calculateOccupiedBedNights);
        $c1Occupancy = $this->getOccupancy($c1StartDate, $c1EndDate, $selectedAccommodationTypes, $reservationStatus, $calculateOccupiedBedNights);

        $dataTemplate = [
            'totalRooms' => 0,
            'availableRooms' => 0,
            'occupiedRooms' => 0,
            'blockedRooms' => 0,
            'bedNights' => 0,
            'totalBedNights' => 0,

            'c1TotalRooms' => 0,
            'c1AvailableRooms' => 0,
            'c1OccupiedRooms' => 0,
            'c1BlockedRooms' => 0,
            'c1BedNights' => 0,
            'c1TotalBedNights' => 0,

            'accommodationTypeId' => '',
            'date' => ''
        ];

        $combinedData = [];
        foreach ($occupancy as $date => $accomms) {
            if (!isset($combinedData[$date])) {
                $combinedData[$date] = [];
            }
            
            $month = new \DateTime($date);
            $month = $month->format('M');
            foreach ($accomms as $id => $data) {
                if (isset($combinedData[$date][$id])) {
                    $combinedData[$date][$id]['totalRooms'] = $data['totalRooms'];
                    $combinedData[$date][$id]['availableRooms'] = $data['availableRooms'];
                    $combinedData[$date][$id]['occupiedRooms'] = $data['occupiedRooms'];
                    $combinedData[$date][$id]['blockedRooms'] = $data['blockedRooms'];
                    $combinedData[$date][$id]['totalBedNights'] = $data['totalBedNights'];
                    $combinedData[$date][$id]['bedNights'] = $data['bedNights'];
                } else {
                    $combinedData[$date][$id] = $dataTemplate;
                    $combinedData[$date][$id]['totalRooms'] = $data['totalRooms'];
                    $combinedData[$date][$id]['availableRooms'] = $data['availableRooms'];
                    $combinedData[$date][$id]['occupiedRooms'] = $data['occupiedRooms'];
                    $combinedData[$date][$id]['blockedRooms'] = $data['blockedRooms'];
                    $combinedData[$date][$id]['totalBedNights'] = $data['totalBedNights'];
                    $combinedData[$date][$id]['bedNights'] = $data['bedNights'];
                    $combinedData[$date][$id]['accommodationTypeId'] = $id;
                    $combinedData[$date][$id]['date'] = $date;
                    $combinedData[$date][$id]['month'] = $month;
                }
            }
        }

        foreach ($c1Occupancy as $date => $accomms) {
            if (!isset($combinedData[$date])) {
                $combinedData[$date] = [];
            }
            $month = new \DateTime($date);
            $month = $month->format('M');
            foreach ($accomms as $id => $data) {
                if (isset($combinedData[$date][$id])) {
                    $combinedData[$date][$id]['c1TotalRooms'] = $data['totalRooms'];
                    $combinedData[$date][$id]['c1AvailableRooms'] = $data['availableRooms'];
                    $combinedData[$date][$id]['c1OccupiedRooms'] = $data['occupiedRooms'];
                    $combinedData[$date][$id]['c1BlockedRooms'] = $data['blockedRooms'];
                    $combinedData[$date][$id]['c1otalBedNights'] = $data['totalBedNights'];
                    $combinedData[$date][$id]['c1BedNights'] = $data['bedNights'];
                } else {
                    $combinedData[$date][$id] = $dataTemplate;
                    $combinedData[$date][$id]['c1TotalRooms'] = $data['totalRooms'];
                    $combinedData[$date][$id]['c1AvailableRooms'] = $data['availableRooms'];
                    $combinedData[$date][$id]['c1OccupiedRooms'] = $data['occupiedRooms'];
                    $combinedData[$date][$id]['c1BlockedRooms'] = $data['blockedRooms'];
                    $combinedData[$date][$id]['c1TotalBedNights'] = $data['totalBedNights'];
                    $combinedData[$date][$id]['c1BedNights'] = $data['bedNights'];
                    $combinedData[$date][$id]['accommodationTypeId'] = $id;
                    $combinedData[$date][$id]['date'] = $date;
                    $combinedData[$date][$id]['month'] = $month;
                }
            }
        }

        $output = [];
        foreach ($combinedData as $date => $accomms) {
            foreach (array_keys($accomms) as $id) {
                $tmpOutput = &$combinedData[$date][$id];
                if ($selectData == 'bedNights') {
                    $tmpOutput['selectedTotal'] = $tmpOutput['totalBedNights'];
                    $tmpOutput['selectedOccupied'] = $tmpOutput['bedNights'];
                    $tmpOutput['c1SelectedTotal'] = $tmpOutput['c1TotalBedNights'];
                    $tmpOutput['c1SelectedOccupied'] = $tmpOutput['c1BedNights'];

                    $tmpOutput['selectedDataLabel'] = 'Bed Nights';
                    $tmpOutput['selectedDataLabelLowerCase'] = 'bed nights';
                } else {
                    $tmpOutput['selectedTotal'] = $tmpOutput['totalRooms'];
                    $tmpOutput['selectedOccupied'] = $tmpOutput['occupiedRooms'];
                    $tmpOutput['c1SelectedTotal'] = $tmpOutput['c1TotalRooms'];
                    $tmpOutput['c1SelectedOccupied'] = $tmpOutput['c1OccupiedRooms'];

                    $tmpOutput['selectedDataLabel'] = 'Rooms';
                    $tmpOutput['selectedDataLabelLowerCase'] = 'rooms';
                }
                $output[] = $combinedData[$date][$id];
            }
        }

        return $output;
    }

    function getOccupancy($startDate, $endDate, $accommodationTypes, $reservationStatus, $calculateOccupiedBedNights = false)
    {
        $bedNights = [];
        $totalRooms = [];
        $blockedRooms = [];
        $occupiedRooms = [];
        $availableRooms = [];
        $totalBedNights = [];
        $accommCapacity = [];

        $capacityResults = $this->em->createQueryBuilder()
            ->select(
                [
                    'accommType.acAccommTypeIx as id',
                    'accommType.acAccommCapacity AS capacity',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accommType')
            ->where('accommType.acAccommTypeIx IN (:accommodationTypes)')
            ->setParameters(
                [
                    'accommodationTypes' => $accommodationTypes,
                ]
            )
            ->getQuery()
            ->getResult();

        if ($calculateOccupiedBedNights) {
            $pax = $this->getPax($startDate, $endDate, $accommodationTypes, $reservationStatus);
        }

        foreach ($capacityResults as $result) {
            $accommCapacity[$result['id']] = $result['capacity'];
        }

        foreach ($accommodationTypes as $id) {
            $stock = new \Resrequest\Application\Stock\Stock($id, $startDate, $endDate);
            $occupiedRooms[$id] = $stock->ToOccupancy(join(':', $reservationStatus['selectedStatuses']));
            $availableRooms[$id] = $stock->ToAvailability(join(':', $reservationStatus['selectedStatuses']));
            $blockedRooms[$id] = $block = $stock->ToBlock();
            $totalRooms[$id] = $accomm = $stock->ToAccomm();
            $dates = $stock->ToDate();
            foreach ($dates as $date) {
                $totalRooms[$id][$date] = $accomm[$date] - $block[$date];
                $totalBedNights[$id][$date] = $accommCapacity[$id] * $totalRooms[$id][$date];
                $bedNights[$id][$date] = isset($pax[$id][$date]) ? $pax[$id][$date] : 0;
            }
        }

        $output = [];
        foreach ($accommodationTypes as $id) {
            foreach ($dates as $date) {
                $output[$date][$id] = [
                    'totalRooms' => $totalRooms[$id][$date],
                    'availableRooms' => $availableRooms[$id][$date],
                    'occupiedRooms' => $occupiedRooms[$id][$date],
                    'blockedRooms' => intval($blockedRooms[$id][$date]),
                    'bedNights' => $bedNights[$id][$date],
                    'totalBedNights' => $totalBedNights[$id][$date],
                    'accommodationTypeId' => $id,
                    'date' => $date
                ];
            }
        }

        return $output;
    }

    public function getPax($startDate, $endDate, $accommodationTypes, $reservationStatus)
    {
        $reservationItems = $this->em->createQueryBuilder()
            ->select(
                [
                    'reservationItem.rvItemDateArrive as arrivalDate',
                    'reservationItem.rvItemDateDepart as departureDate',
                    'reservationItem.rvItemAdultCount as adultCount',
                    'reservationItem.rvItemChildCount as childCount',
                    'reservationItem.rvItemAccommCount as accommCount',
                    'accomm.acAccommTypeIx as accommId',
                    'reservation.rvReservationIx as reservationId',
                    'reservation.rfReservationStatusId',
                    "IF(reservation.rfReservationStatusId = 20, IF(reservation.rvProvisionExpiryDate >= :today, TRUE, FALSE), NULL) AS valid",
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation', 'with', 'reservation.rvReservationIx = reservationItem.rvReservationId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->where('(
                (
                    reservationItem.rvItemDateArrive >= :startDate
                    AND reservationItem.rvItemDateArrive <= :endDate
                ) OR (
                    reservationItem.rvItemDateDepart >= :startDate
                    AND reservationItem.rvItemDateDepart <= :endDate
                ) OR (
                    reservationItem.rvItemDateArrive < :startDate
                    AND reservationItem.rvItemDateDepart > :endDate
                )
            )')
            ->andWhere('accomm.acAccommTypeIx IN(:selectedAccommodationTypes)')
            ->andWhere('reservation.rfReservationStatusId IN(:reservationStatuses)')
            ->having('valid IN(:provisionalFilter) OR valid IS NULL')
            ->setParameters(
                [
                    'startDate' => $startDate,
                    'endDate' => $endDate,
                    'selectedAccommodationTypes' => $accommodationTypes,
                    'reservationStatuses' => $reservationStatus['selectedStatuses'],
                    'provisionalFilter' => $reservationStatus['provisionalFilter'],
                    'today' => (new \DateTime())->format('Y-m-d')
                ]
            )
            ->getQuery()
            ->getResult();

        $output = [];
        foreach ($reservationItems as $reservationItem) {
            $accommId = $reservationItem['accommId'];
            $arrivalDate = $reservationItem['arrivalDate'];
            $departureDate = $reservationItem['departureDate'];

            $adultCount = $reservationItem['adultCount'];
            $childCount = $reservationItem['childCount'];

            $pax = ($adultCount + $childCount) * $reservationItem['accommCount'];

            $interval = \DateInterval::createFromDateString('1 day');
            $reservationItemPeriod = new \DatePeriod($arrivalDate, $interval, $departureDate);

            foreach ($reservationItemPeriod as $date) {
                $day = $date->format('Y-m-d');

                if ($day > $endDate || $day < $startDate) {
                    continue;
                }
                if (!isset($output[$accommId])) {
                    $output[$accommId] = [];
                }
                if (!isset($output[$accommId][$day])) {
                    $output[$accommId][$day] = 0;
                }

                $output[$accommId][$day] += $pax;
            }
        }
        return $output;
    }
}
