<?php

namespace Resrequest\Application\Chart\Dataset;

use Resrequest\Application\Chart\Dataset;
use Resrequest\Application\Chart\Filter\AccommodationAccess;
use Resrequest\Application\Chart\Filter\DateRange;

class ProvisionalsDueToExpire extends Dataset
{
    protected function generateOptions()
    {
        return [
            new AccommodationAccess('accommodationAccess'),
            new DateRange('provisionalExpiryPeriod'),
        ];
    }

    public function buildData()
    {
        $provisionalsDueToExpireQuery = $this->em->createQueryBuilder();

        $reservationStatusFilter = [
            'provisional'
        ];

        $provisionalExpiryPeriod = $this->valueForOption('provisionalExpiryPeriod');
        $accommodationAccess = $this->valueForOption('accommodationAccess');

        $provisionalsDueToExpire = $provisionalsDueToExpireQuery
            ->select(
                [
                    'COUNT(DISTINCT reservation.rvReservationIx) AS reservations',
                    'persona.prNameFirst as consultantFirstName',
                    'persona.prNameLast as consultantLastName',
                    'reservation.rvConsultantId  AS consultantId',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservation.rvConsultantId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->where('reservation.rvProvisionExpiryDate <= :endDate')
            ->andWhere('reservation.rvProvisionExpiryDate >= :startDate')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->groupBy('reservation.rvConsultantId')
            ->setParameters(
                [
                    'reservationStatusFilter' => $reservationStatusFilter,
                    'startDate' => $provisionalExpiryPeriod['startDate'],
                    'endDate' => $provisionalExpiryPeriod['endDate'],
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations'],
                ]
            )
            ->getQuery()
            ->getResult();


        $consultants = [];
        $provisionalsDueToExpireData = [];

        foreach ($provisionalsDueToExpire as $data) {
            $consultantId = $data['consultantId'];
            if (!isset($provisionalsDueToExpireData[$consultantId])) {
                $provisionalsDueToExpireData[$consultantId] = 0;
            }
            $provisionalsDueToExpireData[$consultantId] = $provisionalsDueToExpireData[$consultantId] + (int) $data['reservations'];

            if (!isset($consultants[$data['consultantId']])) {
                $firstName = $data['consultantFirstName'];
                $lastName = $data['consultantLastName'];

                if (is_numeric($consultantId) && $consultantId == 0) {
                    $firstName = "No consultant";
                }
                
                $consultants[$consultantId] = trim("$firstName $lastName");
            }
        }

        $dataset = [];

        foreach ($consultants as $consultantId => $consultantName) {
            if (!isset($provisionalsDueToExpireData[$consultantId])) {
                $provisionalsDueToExpireData[$consultantId] = 0;
            }

            array_push($dataset, [
                'consultantId' => $consultantId,
                'consultantName' => $consultantName,
                'provisionalsDueToExpire' => $provisionalsDueToExpireData[$consultantId],
            ]);
        }

        return $dataset;
    }
}
