<?php

namespace Resrequest\Application\Chart\Dataset;

use Resrequest\Application\Chart\Dataset;
use Resrequest\Application\Chart\Filter\AccommodationAccess;
use Resrequest\Application\Chart\Filter\DateRange;

class Waitlist extends Dataset
{
    protected function generateOptions() {
        return [
            new AccommodationAccess('accommodationAccess'),
            new DateRange('arrivalPeriod'),
        ];
    }

    public function buildData()
    {
        $waitlistsQuery = $this->em->createQueryBuilder();
        
        $arrivalPeriod = $this->valueForOption('arrivalPeriod');
        $startDateString = $arrivalPeriod['startDate']; 
        $endDateString = $arrivalPeriod['endDate'];

        $accommodationAccess = $this->valueForOption('accommodationAccess');

        $reservationStatusFilter = [
            'Wait listed'
        ];

        $waitlistedReservations = $waitlistsQuery
            ->select(
                [
                    'reservation.rvReservationIx AS reservationId',
                    'reservationItem.acAccommTypeId as accommodationTypeId',
                    'accomm.acAccommDesc as accommodationTypeDescription',
                    'persona.prNameFirst as consultantFirstName',
                    'persona.prNameLast as consultantLastName',
                    'reservation.rvConsultantId  AS consultantId',
                    'reservationItem.rvItemDateArrive AS arrivalDate',
                    'reservationItem.rvItemDateDepart AS departureDate'
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation', 'with', 'reservation.rvReservationIx = reservationItem.rvReservationId')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = reservation.rvConsultantId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm', 'with', 'accomm.acAccommTypeIx = reservationItem.acAccommTypeId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\RfReservationStatus', 'reservationStatusTable', 'with', 'reservationStatusTable.rfReservationStatusId = reservation.rfReservationStatusId')
            ->where('reservation.rvDateArrive >= :startDate AND reservation.rvDateArrive <= :endDate')
            ->andWhere('reservationStatusTable.rfReservationStatusDesc IN(:reservationStatusFilter)')
            ->andWhere('
                accomm.acAccommTypeIx IN(:allowedAccommodationTypes)
                AND accomm.acAccommTypeIx IN(:selectedAccommodationTypes)
            ')
            ->orWhere('reservation.rvReservationIx IN(:allowedReservations)')
            ->setParameters(
                [
                    'reservationStatusFilter' => $reservationStatusFilter,
                    'startDate' => $startDateString,
                    'endDate' => $endDateString,
                    'allowedAccommodationTypes' => $accommodationAccess['allowedAccommodationTypes'],
                    'selectedAccommodationTypes' => $accommodationAccess['selectedAccommodationTypes'],
                    'allowedReservations' => $accommodationAccess['allowedReservations'],
                ]
            )
            ->getQuery()
            ->getResult();

        $reservations = []; // List of reservations and their items
        $consultants = []; // List of consultants and their names
        $accommodationTypes = [];
        
        // Used to get availability
        $earliestArrivalDate = new \DateTime();
        $latestDepartureDate = new \DateTime();

        // Process accommodation types and consultants
        foreach ($waitlistedReservations as $reservationItem) {
            $reservationId = $reservationItem['reservationId'];
            $accommodationTypeId = $reservationItem['accommodationTypeId'];
            $consultantId = $reservationItem['consultantId'];
            $arrivalDate = $reservationItem['arrivalDate'];
            $departureDate = $reservationItem['departureDate'];


            if (!isset($consultants[$consultantId])) {
                $firstName = $reservationItem['consultantFirstName'];
                $lastName = $reservationItem['consultantLastName'];

                if (is_numeric($consultantId) && $consultantId == 0) {
                    $firstName = "No consultant";
                }
                
                $consultants[$consultantId]['name'] = trim("$firstName $lastName");

                $consultants[$consultantId]['convertibleWaitlistedReservations'] = 0;
            }

            if (!isset($reservations[$reservationId])) {
                $reservations[$reservationId] = [
                    'consultantId' => $reservationItem['consultantId'],
                    'items' => [],
                ];
            }

            $reservations[$reservationId]['items'][] = [
                'arrivalDate' => $arrivalDate,
                'departureDate' => $departureDate,
                'accommodationType' => $accommodationTypeId
            ];

            // Add unique accommodation types to list
            if (!isset($accommodationTypes[$accommodationTypeId])) {
                $accommodationTypes[$accommodationTypeId] = [
                    'description' => $reservationItem['accommodationTypeDescription']
                ];
            }

            if ($arrivalDate < $earliestArrivalDate) {
                $earliestArrivalDate = $arrivalDate;
            }
            if ($departureDate > $latestDepartureDate) {
                $latestDepartureDate = $departureDate;
            }
        }

        $availability = []; // Availability for each accommodation type

        // Get availability
        foreach ($accommodationTypes as $id => $accommodationType) {
            $stock = new \Resrequest\Application\Stock\Stock($id, $earliestArrivalDate->format('Y-m-d'), $latestDepartureDate->format('Y-m-d'));
            $availability[$id] = $stock->ToAvailability();
        }

        // Check to see if each day of the reservation has availability
        foreach ($reservations as $id => &$reservation) {
            $isConvertible = true; // Whether all the days of the reservation have availability

            // Check availability for each accommodation type in the reservation
            foreach ($reservation['items'] as $item) {
                $interval = \DateInterval::createFromDateString('1 day');
                $arrivalDate = $item['arrivalDate'];
                $departureDate = $item['departureDate'];
                $travelPeriod = new \DatePeriod($arrivalDate, $interval, $departureDate);
                
                // Check availability for each day of the item
                foreach ($travelPeriod as $day) {
                    $dayString = $day->format('Y-m-d');
                    // If there is no availability on any of the days the reservation cannot be converted to confirmed status
                    if ($availability[$item['accommodationType']][$dayString] <= 0) {
                        $isConvertible = false;
                        break;
                    }
                }
            }

            $reservation['convertible'] = $isConvertible;

            if ($isConvertible) {
                $consultants[$reservation['consultantId']]['convertibleWaitlistedReservations']++;
            }
        }

        $dataset = [];

        foreach ($consultants as $consultantId => $consultant) {
            if (!empty($consultant['convertibleWaitlistedReservations'])) {
                array_push($dataset, [
                    'consultantId' => $consultantId,
                    'consultantName' => $consultant['name'],
                    'convertibleWaitlistedReservations' => $consultant['convertibleWaitlistedReservations'],
                ]);
            }
        }

        return $dataset;
    }
}
