<?php

namespace Resrequest\Application\Chart\Filter;

use Resrequest\Application\Chart\Filter;
use Resrequest\Application\Chart\Option;
use Resrequest\Application\Chart\Input\Checkbox;
use Resrequest\Application\Chart\Input\AccommodationFilter;

class AccommodationAccess extends Filter
{
    protected $supportedOutputs = [
        'allowedAccommodationTypes',
        'selectedAccommodationTypes',
        'allowedReservations',
        'reservationAllowedAccommodationTypes',
        'allAccommodationTypes'
    ];

    protected function generateOptions()
    {
        return [
            new Option('accommodationTypes', new AccommodationFilter()),
            new Option('disableAcl', new Checkbox(false)),
        ];
    }

    protected function processOptions()
    {
        $enterprise = $this->sm->get('Resrequest\Setup\Service\Enterprise');
        $userId = $enterprise->userId;
        $accessGroupId = $enterprise->accessGroupId;
        $publicAccessGroup = $enterprise->publicUserAccessGroup();
        $userStatusId = $enterprise->userStatusId;
        $agentId = false;

        $accommodationTypesQuery = $this->em->createQueryBuilder();
        $accommodationTypesQuery
            ->select(
                [
                    'DISTINCT accomm.acAccommTypeIx AS id',
                    'accomm.acAccommDesc AS description',
                    'property.prBusinessId AS propertyId',
                    'accomm.acAccommTypeInactiveYn AS inactive',
                    'IF((scAccomm.scGroupId IS NOT NULL), TRUE, FALSE) AS allowed',
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\AcAccommType', 'accomm')
            ->leftJoin('Resrequest\DB\Enterprise\Entity\ScAccomm', 'scAccomm', 'with', 'scAccomm.acAccommTypeId = accomm.acAccommTypeIx AND scAccomm.scGroupId = :accessGroupId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\PrBusiness', 'property', 'with', 'property.prBusinessId = accomm.prBusinessId')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = property.prBusinessId')
            ->orderBy('accomm.acAccommSequence');

        // Users without any relationship will be defaulted to
        // the public user access group
        if ($userStatusId == 0) {
            $accommodationTypesQuery->setParameters(
                [
                    'accessGroupId' => $publicAccessGroup,
                ]
            );
        } else {
            $accommodationTypesQuery->setParameters(
                [
                    'accessGroupId' => $accessGroupId,
                ]
            );
        }

        $accommodationTypes = $accommodationTypesQuery->getQuery()->getResult();


        $disableAclOption = $this->optionByName('disableAcl');
        $disableAcl = $disableAclOption->getValue();

        $allowedAccommodationTypes = []; // Accommodation types allowed in user acces

        foreach ($accommodationTypes as &$accommodationType) {
            $allowed = boolval($accommodationType['allowed']);
            $accommodationType['allowed'] = $allowed;

            if ($allowed || $disableAcl) {
                $allowedAccommodationTypes[] = $accommodationType['id'];
            }
        }
        unset($accommodationType);

        // Get reservations the agent or linked agent has access to
        // and get the reservations accommodation types
        $allowedReservations = [];
        $reservationAllowedAccommodationTypes = [];

        if (isset($GLOBALS['is_an_agent']) && $GLOBALS['is_an_agent']) {
            $agentId = $userId;
        } elseif (!empty($GLOBALS['pr_agent_link'])) {
            $agentId = $GLOBALS['pr_agent_link'];
        }

        if (!empty($agentId) && $disableAcl === false) {
            $allowedReservationsQuery = $this->em->createQueryBuilder();
            $allowedReservationsResult = $allowedReservationsQuery
                ->select(
                    [
                        'DISTINCT reservation.rvReservationIx AS reservationId',
                        'reservationItem.acAccommTypeId  AS accommodationTypeId',
                    ]
                )
                ->from('Resrequest\DB\Enterprise\Entity\RvReservation', 'reservation')
                ->innerJoin('Resrequest\DB\Enterprise\Entity\RvReservationItem', 'reservationItem', 'with', 'reservationItem.rvReservationId = reservation.rvReservationIx')
                ->andWhere('reservation.rvAgentId = :agentId')
                ->setParameters(
                    [
                        'agentId' => $agentId,
                    ]
                )
                ->getQuery()
                ->getResult();

            foreach ($allowedReservationsResult as $reservation) {
                $reservationId = $reservation['reservationId'];
                $accommodationTypeId = $reservation['accommodationTypeId'];

                if (!in_array($reservationId, $allowedReservations)) {
                    $allowedReservations[] = $reservationId;
                }

                if (!in_array($accommodationTypeId, $reservationAllowedAccommodationTypes)) {
                    $reservationAllowedAccommodationTypes[] = $accommodationTypeId;
                }
            }
        }

        $allowedProperties = [];
        $allAccommodationTypes = []; // All accommodation types including inactive
        $selectedAccommodationTypes = []; // Filtered accommodation types by config

        $accommodationTypesOption = $this->optionByName('accommodationTypes');
        $accommodationTypesOptionValue = $accommodationTypesOption->getValue();

        // Build list of allowed properties and accomodation types
        foreach ($accommodationTypes as $accommodationType) {
            $allAccommodationTypes[] = $accommodationType['id'];

            // Filter properties based on accommodation type filter option.
            // Accommodation types can be selected if
            // the user has been allowed access to it
            // or the user has access to a reservation which contains a property
            // type that they haven't been allowed access to
            if (
                empty($accommodationTypesOptionValue)
            ) {
                // If no accommodation types selected, automatically select
                // all allowed accommodation types
                if (
                    in_array($accommodationType['id'], $allowedAccommodationTypes)
                    || in_array($accommodationType['id'], $reservationAllowedAccommodationTypes)
                ) {
                    $selectedAccommodationTypes[] = $accommodationType['id'];
                }
            } else {
                // If accommodation types have been selected
                // check if they're allowed to be
                if (in_array($accommodationType['id'], $accommodationTypesOptionValue)) {
                    if (
                        in_array($accommodationType['id'], $allowedAccommodationTypes)
                        || in_array($accommodationType['id'], $reservationAllowedAccommodationTypes)
                    ) {
                        $selectedAccommodationTypes[] = $accommodationType['id'];
                    }
                }
            }

            if (!in_array($accommodationType['propertyId'], $allowedProperties)) {
                array_push($allowedProperties, $accommodationType['propertyId']);
            }
        }

        // Get property information for for allowed accommodation types
        // to build UI config
        $propertiesQuery = $this->em->createQueryBuilder();
        $propertiesResult = $propertiesQuery
            ->select(
                [
                    'property.prBusinessId AS id',
                    'property.prBusInactiveYn AS inactive',
                    'persona.prNameLast AS name'
                ]
            )
            ->from('Resrequest\DB\Enterprise\Entity\PrBusiness', 'property')
            ->innerJoin('Resrequest\DB\Enterprise\Entity\PrPersona', 'persona', 'with', 'persona.prPersonaIx = property.prBusinessId')
            ->where('property.prBusinessId IN(:properties)')
            ->orderBy('property.prBusinessSequence')
            ->setParameters(
                [
                    'properties' => $allowedProperties,
                ]
            )
            ->getQuery()
            ->getResult();

        $uiConfig = [];
        // Build UI config
        foreach ($propertiesResult as $property) {
            $accommodations = [];

            foreach ($accommodationTypes as $accommodationType) {
                if (
                    $accommodationType['propertyId'] == $property['id']
                    && (in_array($accommodationType['id'], $allowedAccommodationTypes)
                    || in_array($accommodationType['id'], $reservationAllowedAccommodationTypes))
                ) {
                    $accommodations[] = [
                        'id' => $accommodationType['id'],
                        'name' => $accommodationType['description'],
                        'inactive' => $accommodationType['inactive'],
                    ];
                }
            }

            if (empty($accommodations)) {
                continue;
            }

            $uiConfig[] = [
                'id' => $property['id'],
                'name' => $property['name'],
                'inactive' => $property['inactive'],
                'accommodations' => $accommodations
            ];
         }

        $accommodationTypesOption->input->setConfig($uiConfig);

        $value = [
            'allowedAccommodationTypes' => $allowedAccommodationTypes,
            'selectedAccommodationTypes' => $selectedAccommodationTypes,
            'allowedReservations' => $allowedReservations,
            'reservationAllowedAccommodationTypes' => $reservationAllowedAccommodationTypes,
            'allAccommodationTypes' => $allAccommodationTypes
        ];

        return $value;
    }
}
