<?php

namespace Resrequest\Application\Chart;

use MaglLegacyApplication\Application\MaglLegacy;

abstract class Process
{
    /*
     * The list of supported inputs for the process.
     */
    protected $supportedInputs = [];

    /*
     * The list of supported outputs for the process.
     */
    protected $supportedOutputs = [];

    /**
     * This is the initial value that gets passed to
     * the processData function. It is used to store
     * state in between processing data elements
     */
    const INITIAL_ACCUMULATOR = [];

    /**
     * Zend Service Manager.
     *
     * @var \Zend\ServiceManager\ServiceManager
     */
    protected $sm;

    /**
     * Doctrine Entity Manager.
     *
     * @var \Doctrine\ORM\EntityManager
     */
    protected $em;
    /**
     * The ID of the process.
     *
     * @var string
     */
    protected $id;

    /**
     * The name of a process.
     *
     * @var string
     */
    protected $name;

    /**
     * The type of process.
     *
     * @var string
     */
    protected $type = 'generic';

    /**
     * The process' sub type
     * 
     * @var string.
     */
    protected $subType = 'generic';

    /**
     * Process config.
     *
     * @var array
     */
    protected $config;

    /**
     * Array of options.
     *
     * @var array
     */
    protected $options;

    /**
     * The value returned after processing the-options.
     *
     * @var mixed
     */
    protected $optionsValue;

    /* 
     * Maps supported inputs to source fields.
     */
    protected $inputs = [];

    /* 
     * Maps supported outputs to output fields.
     */
    protected $outputs = [];

    /**
     * Whether to process the options on startu[]
     */
    public $processOptions;

    /**
     * Creates the process.
     *
     * @param string $name The name of the process.
     * @param array $config The config of the process.
     */
    public function __construct(string $name, array $config = [], $processOptions = true)
    {
        $this->sm = MaglLegacy::getServiceManager();
        $this->em = $this->sm->get('doctrine.entitymanager.orm_enterprise');

        $this->name = $name;
        $this->config = $config;
        $this->options = $this->generateOptions();
        $this->processOptions = $processOptions;

        $this->update($config);
    }

    /**
     * Updates option value and sets the ID
     * using the given configuration.
     *
     * @param array $config
     * @return void
     */
    public function update(array $config)
    {
        if (empty($config) && $this->processOptions) {
            $this->optionsValue = $this->processOptions();
            return;
        }

        if (empty($config['id'])) {
            $this->id = '';
        } else {
            $this->id = $config['id'];
        }

        // Add option configuration to array with the option name as the key
        $configByName = [];
        if (!empty($config['options'])) {
            foreach ($config['options'] as $option) {
                $configByName[$option['name']] = $option;
            }
        }

        foreach ($this->options as $option) {
            // If available, update option value
            if (array_key_exists($option->name, $configByName)) {
                // Override option input configuration defaults
                $optionConfig = $configByName[$option->name];

                if (isset($optionConfig['value']) && !is_null($optionConfig['value'])) {
                    $option->setValue($optionConfig['value']);
                }
            }
        }

        if ($this->processOptions) {
            $this->processOptionsAndMappings();
        }
    }

    /**
     * Processes options and returns the final option value.
     *
     * @return mixed
     */
    protected abstract function processOptions();

    public function processOptionsAndMappings() {
        $this->optionsValue = $this->processOptions();
        $this->processMappings();
    }

    /**
     * Transforms the process' options into an array.
     *
     * @return array
     */
    protected function optionsToArray()
    {
        $configs = [];

        foreach ($this->options as $option) {
            $configs[] = $option->toArray();
        }

        return $configs;
    }

    /**
     * Returns the option by name.
     *
     * @param string $name Name of the option.
     * @return Option
     */
    protected function optionByName($name)
    {
        foreach ($this->options as $option) {
            if ($option->name == $name) {
                return $option;
            }
        }

        throw new \Exception("Option '$name' doesn't exist");
    }

    /**
     * Creates the options that are supported by the process.
     *
     * @return array
     */
    protected function generateOptions()
    {
        return [];
    }


    /**
     * Generates config for this process.
     *
     * @return array
     */
    public abstract function toArray();

    /**
     * Process converters mappings configuration.
     *
     * @return void
     */
    protected function processMappings()
    {
        $inputs = [];
        $outputs = [];

        if (!empty($this->config['mappings']['inputs'])) {
            $inputs = $this->config['mappings']['inputs'];
        }
        if (!empty($this->config['mappings']['outputs'])) {
            $outputs = $this->config['mappings']['outputs'];
        }

        foreach ($inputs as $input) {
            if ($this->isInputSupported($input['name'])) {
                $this->inputs[$input['name']] = $input['fields'];
            }
        }

        foreach ($outputs as $output) {
            if ($this->isOutputSupported($output['name'])) {
                $this->outputs[$output['name']] = $output['field'];
            }
        }
    }

    /**
     * Returns the mapped input field.
     *
     * @param string $name
     * @return string field
     */
    protected function inputFieldByName(string $name)
    {
        if (array_key_exists($name, $this->inputs)) {
            return $this->inputs[$name];
        } else {
            throw new \Exception("Input mapping for '$name' doesn't exist");
        }
    }

    /**
     * Returns the mapped output field.
     *
     * @param string $name
     * @return string field
     */
    protected function outputFieldByName(string $name)
    {
        if (array_key_exists($name, $this->outputs)) {
            return $this->outputs[$name];
        } else {
            throw new \Exception("Output mapping for '$name' doesn't exist");
        }
    }

    /**
     * Whether a mapping for an output field
     * is provided.
     *
     * @return boolean
     */
    protected function isInputMapped(string $name)
    {
        return array_key_exists($name, $this->inputs);
    }

    /**
     * Whether a mapping for an output field
     * is provided.
     *
     * @return boolean
     */
    protected function isOutputMapped(string $name)
    {
        return array_key_exists($name, $this->outputs);
    }

    /**
     * Whether the input is supported.
     *
     * @param string $name
     * @return boolean
     */
    protected function isInputSupported(string $name)
    {
        return in_array($name, $this->supportedInputs);
    }

    /**
     * Whether the output is supported.
     *
     * @param string $name
     * @return boolean
     */
    protected function isOutputSupported(string $name)
    {
        return in_array($name, $this->supportedOutputs);
    }

    /**
     * Evalutes the process to return its value.
     *
     * https://www.youtube.com/watch?v=oeX1XXQNi-Q
     * 
     * @param array $data Optional input data
     * @return array
     */
    public function run(array $data = [])
    {
        $accumulator = $this::INITIAL_ACCUMULATOR;

        foreach ($data as $element) {
            $supportedInputData = $this->extractInputs($element);
            $accumulator = $this->processData($element, $supportedInputData, $accumulator);
        }

        $results = $this->finaliseData($accumulator);

        if (empty($results)) {
            $results = [];
        }

        $output = [];
        foreach ($results as &$result) {
            $this->mapOutputs($result['data'], $result['outputs']);
            $output[] = $result['data'];
        }

        return $output;
    }

    /**
     * Takes the defined outputs and the provided output mapping
     * and maps it to the data.
     *
     * @param array $data
     * @param array $outputs
     * @return void
     */
    protected function mapOutputs(array &$data, array &$outputs)
    {
        foreach ($outputs as $key => $value) {
            if (in_array($key, $this->supportedOutputs)) {


                if (isset($this->outputs[$key])) {
                    $field = $this->outputs[$key];
                } else {
                    continue;
                }

                // Complex output
                if (is_array($field)) {
                    $fields = $field;
                    foreach ($fields as $field => $targetField) {
                        if (isset($value[$field])) {
                            $data[$targetField] = $value[$field];
                        } else {
                            throw new \Exception("Output field '$field' missing from process '{$this->subType}' output");
                        }
                    }
                } else {
                    $data[$field] = $value;
                }
            }
        }
    }

    /**
     *extractInputs Extract the supported inputs from the data element.
     *
     * @param array $element
     * @return array
     */
    protected function extractInputs($element)
    {
        $data = [];
        foreach ($this->inputs as $input => $fields) {
            $data[$input] = [];

            if (is_array($fields)) {
                foreach ($fields as $field) {
                    if (!array_key_exists($field, $element)) {
                        throw new \Exception("Input field '$field' is not present in dataset");
                    }
                    $data[$input][$field] = $element[$field];
                }
            } else {
                $field = $fields;
                if (!array_key_exists($field, $element)) {
                    throw new \Exception("Input field '$field' is not present in dataset");
                }
                $data[$input][$field] = $element[$field];
            }
        }

        return $data;
    }

    protected function getSingleInput(array $extractedData)
    {
        foreach ($extractedData as $data) {
            return $data;
        }
    }

    /**
     * Takes a single element of data from the input dataset
     * and processes it. Intermediate results can be stored in the accummulator.
     *
     * @param array $originalData Dataset element.
     * @param array $extractedData Data extracted from the dataset using the supported inputs.
     * @param array $accumulator Used to store state in between processing.
     * @return void
     */
    protected function processData($originalData, $extractedData, $accumulator)
    { }

    /**
     * Takes in the processed state and generates an array
     * where each element contains outputs and data that
     * needs to be mapped.
     *
     * @param array $accumulator
     * @return array
     */
    protected function finaliseData($accumulator)
    {
        return [];
    }

    public function getType()
    {
        return $this->type;
    }

    public function getSubType()
    {
        return $this->subType;
    }

    public function getName()
    {
        return $this->name;
    }
}
