<?php

namespace Resrequest\Application\Chart\Store\Charts;

use Resrequest\Application\Chart\Store\Chart;
use Resrequest\Application\Chart\Config\Chart as ChartConfig;
use Resrequest\Application\Chart\Config\Converter;
use Resrequest\Application\Chart\Config\Dataset;
use Resrequest\Application\Chart\Config\DatasetOption;
use Resrequest\Application\Chart\Config\Filter;
use Resrequest\Application\Chart\Config\InputMapping;
use Resrequest\Application\Chart\Config\Option;
use Resrequest\Application\Chart\Config\OutputMapping;
use Resrequest\Application\Chart\Config\UiConfig;
use Resrequest\Application\Chart\Config\UiOption;
use Resrequest\Application\Chart\Config\UiPage;
use Resrequest\Application\Chart\Config\UiPageSection;
use Resrequest\Application\Chart\Config\UiSection;
use Resrequest\Application\Chart\Config\UiSubOption;

class AgentOverviewRevenue extends Chart
{
    protected $id = '11ea31004cdfaf038fc802366610d15e';
    protected $library = 'FusionCharts';
    protected $chartClass = 'scrollCombiDY2D';
    protected $name = 'Accommodation revenue & yield';
    protected $description = 'This chart compares revenue and yield for a different periods. Chart filters can be set by clicking the Chart Options icon at the top right of the chart. Notes: (1) TBA itineraries are excluded from chart results. (2) If you work with multiple currencies, the chart defaults to the currency defined as the Default Rate Type for Internal Users (defined on the Defaults code table). Only bookings using that currency will be included in the initial results load. To include other currencies: set the exchange rate for each currency in the Chart Option\'s Currency Conversion table. (3) Revenue on itineraries with 0 bed nights is allocated to the Arrival date of the itinerary and the full amount is allocated as yield for the itinerary.';
    protected $reportInfo = 'Use the Financial Analysis report (use the Folio or Day report depending on the chart\'s filters). Set the report\'s date range and filters to match the chart\'s filters. Include Revenue Accommodation and Yield as report fields (in Select data option). Set the Display options to include totals by month.';

    public function toArray()
    {
        $libraryConfig = [
            'chart' => [
                'caption' => 'Accommodation Revenue & Yield',
                'subcaption' => '',
                'xAxisName' => 'Currency',
                'yAxisName' => 'Revenue',
                'sYAxisName' => 'Yield',
                'theme' => 'fusion',
                'captionAlignment' => 'center',
                'baseFont' => 'Helvetica',
                'legendItemFont' => 'Helvetica',
                'legendItemFontSize' => '13',
                'outCnvBaseFontSize' => '12',
                'xAxisNameFontSize' => '12'
            ]
        ];

        $mapping = [
            'category' => 'date',
            'series' => [
                [
                    'dynamicSeriesName' => 'revenueSeries',
                    'data' => 'revenue',
                    'config' => [
                        'plottooltext' => '<b>$value revenue</b> in $label',
                    ],
                    'dynamicConfig' => [
                        'color' => 'revenueColour'
                    ]
                ],
                [
                    'dynamicSeriesName' => 'yieldSeries',
                    'data' => 'yield',
                    'config' => [
                        'plottooltext' => '<b>$value yield</b> in $label',
                        'renderAs' => 'line',
                        'parentYAxis' => 'S',
                    ],
                    'dynamicConfig' => [
                        'color' => 'yieldColour',
                        'lineThickness' => 'yieldLineThickness',
                        'dashed' => 'yieldLineDashed',
                        'lineDashLen' =>  'yieldLineDashLength'
                    ]
                ],
            ]
        ];

        $dateRange = [
            'startDate' => '-6 years',
            'endDate' => '+6 months'
        ];

        $compareDateRange = [
            'startDate' => 'today',
            'endDate' => 'today'
        ];

        $dataset = (new Dataset('bookingRevenue'))
            ->addFilters([
                new Filter('rateType', 'rateType'),
                new Filter('rateComponent', 'rateComponent'),
                (new Filter('accommodationAccess', 'accommodationAccess'))
                    ->addOption(new Option('disableAcl', false)),
                (new Filter('reservationStatus', 'reservationStatus'))
                    ->addOption(new Option('reservationStatuses', ['confirmed', 'provisional'])),
                (new Filter('dateRange', 'dateRange'))
                    ->addOption(new Option('dateRange', $dateRange)),
                (new Filter('compare1DateRange', 'compare1DateRange'))
                    ->addOption(new Option('dateRange', $compareDateRange))
            ])
            ->addDatasetOptions([
                (new DatasetOption('farByFolio', 'farByFolio'))
                    ->addOption(new Option('farByFolio', 'day')),
                new DatasetOption('excludeTba', 'excludeTba'),
                (new DatasetOption('commission', 'commission'))
                    ->addOption(new Option('commission', 'nett')),
                (new DatasetOption('includeAccommTax', 'includeAccommTax'))
                    ->addOption(new Option('includeAccommTax', false)),
                (new DatasetOption('agent', 'agent'))
                    ->addOption(new Option('agent', false)),
            ])
            ->addConverters(
                [
                    (new Converter('additionGroup', 'additionGroup'))
                        ->setSubType('additionGroup')
                        ->addInputMappings([
                            new InputMapping('groupBy', ['date', 'invoiceCurrency', 'comparisonData']),
                            new InputMapping('additionFields', ['revenue', 'bedNights']),
                        ]),
                    (new Converter('dateFormat', 'dateFormat'))
                        ->setSubType('dateFormat')
                        ->addInputMappings([
                            new InputMapping('date', ['date']),
                        ])
                        ->addOutputMappings([
                            new OutputMapping('date', 'date'),
                        ])
                        ->addOption(new Option('format', 'M')),
                    (new Converter('additionGroup', 'additionGroup'))
                        ->setSubType('additionGroup')
                        ->addInputMappings([
                            new InputMapping('groupBy', ['date', 'invoiceCurrency']),
                            new InputMapping('additionFields', ['revenue', 'bedNights']),
                        ]),
                    (new Converter('bedNights'))
                        ->setSubType('max')
                        ->addInputMapping(
                            new InputMapping(
                                'maxFields',
                                [
                                    'bedNights'
                                ]
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'bedNights')
                        )
                        ->addOption(new Option('constants', [1])),
                    (new Converter('yield'))
                        ->setSubType('mathExpression')
                        ->addInputMapping(
                            new InputMapping(
                                'expressionFields',
                                [
                                    'revenue',
                                    'bedNights'
                                ]
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'yield')
                        )
                        ->addOption(new Option('expression', ['revenue', '/', 'bedNights'])),
                    (new Converter('roundYield'))
                        ->setSubType('round')
                        ->addInputMapping(
                            new InputMapping(
                                'field',
                                [
                                    'yield'
                                ]
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'yield')
                        ),
                    (new Converter('roundRevenue'))
                        ->setSubType('round')
                        ->addInputMapping(
                            new InputMapping(
                                'field',
                                [
                                    'revenue'
                                ]
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'revenue')
                        ),
                    (new Converter('revenueTemplate'))
                        ->setSubType('template')
                        ->addInputMapping(
                            new InputMapping(
                                'templateFields',
                                ['invoiceCurrency']
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'revenueSeries')
                        )->addOption(new Option('template', '{{invoiceCurrency}} revenue')),
                    (new Converter('yieldtemplate'))
                        ->setSubType('template')
                        ->addInputMapping(
                            new InputMapping(
                                'templateFields',
                                ['invoiceCurrency']
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'yieldSeries')
                        )->addOption(new Option('template', '{{invoiceCurrency}} yield')),
                    (new Converter('colourDistributor'))
                        ->setSubType('groupDistributor')
                        ->addInputMapping(new InputMapping('groupBy', ['invoiceCurrency']))
                        ->addOption(new Option(
                            'data',
                            [
                                [
                                    'revenueColour' => '#3364b5',
                                    'yieldColour' => '#C9252F',
                                ],
                                [
                                    'revenueColour' => '#cacaca',
                                    'yieldColour' => '#b37478',
                                    'yieldLineDashed' => true,
                                ],
                                [
                                    'revenueColour' => '#418549',
                                    'yieldColour' => '#42f557',
                                    'yieldLineThickness' => '5',
                                ],
                                [
                                    'revenueColour' => '#41857b',
                                    'yieldColour' => '#78d6c8',
                                    'yieldLineDashed' => true,
                                    'yieldLineDashLength' => 1
                                ]
                            ]
                        )),
                ]
            );

        $basicPage = (new UiPage('Basic'))
            ->addSections([
                (new UiPageSection())
                    ->addOption(
                        (new UiOption('dateRange'))
                            ->addSubOption(new UiSubOption('dateRange', 'Date Range (D/M/Y)'))
                    )
                    ->addOption(
                        (new UiOption('compare1DateRange'))
                            ->addSubOption(new UiSubOption('dateRange', 'Comparison Date Range'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('currency'))
                            ->addSubOption(new UiSubOption('currency', 'Currency Conversion'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('accommodationAccess'))
                            ->addSubOption(new UiSubOption('accommodationTypes', 'Property & Accommodation Type'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('reservationStatus'))
                            ->addSubOption(new UiSubOption('reservationStatuses', 'Status'))
                    )
            ]);

        $financialsPage = (new UiPage('Financials'))
            ->addSections([
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('farByFolio'))
                            ->addSubOption(new UiSubOption('farByFolio', 'Revenue By'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('includeAccommTax'))
                            ->addSubOption(new UiSubOption('includeAccommTax', 'Tax'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('commission'))
                            ->addSubOption(new UiSubOption('commission', 'Commission'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('rateComponent'))
                            ->addSubOption(new UiSubOption('components', 'Rate Components'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('rateType'))
                            ->addSubOption(new UiSubOption('rates', 'Rate Types'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('excludeTba'))
                            ->addSubOption(new UiSubOption('excludeTba'))
                    )
            ]);

        $uiConfig = (new UiConfig())
            ->addSection(
                (new UiSection('Options'))
                    ->addPages([
                        $basicPage,
                        $financialsPage
                    ])
            );

        $chart = (new ChartConfig())
            ->setId($this->id)
            ->setLibrary($this->library)
            ->setLibaryConfig($libraryConfig)
            ->setChartClass($this->chartClass)
            ->setName($this->name)
            ->setDescription($this->description)
            ->setReportInfo($this->reportInfo)
            ->addMapping($mapping)
            ->addDataset($dataset)
            ->setUiConfig($uiConfig);

        return  $chart->toArray();
    }
}
