<?php

namespace Resrequest\Application\Chart\Store\Charts;

use Resrequest\Application\Chart\Store\Chart;
use Resrequest\Application\Chart\Config\Dataset;
use Resrequest\Application\Chart\Config\Filter;
use Resrequest\Application\Chart\Config\Option;
use Resrequest\Application\Chart\Config\UiConfig;
use Resrequest\Application\Chart\Config\Chart as ChartConfig;
use Resrequest\Application\Chart\Config\Converter;
use Resrequest\Application\Chart\Config\DatasetOption;
use Resrequest\Application\Chart\Config\InputMapping;
use Resrequest\Application\Chart\Config\OutputMapping;
use Resrequest\Application\Chart\Config\UiOption;
use Resrequest\Application\Chart\Config\UiPage;
use Resrequest\Application\Chart\Config\UiPageSection;
use Resrequest\Application\Chart\Config\UiSection;
use Resrequest\Application\Chart\Config\UiSubOption;

class BusinessByCountry extends Chart
{
    protected $id = '11e9ad14ad08dddf90da02366610d15e';
    protected $library = 'FusionCharts';
    protected $chartClass = 'WorldWithCountries';
    protected $name = 'New sales by source country';
    protected $description = "This heat map shows new and lost sales during a specified sales period, with results grouped by the booking’s source country. Some bookings may have an unknown source country, so the source country is determined by the following (in this order): 1. Booking nationality. 2. The Agent’s physical / postal country (when there is no originator). 3. The Originator’s physical / postal country. 4. If booking has no Agent i.e. a Direct booking, and there is no booking nationality, then the physical / postal address of the booking’s contact person is used.";
    protected $reportInfo = "";

    public function toArray()
    {
        $libraryConfig = [
            'chart' => [
                'caption' => 'New Sales by Source Country',
                'subcaption' => 'Sales revenue by country for this year',
                'theme' => 'fusion',
                'captionAlignment' => 'center',
                'baseFont' => 'Helvetica',
                'legendItemFont' => 'Helvetica',
                'legendItemFontSize' => '13',
                'outCnvBaseFontSize' => '11',
                'xAxisNameFontSize' => '11',
                'nullEntityColor' => '#C2C2D6',
                'nullEntityAlpha' => '50',
                'hoverOnNull' => '0',
                'entitytooltext' => '<b>$dataValue {{invoiceCurrency}} sales revenue</b> in $lName',
                'formatNumber' => '1',
                'formatNumberScale' => '0',
                'hoverOnEmpty' => '0',
                'showLabels' => '0',
                'includeNameInLabels' => '0'
            ],
            'colorRange' => [
                'gradient' => "1",
                'startlabel' => "Lost revenue",
                'endlabel' => "Gained reveue",
                'code' => '#cacaca',
                'minValue' => "{{minSalesRevenue}}",
                'color' => [
                    [
                        'maxValue' => "{{minSalesRevenue}}",
                        'color' => "#c9252f"
                    ],
                    [
                        'minValue' => "0",
                        'maxValue' => "0",
                        'color' => "#cacaca"
                    ],
                    [
                        'maxValue' => "{{maxSalesRevenue}}",
                        'color' => "#3364b5"
                    ],
                ]
            ]
        ];

        $mapping = [
            'country' => 'originCountryCode',
            'value' => 'salesRevenue',
            'config' => [
                'showLabel' => 1
            ]
        ];

        $libraryConfig['annotations'] = [
            'autoscale' => '1',
            'groups' =>
            [
                [
                    'items' =>
                    [
                        [
                            'id' => 'currency-rectangle',
                            'type' => 'rectangle',
                            'radius' => '3',
                            'x' => '10',
                            'y' => '10',
                            'tox' => '80',
                            'toy' => '40',
                            'color' => '#c9c9c9',
                            'alpha' => '70',
                            'origW' => '150',
                            'origH' => '300',
                        ],
                        [
                            'id' => 'currency-text',
                            'type' => 'text',
                            'fillcolor' => '#ffffff',
                            'fontsize' => '10',
                            'x' => '45',
                            'y' => '27',
                            'text' => '{{invoiceCurrency}}',
                            'origW' => '150',
                            'origH' => '300',
                        ],
                    ],
                ],
            ],
        ];

        $dateRange = [
            'startDate' => 'First day of January this year',
            'endDate' => 'Last day of December this year'
        ];

        $dataset = (new Dataset('bookingRevenue'))
            ->addFilters([
                new Filter('rateType', 'rateType'),
                new Filter('rateComponent', 'rateComponent'),
                (new Filter('accommodationAccess', 'accommodationAccess'))
                    ->addOption(new Option('disableAcl', false)),
                (new Filter('reservationStatus', 'reservationStatus'))
                    ->addOption(new Option('reservationStatuses')),
                (new Filter('dateRange', 'dateRange'))
                    ->addOption(new Option('dateRange', $dateRange)),
                (new Filter('saleLostGainedDateRange', 'saleLostGainedDateRange'))
                    ->addOption(new Option('dateRange', $dateRange))
            ])
            ->addDatasetOptions([
                new DatasetOption('farByFolio', 'farByFolio'),
                new DatasetOption('excludeTba', 'excludeTba'),
                (new DatasetOption('commission', 'commission'))
                    ->addOption(new Option('commission', 'nett')),
                (new DatasetOption('includeAccommTax', 'includeAccommTax'))
                    ->addOption(new Option('includeAccommTax', false)),
                (new DatasetOption('calculateLostGainedRevenue', 'calculateLostGainedRevenue'))
                    ->addOption(new Option('calculateLostGainedRevenue', true)),
            ])
            ->addConverters(
                [
                    (new Converter('currency', 'currency'))
                        ->setSubType('currency')
                        ->addInputMappings([
                            new InputMapping('amount', ['revenueGained', 'revenueLost']),
                            new InputMapping('currencyFrom', ['invoiceCurrency'])
                        ])
                        ->addOutputMappings([
                            new OutputMapping('currencyTo', 'invoiceCurrency'),
                            new OutputMapping('amount', [
                                'revenueGained' => 'revenueGained',
                                'revenueLost' => 'revenueLost',
                            ])
                        ]),
                    (new Converter('reservationOriginCountry', 'reservationOriginCountry'))
                        ->setSubType('reservationCountry')
                        ->addInputMappings([
                            new InputMapping('reservationId', ['reservationId'])
                        ])
                        ->addOutputMapping(
                            new OutputMapping('originCountryCode', 'originCountryCode')
                        )
                        ->addOption(new Option('excludeReservationsWithoutCountry', true)),
                    (new Converter('additionGroup', 'additionGroup'))
                        ->setSubType('additionGroup')
                        ->addInputMappings([
                            new InputMapping('groupBy', ['originCountryCode']),
                            new InputMapping('additionFields', ['revenueGained', 'revenueLost']),
                        ]),
                    (new Converter('salesRevenue'))
                        ->setSubType('mathExpression')
                        ->addInputMapping(
                            new InputMapping(
                                'expressionFields',
                                [
                                    'revenueLost',
                                    'revenueGained'
                                ]
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'salesRevenue')
                        )
                        ->addOption(new Option('expression', ['revenueGained', '-', 'revenueLost'])),
                    (new Converter('roundSalesRevenue'))
                        ->setSubType('round')
                        ->addInputMapping(
                            new InputMapping(
                                'field',
                                [
                                    'salesRevenue'
                                ]
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'salesRevenue')
                        ),
                    (new Converter('salesRevenueRange'))
                        ->setSubType('datasetRange')
                        ->addInputMapping(
                            new InputMapping('field', ['salesRevenue'])
                        )
                        ->addOutputMapping(
                            new OutputMapping('minimum', 'minSalesRevenue')
                        )
                        ->addOutputMapping(
                            new OutputMapping('maximum', 'maxSalesRevenue')
                        ),
                    (new Converter('minSalesRevenue'))
                        ->setSubType('mathExpression')
                        ->addInputMapping(
                            new InputMapping(
                                'expressionFields',
                                [
                                    'minSalesRevenue',
                                ]
                            )
                        )
                        ->addOutputMapping(
                            new OutputMapping('result', 'minSalesRevenue')
                        )
                        ->addOption(new Option('expression', ['minSalesRevenue', '-', '0.01'])),
                ]
            );

        $basicPage = (new UiPage('Basic'))
            ->addSections([
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('saleLostGainedDateRange'))
                            ->addSubOption(new UiSubOption('dateRange', 'Sale Date Range'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('currency'))
                            ->addSubOption(new UiSubOption('currency', 'Currency Conversion'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('accommodationAccess'))
                            ->addSubOption(new UiSubOption('accommodationTypes', 'Property & Accommodation Type'))
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('reservationStatus'))
                            ->addSubOption(new UiSubOption('reservationStatuses', 'Status'))
                    )
            ]);

        $uiConfig = (new UiConfig())
            ->addSection(
                (new UiSection('Options'))
                    ->addPages([
                        $basicPage,
                    ])
            );

        $chart = (new ChartConfig())
            ->setId($this->id)
            ->setLibrary($this->library)
            ->setLibaryConfig($libraryConfig)
            ->setChartClass($this->chartClass)
            ->setName($this->name)
            ->setDescription($this->description)
            ->setReportInfo($this->reportInfo)
            ->addMapping($mapping)
            ->addDataset($dataset)
            ->setUiConfig($uiConfig);

        return $chart->toArray();
    }
}
