<?php

namespace Resrequest\Application\Chart\Store;

use Resrequest\Application\Chart\Registry;

class Container
{
    /**
     * The id of the container.
     *
     * @var string
     */
    protected $id;

    /**
     * The name of the container.
     *
     * @var string
     */
    protected $name;

    /**
     * The description of the container.
     *
     * @var string
     */
    protected $description;

    /**
     * The container layout configuration.
     *
     * @var array
     */
    protected $gridsterConfig = [
        'configuration' => [
            'gridType' => 'verticalFixed',
            'compactType' => 'compactUp&Left',
            'margin' => 20,
            'outerMargin' => false,
            'useTransformPositioning' => true,
            'mobileBreakpoint' => 1000,
            'minCols' => 1,
            'maxCols' => 19,
            'minRows' => 1,
            'maxRows' => 100,
            'maxItemCols' => 100,
            'minItemCols' => 1,
            'maxItemRows' => 100,
            'minItemRows' => 1,
            'maxItemArea' => 2500,
            'minItemArea' => 1,
            'defaultItemCols' => 1,
            'defaultItemRows' => 1,
            'fixedColWidth' => 105,
            'fixedRowHeight' => 70,
            'keepFixedHeightInMobile' => true,
            'keepFixedWidthInMobile' => false,
            'scrollSensitivity' => 10,
            'scrollSpeed' => 20,
            'enableEmptyCellClick' => false,
            'enableEmptyCellContextMenu' => false,
            'enableEmptyCellDrop' => false,
            'enableEmptyCellDrag' => false,
            'emptyCellDragMaxCols' => 50,
            'emptyCellDragMaxRows' => 50,
            'ignoreMarginInRow' => false,
            'draggable' => [
                'enabled' => false
            ],
            'resizable' => [
                'enabled' => false
            ],
            'swap' => false,
            'pushItems' => true,
            'disablePushOnDrag' => true,
            'disablePushOnResize' => false,
            'pushDirections' => [
                'north' => true,
                'east' => true,
                'south' => true,
                'west' => true
            ],
            'pushResizeItems' => false,
            'displayGrid' => "never",
            'disableWindowResize' => false,
            'disableWarnings' => false,
            'scrollToNewItems' => false
        ]
    ];

    /**
     * Gridster items
     *
     * @var array
     */
    protected $items;

    /**
     * The charts that are contained within this container;
     *
     * @var array
     */
    protected $charts;

    /**
     * The UI config for the chhart container
     *
     * @var array
     */
    protected $uiConfig;

    protected $options = [];
    protected $converters = [];

    /**
     * Creates a new container instance.
     *
     * @param string $id The id of the container.
     * @param array $charts The charts contained in the container.
     */
    public function __construct(
        $id = '',
        $name = '',
        $description = '',
        $charts = [],
        $uiConfig = null,
        $converters = [],
        $options = [],
        $optionsConfig = []
    ) {
        $this->id = $id;
        $this->name = $name;
        $this->description = $description;
        $this->uiConfig = $uiConfig;
        $this->setCharts($charts);

        if (!empty($converters)) {
            foreach ($converters as $converter) {
                $converterConfig = $converter->toArray();
                $name = $converterConfig['name'];
                $type = $converterConfig['type'];
                if (Registry::hasConverter($type)) {
                    $converterClass = Registry::getConverterClass($type);
                    $config = $converterConfig;
                    $this->converters[] = new $converterClass($name, $config);
                } else {
                    throw new \Exception("Invalid converter '$type'");
                }
            }
        }

        // Update supported options with the provided config
        // if (!empty($config['options'])) {
        //     foreach ($config['options'] as $option) {
        //         if (empty($option['config']['name'])) {
        //             throw new \Exception('Invalid option config');
        //         }

        //         $optionConfig = $option['config'];
        //         $name = $optionConfig['name'];
        //         if ($this->isOptionSupported($name)) {
        //             $option = $this->getOptionByName($name);
        //             // Apply option config
        //             $option->update($optionConfig);
        //         }
        //     }
        // }

        foreach ($options as $option) {
            $name = $option->getName();
            $this->options[$name] = $option;
        }

    }

    /**
     * Sets the gridster configuration for the component.
     *
     * @param array $config
     * @return void
     */
    public function setGridsterConfig($config)
    {
        $this->gridsterConfig = $config;
    }

    /**
     * Sets the charts that are part of the container.
     *
     * @param array $charts
     * @return void
     */
    public function setCharts($charts)
    {
        $this->charts = [];
        $this->items = [];
        foreach ($charts as $chart) {
            $this->charts[] = $chart->getChart();
            $this->items[] = $chart->toArray();
        }
    }

    /**
     * Returns the id of the container.
     *
     * @return string
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Get the charts that are part of the container.
     *
     * @return array
     */
    public function getCharts()
    {
        return $this->charts;
    }

    /**
     * Override in child class
     *
     * @return array Array of options
     */
    protected function generateOptions()
    {
        return [];
    }

    /**
     * The array representation of the container.
     *
     * @return array
     */
    public function toArray($processOptions = false)
    {
        $gridsterConfig = array_merge($this->gridsterConfig, ['items' => $this->items]);

        $optionConfigs = [];

        foreach ($this->options as $option) {
            if ($processOptions) {
                $option->processOptionsAndMappings();
            }
            $optionConfigs[] = [
                'type' => $option->getType(),
                'config' => $option->toArray()
            ];
        }

        $converterConfigs = [];

        foreach ($this->converters as $converter) {
            $converterConfigs[] = $converter->toArray();
        }
        
        if ($this->uiConfig) {
            $uiConfig = $this->uiConfig->toArray();
        } else {
            $uiConfig = null;
        }

        return [
            'id' => $this->id,
            'name' => $this->name,
            'description' => $this->description,
            'config' => [
                'dashboard' => $gridsterConfig,
                'charts' => $this->charts,
                'ui' => $uiConfig,
                'options' => $optionConfigs,
                'converters' => $converterConfigs
            ]
        ];
    }
}
