<?php

namespace Resrequest\Application\Chart\Store;

use Resrequest\Application\Chart\Config\Converter;
use Resrequest\Application\Chart\Config\InputMapping;
use Resrequest\Application\Chart\DatasetOption;
use Resrequest\Application\Chart\Filter\AccommodationAccess;
use Resrequest\Application\Chart\Filter\DateRange;
use Resrequest\Application\Chart\Filter\RateComponent;
use Resrequest\Application\Chart\Filter\RateType;
use Resrequest\Application\Chart\Filter\ReservationStatus;
use Resrequest\Application\Chart\Input;
use Resrequest\Application\Chart\Input\ButtonToggle;
use Resrequest\Application\Chart\Option;
use Resrequest\Application\Chart\Config\OutputMapping;
use Resrequest\Application\Chart\Config\UiConfig;
use Resrequest\Application\Chart\Config\UiOption;
use Resrequest\Application\Chart\Config\UiPage;
use Resrequest\Application\Chart\Config\UiPageSection;
use Resrequest\Application\Chart\Config\UiSection;
use Resrequest\Application\Chart\Config\UiSubOption;
use Resrequest\Application\Chart\Input\RadioGroup;
use Resrequest\Application\Chart\Input\SliderToggle;

class ContainerStore
{
    /**
     * The containers in the store.
     *
     * @var array
     */
    private $containers;

    public function __construct()
    {
        $this->containers = [];
        $this->addContainers();
    }

    /**
     * Whether there is a config for the given chart.
     *
     * @param string $id
     * @return boolean
     */
    public function hasContainer($id)
    {
        return array_key_exists($id, $this->containers);
    }

    /**
     * Retrieves the config for a chart.
     *
     * @param string $id
     * @return array
     */
    public function getContainerConfig($id, $processOptions = false)
    {
        return $this->containers[$id]->toArray($processOptions);
    }

    /**
     * Adds containers to the store.
     *
     * @param Chart $chart
     * @return void
     */
    private function addContainers()
    {
        $this->addReservationManagementContainer();
        $this->addSalesOverviewContainer();
        $this->addProductUsageContainer();
        $this->addTourismTodayContainers();
    }

    private function addContainer($id, $name, $description, $charts = [], $uiConfig = [], $converters = [], $options = [], $optionsConfig = [])
    {
        $this->containers[$id] = new Container($id, $name, $description, $charts, $uiConfig, $converters, $options, $optionsConfig);
    }

    public function getContainerConfigs($processOptions = false)
    {
        $configs = [];

        foreach ($this->containers as $container) {
            $configs[] = $container->toArray($processOptions);
        }

        return $configs;
    }

    private function addReservationManagementContainer()
    {
        $id = '11e9d3a1cbb8051b8f6902399910d15e';
        $name = 'Reservation management';
        $description = '';
        $charts = [
            new GridsterItem(ChartStore::chartIdByName('consultantBookingsByArrival'), 0, 0, 4, 7),
            new GridsterItem(ChartStore::chartIdByName('arrivalsWithoutPayment'), 7, 0, 4, 5),
            new GridsterItem(ChartStore::chartIdByName('arrivalsByConsultantWithoutPayment'), 12, 0, 4, 5),
            new GridsterItem(ChartStore::chartIdByName('newBookingsByConsultants'), 0, 4, 4, 7),
            new GridsterItem(ChartStore::chartIdByName('expiringProvisionals'), 7, 4, 4, 5),
            new GridsterItem(ChartStore::chartIdByName('convertibleWaitlists'), 12, 4, 4, 5),
            new GridsterItem(ChartStore::chartIdByName('newBookingsPerformance'), 0, 8, 4, 17),
        ];

        $this->addContainer($id, $name, $description, $charts);
    }

    private function addSalesOverviewContainer()
    {
        $id = '11ea1c3fae309e1a91f502399910d15e';
        $name = 'Sales overview';
        $description = '';
        $charts = [
            new GridsterItem(ChartStore::chartIdByName('revenueAccommodationThreeYear'), 0, 0, 4, 6),
            new GridsterItem(ChartStore::chartIdByName('revenueAccommodationCumulativeTwoYear'), 0, 6, 4, 6),
            new GridsterItem(ChartStore::chartIdByName('revenueYield'), 9, 0, 8, 8),
            new GridsterItem(ChartStore::chartIdByName('occupancy'), 0, 8, 6, 14),
        ];

        // Add container options
        $radioGroup = new RadioGroup();
        $radioGroup->setValue('nett');
        $radioGroup->addOption('Nett', 'nett');
        $radioGroup->addOption('Gross', 'gross');
        $radioGroup->addOption('Payable', 'payable');

        $accommTaxToggle = new ButtonToggle();
        $accommTaxToggle->setValue(true);
        $accommTaxToggle->addOption('Inclusive', true);
        $accommTaxToggle->addOption('Exclusive', false);

        $farByToggle = new ButtonToggle();
        $farByToggle->setValue('day');
        $farByToggle->addOption('Day', 'day');
        $farByToggle->addOption('Folio', 'folio');

        $options = [
            new RateType('rateType', [], false),
            new RateComponent('rateComponent', [], false),
            new AccommodationAccess('accommodationAccess', [], false),
            new ReservationStatus('reservationStatus', [], false),
            new DatasetOption('farByFolio', [new Option('farByFolio', $farByToggle)], false),
            new DatasetOption('excludeTba', [new Option('excludeTba', new SliderToggle(false, 'Exclude TBAs'))], false),
            new DatasetOption('commission', [new Option('commission', $radioGroup)], false),
            new DatasetOption('includeAccommTax', [new Option('includeAccommTax', $accommTaxToggle)], false),
            new DatasetOption('agent', [new Option('agent', new Input(false))], false),
            new DatasetOption('calculateLostGainedRevenue', [new Option('calculateLostGainedRevenue', new Input(false))], false),
        ];

        $dateRange = [
            'startDate' => 'First day of January this year',
            'endDate' => 'Last day of December this year'
        ];

        $basicPage = (new UiPage('Basic'))
            ->addSections([
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('currency'))
                            ->addSubOption(
                                (new UiSubOption('currency', 'Currency Conversion'))
                                    ->addLinkedSubOption('currency', 'currency', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('currency', 'currency', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('currency', 'currency', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('currency', 'currency', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('currency', 'currency', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('currency', 'currency', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('accommodationAccess'))
                            ->addSubOption(
                                (new UiSubOption('accommodationTypes', 'Property & Accommodation Type'))
                                    ->addLinkedSubOption('accommodationAccess', 'accommodationTypes', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('accommodationAccess', 'accommodationTypes', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('accommodationAccess', 'accommodationTypes', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('accommodationAccess', 'accommodationTypes', '11ea7f6e5e70b330973602366610d15e')
                                    ->addLinkedSubOption('accommodationAccess', 'accommodationTypes', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('accommodationAccess', 'accommodationTypes', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('accommodationAccess', 'accommodationTypes', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('reservationStatus'))
                            ->addSubOption(
                                (new UiSubOption('reservationStatuses', 'Status'))
                                    ->addLinkedSubOption('reservationStatus', 'reservationStatuses', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('reservationStatus', 'reservationStatuses', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('reservationStatus', 'reservationStatuses', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('reservationStatus', 'reservationStatuses', '11ea7f6e5e70b330973602366610d15e')
                                    ->addLinkedSubOption('reservationStatus', 'reservationStatuses', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('reservationStatus', 'reservationStatuses', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('reservationStatus', 'reservationStatuses', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    )
            ]);

        $financialsPage = (new UiPage('Financials'))
            ->addSections([
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('farByFolio'))
                            ->addSubOption(
                                (new UiSubOption('farByFolio', 'Revenue By'))
                                    ->addLinkedSubOption('farByFolio', 'farByFolio', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('farByFolio', 'farByFolio', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('farByFolio', 'farByFolio', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('farByFolio', 'farByFolio', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('farByFolio', 'farByFolio', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('farByFolio', 'farByFolio', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('includeAccommTax'))
                            ->addSubOption(
                                (new UiSubOption('includeAccommTax', 'Tax'))
                                    ->addLinkedSubOption('includeAccommTax', 'includeAccommTax', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('includeAccommTax', 'includeAccommTax', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('includeAccommTax', 'includeAccommTax', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('includeAccommTax', 'includeAccommTax', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('includeAccommTax', 'includeAccommTax', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('includeAccommTax', 'includeAccommTax', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('commission'))
                            ->addSubOption(
                                (new UiSubOption('commission', 'Commission'))
                                    ->addLinkedSubOption('commission', 'commission', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('commission', 'commission', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('commission', 'commission', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('commission', 'commission', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('commission', 'commission', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('commission', 'commission', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('rateComponent'))
                            ->addSubOption(
                                (new UiSubOption('components', 'Rate Components'))
                                    ->addLinkedSubOption('rateComponent', 'components', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('rateComponent', 'components', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('rateComponent', 'components', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('rateComponent', 'components', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('rateComponent', 'components', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('rateComponent', 'components', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('rateType'))
                            ->addSubOption(
                                (new UiSubOption('rates', 'Rate Types'))
                                    ->addLinkedSubOption('rateType', 'rates', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('rateType', 'rates', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('rateType', 'rates', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('rateType', 'rates', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('rateType', 'rates', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('rateType', 'rates', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    ),
                (new UiPageSection(''))
                    ->addOption(
                        (new UiOption('excludeTba'))
                            ->addSubOption(
                                (new UiSubOption('excludeTba'))
                                    ->addLinkedSubOption('excludeTba', 'excludeTba', '11ea31004cdfaf038fc802399910d15e')
                                    ->addLinkedSubOption('excludeTba', 'excludeTba', '11ea7ec64157287a989402399910d15e')
                                    ->addLinkedSubOption('excludeTba', 'excludeTba', '11ea7f6e5e70b330973602399910d15e')
                                    ->addLinkedSubOption('excludeTba', 'excludeTba', '11ea84e029d970eb952002399910d15e')
                                    ->addLinkedSubOption('excludeTba', 'excludeTba', '11ea85450f542779952002399910d15e')
                                    ->addLinkedSubOption('excludeTba', 'excludeTba', '11e9ad14ad08dddf90da02366610d15e')
                            )
                    )
            ]);

        $uiConfig = (new UiConfig())
            ->addSection(
                (new UiSection('Options'))
                    ->addPages([
                        $basicPage,
                        $financialsPage
                    ])
            );

        $converters = [
            (new Converter('currency', 'currency'))
                ->setSubType('currency')
                ->addInputMappings([
                    new InputMapping('amount', ['revenueGained', 'revenueLost']),
                    new InputMapping('currencyFrom', ['invoiceCurrency'])
                ])
                ->addOutputMappings([
                    new OutputMapping('currencyTo', 'invoiceCurrency'),
                    new OutputMapping('amount', [
                        'revenueGained' => 'revenueGained',
                        'revenueLost' => 'revenueLost',
                    ])
                ])
        ];

        $this->addContainer($id, $name, $description, $charts, $uiConfig, $converters, $options);
    }

    /**
     * Adds dataset options to the dataset.
     *
     * @param array $options
     * @return array $options
     */
    public function addDatasetOptions($options)
    {
        $options = array_map(function ($option) {
            return [
                'type' => 'data',
                'config' => $option
            ];
        }, $options);

        return $options;
    }

    /**
     * Adds filters to the dataset.
     *
     * @param array $filters
     * @return array $filters
     */
    public function addFilters($filters)
    {
        $filters = array_map(function ($filter) {
            return [
                'type' => 'filter',
                'config' => $filter
            ];
        }, $filters);

        return $filters;
    }

    private function addProductUsageContainer()
    {
        $id = '11ea1c3faeproductusage399910d15e';
        $name = 'Product usage';
        $description = '';
        $charts = [];

        $this->addContainer($id, $name, $description, $charts);
    }

    private function addTourismTodayContainers() {
        $this->addContainer('11eac9c7a7a83245ab4102ae0e1a0d08', 'Sales & travel movements', '', []);
        $this->addContainer('11eac9c7afb68b7bab4102ae0e1a0d08', 'Travel routes', '', []);
        $this->addContainer('11eac9c7b21d3cb8ab4102ae0e1a0d08', 'Booking tempo', '', []);
        $this->addContainer('11eac9c7c2bf3186ab4102ae0e1a0d08', 'Travel behaviour', '', []);
        $this->addContainer('11ead7940a7d595da11002ae0e1a0d08', 'Occupancies', '', []);
        $this->addContainer('11ead79410027fffa11002ae0e1a0d08', 'Yield, lead times & materialisation', '', []);
    }
}
