<?php

namespace Resrequest\Application\Financial\System;

use Doctrine\ORM\EntityManager;
use Resrequest\DB\Enterprise\Entity\FnOptionData;
use Exception;

class FinancialOptions
{
    /**
     * Entity manager instance for communication with the database.
     *
     * @var EntityManager
     */
    private $entityManager;

    /**
     * The input configuartions for the various financial options.
     *
     * @var array
     */
    private $inputConfigurations;

    /**
     * Input types available.
     */
    const INPUT_SELECT = 'select';
    const INPUT_SEQUENCE = 'sequence';
    const INPUT_CSV = 'csv';
    const INPUT_NUMERIC = 'numeric';
    const INPUT_TEXT = 'text';
    const INPUT_CHECKBOX = 'checkbox';

    /**
     * Create a new instance of the FinancialOptions.
     *
     * @param EntityManager $entityManager
     */
    public function __construct(EntityManager $entityManager)
    {
        $this->entityManager = $entityManager;
        $this->inputConfigurations = [];
        $this->loadConfigurations();
    }

    private function loadConfigurations()
    {
        $this->loadAccountCodeFormat();
        $this->loadTaxPrecision();
        $this->loadJournalFormat();
        $this->loadReferenceFormat();
        $this->loadDebtorRecalculate();
        $this->loadSummaryDate();
        $this->loadExchangeRateFormat();
        $this->loadAccountCodeSeparator();
        $this->loadCompanyName();
        $this->loadQBInvAmount();
        $this->loadMemoFormat();
        $this->loadXeroTrackingLabels();
        $this->loadXeroTrackingValues();
        $this->loadMemoFormatPayment();
        $this->loadMemoFormatInvoice();
        $this->loadTwinfieldsCode();
        $this->loadHeaderLanguage();
        $this->loadZAxisConcatenations();
        $this->loadZAxisTaxGroups();
    }

    private function loadAccountCodeFormat()
    {
        $this->inputConfigurations['RS1'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'Ledger code',
                    'value' => '1'
                ],
                [
                    'label' => 'Ledger code + Cost centre',
                    'value' => '2'
                ],
                [
                    'label' => 'Cost centre + Ledger code',
                    'value' => '3'
                ],
            ]
        ];
    }

    private function loadTaxPrecision()
    {
        $this->inputConfigurations['RS2'] = [
            'type' => self::INPUT_NUMERIC,
            'config' => null
        ];
    }

    private function loadJournalFormat()
    {
        $this->inputConfigurations['RS3'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'Normal',
                    'value' => '1'
                ],
                [
                    'label' => 'With source currency',
                    'value' => '2'
                ],
            ]
        ];
    }

    private function loadReferenceFormat()
    {
        $this->inputConfigurations['RS4'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'Line item type',
                    'value' => '1'
                ],
                [
                    'label' => 'Reservation number',
                    'value' => '2'
                ],
                [
                    'label' => 'Link (invoice or payment item)',
                    'value' => '3'
                ],
                [
                    'label' => 'Debtor code',
                    'value' => '4'
                ],
                [
                    'label' => 'Document number',
                    'value' => '5'
                ],
                [
                    'label' => 'Reservation number + Invoice number',
                    'value' => '6'
                ],
            ]
        ];
    }

    private function loadDebtorRecalculate()
    {
        $this->inputConfigurations['RS5'] = [
            'type' => self::INPUT_CHECKBOX,
            'config' => null
        ];
    }

    private function loadSummaryDate()
    {
        $this->inputConfigurations['RS6'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'Latest',
                    'value' => '1'
                ],
                [
                    'label' => 'Earliest',
                    'value' => '2'
                ],
            ]
        ];
    }

    private function loadExchangeRateFormat()
    {
        $this->inputConfigurations['RS7'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'Home currency into x foreign currency',
                    'value' => '1'
                ],
                [
                    'label' => 'Foreign currency into x home currency',
                    'value' => '2'
                ],
            ]
        ];
    }

    private function loadAccountCodeSeparator()
    {
        $this->inputConfigurations['RS8'] = [
            'type' => self::INPUT_TEXT,
            'config' => null
        ];
    }

    private function loadCompanyName()
    {
        $this->inputConfigurations['RS9'] = [
            'type' => self::INPUT_TEXT,
            'config' => null
        ];
    }

    private function loadQBInvAmount()
    {
        $this->inputConfigurations['RS10'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'Export source currency incl tax',
                    'value' => '1'
                ],
                [
                    'label' => 'Export source currency excl tax',
                    'value' => '2'
                ],
                [
                    'label' => 'Export GL currency incl tax',
                    'value' => '3'
                ],
                [
                    'label' => 'Export GL currency excl tax',
                    'value' => '4'
                ],
            ]
        ];
    }

    private function loadMemoFormat()
    {
        $this->inputConfigurations['RS11'] = [
            'type' => self::INPUT_SEQUENCE,
            'config' => [
                [
                    'label' => 'Transaction type',
                    'value' => '1'
                ],
                [
                    'label' => 'Source document ID',
                    'value' => '2'
                ],
                [
                    'label' => 'Reservation number',
                    'value' => '3'
                ],
                [
                    'label' => 'Reservation name',
                    'value' => '4'
                ],
                [
                    'label' => 'Debtor name',
                    'value' => '5'
                ],
                [
                    'label' => 'Source currency',
                    'value' => '6'
                ],
                [
                    'label' => 'Exchange rate',
                    'value' => '7'
                ],
            ]
        ];
    }

    private function loadXeroTrackingLabels()
    {
        $this->inputConfigurations['RS12'] = [
            'type' => self::INPUT_CSV,
            'config' => null
        ];
    }

    private function loadXeroTrackingValues()
    {
        $this->inputConfigurations['RS13'] = [
            'type' => self::INPUT_SEQUENCE,
            'config' => [
                [
                    'label' => 'Blank',
                    'value' => '1'
                ],
                [
                    'label' => 'Revenue centre 1',
                    'value' => '2'
                ],
                [
                    'label' => 'Revenue centre 2',
                    'value' => '6'
                ],
                [
                    'label' => 'Reservation number',
                    'value' => '3'
                ],
                [
                    'label' => 'Invoice number',
                    'value' => '4'
                ],
                [
                    'label' => 'Debtor code',
                    'value' => '5'
                ],
            ]
        ];
    }

    private function loadMemoFormatPayment()
    {
        $this->inputConfigurations['RS14'] = [
            'type' => self::INPUT_SEQUENCE,
            'config' => [
                [
                    'label' => 'Transaction type',
                    'value' => '1'
                ],
                [
                    'label' => 'Source document ID',
                    'value' => '2'
                ],
                [
                    'label' => 'Payment item ID',
                    'value' => '3'
                ],
                [
                    'label' => 'Reservation number',
                    'value' => '4'
                ],
                [
                    'label' => 'Reservation name',
                    'value' => '5'
                ],
                [
                    'label' => 'Debtor name',
                    'value' => '6'
                ],
                [
                    'label' => 'Source currency',
                    'value' => '7'
                ],
                [
                    'label' => 'Exchange rate',
                    'value' => '8'
                ],
                [
                    'label' => 'Bank/CP abbreviation',
                    'value' => '9'
                ],
                [
                    'label' => 'Payment note',
                    'value' => '20'
                ],
                [
                    'label' => 'Payment reference',
                    'value' => '21'
                ],
            ]
        ];
    }

    private function loadMemoFormatInvoice()
    {
        $this->inputConfigurations['RS15'] = [
            'type' => self::INPUT_SEQUENCE,
            'config' => [
                [
                    'label' => 'Transaction type',
                    'value' => '1'
                ],
                [
                    'label' => 'Source document ID',
                    'value' => '2'
                ],
                [
                    'label' => 'Reservation number',
                    'value' => '3'
                ],
                [
                    'label' => 'Reservation name',
                    'value' => '4'
                ],
                [
                    'label' => 'Debtor name',
                    'value' => '5'
                ],
                [
                    'label' => 'Source currency',
                    'value' => '6'
                ],
                [
                    'label' => 'Exchange rate',
                    'value' => '7'
                ],
                [
                    'label' => 'Revenue item',
                    'value' => '20'
                ],
            ]
        ];
    }

    private function loadTwinfieldsCode()
    {
        $this->inputConfigurations['RS16'] = [
            'type' => self::INPUT_TEXT,
            'config' => null
        ];
    }

    private function loadHeaderLanguage()
    {
        $this->inputConfigurations['RS17'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'No header',
                    'value' => '1'
                ],
                [
                    'label' => 'English',
                    'value' => '2'
                ],
                [
                    'label' => 'Dutch',
                    'value' => '3'
                ],
            ]
        ];
    }

    private function loadZAxisConcatenations() {
        $this->inputConfigurations['RS19'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'None (default)', 
                    'value' => '0'
                ],
                [
                    'label' => 'Revenue centre 1', 
                    'value' => '1'
                ],
                [
                    'label' => 'Revenue centre 2', 
                    'value' => '2'
                ],
                [
                    'label' => 'Revenue centre 1 : Revenue centre 2', 
                    'value' => '3'
                ],
                [
                    'label' => 'Revenue centre 2 : Revenue centre 1', 
                    'value' => '4'
                ]
            ]

        ];
    }

    private function loadZAxisTaxGroups()
    {
        $this->inputConfigurations['RS20'] = [
            'type' => self::INPUT_SELECT,
            'config' => [
                [
                    'label' => 'No (default)',
                    'value' => '0'
                ],
                [
                    'label' => 'Yes',
                    'value' => '1'
                ]
            ]

        ];
    }

    /**
     * Get the input configuration for the given option.
     *
     * @param string $optionId
     * @return array
     */
    private function input(string $optionId)
    {
        return $this->inputConfigurations[$optionId];
    }

    /**
     * Whether the invoicing unit exists and has a financial interface set.
     *
     * @param string $invoicingUnit
     * @return boolean
     */
    public function isValidInvoicingUnit(string $invoicingUnit)
    {
        $builder = $this->entityManager->createQueryBuilder();
        $record = null;

        try {
            $record = $builder
                ->select('prBusiness.fnSystemId')
                ->from('Resrequest\DB\Enterprise\Entity\PrBusiness', 'prBusiness')
                ->where('prBusiness.prBusinessId = :invoicingUnit')
                ->setParameters(['invoicingUnit' => $invoicingUnit])
                ->getQuery()
                ->getSingleResult();
        } catch (Exception $e) {
            //
        }

        return $record['fnSystemId'] ? true : false;
    }

    /**
     * The options for the financial interface that has been set for the
     * invoicing unit.
     *
     * @param string $invoicingUnit
     * @return array
     */
    public function optionsForInvoicingUnit(string $invoicingUnit)
    {
        $financialSystem = $this->financialSystem($invoicingUnit);
        $builder = $this->entityManager->createQueryBuilder();
        $options = $builder
            ->select([
                'fnSystemOption.fnOptionId AS id',
                'fnSystemOption.fnSystemOptionDefault AS default_value',
                'fnOptionData.fnOptionDataContents AS current_value',
                'fnSystemOption.fnSystemOptionDesc AS description',
                'fnOption.fnOptionName AS name',
                'fnOption.fnOptionDesc AS raw_description'
            ])
            ->from('Resrequest\DB\Enterprise\Entity\FnSystemOption', 'fnSystemOption')
            ->innerJoin(
                'Resrequest\DB\Enterprise\Entity\FnOption',
                'fnOption',
                'with',
                'fnOption.fnOptionIx = fnSystemOption.fnOptionId'
            )
            ->leftJoin(
                'Resrequest\DB\Enterprise\Entity\FnOptionData',
                'fnOptionData',
                'with',
                'fnOptionData.fnOptionId = fnSystemOption.fnOptionId AND fnOptionData.prBusinessId = :invoicingUnit'
            )
            ->where('fnSystemOption.fnSystemId = :systemId')
            ->setParameters([
                'invoicingUnit' => $invoicingUnit,
                'systemId' => $financialSystem,
            ])
            ->orderBy('fnOption.fnOptionId')
            ->getQuery()
            ->getResult();

        foreach ($options as &$option) {
            $option['input'] = $this->input($option['id']);
        }

        return $options;
    }

    /**
     * Retrieves the financial system id for the given invoicing unit.
     *
     * @param string $invoicingUnit
     * @return string
     */
    public function financialSystem(string $invoicingUnit)
    {
        $builder = $this->entityManager->createQueryBuilder();
        $record = null;

        try {
            $record = $builder
                ->select('prBusiness.fnSystemId')
                ->from('Resrequest\DB\Enterprise\Entity\PrBusiness', 'prBusiness')
                ->where('prBusiness.prBusinessId = :invoicingUnit')
                ->setParameters(['invoicingUnit' => $invoicingUnit])
                ->getQuery()
                ->getSingleResult();
        } catch (Exception $e) {
            //
        }

        return $record['fnSystemId'] ? $record['fnSystemId'] : '';
    }

    /**
     * Whether the provided options are valid for the invoicing unit's
     * current system.
     *
     * @param string $invoicingUnit
     * @param array $options
     * @return boolean
     */
    public function validOptions(string $invoicingUnit, array $options)
    {
        $validOptions = array_map(
            function($option) {
                return $option['id'];
            },
            $this->optionsForInvoicingUnit($invoicingUnit)
        );

        foreach ($options as $option) {
            if (!in_array($option['id'], $validOptions)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Save the options for the invoicing unit.
     *
     * @param string $invoicingUnit
     * @param array $options
     * @return void
     */
    public function saveOptions(string $invoicingUnit, array $options)
    {
        // Remove existing options
        $builder = $this->entityManager->createQueryBuilder();
        $builder
            ->delete('Resrequest\DB\Enterprise\Entity\FnOptionData', 'fnOptionData')
            ->where('fnOptionData.prBusinessId = :invoicingUnit')
            ->setParameters(['invoicingUnit' => $invoicingUnit])
            ->getQuery()
            ->execute();

        $newOptions = [];

        foreach ($options as $option) {
            $newOption = new FnOptionData();
            $newOption
                ->setFnOptionDataId(0)
                ->setPrBusinessId($invoicingUnit)
                ->setFnOptionId($option['id'])
                ->setFnOptionDataContents($option['value']);
            array_push($newOptions, $newOption);
        }

        $this->save($newOptions);
    }

    /**
     * Persist the new FnOptionData options.
     *
     * @param array $options
     * @return void
     */
    private function save(array $options)
    {
        foreach ($options as $option) {
            $this->entityManager->persist($option);
        }

        $this->entityManager->flush();
    }
}