<?php

namespace Resrequest\Application\Reminder;

use MaglLegacyApplication\Application\MaglLegacy;
use ZF\ContentNegotiation\JsonModel;
use Doctrine\ORM\EntityManager;
use Resrequest\DB\Enterprise\Entity\AwWorkflow;
use Resrequest\DB\Enterprise\Entity\ReReminder;
use Resrequest\DB\Enterprise\Entity\ReReminderPersona;
use Resrequest\DB\Enterprise\Entity\ReReminderLog;
use Exception;
use Resrequest\Application\Workflow\Workflow;

define('REMINDER_SERVER' , dirname(dirname(dirname(dirname(dirname(dirname(__FILE__))))))); //Go back 5 directories
define('REMINDER_ROOT', REMINDER_SERVER. '/Application/src/Resrequest/legacy');

require_once(REMINDER_ROOT . '/functions.php');
require_once(REMINDER_ROOT . '/functions.mail.php');
require_once(REMINDER_ROOT . '/class.audit.php');
require_once(REMINDER_ROOT . '/functions.templates.php');

class Reminder
{
  /**
   * Entity manager instance for communication with the database.
   *
   * @var EntityManager
   */
  private $entityManager;
  private $workflowService;

  private $host;
  private $enterpriseUrl;
  private $overrideCli = false;
  private $url;
  private $reReminder;
  private $prPersona;
  private $now;
  private $content;
  private $mailFrom;
  private $mailFromName;
  private $enterprise;
  private $config;

  /**
   * Create a new instance of the Reminders.
   *
   * @param EntityManager $entityManager
   */
  public function __construct($db, $host, $databaseName, $overrideCli = false)
  {
    $application = MaglLegacy::getInstance()->getApplication();
    $serviceManager = $application->getServiceManager();
    $this->workflowService = new Workflow($db, $host, $databaseName);

    $this->ReReminder = '\Resrequest\DB\Enterprise\Entity\ReReminder';
    $this->PrPersona = '\Resrequest\DB\Enterprise\Entity\PrPersona';
    $this->now = strtotime(date("Y-m-d H:i:s"));
    $this->entityManager = $serviceManager->get('EnterpriseEntityManager');
    $this->db = $this->entityManager->getConnection();
    $this->dbService = $db;
    $this->dbService->setEnterpriseConnection($databaseName);
    $this->enterprise = $serviceManager->get('Resrequest\Setup\Service\Enterprise');
    $this->config = $this->enterprise->setup();
    $this->host = $host;
    $this->enterpriseUrl = $host . '/reservation.php?';
    $this->mailFrom = "noreply@resrequest.com";
    $this->mailFromName = "ResRequest Reminders";
    $this->lastIdSql = "SELECT @last_insert_id;";
    $this->content = [];

    $this->overrideCli = $overrideCli;
  }

  public function fetch($reReminderIx)
  {
    $reReminder = $this->db->fetchAssoc("
      SELECT 
            re_reminder.re_reminder_ix,
            re_reminder.re_reminder_note,
            re_reminder.re_reminder_send_date,
            re_reminder.re_reminder_repeat_yn,
            re_reminder.re_reminder_link_to_ind_id,
            re_reminder.re_reminder_link_to_id
        FROM  
            re_reminder
        WHERE 
            re_reminder.re_reminder_ix = ?
    ",[$reReminderIx]);
    return $reReminder;
  }

  public function fetchRecipients($reReminderIx)
  {
    $recipients = $this->db->fetchAll("
        SELECT 
          re_reminder_persona.re_reminder_id,
          pr_persona.pr_persona_ix,
          pr_persona.pr_name_first,
          pr_persona.pr_name_last,
          pr_persona.pr_email
        FROM  
          re_reminder_persona
          INNER JOIN pr_persona ON pr_persona.pr_persona_ix = re_reminder_persona.pr_persona_id
        WHERE
          re_reminder_persona.re_reminder_id = ?
    ", [$reReminderIx]);
    return $recipients;
  }

  public function create(
    $reNote,
    $reRepeatYn,
    $reReminderStatusId,
    $reReminderSendDate,
    $reReminderSendDateTz,
    $reReferToInd,
    $reReferToId,
    $prUserId,
    array $rePersonaIds,
    $awWorkflowId = null
  )
  {
    $createdTz = date("e");
    $reReminder = new ReReminder();
    $reReminder->setReReminderNote($reNote)
       ->setReReminderRepeatYn($reRepeatYn)
       ->setReReminderStatusId($reReminderStatusId)
       ->setReReminderSendDate($reReminderSendDate)
       ->setReReminderSendDateTz($reReminderSendDateTz)
       ->setReReminderLinkToIndId($reReferToInd)
       ->setReReminderLinkToId($reReferToId)
       ->setAwWorkflowId($awWorkflowId)
       ->setAdCreateUserId($prUserId)
       ->setAdCreateDateTz($createdTz)
       ->setAdModifyDateTz($createdTz);
    $this->entityManager->persist($reReminder);
    $this->entityManager->flush();
    $this->entityManager->clear(); // inside loops Doctrine sometimes thinks some entities have already been persisted
    $reReminderIx = $this->db->fetchAssoc($this->lastIdSql)['@last_insert_id'];

    $entityType = null;
    if($reReferToInd == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION) {
      $entityType = \AuditTrail::TYPE_RESERVATION;
    } elseif($reReferToInd == ReReminder::RE_REMINDER_LINK_TO_PR_PERSONA) {
      $entityType = \AuditTrail::TYPE_PERSONA;
    }
    if (!empty($entityType)) {
      global $lDB;
      $audit = new \AuditTrail($reReferToId, $entityType, [] ,$lDB);

      if (!empty($awWorkflowId)) {
        $workflowTypeId = $this->entityManager->find(AwWorkflow::class, $awWorkflowId)->getAwWorkflowTypeInd();
        $workflowType = AwWorkflow::getWorkflowTypeDescription($workflowTypeId);
        $audit->save("Reminder (Add) - Automated Workflows ({$workflowType})");
      } else {
        $audit->save("Reminder (Add)");
      }
    }

    if(!isset($reReminderIx) || $reReminderIx === "") {
      return $this->response(['status' => 'error', 'message' => 'An internal server error occurred']);
    }

    foreach ($rePersonaIds as $rePersonaId) {
      if ($objPrPersona = $this->entityManager->find($this->PrPersona, $rePersonaId)) {
        $reReminderPersona = new ReReminderPersona();
        $reReminderPersona->setReReminderSentToEmail($objPrPersona->getPrEmail());
        $reReminderPersona->setReReminderId($reReminderIx);
        $reReminderPersona->setPrPersonaId($rePersonaId);
        $this->entityManager->persist($reReminderPersona);
      }
    }
    $this->entityManager->flush();
    $reminder = $this->fetch($reReminderIx);
    return $this->response(['status' => 'success', 'message' => 'Reminder(s) created', 'data' => $reminder]);
  }

  private function consolidate($reminders) {
    // Determine reminders to consolidate.
    $consolidatedReminders = [];
    foreach ($reminders as $i => $reminder) {
      switch ($reminder['aw_workflow_type_ind']) {
        case AwWorkflow::AW_WORKFLOW_TYPE_PROV_EXPIRY_REMINDER:
          $consolidateYn = json_decode($reminder['aw_workflow_config'], true)['consolidateEmailYn'];
          if (
            !empty($reminder['re_reminder_consolidate_id'])
            || $reminder['re_reminder_link_to_ind_id'] == ReReminder::RE_REMINDER_LINK_TO_REMINDERS
            || empty($consolidateYn)
          ) {
            // Skip reminders that have already been consolidated.
            continue;
          }

          $sendDate = (new \DateTime($reminder['re_reminder_original_send_date']))->format('Y-m-d');
          $groupBy = $reminder['rv_corr_persona_id'] . '|' . $sendDate;

          if (!isset($consolidatedReminders[$groupBy])) {
            $consolidatedReminders[$groupBy] = [];
          }

          $consolidatedReminders[$groupBy][] = $i;
          break;
        default:
        break;
      }
    }

    // Create consolidated reminders
    $remindersToFetch = [];
    foreach (array_values($consolidatedReminders) as $reminderIndexes) {
      if (count($reminder) < 2) {
        continue;
      }
      $message = json_decode($reminder['aw_workflow_config'], true)['email']['message'];

      $result = $this->create(
        $message,
        ReReminder::RE_REMINDER_REPEAT_NO,
        ReReminder::RE_REMINDER_STATUS_ACTIVE,
        $reminder['re_reminder_send_date'],
        date("e"),
        ReReminder::RE_REMINDER_LINK_TO_REMINDERS,
        '',
        'RS4',
        [$reminder['rv_corr_persona_id']],
        $reminder['aw_workflow_ix']
      );

      if (!is_array($result)) {
        $result = json_decode($result);
      }

      if ($result['status'] == 'success') {
        // Assign ID to consolidated reminders.
        $remindersToFetch[] = $reminderIx = $result['data']['re_reminder_ix'];
        $consolidatedReminders = [];
        foreach ($reminderIndexes as $i) {
          $consolidatedReminders[] = $reminders[$i]['re_reminder_ix'];
          $remindersToFetch[] = $reminders[$i]['re_reminder_ix'];
          // Clear modified reminders.
          unset($reminders[$i]);
        }

        $result = $this->db->executeQuery("
            UPDATE
                re_reminder
            SET
                re_reminder.re_reminder_consolidate_id = ?
            WHERE
                re_reminder.re_reminder_ix IN(?)
        ",[
            $reminderIx,
            $consolidatedReminders,
        ],[
          \PDO::PARAM_STR,
          \Doctrine\DBAL\Connection::PARAM_INT_ARRAY
        ]);
      }
    }

    // Fetch consolidated and modified reminders.
    if (!empty($remindersToFetch)) {
      $updatedReminders = $this->reminderList(ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION);
      $reminders = array_merge($reminders, $updatedReminders);
    }

    $output = [];
    foreach ($reminders as $i => $reminder) {
      if ($reminder['re_reminder_link_to_ind_id'] == ReReminder::RE_REMINDER_LINK_TO_REMINDERS) {
        $tmpLinkedReminders = array_keys(array_column($reminders,'re_reminder_consolidate_id'), $reminder['re_reminder_ix']);
        $linkedReminders = [];
        foreach ($tmpLinkedReminders as $linkedReminder) {
          $linkedReminders[] = $reminders[$linkedReminder];
        }

        $reminder['linkedReminders'] = $linkedReminders;
        $output[] = $reminder;
      } else if (empty($reminder['re_reminder_consolidate_id'])) {
        // Filter reminders that were not consolidated
        $output[] = $reminder;
      }
    }

    return $output;
  }

  function processPayDueReminder($reminder, $recipient) {
    $reservation = $this->db->fetchAssoc("
      SELECT 
        *,
        (
          rv_amt_accomm_gross + rv_amt_extra_gross + rv_amt_travel_gross
        ) AS grossTotal,
        (
          rv_amt_accomm_nett + rv_amt_extra_nett + rv_amt_travel_nett
        ) AS nettTotal
      FROM rv_reservation where rv_reservation_ix = '{$reminder['rv_reservation_ix']}'
    ");
    $workflow = $this->workflowService->workflowList($reminder['aw_workflow_ix'])[0];
    list($select, $where, $dates) = $this->workflowService->getWorkflowReservationSelectWhere($workflow, '2');

    $processedReservations = $this->workflowService->processPayDueReservations([$reservation], $dates);

    if (!empty($processedReservations)) {
      $reservation = $processedReservations[0];
    }

    $reminder['payPlanDueTotal'] = $reservation['payPlanDueTotal'] ?? 0;

    $whereSql = '';

    if (!empty($reservation) && isset($reservation['payPlanDueNow']) && isset($reservation['onsiteExtrasDueNow'])) {
      if ($reservation['payPlanDueNow'] && !$reservation['onsiteExtrasDueNow']) {
        // Pay plan due without onsite extras
        $whereSql = "AND rf_database.rf_db_env_type_ind != '4'";

        if ($recipient['pr_persona_ix'] != $reminder['rv_billing_persona_id']) {
          $whereSql .= " AND fn_folio.fn_folio_to_id = '{$recipient['pr_persona_ix']}'";
        }
      } else if (!$reservation['payPlanDueNow'] && $reservation['onsiteExtrasDueNow']) {
        // Onsite extras due without payment plan
        $whereSql = "
          AND fn_folio.fn_folio_to_id = '{$recipient['pr_persona_ix']}'
          AND rf_database.rf_db_env_type_ind = '4'";
      } else if ($reservation['payPlanDueNow'] && $reservation['onsiteExtrasDueNow']) {
        // Pay plan and onsite extras due
        $whereSql = "
          AND fn_folio.fn_folio_to_id = '{$recipient['pr_persona_ix']}'";
      }
    }

    $folios = $this->db->executeQuery("
      SELECT
        fn_folio.fn_folio_ix as id,
        rf_database.rf_db_env_type_ind = '4' as onsiteFolio,
        fn_folio.fn_folio_amount as folioAmount
      FROM
        fn_folio
        INNER JOIN rf_database ON rf_database.rf_db_code = fn_folio.fn_folio_db
      WHERE
        fn_folio.rv_reservation_id = '{$reminder['rv_reservation_ix']}'
        {$whereSql}
        AND (
          fn_folio.fn_invoice_id = ''
          OR fn_folio.fn_invoice_id = '0'
          OR fn_folio.fn_invoice_id IS NULL
        )
        AND fn_folio.fn_folio_amount > 0
        AND fn_folio_amt_paid < fn_folio.fn_folio_amount
      ORDER BY
        fn_folio_folio_num")->fetchAll();
        

    $invoices = $this->db->executeQuery("
      SELECT
        fn_invoice_ix as id,
        rf_database.rf_db_env_type_ind = '4' as onsiteFolio
      FROM
        fn_invoice
        INNER JOIN fn_folio ON fn_folio.fn_invoice_id = fn_invoice.fn_invoice_ix
        INNER JOIN rf_database ON rf_database.rf_db_code = fn_folio.fn_folio_db
      WHERE
        fn_invoice.rv_reservation_id = '{$reminder['rv_reservation_ix']}'
        AND fn_invoice.fn_inv_status_ind != 8
        {$whereSql}
        AND fn_folio.fn_folio_amount > 0
        AND fn_folio_amt_paid < fn_folio.fn_folio_amount
      ORDER BY
        fn_invoice.fn_inv_status_ind,
        fn_invoice.fn_inv_void_yn")->fetchAll();


    if (!isset($reminder['folios'])) {
      $reminder['folios'] = [];
    }
    if (!isset($reminder['invoices'])) {
      $reminder['invoices'] = [];
    }

    $reminder['invoices'][$recipient['pr_persona_ix']] = $invoices;
    $reminder['folios'][$recipient['pr_persona_ix']] = $folios;

    return $reminder;
  }

  public function send()
  {
    global $lDB;

    $rvReservationReminders = $this->reminderList(ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION);
    $remindersSentYesterday = $this->remindersSentYesterday();
    $rvReservationReminders = $this->consolidate($rvReservationReminders);

    if (isset($GLOBALS['principal_name'])) {
      $principalName = $GLOBALS['principal_name'];
    } else {
      $principal = $lDB->get("SELECT rf_principal_id, rf_res_office_id FROM rf_default WHERE rf_default_id = '1'", 1);
      $principalName = $lDB->get("SELECT pr_name_last FROM pr_persona WHERE pr_persona_ix = '" . $principal['rf_principal_id'] . "'", 4);
    }

    foreach ($rvReservationReminders as $item){
      $reservationUrl = $this->config['enterprise']['http'].$this->enterpriseUrl."2+".$item['rv_reservation_ix'];

      $rvRecipients = $this->fetchRecipients($item['re_reminder_ix']);

      $item['late_message_status'] = 'none';
      $reminderSendDate = new \DateTime($item['re_reminder_send_date'], new \DateTimeZone($item['re_reminder_send_date_tz']));
      $reminderSendDate->setTimezone(new \DateTimeZone(date("e")));
      if($this->now - $reminderSendDate->format("U") > 1800){
        if (!$item['re_reminder_repeat_yn']) {
            $item['late_message_status'] = '';
        } else {
          if (!$this->hasReminder($item['re_reminder_ix'], $remindersSentYesterday)) {
            // Repeat reminders are late if they weren't sent out yesterday otherwise
            // we assume that they are on time.
            $item['late_message_status'] = '';
          }
        }
      }

      $item['re_reminder_note'] = nl2br($item['re_reminder_note']);
      $item['system_reservation_url'] = $reservationUrl;
      $item['original_send_date'] = date("d M Y", strtotime($item['re_send_date']));
      $item['original_send_time'] = date("h:i A T", strtotime($item['re_reminder_send_date']));
      $item['rv_date_arrive'] = date("d M Y", strtotime($item['rv_date_arrive']));
      $item['rv_date_depart'] = date("d M Y", strtotime($item['rv_date_depart']));

      if ($item['re_reminder_repeat_yn']) {
        $item['scheduled_send_date'] = date('d M Y', strtotime('yesterday'));
      } else {
        $item['scheduled_send_date'] = $item['original_send_date'];
      }

      $item['principal_id'] = $this->config['enterprise']['principalId'];
      $item['principal_name'] = $principalName;
      $item['url'] = $this->config['enterprise']['http'] . $this->host;
      $item['online_checkin_url'] = "/guest/check-in"; // will be appended to base url
      $item['online_images_dir'] = $this->config['enterprise']['http'] . $this->config['enterprise']['images']['online_images_dir'];
      $item['top_client_logo'] = "top_clientlogo.jpg";
      $item['guest_app_img'] = "guest-app_email.jpg";
      $sites = $this->sites($item['rv_reservation_ix']);
      $sitesHTML = [];
      if ($sites['website']) {
        $sitesHTML[] = "<a href='{$sites['website']}' target='_blank' style='text-decoration: none; color: #666666; text-transform: uppercase;'>Visit our website</a>";
      }
      foreach($sites['socials'] as $social) {
        $sitesHTML[] = "<a href='{$social['pr_bus_social_data']}' target='_blank' style='text-decoration: none; color: #666666; text-transform: uppercase;'>{$social['pr_social_label']}</a>";
      }
      $item['sites'] = join("&nbsp;|&nbsp;", $sitesHTML);


      foreach ($rvRecipients as $rvRecipient) {
        if (empty($rvRecipient['pr_name_first'])) {
          $item['guest_name'] = $rvRecipient['pr_name_last'];
        } else {
          $item['guest_name'] = $rvRecipient['pr_name_first'];
        }

        if ($item['aw_workflow_type_ind'] == AwWorkflow::AW_WORKFLOW_TYPE_PAYMENTS_DUE) {
          $item = $this->processPayDueReminder($item, $rvRecipient);

          // Change sent from address for payment due reminders
          if (!empty($item['rv_consultant_id'])) {
            if ($objPrPersona = $this->entityManager->find($this->PrPersona, $item['rv_consultant_id'])) {
              if (!empty($objPrPersona->getPrEmail())) {
                $item['fromEmail'] = $objPrPersona->getPrEmail();
              }
            }
          }
        }

        if (empty($rvRecipient['pr_email'])) {
          continue;
        }
        
        // Change sent from address for prov expiry reminders
        if ($item['aw_workflow_type_ind'] == AwWorkflow::AW_WORKFLOW_TYPE_PROV_EXPIRY_REMINDER) {
          if (empty($item['rv_consultant_id'])) {
            $rf_res_office_id = $this->db->fetchAssoc("select rf_res_office_id from rf_default where rf_default_id = '1'")['rf_res_office_id'];
            $resOfficeContact = $this->db->fetchAssoc("
              SELECT
                pr_email
              FROM
                pr_persona
              WHERE pr_persona_ix = ?
            ", [$rf_res_office_id]);
            $item['fromEmail'] = $resOfficeContact['pr_email'];
          } else {
            if ($objPrPersona = $this->entityManager->find($this->PrPersona, $item['rv_consultant_id'])) {
              if (!empty($objPrPersona->getPrEmail())) {
                $item['fromEmail'] = $objPrPersona->getPrEmail();
              }
            }
          }
        }

        $this->content [] = [
          'rv_reservation_ix' => $item['rv_reservation_ix'],
          're_reminder_ix' => $item['re_reminder_ix'],
          'to' =>  $rvRecipient['pr_email'],
          'toId' => $rvRecipient['pr_persona_ix'],
          'subject' => $this->getReminderEmailSubject($item['rv_reservation_ix'], $item['re_reminder_link_to_ind_id'], $item['aw_workflow_type_ind'], $item, $rvRecipient),
          'message' => $this->mailerTemplate($item, $item['re_reminder_link_to_ind_id'], $item['aw_workflow_type_ind'], $rvRecipient),
          'item' => $item,
        ];
      }
    }

    $prPersonaReminders = $this->reminderList(ReReminder::RE_REMINDER_LINK_TO_PR_PERSONA);

    foreach ($prPersonaReminders as $item) {
      $personaUrl = $this->config['enterprise']['http'].$this->enterpriseUrl."45+".$item['pr_persona_ix'];

      $prRecipients = $prRecipientsArr = $this->db->fetchAll("
        SELECT 
          re_reminder_persona.re_reminder_id,
          pr_persona.pr_persona_ix,
          pr_persona.pr_name_first,
          pr_persona.pr_name_last,
          pr_persona.pr_email
        FROM  
          re_reminder_persona
          INNER JOIN pr_persona ON pr_persona.pr_persona_ix = re_reminder_persona.pr_persona_id
        WHERE
          re_reminder_persona.re_reminder_id = ?",
        [$item['re_reminder_ix']]
      );

      $item['late_message_status'] = 'none';
      if($this->now - strtotime($item['re_reminder_send_date']) > 1800){
        if (!$item['re_reminder_repeat_yn']) {
          $item['late_message_status'] = '';
        } else {
          if (!$this->hasReminder($item['re_reminder_ix'], $remindersSentYesterday)) {
            // Repeat reminders are late if they weren't sent out yesterday otherwise
            // we assume that they are on time.
            $item['late_message_status'] = '';
          }
        }
      }

      $item['system_reservation_url'] = $personaUrl;
      $item['original_send_date'] = date("d M Y", strtotime($item['re_send_date']));
      $item['original_send_time'] = date("h:i A T", strtotime($item['re_reminder_send_date']));

      if ($item['re_reminder_repeat_yn']) {
        $item['scheduled_send_date'] = date('d M Y', strtotime('yesterday'));
      } else {
        $item['scheduled_send_date'] = $item['original_send_date'];
      }
      $item['re_reminder_note'] = nl2br($item['re_reminder_note']);
      $item['principal_name'] = $principalName;
      $item['url'] = $this->config['enterprise']['http'] . $this->host;
      $item['host'] = $this->host;

      foreach ($prRecipients as $prRecipient) {
        $this->content [] = [
          'rv_reservation_ix' => "",
          're_reminder_ix' => $item['re_reminder_ix'],
          'to' =>  $prRecipient['pr_email'],
          'subject' => $this->getReminderEmailSubject($item['pr_persona_ix'], $item['re_reminder_link_to_ind_id'], null, $item),
          'message' => $this->mailerTemplate($item, $item['re_reminder_link_to_ind_id'], $item['aw_workflow_type_ind']),
          'item' => $item
        ];
      }
    }

    $this->dbService->masterOverride(true);
    $return_data = [];

    //send out emails
    foreach ($this->content as $reminderItem) {
      // Custom mail from
      $mailFrom = $this->mailFrom;
      if (!empty($reminderItem['item']['fromEmail'])) {
          $mailFrom = $reminderItem['item']['fromEmail'];
      }
      
      $stringAttachment = [];

      if (
        $reminderItem['item']['aw_workflow_type_ind'] == AwWorkflow::AW_WORKFLOW_TYPE_PAYMENTS_DUE
        && !empty(json_decode($reminderItem['item']['aw_workflow_config'], true)['attachFoliosYn'])
      ) {
          global $lDB, $userStatusId, $action;
          $dbcode = $GLOBALS['dbcode'];
          $GLOBALS['sc_group_id'] = 3;
          require_once(__DIR__ . '/../../legacy/PDF/pdf.php');
          require_once(__DIR__ . '/../../legacy/functions.system.php');
          require_once(__DIR__ . '/../../legacy/init.form215.func.php');
          

          $invoices = $reminderItem['item']['invoices'][$reminderItem['toId']];
          $folios = $reminderItem['item']['folios'][$reminderItem['toId']];


          $folioTotal = array_sum(array_column($folios, 'folioAmount'));

          if (!empty($folios)) {
            $customNotes = $this->db->fetchAll("
                SELECT
                  rf_note_custom.rf_note_custom_ix as custNoteId,
                  rf_note_custom.rf_note_custom_desc as custNoteDesc,
                  rf_note_custom.rf_note_custom_hdg as custNoteHdg
                FROM
                  rf_note_custom
                ORDER BY
                  rf_note_custom.rf_note_custom_desc
              ");
            $customIds = array();
            foreach ($customNotes as $custom) {
              $def = $this->db->fetchAll("
                  SELECT
                    rf_note_cust_def_ext_inv_yn,
                    rf_note_cust_def_ext_summ_yn,
                    rf_note_cust_def_ext_pro_yn,
                    rf_note_cust_def_int_inv_yn,
                    rf_note_cust_def_int_summ_yn,
                    rf_note_cust_def_int_pro_yn
                  FROM
                    rf_note_cust_default
                  WHERE
                    rf_note_cust_default.rf_note_custom_id = '$custom[custNoteId]'
                    AND rf_note_cust_default.rf_db_code = ''
                ")[0];

              if ($def['rf_note_cust_def_int_pro_yn'] == 1) {
                array_push($customIds, $custom['custNoteId']);
              }
            }

            $rfDefault = $this->db->fetchAll("
                SELECT
                  rf_doc_tax_table_yn,
                  rf_doc_tax_table_rate_yn,
                  rf_doc_tax_table_total_yn
                FROM
                  rf_default
              ")[0];

            $prBus = $this->db->fetchAll("
                SELECT
                  pr_bus_opt_special,
                  pr_bus_opt_special_note
                FROM
                  pr_business
              ")[0];

            foreach ($folios as $i => $folio) {
              $displayOptions = 549978112;

              if ($folioTotal >= $reminderItem['item']['payPlanDueTotal']) {
                if ($folio['onsiteFolio'] == "0") {
                  $displayOptions += 1;
                }
              }

              $businessArray = $this->db->fetchAll("
                SELECT DISTINCT
                  pr_business_id
                FROM
                  fn_folio
                WHERE
                  fn_folio.fn_folio_ix = '{$folio['id']}'
              ");
              $currencyId = array_column($this->db->fetchAll("SELECT fn_folio.rf_currency_id FROM fn_folio WHERE fn_folio_ix = '{$folio['id']}'"), 'rf_currency_id')[0];

              $businessArray = array_column($businessArray, 'pr_business_id');

              $busBankSql = $this->db->fetchAll("
                SELECT DISTINCT
                  rf_bank.rf_bank_ix as busBankId
                FROM
                  pr_bus_bank
                  INNER JOIN rf_bank ON rf_bank.rf_bank_ix = pr_bus_bank.rf_bank_id
                  INNER JOIN rf_currency ON rf_currency.rf_currency_ix = rf_bank.rf_currency_id
                WHERE
                  pr_bus_bank.pr_business_id IN ('".join("','",$businessArray)."')
                  AND rf_bank.rf_bank_ind = '0'
                  AND rf_bank.rf_bank_sys_code = '0'
                  AND rf_bank.rf_currency_id = '$currencyId'
              ");

              $linkedBusinessBanks = array_column($busBankSql, 'busBankId');

              if (!empty($linkedBusinessBanks)) {
                // Display bank list
                $displayOptions += 4;
              }

              $_SERVER['argv'] = array(
                /* 00 */ "reservation.php?642",
                /* 01 */ $reminderItem['item']['rv_reservation_ix'], // Reservation Ids
                /* 02 */ "1", // Options (view PDF)
                /* 03 */ $folio['id'], // Folio Ids
                /* 04 */ '',
                /* 05 */ "",
                /* 06 */ $displayOptions,
                /* 07 */ "",
                /* 08 */ "",
                /* 09 */ "",
                /* 10 */ join(',', $linkedBusinessBanks),
                /* 11 */ join(',', $customIds),
                /* 12 */ "",
                /* 13 */ "",
                /* 14 */ "1", // Use invoice defaults
                /* 15 */ "",
                /* 16 */ "",
                /* 17 */ "",
                /* 18 */ "",
                /* 19 */ "",
                /* 20 */ "",
                /* 21 */ "",
                /* 22 */ $rfDefault['rf_doc_tax_table_yn'],
                /* 23 */ "1",
                /* 24 */ $rfDefault['rf_doc_tax_table_rate_yn'],
                /* 25 */ $rfDefault['rf_doc_tax_table_total_yn'],
                /* 26 */ $prBus['pr_bus_opt_special'],
                /* 27 */ $prBus['pr_bus_opt_special_note'],
              );
  
              $_SERVER['argc'] = sizeof($_SERVER['argv']);
              $GLOBALS['pdfEcho'] = true;
              ob_start();
              $form = 215;
              include(__DIR__ ."/../../legacy/init.form215.php");
              $pdf = ob_get_contents();
              ob_end_clean();
              $i++;
              $stringAttachment[] = [ 'data' => $pdf, 'filename' => "folio {$i}.pdf"];
            }
          }

          foreach ($invoices as $i => $invoice) {
            $_SERVER['argv'] = array(
              /* 00 */ "reservation.php?642",
              /* 01 */ $reminderItem['item']['rv_reservation_ix'], // Reservation Ids
              /* 02 */ "1", // Options (view PDF)
              /* 03 */ '', // Folio Ids
              /* 04 */ $invoice['id'],
              /* 05 */ "",
              /* 06 */ "",
              /* 07 */ "",
              /* 08 */ "",
              /* 09 */ "",
              /* 10 */ "",
              /* 11 */ "",
              /* 12 */ "",
              /* 13 */ "",
              /* 14 */ "1" // Use invoice defaults
            );
            $_SERVER['argc'] = sizeof($_SERVER['argv']);
            $GLOBALS['pdfEcho'] = true;
            ob_start();
            $form = 215;
            include(__DIR__ ."/../../legacy/init.form215.php");
            $pdf = ob_get_contents();
            ob_end_clean();
            $i++;
            $stringAttachment[] = [ 'data' => $pdf, 'filename' => "invoice {$i}.pdf"];
          }
      }

      $item = $reminderItem;

      if (empty($stringAttachment)) {
        $stringAttachment = '';
      }

      $showAttachmentMessage = true;
      if (!empty($reminderItem['item']['aw_workflow_type_ind'])) {
        $showAttachmentMessage = false;
      }

      $data = email(
        $mailFrom,
        $item['to'],
        $item['subject'], 
        "", //headers
        $item['message'],
        0, //action
        0, //email id
        $item['rv_reservation_ix'],
        $this->mailFromName,
        $stringAttachment, //attach string
        '', //attach string file name
        false, //pf_object_id
        false, //attachments
        50, // wordwrap
        true, // isHtml
        [],
        $showAttachmentMessage
      );

      $code = (!$data) ? ReReminder::RE_REMINDER_STATUS_SENT : ReReminder::RE_REMINDER_STATUS_FAILED;

      $ReReminderLog = new ReReminderLog();
      $ReReminderLog->setReReminderId($item['re_reminder_ix'])->setReReminderLogStatus($code);
      $this->entityManager->persist($ReReminderLog);

      if ($item['item']['re_reminder_link_to_ind_id'] == ReReminder::RE_REMINDER_LINK_TO_REMINDERS && false) {
        // Set status for consolidated reminders
        $consolidatedIds = array_column($item['item']['linkedReminders'], 're_reminder_ix');
        $result = $this->db->executeQuery("
            UPDATE
                re_reminder
            SET
                re_reminder.re_reminder_status_id = ?
            WHERE
                re_reminder.re_reminder_ix IN(?)
        ",[
            $code,
            $consolidatedIds,
        ],[
          \PDO::PARAM_STR,
          \Doctrine\DBAL\Connection::PARAM_INT_ARRAY
        ]);
      }

      $reReminder = $this->entityManager->find($this->ReReminder, $item['re_reminder_ix']);
      $isRepeatReminder = $reReminder->getReReminderRepeatYn();

      if ($isRepeatReminder) {
        if (
          $reReminder->getReReminderStatusId() == ReReminder::RE_REMINDER_STATUS_FAILED
          && $code == ReReminder::RE_REMINDER_STATUS_SENT
        ) {
          // If the reminder had previously failed and now succeeded we revert it's status back to
          // active (not sent eitherwise they will not continue to be sent).
          $reReminder->setReReminderStatusId(ReReminder::RE_REMINDER_STATUS_ACTIVE);
        } elseif ($code != ReReminder::RE_REMINDER_STATUS_SENT) {
          $reReminder->setReReminderStatusId($code);
        }
      } else {
        $reReminder->setReReminderStatusId($code);
      }

      // Resort to using old-school $lDB for this update, due to issues overriding master on a slave environment
      $this->dbService->masterOverride(true);
      $lDB->put("
        UPDATE
          re_reminder
        SET
          re_reminder_status_id = " . $reReminder->getReReminderStatusId() . "
        WHERE
          re_reminder_ix = '" . $reReminder->getReReminderIx() . "'
      ");

      $lDB->put("
        INSERT INTO re_reminder_log (
          re_reminder_log_ix,
          re_reminder_log_db,
          re_reminder_log_id,
          re_reminder_id,
          re_reminder_log_status
        ) VALUES (
          (SELECT GET_UUID()),
          'WB',
          '0',
          '" . $reReminder->getReReminderIx() . "',
          " . $code . "
        )
      ");
      $this->dbService->masterOverride(false);    

      $audit = new \AuditTrail($reReminder->getReReminderLinkToId(),$reReminder->getReReminderLinkToIndId(),[],$lDB);

      if (!empty($reReminder->getAwWorkflowId())) {
        $workflowTypeId = $this->entityManager->find(AwWorkflow::class, $reReminder->getAwWorkflowId())->getAwWorkflowTypeInd();
        $workflowType = AwWorkflow::getWorkflowTypeDescription($workflowTypeId);
        $audit->save("Reminder (Send) - Automated Workflows ({$workflowType})");
      } else {
        $audit->save("Reminder (Send)");
      }
      
      $return_data [] = [
        'status' => $code,
        'to' => $item['to'],
        'subject' => $item['subject'],
        'message' => $this->messageForCode($code)
      ];
    }

    $this->dbService->masterOverride(false);

    return $this->response(['data'  => $return_data, 'error' => '', 'code'  => 200]);
  }

  private function reminderList($reReminderLinkToIndId, $ids = [])
  {
    $select = "";
    $join = "";
    $where = "";
    $groupBy = "";
    switch ($reReminderLinkToIndId){
      case ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION:
        $select = "
          rv_reservation.rv_reservation_ix,
          rv_reservation.rv_create_expiry_date,
          rv_reservation.rv_date_arrive,
          rv_reservation.rv_date_depart,
          rv_reservation.rv_note_general,
          rv_reservation.rv_res_name,
          rv_reservation.rf_reservation_status_id,
          rv_reservation.rv_note_internal,
          rv_reservation.rv_note_guests,
          rv_reservation.rv_corr_persona_id,
          rv_reservation.rv_billing_persona_id,
          rv_reservation.rv_agent_id,
          rv_reservation.rv_consultant_id,
          SUM(rv_reservation_item.rv_item_adult_count + rv_reservation_item.rv_item_child_count) AS rv_pax_count";
        $join = "
          LEFT JOIN rv_reservation ON rv_reservation.rv_reservation_ix = re_reminder.re_reminder_link_to_id
          LEFT JOIN rv_reservation_item ON rv_reservation_item.rv_reservation_id = rv_reservation.rv_reservation_ix";
        $where = "(re_reminder.re_reminder_link_to_ind_id = ".ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION." OR re_reminder.re_reminder_link_to_ind_id = ".ReReminder::RE_REMINDER_LINK_TO_REMINDERS.") AND";
        $groupBy = "re_reminder.re_reminder_ix, rv_reservation.rv_reservation_ix";
      break;
      case ReReminder::RE_REMINDER_LINK_TO_PR_PERSONA:
        $select = "
          pr_persona.pr_persona_ix,
          pr_persona.pr_name_first,
          pr_persona.pr_name_last,
          pr_persona.pr_email AS pr_persona_email,
          CONCAT_WS(' ',pr_persona.pr_name_first, pr_persona.pr_name_last) AS pr_persona_name";
        $join = "INNER JOIN pr_persona ON pr_persona.pr_persona_ix = re_reminder.re_reminder_link_to_id";
        $where = "re_reminder.re_reminder_link_to_ind_id = ".ReReminder::RE_REMINDER_LINK_TO_PR_PERSONA." AND";
      break;
      default:
        return false;
    }

    if (!empty($ids)) {
      $where = $where . ' re_reminder.re_reminder_ix IN(' . join(',', $ids) . ') AND';
    }

    $groupBy = \strlen($groupBy) > 0 ? " GROUP BY ".$groupBy : "";

    $SQL = "
      SELECT
        re_reminder.re_reminder_ix,
        re_reminder.re_reminder_note,
        re_reminder_status.re_reminder_status_desc,
        re_reminder.re_reminder_repeat_yn,
        re_reminder.re_reminder_send_date,
        re_reminder.re_reminder_send_date_tz,
        re_reminder.re_reminder_send_date AS re_reminder_original_send_date,
        re_reminder.re_reminder_link_to_ind_id,
        re_reminder.re_reminder_consolidate_id,
        aw_workflow.aw_workflow_ix,
        aw_workflow.aw_workflow_type_ind,
        aw_workflow.aw_workflow_config,
        DATE(re_reminder.re_reminder_send_date) AS re_send_date,
        TIME(re_reminder.re_reminder_send_date) AS re_send_time,
        CONCAT_WS(' ',created_by.pr_name_first, created_by.pr_name_last) AS created_by_name,
        {$select}
      FROM
        re_reminder
        LEFT JOIN pr_persona AS created_by ON created_by.pr_persona_ix = re_reminder.ad_create_user_id
        LEFT JOIN re_reminder_status ON re_reminder_status.re_reminder_status_id = re_reminder.re_reminder_status_id
        LEFT JOIN aw_workflow ON aw_workflow.aw_workflow_ix = re_reminder.aw_workflow_id
        {$join}
      WHERE
        {$where}
        re_reminder.re_reminder_status_id IN ('" . ReReminder::RE_REMINDER_STATUS_ACTIVE . "', '" . ReReminder::RE_REMINDER_STATUS_FAILED . "') AND 
        (
          (
            re_reminder.re_reminder_repeat_yn = " . ReReminder::RE_REMINDER_REPEAT_YES . " AND
            TIME(CONVERT_TZ(re_reminder.re_reminder_send_date, re_reminder_send_date_tz, '" . date('e') . "')) <=  TIME(NOW()) AND
            DATE(CONVERT_TZ(re_reminder.re_reminder_send_date, re_reminder_send_date_tz, '" . date('e') . "')) <=  DATE(NOW()) AND
            re_reminder.re_reminder_ix NOT IN (
              SELECT
                re_reminder_id
              FROM
                re_reminder_log
              WHERE
                re_reminder_log_status = " . ReReminder::RE_REMINDER_STATUS_SENT . "
                AND DATE(re_reminder_log.re_reminder_log_date) = DATE(NOW())
            )
          ) 
          OR
          (
              re_reminder.re_reminder_repeat_yn = " . ReReminder::RE_REMINDER_REPEAT_NO . " AND
              re_reminder.re_reminder_ix NOT IN (
                SELECT
                  re_reminder_id
                FROM
                  re_reminder_log
                WHERE
                  re_reminder_log_status = " . ReReminder::RE_REMINDER_STATUS_SENT . "
              )
                  AND 
              (
                TIME(CONVERT_TZ(re_reminder.re_reminder_send_date, re_reminder_send_date_tz, '" . date('e') . "')) <=  TIME(NOW()
                AND DATE(CONVERT_TZ(re_reminder.re_reminder_send_date, re_reminder_send_date_tz, '" . date('e') . "')) BETWEEN DATE(NOW() - INTERVAL " . ReReminder::RE_REMINDER_INTERVAL . " DAY) AND DATE(NOW())
              ) OR CONVERT_TZ(re_reminder.re_reminder_send_date, re_reminder_send_date_tz, '" . date('e') . "') BETWEEN (NOW() - INTERVAL " . ReReminder::RE_REMINDER_INTERVAL . " DAY) AND NOW()
              )
          )
        )
        {$groupBy}
    ";
    return $this->db->fetchAll($SQL);
  }

  private function sites($rv_reservation_id)
  {
    $sites = [];
    $websiteSQL = "
      SELECT
        rf_default.rf_prn_web_site
      FROM
        rf_default";
    $website = $this->db->fetchAssoc($websiteSQL);
    $sites['website'] = $this->config['enterprise']['http'].$website['rf_prn_web_site'] ?? "";
    $socialsSQL = "
      SELECT
        pr_social.pr_social_name,
        pr_social.pr_social_label,
        pr_social.pr_social_sort_seq,
        pr_bus_social.pr_bus_social_data
      FROM
        pr_social
        INNER JOIN pr_bus_social ON pr_bus_social.pr_social_id = pr_social.pr_social_ix
        INNER JOIN rv_reservation_item ON rv_reservation_item.pr_business_id = pr_bus_social.pr_business_id
      WHERE
        rv_reservation_item.rv_reservation_id = '{$rv_reservation_id}'
      ORDER BY pr_social.pr_social_sort_seq";
    $socials = $this->db->fetchAll($socialsSQL);
    $sites['socials'] = $socials == false ? [] : $socials;
    return $sites;
  }

  private function remindersSentYesterday()
  {
    $sql = "
      SELECT
        re_reminder_id
      FROM
        re_reminder_log
      WHERE
        re_reminder_log_date BETWEEN SUBDATE(CURRENT_DATE(), 1) AND CURRENT_DATE()
    ";

    return $this->db->fetchAll($sql);
  }

  private function hasReminder($reminderId, $reminders)
  {
    foreach ($reminders as $reminder) {
      if ($reminderId == $reminder['re_reminder_id']) return true;
    }

    return false;
  }

  private function mailerTemplate($item, $type, $workflowTypeInd = null, $rvRecipient = null)
  {
    if ($type == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION
      && !is_null($workflowTypeInd) && ($workflowTypeInd == AwWorkflow::AW_WORKFLOW_TYPE_GUEST_CHECKIN_REMINDER)
    ) {
      $template = file_get_contents(REMINDER_SERVER."/Application/public/html/template_re_reminder_rv_reservation_guest_checkin.htm");
    } elseif ($type == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION
      && !is_null($workflowTypeInd) && ($workflowTypeInd == AwWorkflow::AW_WORKFLOW_TYPE_PROV_EXPIRY_REMINDER)
    ) {
      // Prov expiry reminders
      $subject = json_decode($item['aw_workflow_config'], true)['email']['subject'];
      \parseTemplate($subject, $item['rv_reservation_ix']);
      $item['subject'] = $subject;
      $template = file_get_contents(REMINDER_SERVER."/Application/public/html/template_re_reminder_rv_reservation_prov_expiry.htm");
      $message = json_decode($item['aw_workflow_config'], true)['email']['message'];
      \parseTemplate($message, $item['rv_reservation_ix']);

      $item['re_reminder_note'] = nl2br($message);
    } elseif ($type == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION
      && !is_null($workflowTypeInd) && ($workflowTypeInd == AwWorkflow::AW_WORKFLOW_TYPE_PAYMENTS_DUE)
    ) {
      // Payments due reminders
      $subject = json_decode($item['aw_workflow_config'], true)['email']['subject'];
      \parseTemplate($subject, $item['rv_reservation_ix'], array_column($item['folios'][$rvRecipient['pr_persona_ix']], 'id'), array_column($item['invoices'][$rvRecipient['pr_persona_ix']], 'id'));
      $item['subject'] = $subject;
      $template = file_get_contents(REMINDER_SERVER."/Application/public/html/template_re_reminder_rv_reservation_payments_due.htm");
      $message = json_decode($item['aw_workflow_config'], true)['email']['message'];
      \parseTemplate($message, $item['rv_reservation_ix'], array_column($item['folios'][$rvRecipient['pr_persona_ix']], 'id'), array_column($item['invoices'][$rvRecipient['pr_persona_ix']], 'id'));

      $item['re_reminder_note'] = nl2br($message);
    } elseif ($type == ReReminder::RE_REMINDER_LINK_TO_REMINDERS
      && !is_null($workflowTypeInd) && ($workflowTypeInd == AwWorkflow::AW_WORKFLOW_TYPE_PROV_EXPIRY_REMINDER)
    ) {
      // Consolidated prov expiry reminders
      $resIds = array_column($item['linkedReminders'], 'rv_reservation_ix');
      $template = file_get_contents(REMINDER_SERVER."/Application/public/html/template_re_reminder_rv_reservation_prov_expiry.htm");
      \parseConsolidatedTemplate($item['re_reminder_note'], $resIds);
    } elseif ($type == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION) {
      $template = file_get_contents(REMINDER_SERVER."/Application/public/html/template_re_reminder_rv_reservation.htm");
    } elseif($type == ReReminder::RE_REMINDER_LINK_TO_PR_PERSONA) {
      $template = file_get_contents(REMINDER_SERVER."/Application/public/html/template_re_reminder_pr_persona.htm");
    }
    else return false;

    foreach ($item as $key => $value){
      if (is_string($value) || is_numeric($value)) {
        $template = str_replace("!$key!", $value, $template);
      }
    }

    return $template;
  }

  private function response($data) {
    if ($this->isCli()) {
      return $data;
    } else {
      return new JsonModel($data);
    }
  }

  private function messageForCode($code)
  {
    switch ($code) {
    case 10:
      return 'Active';
    case 20:
      return 'Suspended';
    case 30:
      return 'Failed';
    case 40:
      return 'Sent';
    default:
      return $code;
    }
  }

  private function getReminderEmailSubject($entityId, $type, $workflowTypeInd = null, $item, $rvRecipient = null)
  {
    $subject = "ResRequest: Reminder {{$entityId}}";
    if (
      $type == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION &&
      !is_null($workflowTypeInd) &&
      $workflowTypeInd == AwWorkflow::AW_WORKFLOW_TYPE_GUEST_CHECKIN_REMINDER
    ) {
      $subject = "ResRequest: Check-In Reminder {{$entityId}}";
    } elseif (
      $type == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION &&
      !is_null($workflowTypeInd) &&
      $workflowTypeInd == AwWorkflow::AW_WORKFLOW_TYPE_PROV_EXPIRY_REMINDER
    ) {
      $subject = json_decode($item['aw_workflow_config'], true)['email']['subject'];
      \parseTemplate($subject, $item['rv_reservation_ix']);
    } elseif (
      $type == ReReminder::RE_REMINDER_LINK_TO_REMINDERS &&
      !is_null($workflowTypeInd) &&
      $workflowTypeInd == AwWorkflow::AW_WORKFLOW_TYPE_PROV_EXPIRY_REMINDER
    ) {
      $subject = json_decode($item['aw_workflow_config'], true)['email']['subject'];
      \parseTemplate($subject);
    } elseif (
      $type == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION &&
      !is_null($workflowTypeInd) &&
      $workflowTypeInd == AwWorkflow::AW_WORKFLOW_TYPE_PAYMENTS_DUE
    ) {
      $subject = json_decode($item['aw_workflow_config'], true)['email']['subject'];
      \parseTemplate($subject, $item['rv_reservation_ix'], array_column($item['folios'][$rvRecipient['pr_persona_ix']], 'id'), array_column($item['invoices'][$rvRecipient['pr_persona_ix']], 'id'));
    } elseif ($type == ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION) {
      $subject = "ResRequest: Reservation Reminder {" . $entityId . " - " . $item['rv_res_name'] . "}";
    } elseif ($type == ReReminder::RE_REMINDER_LINK_TO_PR_PERSONA) {
      $subject = "ResRequest: Contact Reminder {" . trim($item['pr_persona_name']) . "}";
    }
    return $subject;
  }

  private function isCli()
  {
    if(defined('STDIN')) {
      return true;
    }

    if(empty($_SERVER['REMOTE_ADDR']) && !isset($_SERVER['HTTP_USER_AGENT']) && count($_SERVER['argv']) > 0) {
      return true;
    } 

    if ($this->overrideCli == true) {
      return true;
    }

    return false;
  }
}
