<?php

namespace Resrequest\Application\Workflow;

use MaglLegacyApplication\Application\MaglLegacy;
use ZF\ContentNegotiation\JsonModel;
use Doctrine\ORM\EntityManager;
use Resrequest\DB\Enterprise\Entity\AwWorkflow;
use Resrequest\Application\Reminder\Reminder;
use Resrequest\DB\Enterprise\Entity\ReReminder;
use Resrequest\DB\Enterprise\Entity\ReReminderLog;
use Exception;

define('WORKFLOW_SERVER' , dirname(dirname(dirname(dirname(dirname(dirname(__FILE__))))))); //Go back 5 directories
define('WORKFLOW_ROOT', WORKFLOW_SERVER. '/Application/src/Resrequest/legacy');

require_once(WORKFLOW_ROOT . '/functions.php');
require_once(WORKFLOW_ROOT . '/functions.mail.php');
require_once(WORKFLOW_ROOT . '/class.audit.php');

class Workflow
{
  /**
   * Entity manager instance for communication with the database.
   *
   * @var EntityManager
   */
  private $entityManager;

  private $protocol;
  private $host;
  private $ReReminder;
  private $PrPersona;
  private $now;
  private $content;
  private $mailFrom;
  private $mailFromName;
  private $databaseName;
  private $enterprise;

  /**
   * Create a new instance of the Reminders.
   *
   * @param EntityManager $entityManager
   */
  public function __construct($db, $host, $databaseName)
  {
    $application = MaglLegacy::getInstance()->getApplication();
    $serviceManager = $application->getServiceManager();

    $this->protocol = 'https://';
    $this->host = $host."/reservation.php?";
    $this->ReReminder = '\Resrequest\DB\Enterprise\Entity\ReReminder';
    $this->PrPersona = '\Resrequest\DB\Enterprise\Entity\PrPersona';
    $this->now = date("Y-m-d H:i:s");
    $this->entityManager = $serviceManager->get('EnterpriseEntityManager');
    $this->db = $this->entityManager->getConnection();
    $this->dbService = $db;
    $this->databaseName = $databaseName;
    $this->mailFrom = "noreply@resrequest.com";
    $this->mailFromName = "ResRequest Reminders";
    $this->content = [];
    $this->enterprise = $serviceManager->get('Resrequest\Setup\Service\Enterprise');
  }

  public function run($awWorkflowIx = null, $awWorkflowTypeInd = null)
  {
    $createdTz = date("e");
    $reminders = [];
    $reminder = new Reminder($this->dbService, $this->host, $this->databaseName);

    if ($awWorkflowTypeInd === null || $awWorkflowTypeInd == '0') { // Guest check-in reminders
      $prUserId = $_SESSION['userid'] ?? $this->enterprise->systemUser['userid'];
      $workflows = $this->workflowList($awWorkflowIx, 0);
      $reservations = $this->reservationList($workflows);

      foreach ($reservations as $reservation) {
        $note = "Automated Workflow: \nName: {$reservation['aw_workflow_name']}\nType: " . AwWorkflow::getWorkflowTypeDescription(0);
        $reminders[] = $reminder->create(
          $note,
          ReReminder::RE_REMINDER_REPEAT_NO,
          ReReminder::RE_REMINDER_STATUS_ACTIVE,
          $reservation['reminder_send_date'],
          $createdTz,
          ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION,
          $reservation['rv_reservation_ix'],
          $prUserId,
          [$reservation['pr_guest_id']],
          $reservation['aw_workflow_ix']
        );
      }
    }
    
    if ($awWorkflowTypeInd === null || $awWorkflowTypeInd == '1') { // Provisional expiry reminders
      $provExpiryReminders = $this->createProvExpiryReminders($reminder, $awWorkflowIx);
      $reminders = array_merge($reminders, $provExpiryReminders);
    }

    if ($awWorkflowTypeInd === null || $awWorkflowTypeInd == '2') { // Provisional expiry reminders
      $paymentsDueReminders = $this->createPaymentsDueReminders($reminder, $awWorkflowIx);
      $reminders = array_merge($reminders, $paymentsDueReminders);
    }

    return $this->response(['data'  => $reminders, 'error' => '', 'code'  => 200]);
  }

  private function createProvExpiryReminders($reminder, $awWorkflowIx = null) {
    $reminders = [];
    $createdTz = date("e");

    $workflows = $this->workflowList($awWorkflowIx, 1);
    $reservations = $this->reservationList($workflows);

    foreach ($reservations as $reservation) {
      $this->db->insert('ad_reservation', [
        'ad_res_time' => date("Y-m-d H:i:s"),
        'ad_res_time_tz' => date("e"),
        'ad_res_form' => 'Sent Prov Notification',
        'rv_reservation_id' => $reservation['rv_reservation_ix'],
        'pr_user_id' => 'RS1',
        'rf_database_id' => $GLOBALS['environment_id']
      ]);
      $note = "Automated Workflow: \nName: {$reservation['aw_workflow_name']}\nType: " . AwWorkflow::getWorkflowTypeDescription(1);
      $reminders[] = $reminder->create(
        $note,
        ReReminder::RE_REMINDER_REPEAT_NO,
        ReReminder::RE_REMINDER_STATUS_ACTIVE,
        $reservation['reminder_send_date'],
        $createdTz,
        ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION,
        $reservation['rv_reservation_ix'],
        'RS4',
        [$reservation['rv_corr_persona_id']],
        $reservation['aw_workflow_ix']
      );
    }

    return $reminders;
  }

  private function createPaymentsDueReminders($reminder, $awWorkflowIx = null) {
    $reminders = [];
    $createdTz = date("e");

    $workflows = $this->workflowList($awWorkflowIx, 2);
    $reservations = $this->reservationList($workflows);

    foreach ($reservations as $reservation) {
      $recipients = [];

      $sendTo = '';
      if ($reservation['payPlanDueNow']) {
        $sendTo = $reservation['rv_billing_persona_id'];
        $recipients[] = $sendTo;
      }

      if ($reservation['onsiteExtrasDueNow']) {
        $rawRecipients = $this->db->fetchAll("
          SELECT DISTINCT
            fn_folio.fn_folio_to_id
          FROM
            fn_folio
            INNER JOIN rv_reservation ON rv_reservation.rv_reservation_ix = fn_folio.rv_reservation_id
            LEFT JOIN pr_persona ON pr_persona.pr_persona_ix = fn_folio.fn_folio_to_id
            LEFT JOIN rf_title ON rf_title.rf_title_ix = pr_persona.pr_title_id
            LEFT JOIN rf_name_suffix ON rf_name_suffix.rf_name_suffix_ix = pr_persona.pr_name_suffix_id
            INNER JOIN rf_currency ON rf_currency.rf_currency_ix = fn_folio.rf_currency_id
            LEFT JOIN rv_res_item_group ON rv_res_item_group.rv_res_item_group_ix = fn_folio.rv_res_item_group_id
            LEFT JOIN ac_accomm_room ON ac_accomm_room.ac_accomm_room_ix = rv_res_item_group.ac_accomm_room_id
            INNER JOIN rf_database ON rf_database.rf_db_code = fn_folio.fn_folio_db
          WHERE
            fn_folio.rv_reservation_id = '{$reservation['rv_reservation_ix']}'
            AND pr_persona.pr_email IS NOT NULL
            AND pr_persona.pr_email != ''
            AND rf_database.rf_db_env_type_ind = '4'
            AND (
              fn_folio.fn_folio_to_id NOT LIKE 0
            )
            AND fn_folio_amt_paid < fn_folio.fn_folio_amount
            AND fn_folio.fn_folio_amount > 0
          ORDER BY
            fn_folio_folio_num");

        foreach ($rawRecipients as $recipient) {
          $recipient = $recipient['fn_folio_to_id'];

          if (!in_array($recipient, $recipients)) {
            $recipients[] = $recipient;
          }
        }
      }

      $note = "Automated Workflow: \nName: {$reservation['aw_workflow_name']}\nType: " . AwWorkflow::getWorkflowTypeDescription(2);
      $reminders[] = $reminder->create(
        $note,
        ReReminder::RE_REMINDER_REPEAT_NO,
        ReReminder::RE_REMINDER_STATUS_ACTIVE,
        $reservation['reminder_send_date'],
        $createdTz,
        ReReminder::RE_REMINDER_LINK_TO_RV_RESERVATION,
        $reservation['rv_reservation_ix'],
        'RS4',
        $recipients,
        $reservation['aw_workflow_ix']
      );
    }

    return $reminders;
  }

  function workflowList($awWorkflowIx = null, $awWorkflowTypeInd = null)
  {
    $awWorkflowTriggers = require_once(__DIR__ . '/../../../../src/Resrequest/legacy/data.awftriggers.php');

    $select = "
      aw_workflow.aw_workflow_db,
      aw_workflow.aw_workflow_ix,
      aw_workflow.aw_workflow_name,
      aw_workflow.aw_workflow_type_ind,
      aw_workflow.aw_workflow_schedule_days,
      aw_workflow.aw_workflow_before_after_ind,
      aw_workflow.aw_workflow_trigger_id,
      aw_workflow.aw_workflow_config,
      aw_workflow.aw_workflow_time,
      aw_workflow.aw_workflow_time_tz";
    $where = "aw_workflow.aw_workflow_inactive_yn = 0";
    if (!empty($awWorkflowIx)) {
      $where .= " AND aw_workflow.aw_workflow_ix = '$awWorkflowIx'";
    }
    if (!is_null($awWorkflowTypeInd)) {
      $where .= " AND aw_workflow.aw_workflow_type_ind = '$awWorkflowTypeInd'";
    }

    $SQL = "
      SELECT
        {$select}
      FROM
        aw_workflow
      WHERE
        {$where}";
    $workflows = $this->db->fetchAll($SQL);

    foreach ($workflows as $key => $workflow) {
      $workflows[$key]['aw_workflow_trigger_name'] = $awWorkflowTriggers[$workflow['aw_workflow_trigger_id']]['awWorkflowTriggerName'];
    }

    return $workflows;
  }

  private function reservationList($workflows)
  {
    $awWorkflowTriggers = require_once(__DIR__ . '/../../../../src/Resrequest/legacy/data.awftriggers.php');

    $reservations = [];
    foreach($workflows as $workflow) {
      $this->setLastRun($workflow['aw_workflow_ix']);
      $workflowDetailsSelect = "
        '{$workflow['aw_workflow_ix']}' AS aw_workflow_ix,
        '{$workflow['aw_workflow_name']}' AS aw_workflow_name,
        '{$workflow['aw_workflow_type_ind']}' AS aw_workflow_type_ind,
        '{$workflow['aw_workflow_schedule_days']}' AS aw_workflow_schedule_days,
        '{$workflow['aw_workflow_before_after_ind']}' AS aw_workflow_before_after_ind,
        '{$workflow['aw_workflow_trigger_id']}' AS aw_workflow_trigger_id,
        '{$awWorkflowTriggers[ $workflow['aw_workflow_trigger_id'] ]['awWorkflowTriggerName']}' AS aw_workflow_trigger_name,
        '{$workflow['aw_workflow_time']}' AS aw_workflow_time,
        '{$workflow['aw_workflow_time_tz']}' AS aw_workflow_time_tz,";

      if ($workflow['aw_workflow_type_ind'] == '0') {
        list($select, $where) = $this->getWorkflowReservationSelectWhere($workflow, '0');
        $workflowReminderSQL = "
          SELECT DISTINCT
            rv_reservation.rv_reservation_ix
          FROM
            rv_reservation
            INNER JOIN rv_reservation_item ON rv_reservation_item.rv_reservation_id = rv_reservation.rv_reservation_ix
            INNER JOIN re_reminder ON re_reminder.re_reminder_link_to_id = rv_reservation.rv_reservation_ix
            INNER JOIN rv_reservation_guest ON rv_reservation_guest.rv_reservation_id = rv_reservation.rv_reservation_ix
            INNER JOIN pr_persona ON pr_persona.pr_persona_ix = rv_reservation_guest.pr_guest_id
            INNER JOIN pr_guest ON pr_guest.pr_guest_id = rv_reservation_guest.pr_guest_id
          WHERE
              pr_persona.pr_email IS NOT NULL
              AND TRIM(IFNULL(pr_persona.pr_email, '')) != ''
              AND pr_guest.pr_guest_inactive_yn = 0
              AND rv_reservation.rf_reservation_status_id IN (30)
              AND re_reminder.aw_workflow_id = '{$workflow['aw_workflow_ix']}'
              AND {$where}
        ";
        $results = $this->db->fetchAll($workflowReminderSQL);
        $workflowReservations = count($results) > 0 ? array_column($results, 'rv_reservation_ix') : [];
        $reservationSQL = "
          SELECT DISTINCT
            rv_reservation.rv_reservation_ix,
            rv_reservation.rv_date_arrive,
            rv_reservation.rv_date_depart,
            rv_reservation.rv_pax_count,
            pr_persona.pr_email,
            pr_guest.pr_guest_id,
            rv_reservation.rv_corr_persona_id,
            {$workflowDetailsSelect}
            {$select}
          FROM
            rv_reservation
            INNER JOIN rv_reservation_item ON rv_reservation_item.rv_reservation_id = rv_reservation.rv_reservation_ix
            INNER JOIN rv_reservation_guest ON rv_reservation_guest.rv_reservation_id = rv_reservation.rv_reservation_ix
            INNER JOIN pr_persona ON pr_persona.pr_persona_ix = rv_reservation_guest.pr_guest_id
            INNER JOIN pr_guest ON pr_guest.pr_guest_id = rv_reservation_guest.pr_guest_id
          WHERE
              pr_persona.pr_email IS NOT NULL
              AND TRIM(IFNULL(pr_persona.pr_email, '')) != ''
              AND pr_guest.pr_guest_inactive_yn = 0
              AND rv_reservation.rf_reservation_status_id IN (30)
              AND rv_reservation.rv_reservation_ix NOT IN ('".join("','", $workflowReservations)."')
              AND {$where}
        ";

        $reservations = array_merge($reservations, $this->db->fetchAll($reservationSQL));
      }

      if ($workflow['aw_workflow_type_ind'] == '1') {
        list($select, $where) = $this->getWorkflowReservationSelectWhere($workflow, '1');

        $reservationSQL = "
          SELECT DISTINCT
            rv_reservation.rv_provision_expiry_date,
            rv_reservation.rv_reservation_ix,
            rv_reservation.rv_date_arrive,
            rv_reservation.rv_date_depart,
            rv_reservation.rv_pax_count,
            rv_reservation.rv_consultant_id,
            rv_reservation.rv_corr_persona_id,
            {$workflowDetailsSelect}
            {$select}
          FROM
            rv_reservation
            INNER JOIN rv_reservation_item ON rv_reservation_item.rv_reservation_id = rv_reservation.rv_reservation_ix
            LEFT JOIN re_reminder ON re_reminder.re_reminder_link_to_id = rv_reservation.rv_reservation_ix
          WHERE
              rv_reservation.rf_reservation_status_id IN (20)
              AND {$where}
        ";

        $reservations = array_merge($reservations, $this->db->fetchAll($reservationSQL));
      }

      if ($workflow['aw_workflow_type_ind'] == '2') {
          list($select, $where, $dates) = $this->getWorkflowReservationSelectWhere($workflow, '2');

          $reservationSQL = "
            SELECT DISTINCT
              rv_agent_id,
              rv_billing_persona_id,
              pr_persona.pr_name_last,
              rf_currency_symbol,
              rv_invoice_currency_id,
              (
                rv_amt_accomm_gross + rv_amt_extra_gross + rv_amt_travel_gross
              ) AS grossTotal,
              (
                rv_amt_accomm_nett + rv_amt_extra_nett + rv_amt_travel_nett
              ) AS nettTotal,
              (rv_amt_paid) as rv_amt_paid,
              rv_commission_deduct_yn,
              pr_persona.pr_persona_ix,
              rv_reservation.rv_reservation_ix,
              rv_reservation.rv_corr_persona_id,
              rv_reservation.rv_date_depart,
              {$workflowDetailsSelect}
              {$select}
            FROM
              rv_reservation
              LEFT JOIN rv_pay_plan_item ON rv_reservation.rv_reservation_ix = rv_pay_plan_item.rv_reservation_id
              LEFT JOIN pr_persona ON rv_reservation.rv_agent_id = pr_persona.pr_persona_ix
              LEFT JOIN rf_currency ON rv_reservation.rv_invoice_currency_id = rf_currency.rf_currency_ix
              LEFT JOIN re_reminder ON re_reminder.re_reminder_link_to_id = rv_reservation.rv_reservation_ix
            WHERE
              {$where}
              AND (
                (
                  (
                    rv_reservation.rv_amt_accomm_payable + rv_reservation.rv_amt_extra_payable + rv_reservation.rv_amt_travel_payable
                  ) - rv_amt_paid > 0
                ) OR (
                  (
                    rv_reservation.rv_amt_accomm_payable + rv_reservation.rv_amt_extra_payable + rv_reservation.rv_amt_travel_payable
                  ) - rv_amt_paid < 0
                )
              )
        ";

        $reservationList = $this->db->fetchAll($reservationSQL);
        $payDueReservations = $this->processPayDueReservations($reservationList, $dates);
        $reservations = array_merge($reservations, $payDueReservations);
      }
    }

    return $reservations;
  }

  private function setLastRun($awWorkflowIx)
  {
    $workflow = $this->entityManager->find('\Resrequest\DB\Enterprise\Entity\AwWorkflow', $awWorkflowIx);
    $workflow->setAwWorkflowLastRunTimestamp($this->now)
      ->setAwWorkflowLastRunTimestampTz(date("e"));
    $this->entityManager->persist($workflow);
    $this->entityManager->flush();
    $this->entityManager->clear(); // inside loops Doctrine sometimes thinks some entities have already been persisted
  }

  function getWorkflowReservationSelectWhere($workflow, $awWorkflowTypeInd) {
    if ($awWorkflowTypeInd == '0') {
      $dateFunction = $this->getDateFunction($workflow['aw_workflow_before_after_ind']);
      $days = $workflow['aw_workflow_schedule_days'];
      $time = $workflow['aw_workflow_time'];

      switch ($workflow['aw_workflow_trigger_id']) {
        case 1: // itinerary arrival
          $date = "rv_reservation_item.rv_item_date_arrive";
          break;
        case 2: // itinerary departure
          $date = "rv_reservation_item.rv_item_date_depart";
          break;
        case 3: // reservation arrival
          $date = "rv_reservation.rv_date_arrive";
          break;
        case 4: // reservation departure
          $date = "rv_reservation.rv_date_depart";
          break;
      }
      $select = "
        CONCAT({$dateFunction}({$date}, INTERVAL {$days} DAY), ' ', '{$time}') AS reminder_send_date";
      $where = "{$dateFunction}({$date}, INTERVAL {$days} DAY) = DATE(NOW())";
      return [$select, $where];
    } else if ($awWorkflowTypeInd == '1') {
      $time = $workflow['aw_workflow_time'];
      $config = json_decode($workflow['aw_workflow_config'], true);

      $schedule = [
        'dayOfExpiry' => ($config['schedule']['dayOfExpiry'] == true) ? 'TRUE' : 'FALSE',
        'dayBeforeExpiry' => ($config['schedule']['dayBeforeExpiry'] == true) ? 'TRUE' : 'FALSE',
      ];
      $periods = $config['schedule']['periods'];
      foreach ($periods as &$period) {
        if ($period['interval'] <= 0) {
          $period['interval'] = 1;
        }
        $period['enabled'] = ($period['enabled'] == true) ? 'TRUE' : 'FALSE';
      }

      $select = "NOW() AS reminder_send_date";
      $where = "
        (
          (
            {$schedule['dayOfExpiry']}
            AND rv_reservation.rv_provision_expiry_date = DATE(NOW())
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATE(re_reminder.re_reminder_send_date) != DATE(NOW())
            )
          ) OR (
            {$schedule['dayBeforeExpiry']}
            AND rv_reservation.rv_provision_expiry_date = DATE(DATE_ADD(DATE(NOW()), INTERVAL 1 DAY))
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATE(re_reminder.re_reminder_send_date) != DATE(NOW())
            )
          ) OR (
            {$periods[0]['enabled']}
            AND DATEDIFF(rv_reservation.rv_provision_expiry_date, NOW()) >= 2
            AND DATEDIFF(rv_reservation.rv_provision_expiry_date, NOW()) <= 50
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATEDIFF(NOW(), re_reminder.re_reminder_send_date) >= {$periods[0]['interval']}
            )
          ) OR (
            {$periods[1]['enabled']}
            AND DATEDIFF(rv_reservation.rv_provision_expiry_date, NOW()) >= 51
            AND DATEDIFF(rv_reservation.rv_provision_expiry_date, NOW()) <= 100
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATEDIFF(NOW(), re_reminder.re_reminder_send_date) >= {$periods[1]['interval']}
            )
          ) OR (
            {$periods[2]['enabled']}
            AND DATEDIFF(rv_reservation.rv_provision_expiry_date, NOW()) >= 101
            AND DATEDIFF(rv_reservation.rv_provision_expiry_date, NOW()) <= 365
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATEDIFF(NOW(), re_reminder.re_reminder_send_date) >= {$periods[2]['interval']}
            )
          )
        ) AND (
          re_reminder.re_reminder_ix = (
            SELECT re_reminder_ix
            FROM re_reminder
            WHERE
              re_reminder.re_reminder_link_to_id = rv_reservation.rv_reservation_ix
              AND re_reminder.aw_workflow_id = '{$workflow['aw_workflow_ix']}'
            ORDER BY re_reminder_send_date DESC
            LIMIT 1
          ) OR re_reminder.re_reminder_ix IS NULL
        )
      ";

      return [$select, $where];
    } else if ($awWorkflowTypeInd == '2') {
      $time = $workflow['aw_workflow_time'];
      $config = json_decode($workflow['aw_workflow_config'], true);

      $schedule = [
        'dayOfPayment' => ($config['schedule']['dayOfPayment'] == true) ? 'TRUE' : 'FALSE',
      ];
      $periods = $config['schedule']['periods'];

      $dates = [];
      if ($config['schedule']['dayOfPayment'] == true) {
        $today = date('Y-m-d');
        $dates[] = [
          'startDate' => $today,
          'endDate' => $today
        ];
      }
      if ($periods[0]['enabled'] == true) {
        $dates[] = [
          'endDate' => (new \DateTime())->modify('-1 day')->format('Y-m-d'),
          'startDate' => (new \DateTime())->modify('-50 day')->format('Y-m-d')
        ];
      }
      if ($periods[1]['enabled'] == true) {
        $dates[] = [
          'endDate' => (new \DateTime())->modify('-51 day')->format('Y-m-d'),
          'startDate' => (new \DateTime())->modify('-100 day')->format('Y-m-d')
        ];
      }
      if ($periods[2]['enabled'] == true) {
        $dates[] = [
          'endDate' => (new \DateTime())->modify('-101 day')->format('Y-m-d'),
          'startDate' => (new \DateTime())->modify('-365 day')->format('Y-m-d')
        ];
      }

      foreach ($periods as &$period) {
        $period['enabled'] = ($period['enabled'] == true) ? 'TRUE' : 'FALSE';
      }
      
      $sourceIds = $config['filters']['source'];

      if (empty($sourceIds)) {
        $sourceIds = [];
      }

      $statusIds = [];
      $statuses = $config['filters']['status'];
      $statusToId = [
        'cancelled' => 90,
        'confirmed' => 30,
        'deleted' => 95,
        'inProgress' => 25,
        'validProvisional' => 20,
        'expiredProvisional' => 20,
        'quotation' => 0,
        'waitlisted' => 10,
    ];

    $provFilter = 'TRUE';
    $provStatusFilter = ['3'];
    foreach ($statuses as $status => $enabled) {
      if (!empty($enabled && in_array($status, array_keys($statusToId)))) {
        $statusIds[] = $statusToId[$status];

        if ($status == 'validProvisional') {
          $provStatusFilter[] = '1';
          $provFilter = 'FALSE';
        }
        if ($status == 'expiredProvisional') {
          $provStatusFilter[] = '0';
          $provFilter = 'FALSE';
        }
      }
    }


      $sourceFilter = (!empty($sourceIds)) ? 'FALSE' : 'TRUE';
      $statusFilter = (!empty($statusIds)) ? 'FALSE' : 'TRUE';

      $select = "NOW() AS reminder_send_date,
        IF(rv_reservation.rf_reservation_status_id = 20, IF(rv_reservation.rv_provision_expiry_date >= CURDATE(), TRUE, FALSE), NULL) AS valid";
      $where = "
        (
          (
            $sourceFilter
            OR rv_reservation.rf_source_ix IN ('".join("','", $sourceIds)."')
          )
          AND (
            $statusFilter
            OR rv_reservation.rf_reservation_status_id IN ('".join("','", $statusIds)."')
          )
          AND (
            $provFilter
            OR IF(rv_reservation.rf_reservation_status_id = 20, IF(rv_reservation.rv_provision_expiry_date >= CURDATE(), TRUE, FALSE), '3') IN ('".join("','", $provStatusFilter)."')
          )
        )
        AND (
          (
            {$schedule['dayOfPayment']}
            AND (
              rv_pay_plan_item.rv_pay_plan_date = DATE(NOW())
              OR (
                rv_reservation.rv_date_depart = DATE(NOW())
              )
            )
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATE(re_reminder.re_reminder_send_date) != DATE(NOW())
            )
          ) OR (
            {$periods[0]['enabled']}
            AND (
              (
                DATEDIFF(NOW(), rv_pay_plan_item.rv_pay_plan_date) >= 1
                AND DATEDIFF(NOW(), rv_pay_plan_item.rv_pay_plan_date) <= 50
              )
              OR (
                DATEDIFF(NOW(), rv_reservation.rv_date_depart) >= 1
                AND DATEDIFF(NOW(), rv_reservation.rv_date_depart) <= 50
              )
            )
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATEDIFF(NOW(), re_reminder.re_reminder_send_date) >= {$periods[0]['interval']}
            )
          ) OR (
            {$periods[1]['enabled']}
            AND (
              DATEDIFF(NOW(), rv_pay_plan_item.rv_pay_plan_date) >= 51
              AND DATEDIFF(NOW(), rv_pay_plan_item.rv_pay_plan_date) <= 100
              OR (
                DATEDIFF(NOW(), rv_reservation.rv_date_depart) >= 51
                AND DATEDIFF(NOW(), rv_reservation.rv_date_depart) <= 100
              )
            )
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATEDIFF(NOW(), re_reminder.re_reminder_send_date) >= {$periods[1]['interval']}
            )
          ) OR (
            {$periods[2]['enabled']}
            AND (
              (
                DATEDIFF(NOW(), rv_pay_plan_item.rv_pay_plan_date) >= 101
                AND DATEDIFF(NOW(), rv_pay_plan_item.rv_pay_plan_date) <= 150
              ) OR (
                DATEDIFF(NOW(), rv_reservation.rv_date_depart) >= 101
                AND DATEDIFF(NOW(), rv_reservation.rv_date_depart) <= 150
              )
            )
            AND (
              re_reminder.re_reminder_send_date IS NULL
              OR DATEDIFF(NOW(), re_reminder.re_reminder_send_date) >= {$periods[2]['interval']}
            )
          )
        ) AND (
          re_reminder.re_reminder_ix = (
            SELECT re_reminder_ix
            FROM re_reminder
            WHERE
              re_reminder.re_reminder_link_to_id = rv_reservation.rv_reservation_ix
              AND re_reminder.aw_workflow_id = '{$workflow['aw_workflow_ix']}'
            ORDER BY re_reminder_send_date DESC
            LIMIT 1
          ) OR re_reminder.re_reminder_ix IS NULL
        )
      ";

      return [$select, $where, $dates];
    }
  }

  public function processPayDueReservations($reservations, array $dates) {
    $reservationIds = array_unique(array_column($reservations, 'rv_reservation_ix'));
    $plans = $this->db->fetchAll("
      SELECT 
          rv_reservation_id,
          rv_pay_plan_date,
          rv_pay_plan_amount AS payDue
      FROM
        rv_pay_plan_item
      WHERE
        rv_reservation_id IN ('" . join("','", $reservationIds) . "')
      ORDER BY rv_pay_plan_date
    ");

    $paymentPlans = [];
    foreach ($plans  as $plan) {
      $key = $plan['rv_reservation_id'];
      if (!isset($paymentPlans[$key])) {
        $paymentPlans[$key] = [];
      }
      $paymentPlans[$key][] = $plan;
    }

    $reservationOutput = [];
    foreach ($reservations as $reservation) {
      $key = $reservation['rv_reservation_ix'];

      if (isset($reservationOutput[$key])) {
        continue;
      }

      foreach ($dates as $dateRange) {
        $date = $dateRange['startDate'];
        $endDate = $dateRange['endDate'];

        $plans = [];
        if (isset($paymentPlans[$key])) {
          $plans = $paymentPlans[$key];
        }

        if ($reservation['rv_commission_deduct_yn'] == "1") {
          $invoiceTotal = $reservation['nettTotal'];
        } else {
          $invoiceTotal = $reservation['grossTotal'];
        }

        if ($reservation['rv_commission_deduct_yn'] == "1") {
          $invoiceTotal = $reservation['nettTotal'];
          $onsiteExtraAmt = $this->db->fetchColumn("
            SELECT
              SUM(rv_extra.rv_extra_amt_nett)
            FROM
              rv_extra
              INNER JOIN rf_database ON rf_database.rf_db_code = rv_extra.rv_extra_db
            WHERE
              rv_extra.rv_reservation_id = '$reservation[rv_reservation_ix]'
              AND rv_extra.rv_extra_void_ind = 0
              AND rf_database.rf_db_env_type_ind = '4'
          ");
        } else {
          $invoiceTotal = $reservation['grossTotal'];
          $onsiteExtraAmt = $this->db->fetchColumn("
            SELECT
              SUM(rv_extra.rv_extra_amt_gross)
            FROM
              rv_extra
              INNER JOIN rf_database ON rf_database.rf_db_code = rv_extra.rv_extra_db
            WHERE
            rv_extra.rv_reservation_id = '$reservation[rv_reservation_ix]'
              AND rv_extra.rv_extra_void_ind = 0
              AND rf_database.rf_db_env_type_ind = '4'
          ");
        }

        // Only send reminders on a payment date, regardless if an amount is owed.
        $isPaymentDate = false;
        $payPlanDueNow = false;
        $dueNow = 0;
        $dueTotal = 0;
        $dueTotal += $onsiteExtraAmt;
        $lastPaymentDate = false;
        if (isset($paymentPlans[$key])) {
          foreach ($paymentPlans[$key] as $dueItem) {
            $dueTotal += $dueItem['payDue'];

            if (
              $dueItem['rv_pay_plan_date'] <= $endDate
            ) {
              $dueNow += $dueItem['payDue'];
            }

            if (
              $dueItem['rv_pay_plan_date'] <= $endDate
              && $dueItem['rv_pay_plan_date'] >= $date
            ) {
              $isPaymentDate = true;
              $payPlanDueNow = true;
            }

            $lastPaymentDate = $dueItem['rv_pay_plan_date'];
          }
        }

        $unalloc = $invoiceTotal - $dueTotal;

        if ($unalloc > 0) {
          if (
            $lastPaymentDate === false
            && $reservation['rv_date_depart'] <= $endDate
          ) {
            $dueNow += $unalloc;
            
            if (
              $reservation['rv_date_depart'] <= $endDate
              && $reservation['rv_date_depart'] >= $date
            ) {
              $isPaymentDate = true;
              $payPlanDueNow = true;
            }
          } else if (
            $lastPaymentDate <= $endDate
          ) {
            $dueNow += $unalloc;

            if (
              $lastPaymentDate <= $date
              && $lastPaymentDate >= $endDate
            ) {
              $isPaymentDate = true;
              $payPlanDueNow = true;
            }
          }
        }

        $onsiteExtrasDueNow = false;
        if (!empty($onsiteExtraAmt) && $onsiteExtraAmt > 0) {
          if (
            $reservation['rv_date_depart'] <= $endDate
          ) {
            $dueNow += $onsiteExtraAmt;

            if (
              $reservation['rv_date_depart'] <= $endDate
              && $reservation['rv_date_depart'] >= $date
            ) {
              $isPaymentDate = true;
              $onsiteExtrasDueNow = true;
            }
          } else {
            $invoiceTotal = $invoiceTotal - $onsiteExtraAmt;
          }
        }


        $reservation['payPlanDueTotal'] = $dueTotal;

        if (empty($reservation['payPlanDueNow'])) {
          $reservation['payPlanDueNow'] = $payPlanDueNow;
        }
        if (empty($reservation['onsiteExtrasDueNow'])) {
          $reservation['onsiteExtrasDueNow'] = $onsiteExtrasDueNow;
        }

        if (
          $isPaymentDate
          && $dueNow - $reservation['rv_amt_paid'] > 0
        ) {
          $reservationOutput[$key] = $reservation;
        }
      }
    }

    return array_values($reservationOutput);
  }


  private function getDateFunction($awWorkflowBeforeAfterInd) {
    $function = "";
    switch ($awWorkflowBeforeAfterInd) {
      case AwWorkflow::AW_WORKFLOW_BEFORE:
        return "DATE_SUB";
        break;
      case AwWorkflow::AW_WORKFLOW_AFTER:
        return "DATE_ADD";
        break;
    }
  }

  private function response($data) {
    if ($this->isCli()) {
      return $data;
    } else {
      return new JsonModel($data);
    }
  }

  private function isCli()
  {
    if(defined('STDIN')) {
      return true;
    }

    if(empty($_SERVER['REMOTE_ADDR']) && !isset($_SERVER['HTTP_USER_AGENT']) && count($_SERVER['argv']) > 0) {
      return true;
    } 

    return false;
  }
}
