<?php

/*
 * $Id: api.rate.func.php,v 1.1 2013-08-19 17:02:52 light Exp $
 *
 * $Log: api.rate.func.php,v $
 * Revision 1.1  2013-08-19 17:02:52  light
 * Bug #8252 - API extensions for TVL
 *
 */

/**
 * api/api.rate.func.php - API functions related to rates
 */

require_once(__DIR__ . '/../db.ac_accomm_type.php');
require_once(__DIR__ . '/../db.rt_rate_type.php');
require_once(__DIR__ . '/../db.pr_agent.php');
require_once(__DIR__ . '/../class.rate.php');
require_once(__DIR__ . '/../functions.reservation.itinerary.php');

function api_ac_get_rate($search="",$default=0,$agent="",$extra=false) {
	api_rt_get_rate($search,$default,$agent,$extra);
}

function api_rt_get_rate($search="",$default=0,$agent="",$extra=false) {
	global $api_instance;
	global $userStatusId;

	if(!empty($agent) && $agent != "public") {
		// ignore "linked" for external users
		if($userStatusId < 2 && $agent == "linked") {
			$agent = "";
		} else if($userStatusId > 1) {
			if($agent == "linked") {
				$agent = $GLOBALS['pr_agent_link'];
			}
			if(!empty($agent) && !db_pr_agent_exists($agent)) {
				return $api_instance->Error("Agent does not exist.");
			}
		} else {
			return $api_instance->Error("Access denied.");
		}
	}

	if($extra === false) {
		$extra = array(0,0,1,1,0);
	}

	if(!is_array($extra)) {
		return $api_instance->Error("Invalid request for additional information. This parameter must be an array.");
	}


	$extraFields = array();
	if((array_key_exists(0,$extra) && !empty($extra[0])) || (array_key_exists("abbreviation",$extra) && !empty($extra['abbreviation']))) {
		$extraFields['abbreviation'] = "rt_rate_type_abbrv";
	}
	if((array_key_exists(1,$extra) && !empty($extra[1])) || (array_key_exists("colour",$extra) && !empty($extra['colour']))) {
		if ($userStatusId < 2) {
			return $api_instance->Error("Access denied.");
		}
		$extraFields['colour'] = "rt_rate_type_colour";
	}
	if((array_key_exists(2,$extra) && !empty($extra[2])) || (array_key_exists("currency",$extra) && !empty($extra['currency']))) {
		$extraFields['currency'] = "rf_currency_symbol";
	}

	if((array_key_exists(3,$extra) && !empty($extra[3])) || (array_key_exists("default",$extra) && !empty($extra['default']))) {
		$extraFields['default'] = "";
	}

	if((array_key_exists(4,$extra) && !empty($extra[4])) || (array_key_exists("note",$extra) && !empty($extra['note']))) {
		if ($userStatusId < 2) {
			return $api_instance->Error("Access denied.");
		}
		$extraFields['note'] = "rt_rate_type_note";
	}

	$extraFieldFull = array();
	foreach($extraFields as $field) {
		if(empty($field)) {
			continue;
		} elseif($field == "rf_currency_symbol") {
			$extraFieldFull[] = "rf_currency.$field";
		} else {
			$extraFieldFull[] = "rt_rate_type.$field";
		}
	}

	$list = db_rt_rate_type_get_list($search,$agent,$extraFieldFull);
	$newList = array();

	foreach($list as $item) {
		$newItem = array('id'=>$item['rt_rate_type_ix'],'name'=>$item['rt_rate_type_desc']);

		if (isset($item['pr_agent_rate_default_yn'])) {
			$defaultValue = $item['pr_agent_rate_default_yn'];
		} elseif (isset($item['rf_default_rate_id'])) {
			$defaultValue = 1;
		} elseif (isset($item['rf_default_rt_rate_type_id'])) {
			$defaultValue = 1;
		} else {
			$defaultValue = 0;
		}

		foreach($extraFields as $name=>$field) {
			if($name == "default") {
				$newItem[$name] = $defaultValue;
			} else {
				$newItem[$name] = $item[$field];
			}
		}

		if($default != 1 || ($default == 1 && $defaultValue == 1)) {
			$newList[] = $newItem;
		}
	}

	return $newList;
}

function api_rt_get_rate_total($ac_accomm_type_id,$startDate=false,$endDate=false,$rateGroups=false,$rt_rate_type_id=false,$amount_type=false, $pr_agent_id=false,$pr_agent_commission_perc=false) {
	global $api_instance;
	global $lDB;

	if(!db_ac_accomm_type_exists($ac_accomm_type_id)) {
		return $api_instance->Error("Invalid accommodation type.");
	}

	if ($rateGroups === false){
		$rateGroups = array();
	}

	$check = $lDB->get("
		SELECT
			COUNT(*)
		FROM
			sc_accomm
			INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = sc_accomm.ac_accomm_type_id
		WHERE
			sc_accomm.ac_accomm_type_id = '$ac_accomm_type_id'
			AND sc_accomm.sc_group_id = '$GLOBALS[sc_group_id]'
			AND	ac_accomm_type.ac_accomm_type_inactive_yn=0
	",4);
	if($check < 1) {
		return $api_instance->Error("Invalid accommodation type.");
	}

	if(!empty($startDate)) {
		$startDate = strtotime($startDate);
		if($startDate == -1) {
			return $api_instance->Error("Invalid start date");
		}
		$startDate = date("Y-m-d",$startDate);
	}

	if(!empty($endDate)) {
		$endDate = strtotime($endDate);
		if($endDate == -1) {
			return $api_instance->Error("Invalid end date");
		}
		$endDate = date("Y-m-d",$endDate);
	}

	$newRateGroups = cleanRateGroups($rateGroups);

	$maxpax = $lDB->get("
		SELECT
			ac_accomm_type.ac_accomm_max_capacity,
			ac_accomm_type.ac_accomm_max_adults
		FROM
			ac_accomm_type
		WHERE
			ac_accomm_type_ix = '$ac_accomm_type_id'
	",1);
	if ($GLOBALS['userStatusId'] < 2) {
		if (getTotalAdults($newRateGroups) > $maxpax['ac_accomm_max_adults']) {
			return $api_instance->Error("Adult capacity ($maxpax[ac_accomm_max_adults]) exceeded for accommodation type ($ac_accomm_type_id)");
		}

		if (getTotalPax($newRateGroups) > $maxpax['ac_accomm_max_capacity']) {
			return $api_instance->Error("Room capacity ($maxpax[ac_accomm_max_capacity]) exceeded for accommodation type ($ac_accomm_type_id)");
		}

		if( (!empty($amount_type) && $amount_type != "payable") || (!empty($pr_agent_id) && $pr_agent_id != "linked") || !empty($pr_agent_commission_perc)) {
			return $api_instance->Error("Access Denied. Agent or commission override only allowed for internal users.");
		}
	}

	if(!empty($rt_rate_type_id)) {
		$list = db_rt_rate_type_get_list();

		$found = false;
		foreach($list as $item) {
			if($item['rt_rate_type_ix'] == $rt_rate_type_id) {
				$found = true;
			}
		}
		if(!$found) {
			return $api_instance->Error("Invalid rate type");
		}
	}

	if($amount_type != "gross" && $amount_type != "nett" && $amount_type != "payable" && $amount_type != "comm" && $amount_type != "tax") {
		$amount_type = "payable";
	}

	if(!is_numeric($pr_agent_commission_perc)) {
		if($pr_agent_commission_perc === "") {
			$pr_agent_commission_perc = false;
		}
	}

	$deduct = false;
	if(
		$pr_agent_id == "linked"
		|| (
			empty($pr_agent_id)
			&& $GLOBALS['userStatusId'] < 2
		)
	) {
		$pr_agent_id = $GLOBALS['pr_agent_link'];
	}
	if(!empty($pr_agent_id)) {
		if(!db_pr_agent_exists($pr_agent_id)) {
			return $api_instance->Error("Agent does not exist.");
		}
		$deduct = $lDB->get("
			SELECT
				pr_agent.pr_agent_comm_deduct_yn
			FROM
				pr_agent
			WHERE
				pr_agent.pr_agent_id = '$pr_agent_id'
		",4);
	}

	$pr_business_id = $lDB->get("SELECT pr_business_id FROM ac_accomm_type WHERE ac_accomm_type_ix = '" . $lDB->escape($ac_accomm_type_id) . "'",4);
	$agentTerms = getAgentPropertyTerms($pr_agent_id, false, $pr_business_id);
	if(empty($rt_rate_type_id)) {
		$rt_rate_type_id = $agentTerms['rt_rate_type_id'];
	}
	if(empty($rt_rate_type_id)) {
		return $api_instance->Error("Agent does not have a default rate set");
	}
	if($pr_agent_commission_perc === false) {
		$pr_agent_commission_perc = $agentTerms['rv_commission_perc'];
	}

	$getRate = new Rate($ac_accomm_type_id,$startDate,$endDate,$newRateGroups,$rt_rate_type_id,false,false,false,$pr_agent_commission_perc,false,$deduct);
	return $getRate->ToTotal($amount_type);

}

function api_ac_get_rate_total($ac_accomm_type_id,$startDate=false,$endDate=false,$units=false,$adults=false,$children=false,$rt_rate_type_id=false) {
	global $api_instance;
	global $lDB;


	if(!empty($units) && !is_numeric($units)) {
		return $api_instance->Error("Invalid number of units");
	}

	if(!empty($adults) && !is_numeric($adults)) {
		return $api_instance->Error("Invalid number of adults");
	}

	if(!empty($children) && !is_numeric($children)) {
		return $api_instance->Error("Invalid number of children");
	}

	// Convert Units, Adults and Children into their applicable rate groups
	$unitCode = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1",4);
	$adultCode = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 2",4);
	// Note: This will return the first child rate groups (there could be many)
	$childCode = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_class = 2",4);

	$rateGroups[] = array($unitCode, $units);
	$rateGroups[] = array($adultCode, $adults);
	$rateGroups[] = array($childCode, $children);

	return api_rt_get_rate_total($ac_accomm_type_id,$startDate,$endDate,$rateGroups,$rt_rate_type_id);

}

function api_rt_get_rate_detail($ac_accomm_type_id,$startDate=false,$endDate=false,$rateGroups=false,$rt_rate_type_id=false,$amount_type=false, $pr_agent_id=false,$pr_agent_commission_perc=false,$split=false) {
	global $api_instance;
	global $lDB;

	if(!db_ac_accomm_type_exists($ac_accomm_type_id)) {
		return $api_instance->Error("Invalid accommodation type.");
	}

	if ($rateGroups === false){
		$rateGroups = array();
	}

	$check = $lDB->get("
		SELECT
			COUNT(*)
		FROM
			sc_accomm
			INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = sc_accomm.ac_accomm_type_id
		WHERE
			sc_accomm.ac_accomm_type_id = '$ac_accomm_type_id'
			AND sc_accomm.sc_group_id = '$GLOBALS[sc_group_id]'
			AND	ac_accomm_type.ac_accomm_type_inactive_yn=0
	",4);
	if($check < 1) {
		return $api_instance->Error("Invalid accommodation type.");
	}

	if(!empty($startDate)) {
		$startDate = strtotime($startDate);
		if($startDate == -1) {
			return $api_instance->Error("Invalid start date");
		}
		$startDate = date("Y-m-d",$startDate);
	}

	if(!empty($endDate)) {
		$endDate = strtotime($endDate);
		if($endDate == -1) {
			return $api_instance->Error("Invalid end date");
		}
		$endDate = date("Y-m-d",$endDate);
	}

	$newRateGroups = cleanRateGroups($rateGroups);

	$maxpax = $lDB->get("
		SELECT
			ac_accomm_type.ac_accomm_max_capacity,
			ac_accomm_type.ac_accomm_max_adults
		FROM
			ac_accomm_type
		WHERE
			ac_accomm_type_ix = '$ac_accomm_type_id'
	",1);
	if ($GLOBALS['userStatusId'] < 2) {
		if (getTotalAdults($newRateGroups) > $maxpax['ac_accomm_max_adults']) {
			return $api_instance->Error("Adult capacity ($maxpax[ac_accomm_max_adults]) exceeded for accommodation type ($ac_accomm_type_id)");
		}

		if (getTotalPax($newRateGroups) > $maxpax['ac_accomm_max_capacity']) {
			return $api_instance->Error("Room capacity ($maxpax[ac_accomm_max_capacity]) exceeded for accommodation type ($ac_accomm_type_id)");
		}

		if( (!empty($amount_type) && $amount_type != "payable") || (!empty($pr_agent_id) && $pr_agent_id != "linked") || !empty($pr_agent_commission_perc)) {
			return $api_instance->Error("Access Denied. Agent or commission override only allowed for internal users.");
		}
	}

	if(!empty($rt_rate_type_id)) {
		$list = db_rt_rate_type_get_list();

		$found = false;
		foreach($list as $item) {
			if($item['rt_rate_type_ix'] == $rt_rate_type_id) {
				$found = true;
			}
		}
		if(!$found) {
			return $api_instance->Error("Invalid rate type");
		}
	}

	if(!is_array($amount_type)) {
		if($amount_type != "gross" && $amount_type != "nett" && $amount_type != "payable" && $amount_type != "comm" && $amount_type != "tax") {
			$amount_type = "payable";
		}
		$amount_type = array($amount_type=>1);
	}

	$hasAmount = false;
	foreach($amount_type as $enabled) {
		if(!empty($enabled)) {
			$hasAmount = true;
		}
	}
	if(!$hasAmount) {
		$amount_type = array('payable'=>1);
	}

	$amount_fields = array();
	if((array_key_exists(0,$amount_type) && !empty($amount_type[0])) || (array_key_exists("payable",$amount_type) && !empty($amount_type['payable']))) {
		$amount_fields[] = "payable";
	}
	if((array_key_exists(1,$amount_type) && !empty($amount_type[1])) || (array_key_exists("gross",$amount_type) && !empty($amount_type['gross']))) {
		$amount_fields[] = "gross";
	}
	if((array_key_exists(2,$amount_type) && !empty($amount_type[2])) || (array_key_exists("nett",$amount_type) && !empty($amount_type['nett']))) {
		$amount_fields[] = "nett";
	}
	if((array_key_exists(3,$amount_type) && !empty($amount_type[3])) || (array_key_exists("comm",$amount_type) && !empty($amount_type['comm']))) {
		$amount_fields[] = "comm";
	}
	if((array_key_exists(4,$amount_type) && !empty($amount_type[4])) || (array_key_exists("tax",$amount_type) && !empty($amount_type['tax']))) {
		$amount_fields[] = "tax";
	}

	if(!is_numeric($pr_agent_commission_perc)) {
		if($pr_agent_commission_perc === "") {
			$pr_agent_commission_perc = false;
		}
	}

	$deduct = false;
	if(
		$pr_agent_id == "linked"
		|| (
			empty($pr_agent_id)
			&& $GLOBALS['userStatusId'] < 2
		)
	) {
		$pr_agent_id = $GLOBALS['pr_agent_link'];
	}
	if(!empty($pr_agent_id)) {
		if(!db_pr_agent_exists($pr_agent_id)) {
			return $api_instance->Error("Agent does not exist.");
		}
		$deduct = $lDB->get("
			SELECT
				pr_agent.pr_agent_comm_deduct_yn
			FROM
				pr_agent
			WHERE
				pr_agent.pr_agent_id = '$pr_agent_id'
		",4);
	}

	$pr_business_id = $lDB->get("SELECT pr_business_id FROM ac_accomm_type WHERE ac_accomm_type_ix = '" . $lDB->escape($ac_accomm_type_id) . "'",4);
	$agentTerms = getAgentPropertyTerms($pr_agent_id, false, $pr_business_id);
	if(empty($rt_rate_type_id)) {
		$rt_rate_type_id = $agentTerms['rt_rate_type_id'];
	}
	if(empty($rt_rate_type_id)) {
		return $api_instance->Error("Agent does not have a default rate set");
	}
	if($pr_agent_commission_perc === false) {
		$pr_agent_commission_perc = $agentTerms['rv_commission_perc'];
	}

	if($split === false) {
		$split = array();
	}

	if(!is_array($split)) {
		return $api_instance->Error("Invalid request for rate split. This parameter must be an array.");
	}

	$split_fields = array();
	if((array_key_exists(0,$split) && !empty($split[0])) || (array_key_exists("period",$split) && !empty($split['period']))) {
		$split_fields[] = "period";
	}
	if((array_key_exists(1,$split) && !empty($split[1])) || (array_key_exists("rate_group",$split) && !empty($split['rate_group']))) {
		$split_fields[] = "rate_group";
	}

	$getRate = new Rate($ac_accomm_type_id,$startDate,$endDate,$newRateGroups,$rt_rate_type_id,false,false,false,$pr_agent_commission_perc,false,$deduct);

	// New vars
	//$split_rate = array('period'=>1,'rate_group'=>1);
	//$amount_types = array("payable");
	//$amount_types = array("payable","gross","nett","comm","tax");

	$periods = $getRate->ToPeriods();

	$all_amount_types = array("payable","gross","nett","comm","tax");

	$rate_detail = array();
	$rate_detail['accommodation_id'] = $ac_accomm_type_id;
	$rate_detail['property_id'] = $pr_business_id;

	foreach($all_amount_types as $type) {
		if(in_array($type,$amount_fields)) {
			$rate_detail[$type] = 0;
		}
	}
	$rate_detail['periods'] = array();

	$rate_group_qty = array();
	foreach($newRateGroups as $rateGroup) {
		$rate_group_qty[$rateGroup['rt_rate_group_ix']] = $rateGroup['qty'];
	}

	$showDetail = false;
	foreach ($amount_type as $type => $enabled) {
		if ($enabled == 1 && $type == "detail") {
			$showDetail = true;
		}
	}
	if (
		$GLOBALS['userStatusId'] < 2 &&
		(
			(!empty($pr_agent_id) && $pr_agent_id != "linked")
		)
	) {
		$showDetail = false;
	}

	$first = true;
	foreach($periods as $period) {
		if($first || in_array("period",$split_fields)) {
			$period_detail = array(
				'period_from'=>$period['rt_period_date_from'],
				'period_to'=>$period['rt_period_date_to']
			);
			foreach($all_amount_types as $type) {
				if(in_array($type,$amount_fields)) {
					$period_detail[$type] = 0;
				}
			}
			$rate_group_detail = array();
		} else {
			if($period['rt_period_date_from'] < $period_detail['period_from']) {
				$period_detail['period_from'] = $period['rt_period_date_from'];
			}
			if($period['rt_period_date_to'] > $period_detail['period_to']) {
				$period_detail['period_to'] = $period['rt_period_date_to'];
			}
		}
		if ($showDetail) {
			$period_detail['components'] = [];
		}

		foreach($period['components'] as $component) {
			foreach($component['rate_groups'] as $rate_group_id=>$rate_group) {
				if(!array_key_exists($rate_group_id,$rate_group_detail)) {
					$rate_group_detail[$rate_group_id] = array(
						'qty' => $rate_group_qty[strtoupper($rate_group_id)],
					);
					foreach($all_amount_types as $type) {
						if(in_array($type,$amount_fields)) {
							$rate_group_detail[$rate_group_id][$type] = 0;
						}
					}
				}

				foreach($all_amount_types as $type) {
					if(in_array($type,$amount_fields)) {
						$value = db_round($rate_group[$type],2);
						$rate_group_detail[$rate_group_id][$type] += $value;
						$period_detail[$type] += $value;
						$rate_detail[$type] += $value;
					}
				}
			}
			
			$taxRates = [];
			foreach ($component['tax_detail'] as $id => $rate) {
				$taxRates[] = [
					'name' => $lDB->get("SELECT rf_tax_rate_desc FROM rf_tax_rate WHERE rf_tax_rate_ix = '" . $id . "'", 4),
					'percentage' => $rate['perc'],
					'amount' => db_round($rate['amount'], 2),
					'id' => $id,
				];
			}

			if ($component['rt_tax_ind'] == 10) {
				$name = $lDB->get("SELECT rf_tax_rate_desc FROM rf_tax_rate WHERE rf_tax_rate_ix = '" . $component['rt_tax_id'] . "'", 4);
			} else {
				$name = $lDB->get("SELECT rt_tax_group_desc FROM rt_tax_group WHERE rt_tax_group_ix = '" . $component['rt_tax_id'] . "'", 4);
			}

			if ($showDetail) {
				$period_detail['components'][] = [
					'component_name' => $lDB->get("SELECT rt_component_desc FROM rt_component WHERE rt_component_ix = '" . $component['rt_component_id'] . "'", 4),
					'component_id' => $component['rt_component_id'],
					'component_commission' => $component['rv_item_comp_amt_comm_calc'],
					'component_gross' => $component['rv_item_comp_amt_gross'],
					'component_nett' => $component['rv_item_comp_amt_nett'],
					'component_payable' => $component['rv_item_comp_amt_payable'],
					'component_tax_amount' => $component['rv_item_comp_amt_tax'],
					'component_tax_percentage' => $component['rf_tax_rate_perc'],
					'tax_type' => $component['rt_tax_ind'] == 20 ? 'Group' : 'Rate',
					'tax_name' => $name,
					'tax_id' => $component['rt_tax_id'],
					'tax_rates' => $taxRates
				];
			}
		}

		if(in_array("period",$split_fields)) {
			if(in_array("rate_group",$split_fields)) {
				$period_detail['rate_groups'] = $rate_group_detail;
			}
			$rate_detail['periods'][] = $period_detail;
		}
		$first = false;
	}
	if(!in_array("period",$split_fields)) {
		if(in_array("rate_group",$split_fields)) {
			$period_detail['rate_groups'] = $rate_group_detail;
		}
		$rate_detail['periods'][] = $period_detail;
	}

	return $rate_detail;
}

function api_rt_get_rate_total_range_batch($dataArray) {
	global $lDB, $api_instance;

	$results = [];

	if (!is_array($dataArray)) {
		return $api_instance->Error("No array of data provided.");
	}

	foreach($dataArray as $item) {
		if (!isset($item[0]) || empty($item[0])) {
			$results[] = [
				"error" => [
					"faultCode" => 800,
					"faultString" => "No accommodation type provided."
				]
			];
			continue;
		}
		if (!isset($item[1]) || empty($item[1])) {
			$results[] = [
				"error" => [
					"faultCode" => 800,
					"faultString" => "No start date provided."
				]
			];
			continue;
		}
		if (!isset($item[2]) || empty($item[2])) {
			$results[] = [
				"error" => [
					"faultCode" => 800,
					"faultString" => "No end date provided."
				]
			];
			continue;
		}
		if (!isset($item[3]) || empty($item[3]) || !is_array($item[3])) {
			$results[] = [
				"error" => [
					"faultCode" => 800,
					"faultString" => "No or invalid rate groups provided."
				]
			];
			continue;
		}

		$ac_accomm_type_id = $item[0];
		$startDate = $item[1];
		$endDate = $item[2];
		$rateGroups = $item[3];
		$rt_rate_type_id = $item[4] ?? false;
		$pr_agent_id = $item[5] ?? false;
		$pr_agent_commission_perc = $item[6] ?? false;
		$result = api_rt_get_rate_total_range($ac_accomm_type_id, $startDate, $endDate, $rateGroups, $rt_rate_type_id, $pr_agent_id, $pr_agent_commission_perc);

		if (isset($result->errno) && isset($result->errstr)) {
			$results[] = [
				"error" => [
					"faultCode" => $result->errno,
					"faultString" => $result->errstr
				]
			];
		} else {
			$results[] = $result;
		}

	}

	return $results;
}

function api_rt_get_rate_total_range($ac_accomm_type_id,$startDate=false,$endDate=false,$rateGroups=false,$rt_rate_type_id=false, $pr_agent_id=false, $pr_agent_commission_perc=false) {
	global $api_instance;
	global $lDB;

	$output = array();

	if(!db_ac_accomm_type_exists($ac_accomm_type_id)) {
		return $api_instance->Error("Invalid accommodation type: " . $ac_accomm_type_id);
	}

	$check = $lDB->get("
		SELECT
			COUNT(*)
		FROM
			sc_accomm
			INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = sc_accomm.ac_accomm_type_id
		WHERE
			sc_accomm.ac_accomm_type_id = '$ac_accomm_type_id'
			AND sc_accomm.sc_group_id = '$GLOBALS[sc_group_id]'
			AND	ac_accomm_type.ac_accomm_type_inactive_yn=0
	",4);
	if($check < 1) {
		return $api_instance->Error("Invalid accommodation type: " . $ac_accomm_type_id);
	}

	if(!empty($startDate)) {
		$startDate = strtotime($startDate);
		if($startDate == -1) {
			return $api_instance->Error("Invalid start date");
		}
		$startDate = date("Y-m-d",$startDate);
	}

	if(!empty($endDate)) {
		$endDate = strtotime($endDate);
		if($endDate == -1) {
			return $api_instance->Error("Invalid end date");
		}
		$endDate = date("Y-m-d",$endDate);
	}

	if(!empty($rt_rate_type_id)) {
		$list = db_rt_rate_type_get_list();

		$found = false;
		foreach($list as $item) {
			if($item['rt_rate_type_ix'] == $rt_rate_type_id) {
				$found = true;
			}
		}
		if(!$found) {
			return $api_instance->Error("Invalid rate type");
		}
	}

	if ($GLOBALS['userStatusId'] < 2 && ((!empty($pr_agent_id) && $pr_agent_id != "linked") || !empty($pr_agent_commission_perc))) {
		return $api_instance->Error("Access Denied. Agent or commission override only allowed for internal users.");
	}

	if(!is_numeric($pr_agent_commission_perc)) {
		if($pr_agent_commission_perc === "") {
			$pr_agent_commission_perc = false;
		}
	}

	$deduct = false;
	if(
		$pr_agent_id == "linked"
		|| (
			empty($pr_agent_id)
			&& $GLOBALS['userStatusId'] < 2
		)
	) {
		$pr_agent_id = $GLOBALS['pr_agent_link'];
	}
	if(!empty($pr_agent_id)) {
		if(!db_pr_agent_exists($pr_agent_id)) {
			return $api_instance->Error("Agent does not exist.");
		}
		$deduct = $lDB->get("
			SELECT
				pr_agent.pr_agent_comm_deduct_yn
			FROM
				pr_agent
			WHERE
				pr_agent.pr_agent_id = '$pr_agent_id'
		",4);
	}

	$pr_business_id = $lDB->get("SELECT pr_business_id FROM ac_accomm_type WHERE ac_accomm_type_ix = '" . $lDB->escape($ac_accomm_type_id) . "'",4);
	$agentTerms = getAgentPropertyTerms($pr_agent_id, false, $pr_business_id);
	if(empty($rt_rate_type_id)) {
		$rt_rate_type_id = $agentTerms['rt_rate_type_id'];
	}
	if(empty($rt_rate_type_id)) {
		return $api_instance->Error("Agent does not have a default rate set");
	}
	if($pr_agent_commission_perc === false) {
		$pr_agent_commission_perc = $agentTerms['rv_commission_perc'];
	}

	// [START] Do the work

	// Create rate object
	$rate = new Rate($ac_accomm_type_id,$startDate,$endDate,false,$rt_rate_type_id,false,false,false,$pr_agent_commission_perc,false,$deduct);

	$dailyAmounts = array();
	$newRateGroups = array();

	// Get max pax
	$maxCapacity = getAccommTypeMaxCapacity($rate->AccommTypeId);
	$maxAdults = getAccommTypeMaxAdults($rate->AccommTypeId);

	// Get room rate group
	list($unitGroup,$unitDesc) = $GLOBALS['lDB']->get("SELECT rt_rate_group_ix, rt_rate_group_desc FROM rt_rate_group WHERE rt_rate_group_sys_code = 1", 1);
	list($adultGroup,$adultDesc) = $GLOBALS['lDB']->get("SELECT rt_rate_group_ix, rt_rate_group_desc FROM rt_rate_group WHERE rt_rate_group_sys_code = 2", 1);
	list($childGroup,$childDesc) = $GLOBALS['lDB']->get("SELECT rt_rate_group_ix, rt_rate_group_desc  FROM rt_rate_group WHERE rt_rate_group_sys_code = 3", 1);

	// Rate Groups
	if (is_array($rateGroups)) {
		if (is_array($rateGroups[0])) { // Assume the standard rate groups format
			$rate->SetRateGroups(cleanRateGroups($rateGroups));
			$rate->SetUnits();
			$rate->PeriodSplit = 2;
			$rate->PeriodSplitSingleDay = true;
			$rate->ToPeriods();
			$dailyAmounts[0]['principal_id'] = $GLOBALS['principal_id'];
			$dailyAmounts[0]['accommodation_id'] = $ac_accomm_type_id;
			$dailyAmounts[0]['property_id'] = $pr_business_id;
			$dailyAmounts[0]['rateGroups'] = $rate->RateGroups;
			$dailyAmounts[0]['periods'] = $rate->CalculateDailyTotals();
		} else { // New type of array where the first element is a string describing the rate groups in the second element
			if ($rateGroups[0] == "all") {
				if (isset($rateGroups[1]) && is_array($rateGroups[1])){ // We want all rate group combinations but we have also defined some limits
					$specifiedRateGroups = cleanRateGroups($rateGroups[1]);
					$allRateGroups = array();
					$rateGroupCombinations = array();
					$arr = array();

					foreach($specifiedRateGroups as $rateGroup){
						$newRateGroups[$rateGroup['rt_rate_group_ix']] = $rateGroup['qty'];
					}

					// Check to see if these rate groups are valid across all periods in range, if not, cockblock.
					// We do this by calling SetRateGroups with the set of rate groups. If there are any conflicts within the periods
					// the rate object will set a default set of rate groups only. This is a hack, I hate it.
					$rate->SetRateGroups($specifiedRateGroups);
					$rate->SetUnits();

					foreach($rate->RateGroups as $classRateGroup){
						if(!array_key_exists('qty',$classRateGroup)){
							return $api_instance->Error("The periods in the given date range have inconsistent rate groups");
						}
					}

					$allRateGroups = array();
					foreach($newRateGroups as $rt_rate_group_id=>$qty) {
							$class = $lDB->get("SELECT rt_rate_group_class FROM rt_rate_group WHERE rt_rate_group_ix = '$rt_rate_group_id'",4);
							$max = 1;
							if($class == 1) {
								$max = $maxAdults;
							} elseif($class == 2) {
								$max = $maxCapacity;
							}

							if($qty < $max) {
								$max = $qty;
							}
							$allRateGroups[$rt_rate_group_id] = array(
								'class'=>$class,
								'max'=>$max
							);

					}

					$requiredCombinations = array();
					$totalCombinations = 1;
					foreach($allRateGroups as $rt_rate_group_id=>$rateGroup) {
						if($rt_rate_group_id == $unitGroup) {
							$requiredCombinations[$rt_rate_group_id] = array(1);
						} else {
							$requiredCombinations[$rt_rate_group_id] = range(0,$rateGroup['max']);
							$totalCombinations *= $rateGroup['max']+1;
						}
					}

					if($totalCombinations > 20000) {
						return $api_instance->Error("Too many possible combinations. No more than 20000 rate group combinations are allowed per call.");
					}

					$rawRateGroupCombinations = array_combinations($requiredCombinations);

					$rateGroupCombinations = array();
					foreach($rawRateGroupCombinations as $rawRateGroupCombination) {
						$rateGroupCombination = array();
						foreach($rawRateGroupCombination as $rt_rate_group_id=>$qty) {
							$rateGroupCombination[$rt_rate_group_id] = array(
								'rt_rate_group_ix'=>$rt_rate_group_id,
								'qty'=>$qty
							);
						}
						$rateGroupCombinations[] = $rateGroupCombination;
					}

					// Now that we have a set of all possible rate groups, do the work
					foreach($rateGroupCombinations as $rgkey => $rateGroupComb){
						$rate->SetRateGroups(cleanRateGroups($rateGroupComb));
						$rate->SetUnits();
						$rate->PeriodSplit = 2;
						$rate->PeriodSplitSingleDay = true;
						$rate->ToPeriods();
						$dailyAmounts[$rgkey]['principal_id'] = $GLOBALS['principal_id'];
						$dailyAmounts[$rgkey]['accommodation_id'] = $ac_accomm_type_id;
						$dailyAmounts[$rgkey]['property_id'] = $pr_business_id;
						$dailyAmounts[$rgkey]['rateGroups'] = $rate->RateGroups;
						$dailyAmounts[$rgkey]['periods'] = $rate->CalculateDailyTotals();
					}
				} else { // We want all rate group combinations

					$rate->PeriodSplit = 2;
					$ratePeriods = $rate->ToPeriods();

					// Get all common rate groups, across all periods (remember ,these could be diff. for each period within the range
					$allRateGroups = array();
					foreach($ratePeriods as $key => $period) {
						if($period['rt_rate_ix'] === false) {
							$allRateGroups[$unitGroup] = array('desc'=>$unitDesc,'class'=>0,'max'=>1);
							$allRateGroups[$adultGroup] = array('desc'=>$adultDesc,'class'=>1,'max'=>$maxAdults);
							$allRateGroups[$childGroup] = array('desc'=>$childDesc,'class'=>2,'max'=>$maxCapacity);
						} else {
							$rateCompGroupHeaders = getRateComponentGroupHeaders($period['rt_rate_ix']);
							foreach($rateCompGroupHeaders as $rateCompGroupHeader) {
								$class = $lDB->get("SELECT rt_rate_group_class FROM rt_rate_group WHERE rt_rate_group_ix = '$rateCompGroupHeader[rt_rate_group_ix]'",4);
								$max = 1;
								if($class == 1) {
									$max = $maxAdults;
								} elseif($class == 2) {
									$max = $maxCapacity;
								}
								$allRateGroups[$rateCompGroupHeader['rt_rate_group_ix']] = array(
									'desc'=>$rateCompGroupHeader['rt_rate_group_desc'],
									'class'=>$class,
									'max'=>$max
								);
							}
						}
					}

					$requiredCombinations = array();
					$totalCombinations = 1;
					foreach($allRateGroups as $rt_rate_group_id=>$rateGroup) {
						if($rt_rate_group_id == $unitGroup) {
							$requiredCombinations[$rt_rate_group_id] = array(1);
						} else {
							$requiredCombinations[$rt_rate_group_id] = range(0,$rateGroup['max']);
							$totalCombinations *= $rateGroup['max']+1;
						}
					}

					if($totalCombinations > 20000) {
						return $api_instance->Error("Too many possible combinations. No more than 20000 rate group combinations are allowed per call.");
					}

					$rawRateGroupCombinations = array_combinations($requiredCombinations);

					$rateGroupCombinations = array();
					foreach($rawRateGroupCombinations as $rawRateGroupCombination) {
						$rateGroupCombination = array();
						foreach($rawRateGroupCombination as $rt_rate_group_id=>$qty) {
							$rateGroupCombination[$rt_rate_group_id] = array(
								'rt_rate_group_ix'=>$rt_rate_group_id,
								'qty'=>$qty
							);
						}
						$rateGroupCombinations[] = $rateGroupCombination;
					}

					// Now that we have a set of all possible rate groups, do the work
					foreach($rateGroupCombinations as $rgkey => $rateGroupComb) {
						$rate->SetRateGroups(cleanRateGroups($rateGroupComb));
						$rate->SetUnits();
						$rate->PeriodSplit = 2;
						$rate->PeriodSplitSingleDay = true;
						$rate->ToPeriods();
						$dailyAmounts[$rgkey]['principal_id'] = $GLOBALS['principal_id'];
						$dailyAmounts[$rgkey]['accommodation_id'] = $ac_accomm_type_id;
						$dailyAmounts[$rgkey]['property_id'] = $pr_business_id;
						$dailyAmounts[$rgkey]['rateGroups'] = $rate->RateGroups;
						$dailyAmounts[$rgkey]['periods'] = $rate->CalculateDailyTotals();
					}
				}
			} else {
				return $api_instance->Error("Invalid rate group array");
			}
		}
	} else {
		return $api_instance->Error("Invalid rate group array");
	}
	// [END] Do the work

	$output = $dailyAmounts;

	return $output;
}

function api_rt_get_rate_group($ac_accomm_type_id,$startDate=false,$endDate=false,$rt_rate_type_id=false) {
	global $api_instance;
	global $lDB;

	$result = array();

	if(!db_ac_accomm_type_exists($ac_accomm_type_id)) {
		return $api_instance->Error("Invalid accommodation type.");
	}

	$check = $lDB->get("
		SELECT
			COUNT(*)
		FROM
			sc_accomm
			INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = sc_accomm.ac_accomm_type_id
		WHERE
			sc_accomm.ac_accomm_type_id = '$ac_accomm_type_id'
			AND sc_accomm.sc_group_id = '$GLOBALS[sc_group_id]'
			AND	ac_accomm_type.ac_accomm_type_inactive_yn=0
	",4);
	if($check < 1) {
		return $api_instance->Error("Invalid accommodation type.");
	}

	if(!empty($startDate)) {
		$startDate = strtotime($startDate);
		if($startDate == -1) {
			return $api_instance->Error("Invalid start date");
		}
		$startDate = date("Y-m-d",$startDate);
	}

	if(!empty($endDate)) {
		$endDate = strtotime($endDate);
		if($endDate == -1) {
			return $api_instance->Error("Invalid end date");
		}
		$endDate = date("Y-m-d",$endDate);
	}

	if(!empty($rt_rate_type_id)) {
		$list = db_rt_rate_type_get_list();

		$found = false;
		foreach($list as $item) {
			if($item['rt_rate_type_ix'] == $rt_rate_type_id) {
				$found = true;
			}
		}
		if(!$found) {
			return $api_instance->Error("Invalid rate type");
		}
	}

	$rate = new Rate($ac_accomm_type_id,$startDate,$endDate,false,$rt_rate_type_id);

	foreach ($rate->RateGroups as $rateGroup){
		$tempRateGroup['id'] = $rateGroup['rt_rate_group_ix'];
		$tempRateGroup['desc'] = $rateGroup['rt_rate_group_desc'];
		$tempRateGroup['abbrv'] = $rateGroup['rt_rate_group_abbrv'];
		$tempRateGroup['class'] = $rateGroup['rt_rate_group_class'];
		if ($rateGroup['rt_rate_group_sys_code'] > 0){
			$tempRateGroup['default'] = 1;
		} else {
			$tempRateGroup['default'] = 0;
		}
		array_push($result,$tempRateGroup);
	}

	return $result;

}

function api_ac_get_payment_plan($searchFilter=false) {
	api_rt_get_payment_plan($searchFilter);
}

function api_rt_get_payment_plan($searchFilter=false) {
	global $lDB;
	global $userStatusId;
	global $calAgentId;

	$result = array();
	$filter = array();

	if ($searchFilter){
		$filter[] = "ac_pay_plan_desc LIKE '".str_replace("*", "%", $searchFilter)."'";
	}

	if($userStatusId < 2) {
		$ac_pay_plan_id = $lDB->get("SELECT ac_pay_plan_id FROM pr_agent WHERE pr_agent_id = '$calAgentId'",4);
        $filter[] = "ac_pay_plan.ac_pay_plan_ix = '$ac_pay_plan_id'";
	}


	if(sizeof($filter) > 0) {
		$filterWhere = "
			WHERE
				".join(" AND ",$filter)."
		";
	} else {
		$filterWhere = "";
	}

	$planArray = $lDB->get("
		SELECT
			ac_pay_plan_ix,
			ac_pay_plan_desc
		FROM
			ac_pay_plan
		$filterWhere

	",2);

	foreach ($planArray as $item){
		$paymentPlan['id'] = $item['ac_pay_plan_ix'];
		$paymentPlan['name'] = $item['ac_pay_plan_desc'];

		array_push($result,$paymentPlan);
	}

	return $result;

}
