<?php

/*
 * $Id: api.reservation.func.php,v 1.1 2013-08-19 17:02:52 light Exp $
 *
 * $Log: api.reservation.func.php,v $
 * Revision 1.1  2013-08-19 17:02:52  light
 * Bug #8252 - API extensions for TVL
 *
 */

/**
 * api/api.reservation.func.php - API functions related to reservations
 */

require_once(__DIR__ . '/../class.audit.php');
require_once(__DIR__ . '/../class.stock.php');
require_once(__DIR__ . '/../class.rate.php');
require_once(__DIR__ . '/../db.ac_pay_plan.php');
require_once(__DIR__ . "/../db.ad_persona_detail.php");
require_once(__DIR__ . '/../db.pr_guest.php');
require_once(__DIR__ . '/../db.pr_persona.php');
require_once(__DIR__ . '/../db.pr_user.php');
require_once(__DIR__ . '/../db.rf_source.php');
require_once(__DIR__ . '/../db.rf_country.php');
require_once(__DIR__ . '/../db.rf_obj_trip.php');
require_once(__DIR__ . '/../db.rt_rate_type.php');
require_once(__DIR__ . '/../db.rv_reservation.php');
require_once(__DIR__ . '/../db.rv_note.php');
require_once(__DIR__ . '/../db.rv_reservation_guest.php');
require_once(__DIR__ . '/../db.rv_reservation_item.php');
require_once(__DIR__ . '/../db.rv_res_item_group.php');
require_once(__DIR__ . '/../db.rv_res_item_guest.php');
require_once(__DIR__ . '/../db.rv_trip_obj_data.php');
require_once(__DIR__ . '/../db.sc_group.php');
require_once(__DIR__ . '/../functions.financial.php');
require_once(__DIR__ . '/../functions.mail.php');
require_once(__DIR__ . '/../functions.persona.php');
require_once(__DIR__ . '/../functions.reservation.php');
require_once(__DIR__ . '/../functions.reservation.advanced.php');
require_once(__DIR__ . '/../functions.reservation.itinerary.php');

function api_rv_create($items,$rt_rate_type_id=false,$rf_reservation_status_id=false,$rv_res_name=false,$rv_agent_ref=false,$rv_note_general=false,$rv_note_guests=false,$agent_id=false,$origin_agent_id=false,$commission_type=false,$commission_perc=false,$payment_plan=false,$source=false,$consultant=false,$allowOverbooking=false, $contactId=false, $environment=false) {
	global $api_instance;
	global $lDB;
	global $wizResId;
	global $userStatusId;

	if (!empty($environment)) {
		$GLOBALS['dbcode'] = $environment;
	}	

	if (
		!canAccessEnvironment($GLOBALS['sc_group_id'], "", $GLOBALS['dbcode']) ||
		!isEnvironmentActive("", $GLOBALS['dbcode'])
	) {
		return $api_instance->Error("No access to the specified environment.");
	}
	

	/*
	 * $items format (OLD):
	 *
	 * 0/'accommodation_type'
	 * 1/'date_arrive'
	 * 2/'date_depart'
	 * 3/'unit_count'
	 * 4/'adult_count'
	 * 5/'child_count'
	 * 6/'rate_type'
	 * 7/'use_allocations'
	 */
	 
	/*
	 * $items format (NEW):
	 *
	 * 0/'accommodation_type'
	 * 1/'date_arrive'
	 * 2/'date_depart'
	 * 3/array(array(),array())
	 * 4/override array
	 * 5/rate type
	 * 6/'use_allocations'
	 */	 

	 if(!is_array($items)) {
		return $api_instance->Error("Invalid itinerary specified.");
	}

	$rateAutoDetect = false;
	$rateList = db_rt_rate_type_get_list();
	if(!empty($rt_rate_type_id)) {

		$found = false;
		foreach($rateList as $item) {
			if($item['rt_rate_type_ix'] == $rt_rate_type_id) {
				$found = true;
			}
		}
		if(!$found) {
			return $api_instance->Error("Invalid rate type");
		}
	} else {
		if(sizeof($rateList) > 0) {
			$tmp_rt_rate_type_id = $rateList[0]['rt_rate_type_ix'];
		} else {
			return $api_instance->Error("Invalid rate type");
		}
	}

	if($userStatusId < 2) {
		// for external users: ignore 'linked', disallow override, use the linked agent if one exists, otherwise direct
		if($agent_id == "linked") {
			$agent_id = "";
		}

		if(!empty($agent_id)) {
			return $api_instance->Error("Access denied for agent override.");
		}

		if(empty($GLOBALS['pr_agent_link'])) {
			$pr_agent_id = false;
		} else {
			$pr_agent_id = $GLOBALS['pr_agent_link'];
		}
	} else {
		// for internal users: blank for direct 'linked' to check for a linked agent, otherwise use specified agent if it exists
		if(empty($agent_id)) {
			$pr_agent_id = false;
		} else {
			if($agent_id == "linked") {
				$pr_agent_id = $GLOBALS['pr_agent_link'];
			} else {
				$pr_agent_id = $agent_id;
			}
			if(!empty($pr_agent_id)) {
				if(!db_pr_agent_exists($pr_agent_id)) {
					return $api_instance->Error("Agent does not exist.");
				}
				if(empty($rt_rate_type_id)) { // Use the default rate of the linked agent/user
					$link_default_rate = $lDB->get("
						SELECT
							rt_rate_type_id
						FROM
							pr_agent_rate
						WHERE
							pr_agent_rate.pr_agent_id = '$pr_agent_id'
							AND  pr_agent_rate_default_yn = '1'
					",4);
					if(empty($link_default_rate)) {
						return $api_instance->Error("Agent does not have a default rate set");
					} else {
						$rateAutoDetect = true;
						$rt_rate_type_id = $link_default_rate;
					}
				}
			}
		}
	}

	if(empty($rt_rate_type_id)) {
		$rateAutoDetect = true;
		$rt_rate_type_id = $tmp_rt_rate_type_id;
	}

	// TODO: Should we load the agent commission and payment plan details here?

	if(sizeof($items) < 1) {
		return $api_instance->Error("No itinerary specified.");
	}
	
	if($rf_reservation_status_id === false || $rf_reservation_status_id == "") {
		$rf_reservation_status_id = "20";
	}

	if($rf_reservation_status_id != "0" && $rf_reservation_status_id != "10" && $rf_reservation_status_id != "20" && $rf_reservation_status_id != "30") {
		return $api_instance->Error("Invalid reservation status");
	}

	if (empty($commission_perc)){
		$pr_agent_commission_perc = false;
	} elseif ($userStatusId > 1) {
		if (!is_numeric($commission_perc)){
			return $api_instance->Error("Invalid commission percentage.");
		} else {
			$pr_agent_commission_perc = $commission_perc;
		}
	} else {
		return $api_instance->Error("Access denied for commission percentage override.");
	}

	foreach($items as $key=>$item) {
		if($rateAutoDetect) {
			$item_rate_type_id = false;
		} else {
			$item_rate_type_id = $rt_rate_type_id;
		}

		$newItem = processItem($item,$item_rate_type_id,$pr_agent_id,$pr_agent_commission_perc);
		if($newItem['error']) {
			return $api_instance->Error($newItem['error_message']);
		}	

		$allocations = array();
		if($rf_reservation_status_id != "0" && $rf_reservation_status_id != "10") {
			$dateArray = explode("-",$newItem['rv_item_date_depart']);
			$dateDepart = date("Y-m-d",mktime(0,0,0,$dateArray[1],$dateArray[2]-1,$dateArray[0]));
			
			$stock = new Stock($newItem['ac_accomm_type_id'],$newItem['rv_item_date_arrive'],$dateDepart,false,$pr_agent_id);

			if($newItem['auto_allocate']) {
				$rawAllocations = $stock->ToReservationAllocation(false, true);
				foreach($rawAllocations as $day) {
					foreach($day as $allocation) {
						if(!$allocation['expired'] && $allocation['allocation_count'] > 0) {
							$allocations[] = $allocation['rv_reservation_ix'];
						}
					}
				}
				$allocations = array_unique($allocations);
			}			
		}
		$newItem['allocations'] = $allocations;

		$items[$key] = $newItem;
	}	 
	
	if(
		$rf_reservation_status_id != "0"
		&& $rf_reservation_status_id != "10"
		&& (!$allowOverbooking || db_sc_group_get_user_setting("sc_grp_res_ovr_overbooking_yn") != "1")
		&& !check_multi_item_availability($items)
	) {
		return $api_instance->Error("Not enough stock available.");
	}

	if(empty($rv_res_name)) {
		$rv_res_name = "API: " . date("Y-m-d h:i:s");
	} else {
		$rv_res_name = addslashes(str_replace("*","%",$rv_res_name));
	}

	if(empty($rv_agent_ref)) {
		$rv_agent_ref = "";
	} else {
		$rv_agent_ref = addslashes(str_replace("*","%",$rv_agent_ref));
	}

	if(empty($rv_note_general)) {
		$rv_note_general = "";
	} else {
		$rv_note_general = addslashes(str_replace("*","%",$rv_note_general));
	}

	if(empty($rv_note_guests)) {
		$rv_note_guests = "";
	} else {
		$rv_note_guests = addslashes(str_replace("*","%",$rv_note_guests));
	}


	if (empty($origin_agent_id)){
		$rv_origin_agent_id = false;
	} elseif ($userStatusId > 1) {
		if (db_pr_persona_exists($origin_agent_id)){
			$rv_origin_agent_id = addslashes($origin_agent_id);
		} else {
			return $api_instance->Error("Originator does not exist.");
		}
	} else {
		return $api_instance->Error("Access denied for originator override.");
	}

	if (empty($commission_type)){
		$pr_agent_comm_deduct_yn = false;
	} elseif ($userStatusId > 1) {
		if ($commission_type != "1" && $commission_type != "0"){
			return $api_instance->Error("Invalid commission type.");
		} else {
			$pr_agent_comm_deduct_yn = $commission_type;
		}
	} else {
		return $api_instance->Error("Access denied for commission type override.");
	}
	
	if (empty($payment_plan)){
		$ac_pay_plan_id = false;
	} elseif ($userStatusId > 1) {
		if (!db_ac_pay_plan_exists($payment_plan)){
			return $api_instance->Error("Payment plan does not exist.");
		} else {
			$ac_pay_plan_id = $payment_plan;
		}
	} else {
		return $api_instance->Error("Access denied for payment plan override.");
	}	
	
	if (empty($source)){
		$rf_source_id = false;
	} elseif ($userStatusId > 1) {
		if (!db_rf_source_exists($source)){
			return $api_instance->Error("Source does not exist.");
		} else {
			$rf_source_id = $source;
		}
	} else {
		return $api_instance->Error("Access denied for source override.");
	}		

	if (empty($consultant) || ($userStatusId < 2 && $consultant === "none")) {
		$consultant = false;
	} elseif ($userStatusId > 1) {
		if ($consultant == "none"){
			// Continue
		} elseif (!db_pr_consultant_exists($consultant)){
			return $api_instance->Error("Consultant does not exist.");
		} else {
			// Continue
		}
	} else {
		return $api_instance->Error("Access denied for consultant override.");
	}

	$contact = $GLOBALS['userid'];

	if (!empty($contactId)) {
		if ($userStatusId < 2) {
			return $api_instance->Error("Access denied for contact override.");
		}

		if (db_pr_persona_exists($contactId)) {
			$isPropertyContact = $lDB->get("SELECT
				COUNT(*)
				FROM pr_business
				WHERE pr_business.pr_business_id = '$contactId'
			", 4);

			if ($isPropertyContact) {
				return $api_instance->Error("Invalid contact id specified. A Property contact can not be used as a booking contact.");
			}

			$isInactive = $lDB->get("SELECT
				COUNT(*)
				FROM pr_persona
				WHERE pr_persona.pr_persona_ix = '$contactId'
				AND pr_persona.pr_persona_inactive_yn = 1
			", 4);

			if ($isInactive) {
				return $api_instance->Error("Invalid contact id specified. The contact has been deactivated.");
			}
	
			$contact = $contactId;
		} else {
			return $api_instance->Error("Invalid contact id specified. The contact does not exist.");
		}
	}

	$rv_reservation_id = db_rv_reservation_insert($rt_rate_type_id,$contact,$pr_agent_id,$pr_agent_commission_perc,$pr_agent_comm_deduct_yn,false,false,false,false,false,false,false,false,$ac_pay_plan_id,false,$rf_source_id,false,false,false,false,$consultant);

	foreach($items as $item) {
		$itemDataRaw = convertItemToRaw($item['ac_accomm_type_id'],$item['rv_item_date_arrive'],$item['rv_item_date_depart'],$item['rateGroups']);

		$createItems = calCreateItem($rv_reservation_id, $itemDataRaw, $item['rt_rate_type_id'], "defaultOnly", $item['override'], false, "0", false, $item['allocations'], false, false, false, false, false, $item['rv_item_comm_perc']);
		if(sizeof($item['guests']) > 0) {
			processGuests($rv_reservation_id, $createItems, $item['guests']);
		}
	}
	
	$today = date("Y-m-d");
	$statusSQL = "";
	switch($rf_reservation_status_id) {
	case "10":
		$statusSQL = "
			rv_wait_list_date = '$today',
			rv_wait_list_user_id = '$GLOBALS[userid]',
		";
		break;
	case "20":
		$defaultExpiryDays = $lDB->get("SELECT rf_prov_expiry_days FROM rf_default",4);
		$provExpiryDate = date("Y-m-d",mktime( 0, 0, 0, date("m"), ( date("d") + $defaultExpiryDays ), date("Y")));
		$statusSQL = "
			rv_prov_date = '$today', rv_provision_expiry_date = '$provExpiryDate',
		";
		break;
	case "30":
		$statusSQL = "
			rv_confirmation_date = '$today',
			rv_confirmation_user_id = '$GLOBALS[userid]',
		";
		break;
	}

	if (empty($rf_source_id)){
		$rf_source_id = $lDB->get("SELECT rf_source.rf_source_ix FROM rf_source WHERE rf_source_sys_code = '1'",4);
	}
	$lDB->put("
		UPDATE rv_reservation SET
			rf_reservation_status_id = '$rf_reservation_status_id',
			$statusSQL
			rv_res_name = '$rv_res_name',
			rv_agent_ref = '$rv_agent_ref',
			rf_source_ix = '$rf_source_id',
			rv_origin_agent_id = '$rv_origin_agent_id'
		WHERE
			rv_reservation_ix = '$rv_reservation_id'
	");
	db_rv_note_insert($rv_reservation_id,false,false,$rv_note_general,$rv_note_guests,false);

	regenBilling($rv_reservation_id, false, true);

	$ad_info = array();

	$agentName = getContactFullName($pr_agent_id,false);
	if ($agentName != ""){
		$ad_info[] = "Agent: $agentName";
	}	
	$contactName = getContactFullName($contact,false);
	if ($contactName != ""){
		$ad_info[] = "Contact: $contactName";
	}	
	$rateTypeDesc = getRateTypeDesc($rt_rate_type_id);
	if ($rateTypeDesc != ""){
		$ad_info[] = "Rate Type: $rateTypeDesc";
	}	
	ammendReservation($rv_reservation_id,"Create (".join(", ",$ad_info).")");		

	return $rv_reservation_id;
}

function processItem($item,$rt_rate_type_id=false,$pr_agent_id=false,$pr_agent_commission_perc=false) {
	global $lDB;
	global $userStatusId;

	$unitCode = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1",4);
	$adultCode = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 2",4);
	
	$newItem = array();
	$newItem['error'] = false;
	$newItem['error_message'] = "";
	
	// If $item[3] is an array, we assume the new format
	if (!is_array($item[3])){

		if(array_key_exists('accommodation_type',$item)) {
			$newItem['ac_accomm_type_id'] = $item['accommodation_type'];
			$newItem['rv_item_date_arrive'] = $item['date_arrive'];
			$newItem['rv_item_date_depart'] = $item['date_depart'];
			$newItem['rateGroups'] = convertToRateGroups($item['unit_count'],$item['adult_count'],$item['child_count']);
			if (array_key_exists('rate_type', $item) && !empty($item['rate_type'])){
				$newItem['rt_rate_type_id'] = $item['rate_type'];	
			} else {
				// Inject reservation-level Rate Type into Itinerary array
				$newItem['rt_rate_type_id'] = $rt_rate_type_id;
			}			
			if(array_key_exists("use_allocations",$item) && !empty($item['use_allocations'])) {
				$newItem['auto_allocate'] = true;
			} else {
				$newItem['auto_allocate'] = false;
			}
			if (array_key_exists("guests", $item)) {
				$newItem['guests'] = $item['guests'];
			}
		} else {
			$item = array_pad($item,9,"");
			$newItem['ac_accomm_type_id'] = $item[0];
			$newItem['rv_item_date_arrive'] = $item[1];
			$newItem['rv_item_date_depart'] = $item[2];
			$newItem['rateGroups'] = convertToRateGroups($item[3],$item[4],$item[5]);
			if (array_key_exists(6, $item) && !empty($item[6])){
				$newItem['rt_rate_type_id'] = $item[6];	
			} else {
				// Inject reservation-level Rate Type into Itinerary array
				$newItem['rt_rate_type_id'] = $rt_rate_type_id;
			}			
			if(array_key_exists(7,$item) && !empty($item[7])) {
				$newItem['auto_allocate'] = true;
			} else {
				$newItem['auto_allocate'] = false;
			}
			$newItem['guests'] = $item[8];
		}	
	
	} else {
	
		if(array_key_exists('accommodation_type',$item)) {
			$newItem['ac_accomm_type_id'] = $item['accommodation_type'];
			$newItem['rv_item_date_arrive'] = $item['date_arrive'];
			$newItem['rv_item_date_depart'] = $item['date_depart'];
			$newItem['rateGroups'] = $item['rateGroups'];
			if(empty($newItem['rateGroups'])) {
				$newItem['error'] = true;
				$newItem['error_message'] = "Empty rate group parameter.";
				return $newItem;
			}
			foreach ($newItem['rateGroups'] as $key => $rateGroup) {
				if(!is_array($rateGroup)) {
					$newItem['error'] = true;
					$newItem['error_message'] = "Invalid rate group parameter. Rate groups should be defined as an outer array with inner arrays for each rate group.";
					return $newItem;
				}
			}
			$newItem['rateGroups'] = cleanRateGroups($newItem['rateGroups']);
			$newItem['override'] = convertToOrideArray($newItem['ac_accomm_type_id'],$newItem['rv_item_date_arrive'],$newItem['rv_item_date_depart'],$newItem['rateGroups'],$rt_rate_type_id,$item['override']);
			if (array_key_exists('rate_type', $item) && !empty($item['rate_type'])) {
				$newItem['rt_rate_type_id'] = $item['rate_type'];	
			} else {
				// Inject reservation-level Rate Type into Itinerary array
				$newItem['rt_rate_type_id'] = $rt_rate_type_id;
			}
			if(array_key_exists("use_allocations",$item) && !empty($item['use_allocations'])) {
				$newItem['auto_allocate'] = true;
			} else {
				$newItem['auto_allocate'] = false;
			}
			if (array_key_exists("guests", $item)) {
				$newItem['guests'] = $item['guests'];
			}
			
		} else {
			$item = array_pad($item,8,"");
			$newItem['ac_accomm_type_id'] = $item[0];
			$newItem['rv_item_date_arrive'] = $item[1];
			$newItem['rv_item_date_depart'] = $item[2];
			$newItem['rateGroups'] = $item[3];
			if(empty($newItem['rateGroups'])) {
				$newItem['error'] = true;
				$newItem['error_message'] = "Empty rate group parameter.";
				return $newItem;
			}
			foreach ($newItem['rateGroups'] as $key => $rateGroup) {
				if(!is_array($rateGroup)) {
					$newItem['error'] = true;
					$newItem['error_message'] = "Invalid rate group parameter. Rate groups should be defined as an outer array with inner arrays for each rate group.";
					return $newItem;
				}
			}
			$newItem['rateGroups'] = cleanRateGroups($newItem['rateGroups']);
			$newItem['override'] = convertToOrideArray($newItem['ac_accomm_type_id'],$newItem['rv_item_date_arrive'],$newItem['rv_item_date_depart'],$newItem['rateGroups'],$rt_rate_type_id,$item[4]);
			if (array_key_exists(5, $item) && !empty($item[5])){
				$newItem['rt_rate_type_id'] = $item[5];	
			} else {
				// Inject reservation-level Rate Type into Itinerary array
				$newItem['rt_rate_type_id'] = $rt_rate_type_id;
			}
			if(array_key_exists(6,$item) && !empty($item[6])) {
				$newItem['auto_allocate'] = true;
			} else {
				$newItem['auto_allocate'] = false;
			}
			$newItem['guests'] = $item[7];
		}		
	}
	$newItem['rv_item_accomm_count'] = getTotalUnits($newItem['rateGroups']);


	// Check if requested rate type exists
	// Note, at this point, it could be the already-checked reservation-level rate type, or the itinerary-level rate type
	$rateList = db_rt_rate_type_get_list();
	if(!empty($newItem['rt_rate_type_id'])) {

		$found = false;
		foreach($rateList as $item) {
			if($item['rt_rate_type_ix'] == $newItem['rt_rate_type_id']) {
				$found = true;
			}
		}
		if(!$found) {
			$newItem['error'] = true;
			$newItem['error_message'] = "Invalid rate type.";
			return $newItem;	
		}

		// Check if requested rate type is allowed as per User Access rules
		if(db_sc_group_get_user_setting("sc_grp_res_ovr_rate_type_yn") != "1") {
			$agentCount = $lDB->get("SELECT COUNT(pr_agent_id) FROM pr_agent WHERE pr_agent_id = '$pr_agent_id'",4);
			if($agentCount > 0) {
				$rateList = $GLOBALS['lDB']->get("SELECT rt_rate_type_id FROM pr_agent_rate WHERE pr_agent_rate.pr_agent_id = '$pr_agent_id'",3);
			} else {
				$rateList = $GLOBALS['lDB']->get("SELECT rf_default_rate_id FROM rf_default_rate",3);
			}
			if(!in_array($newItem['rt_rate_type_id'],$rateList)) {
				$newItem['error'] = true;
				$newItem['error_message'] = "Access denied for rate type override.";
				return $newItem;		
			}
		}
	}

	if(!db_ac_accomm_type_exists($newItem['ac_accomm_type_id'])) {
		$newItem['error'] = true;
		$newItem['error_message'] = "Invalid accommodation type ($newItem[ac_accomm_type_id])";
		return $newItem;
	}

	$pr_business_id = $lDB->get("SELECT pr_business_id FROM ac_accomm_type WHERE ac_accomm_type_ix = '" . $lDB->escape($newItem['ac_accomm_type_id']) . "'",4);
	$agentTerms = getAgentPropertyTerms($pr_agent_id, false, $pr_business_id);

	if(empty($newItem['rt_rate_type_id'])) {
		$newItem['rt_rate_type_id'] = $agentTerms['rt_rate_type_id'];
	}
	if($pr_agent_commission_perc !== false) {
		$newItem['rv_item_comm_perc'] = $pr_agent_commission_perc;
	} else {
		$newItem['rv_item_comm_perc'] = $agentTerms['rv_commission_perc'];
	}


	if(empty($newItem['rv_item_date_arrive'])) {
		$newItem['error'] = true;
		$newItem['error_message'] = "Invalid arrival date ($newItem[rv_item_date_arrive])";
		return $newItem;
	} else {
		$newItem['rv_item_date_arrive'] = strtotime($newItem['rv_item_date_arrive']);
		if($newItem['rv_item_date_arrive'] == -1) {
			$newItem['error'] = true;
			$newItem['error_message'] = "Invalid arrival date ($newItem[rv_item_date_arrive])";
			return $newItem;
		}
		$newItem['rv_item_date_arrive'] = date("Y-m-d",$newItem['rv_item_date_arrive']);
	}
	
	if(empty($newItem['rv_item_date_depart'])) {
		$newItem['error'] = true;
		$newItem['error_message'] = "Invalid departure date ($newItem[rv_item_date_depart])";
		return $newItem;
	} else {
		$newItem['rv_item_date_depart'] = strtotime($newItem['rv_item_date_depart']);
		if($newItem['rv_item_date_depart'] == -1) {
			$newItem['error'] = true;
			$newItem['error_message'] = "Invalid departure date ($newItem[rv_item_date_depart])";
			return $newItem;
		}
		$newItem['rv_item_date_depart'] = date("Y-m-d",$newItem['rv_item_date_depart']);
	}

	if($newItem['rv_item_date_depart'] <= $newItem['rv_item_date_arrive']) {
		$newItem['error'] = true;
		$newItem['error_message'] = "The departure date must be greater than the arrival date";
		return $newItem;
	}

	if($userStatusId < 2 && $newItem['rv_item_date_arrive'] < date("Y-m-d")) {
		$newItem['error'] = true;
		$newItem['error_message'] = "Booking may not be made for past dates";
	}	

	$rf_date_res_limit = $lDB->get("SELECT rf_date_res_limit FROM rf_default",4);
	if($newItem['rv_item_date_depart'] > $rf_date_res_limit) {
		$newItem['error'] = true;
		$newItem['error_message'] = "The date may not be past the current system limit of $rf_date_res_limit";
	}

	foreach ($newItem['rateGroups'] as $key => $rateGroup) {
		if(!is_numeric($rateGroup['qty'])) {
			$newItem['error'] = true;
			$newItem['error_message'] = "Invalid rate group quantity for ".$rateGroup['rt_rate_group_ix']."(".$rateGroup['qty'].")";
			return $newItem;
		}
	}
	

	$check = $lDB->get("
		SELECT
			COUNT(*)
		FROM
			sc_accomm
			INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = sc_accomm.ac_accomm_type_id
		WHERE
			sc_accomm.ac_accomm_type_id = '$newItem[ac_accomm_type_id]'
			AND sc_accomm.sc_group_id = '$GLOBALS[sc_group_id]'
			AND	ac_accomm_type.ac_accomm_type_inactive_yn=0
	",4);
	if($check < 1) {
		$newItem['error'] = true;
		$newItem['error_message'] = "Invalid accommodation type ($newItem[ac_accomm_type_id])";
		return $newItem;
	}	

	$maxpax = $lDB->get("
		SELECT
			ac_accomm_type.ac_accomm_max_capacity,
			ac_accomm_type.ac_accomm_max_adults 
		FROM
			ac_accomm_type
		WHERE
			ac_accomm_type_ix = '$newItem[ac_accomm_type_id]'
	",1);	
	if ($GLOBALS['userStatusId'] < 2) {
		if (getTotalAdults($newItem['rateGroups']) > $maxpax['ac_accomm_max_adults']) {
			$newItem['error'] = true;
			$newItem['error_message'] = "Adult capacity ($maxpax[ac_accomm_max_adults]) exceeded for accommodation type ($newItem[ac_accomm_type_id])";
			return $newItem;		
		}
		
		if (getTotalPax($newItem['rateGroups']) > $maxpax['ac_accomm_max_capacity']) {
			$newItem['error'] = true;
			$newItem['error_message'] = "Room capacity ($maxpax[ac_accomm_max_capacity]) exceeded for accommodation type ($newItem[ac_accomm_type_id])";
			return $newItem;		
		}
	}		

	if(!empty($newItem['override'])) {
		if ($GLOBALS['userStatusId'] > 1 && db_sc_group_get_user_setting("sc_grp_res_ovr_amount_yn") == "1") {
			// If Unit override, make sure the rate permits it
			if ($newItem['override']['level'] == 2){
				$rate = new Rate($newItem['ac_accomm_type_id'],$newItem['rv_item_date_arrive'],$newItem['rv_item_date_depart'],$newItem['rateGroups'],$rt_rate_type_id);
				$rate->ToPeriods();
				$canOverride = checkRateGroupOverride($rate->Periods[0]['rt_rate_ix'], $unitCode);
				if (!$canOverride){
					$newItem['error'] = true;
					$newItem['error_message'] = "Rate does not permit unit override.";
					return $newItem;					
				}
			}
		} else {
			$newItem['error'] = true;
			$newItem['error_message'] = "Access denied for override.";
			return $newItem;		
		}
	} 	

	if(!array_key_exists("guests", $newItem) || empty($newItem['guests'])) {
		$newItem['guests'] = [];
	}
	if(!is_array($newItem['guests'])) {
		$newItem['error'] = true;
		$newItem['error_message'] = "Invalid guests parameter. Guests should be defined as an outer array with inner arrays for each room containing a list of guests for that room.";
		return $newItem;
	}
	if(sizeof($newItem['guests']) > 0) {
		if(!is_array($newItem['guests'][0])) { // Add outer array if a single room of guests is sent
			$newItem['guests'] = [$newItem['guests']];
		}

		if(sizeof($newItem['guests']) > $newItem['rv_item_accomm_count']) {
			$newItem['error'] = true;
			$newItem['error_message'] = "Too many guest groups (" . sizeof($newItem['guests']) . ") for units booked (" . $newItem['rv_item_accomm_count'] . "). Guests should be defined as an outer array with inner arrays for each room containing a list of guests for that room.";
			return $newItem;
		}

		$totalPax = getTotalPax($newItem['rateGroups']);
		foreach($newItem['guests'] as $roomGuests) {
			if(sizeof($roomGuests) > $totalPax) {
				$newItem['error'] = true;
				$newItem['error_message'] = "Too many guests (" . sizeof($roomGuests) . ") in group for PAX ($totalPax). Guests should be defined as an outer array with inner arrays for each room containing a list of guests for that room.";
				return $newItem;
			}
			foreach($roomGuests as $pr_guest_id) {
				if(!db_pr_guest_exists($pr_guest_id)) {
					$newItem['error'] = true;
					$newItem['error_message'] = "Guest does not exist (" . $pr_guest_id . "). Guests should be defined as an outer array with inner arrays for each room containing a list of guests for that room.";
					return $newItem;
				}
			}
		}
	}

	return $newItem;	
}

function processGuests($rv_reservation_id, $items, $groups) {
	foreach($items as $rv_reservation_item_id) {
		foreach($groups as $group) {
			$rv_res_item_group_id = db_rv_res_item_group_insert($rv_reservation_item_id);
			foreach($group as $pr_guest_id) {
				db_rv_reservation_guest_insert($rv_reservation_id, $pr_guest_id);
				db_rv_res_item_guest_insert($pr_guest_id, $rv_res_item_group_id);
			}
		}
	}
}

function api_rf_get_source($search="") {
	global $api_instance;
	global $lDB;
	global $userStatusId;
	
	if ($userStatusId < 2){
		return $api_instance->Error("Access denied.");
	}

	$searchSQL = "";
	if(trim($search) != "") {
		$searchSQL = "
			AND rf_source.rf_source_desc LIKE '".addslashes(str_replace("*","%",$search))."'
		";
	}

	$list = $lDB->get("
		SELECT DISTINCT
			rf_source.rf_source_ix,
			rf_source.rf_source_desc
		FROM
			rf_source
		WHERE
			rf_source.rf_source_inactive_yn=0
			$searchSQL
	",2);

	$newList = array();
	foreach($list as $item) {
		array_push($newList,array('id'=>$item['rf_source_ix'],'name'=>$item['rf_source_desc']));
	}

	return $newList;
}

function api_rf_get_reservation_status($search="") {
	global $lDB;

	$searchSQL = "";
	if(trim($search) != "") {
		$searchSQL = "
			AND rf_reservation_status.rf_reservation_status_desc LIKE '".addslashes(str_replace("*","%",$search))."'
		";
	}

	$list = $lDB->get("
		SELECT DISTINCT
			rf_reservation_status.rf_reservation_status_id,
			rf_reservation_status.rf_reservation_status_desc
		FROM
			rf_reservation_status
		WHERE
			1
			$searchSQL
	",2);
	
	$newList = array();
	foreach($list as $item) {
		array_push($newList,array('id'=>$item['rf_reservation_status_id'],'name'=>$item['rf_reservation_status_desc']));
	}

	return $newList;
}

function api_rv_get_reservation($rv_reservation_id=false,$rv_res_name=false,$startDate=false,$endDate=false,$rf_reservation_status_id=false,$agent_id=false,$contact_id=false) {
	global $api_instance;
	global $lDB;
	global $userStatusId;
	$searchSQL = "";

	if(!empty($rv_reservation_id)) {
		$searchSQL .= " AND rv_reservation.rv_reservation_ix = '".addslashes($rv_reservation_id)."'";
	}	

	if(!empty($rv_res_name)) {
		$searchSQL .= " AND rv_reservation.rv_res_name LIKE '".addslashes(str_replace("*","%",$rv_res_name))."'";
	}		

	if(!empty($startDate) && !empty($endDate)) {
		$startDate = strtotime($startDate);
		if($startDate == -1) {
			return $api_instance->Error("Invalid start date");
		}
		$startDate = date("Y-m-d",$startDate);	
		
		$endDate = strtotime($endDate);
		if($endDate == -1) {
			return $api_instance->Error("Invalid end date");
		}
		$endDate = date("Y-m-d",$endDate);		
		
		$searchSQL .= "
				AND ( 
					(
						rv_reservation.rv_date_arrive >= '$startDate'
						AND rv_reservation.rv_date_arrive <= '$endDate'
					) OR (
						rv_reservation.rv_date_depart >= '$startDate'
						AND rv_reservation.rv_date_depart <= '$endDate'
					) OR (
						rv_reservation.rv_date_arrive <= '$startDate'
						AND rv_reservation.rv_date_depart >= '$endDate'
					)
				)
		";
	
	} else {
	
		if(!empty($startDate)) {
			$startDate = strtotime($startDate);
			if($startDate == -1) {
				return $api_instance->Error("Invalid start date");
			}
			$startDate = date("Y-m-d",$startDate);
			
			$searchSQL .= " AND rv_reservation.rv_date_depart > '$startDate'";
		}

		if(!empty($endDate)) {
			$endDate = strtotime($endDate);
			if($endDate == -1) {
				return $api_instance->Error("Invalid end date");
			}
			$endDate = date("Y-m-d",$endDate);
			
			$searchSQL .= " AND rv_reservation.rv_date_arrive < '$endDate'";
			
		}
	}
	
	if(!empty($rf_reservation_status_id)) {	
		if($rf_reservation_status_id != "0" && $rf_reservation_status_id != "10" && $rf_reservation_status_id != "20" && $rf_reservation_status_id != "30") {
			return $api_instance->Error("Invalid reservation status");
		} else {
			$searchSQL .= " AND rv_reservation.rf_reservation_status_id = '$rf_reservation_status_id'";
		}
	}
	
	if ($userStatusId > 1){
		if(!empty($agent_id)) {
			$searchSQL .= " AND rv_reservation.rv_agent_id = '".addslashes($agent_id)."'";	
		}
	} else {
		if(!empty($agent_id)) {
			return $api_instance->Error("Access denied for agent override.");
		}

		if(empty($GLOBALS['pr_agent_link'])) {
			return $api_instance->Error("Access denied for unlinked external user.");
		}
		
		$searchSQL .= " AND rv_reservation.rv_agent_id = '".addslashes($GLOBALS['pr_agent_link'])."'";
	}

	if(!empty($contact_id)) {
		$searchSQL .= " AND rv_reservation.rv_corr_persona_id = '".addslashes($contact_id)."'";
	}		

	$accommACL= $lDB->get("
		SELECT DISTINCT
			ac_accomm_type.ac_accomm_type_ix
		FROM
			sc_accomm, ac_accomm_type
		WHERE
			sc_accomm.sc_group_id= '" . $GLOBALS['sc_group_id'] . "'
			AND sc_accomm.ac_accomm_type_id = ac_accomm_type.ac_accomm_type_ix
	",3);
	
	require_once(__DIR__ . '/../functions.report.php');
	$accommFilter = useFilter("accomm",
		"0:10:15:20:30:40:50:90:95",
		"SELECT CONCAT(pr_business.pr_bus_name_short,CONCAT(' - ',ac_accomm_type.ac_accomm_desc)) FROM ac_accomm_type INNER JOIN pr_business ON pr_business.pr_business_id = ac_accomm_type.pr_business_id WHERE ac_accomm_type_ix =",
		"rv_reservation_item.ac_accomm_type_id",
		$accommACL,
		false,
		false
	);

	$list = $lDB->get("
		SELECT 
			DISTINCT(rv_reservation.rv_reservation_ix),
			rv_reservation.rv_res_name,
			rv_reservation.rv_date_arrive,
			rv_reservation.rv_date_depart,
			rv_reservation.rf_reservation_status_id,
			rv_reservation.rt_rate_type_id,
			rv_reservation.rv_agent_id,
			rv_reservation.rv_origin_agent_id,
			rv_reservation.rv_corr_persona_id,
			rv_reservation.rv_provision_expiry_date,
			rv_reservation.rf_source_ix
		FROM
			rv_reservation
			LEFT JOIN rv_reservation_item ON rv_reservation_item.rv_reservation_id = rv_reservation.rv_reservation_ix
		WHERE
			1
			$searchSQL
			$accommFilter[where]
	",2);
	
	$newList = array();
	foreach($list as $item) {
		array_push($newList,array(
			'id'=>$item['rv_reservation_ix'],
			'name'=>$item['rv_res_name'],
			'arrive_date'=>$item['rv_date_arrive'],
			'depart_date'=>$item['rv_date_depart'],
			'status'=>$item['rf_reservation_status_id'],
			'provisional_expiry_date'=>$item['rv_provision_expiry_date'],
			'rate_type'=>$item['rt_rate_type_id'],
			'agent'=>$item['rv_agent_id'],
			'origin'=>$item['rv_origin_agent_id'],
			'contact'=>$item['rv_corr_persona_id'],
			'source'=>$item['rf_source_ix']
		));
	}

	return $newList;	

}

function api_rv_get_note($rv_reservation_id) {
	global $api_instance;
	global $lDB;
	global $userStatusId;

	if ($userStatusId < 2) {
		return $api_instance->Error("Access denied.");
	}

	if(!db_rv_reservation_exists($rv_reservation_id)) {
		return $api_instance->Error("Reservation does not exist.");
	}
	
	$rawNotes = db_rv_note_get_by_reservation($rv_reservation_id);

	$notes = array(
		'note'=>$rawNotes['rv_note_general'],
		'memo'=>$rawNotes['rv_note_internal'],
		'guest'=>$rawNotes['rv_note_guests']
	);

	if(db_rf_country_exists($rawNotes['rf_country_id'])) {
		$notes['country_id'] = $rawNotes['rf_country_id'];
		$notes['country_name'] = $lDB->get("SELECT rf_country_name FROM rf_country WHERE rf_country_ix = '$rawNotes[rf_country_id]'",4);
	} else {
		$notes['country_id'] = "";
		$notes['country_name'] = "";
	}

	return $notes;	
}

function api_rv_update_note($rv_reservation_id,$rv_note_general,$rv_note_guests=false,$rv_note_internal=false,$rf_country_id=false) {
	global $api_instance;
	global $lDB;
	global $userStatusId;

	if ($userStatusId < 2) {
		return $api_instance->Error("Access denied.");
	}

	if(!db_rv_reservation_exists($rv_reservation_id)) {
		return $api_instance->Error("Reservation does not exist.");
	}

	$canEditReservation = canEditReservation($rv_reservation_id,true,137);
	$shared = $lDB->get("SELECT rf_share_notes_yn FROM rf_default",4);

	if(!$canEditReservation && $shared != "1") {
		return $api_instance->Error("Note edit access denied.");
	}

	if(!empty($rf_country_id) && !db_rf_country_exists($rf_country_id)) {
		return $api_instance->Error("Country does not exist.");
	}

	$rv_note_id = db_rv_note_insert($rv_reservation_id,$rf_country_id,false,$rv_note_general,$rv_note_guests,$rv_note_internal);

	ammendReservation($rv_reservation_id,"Notes");

	return $rv_note_id;
}

function api_rv_update_status(
	$rv_reservation_id,
	$rf_reservation_status_id,
	$rv_provision_expiry_date=false,
	$rv_cancel_reason=false,
	$allowOverbooking=false,
	$sendModificationRequest=false,
	$customSubject='',
	$customModificationText=''
) {
	global $api_instance;
	global $lDB;
	global $userStatusId;
	global $userid;


	$canChangeStatuses = canChangeStatus($rv_reservation_id, $rf_reservation_status_id);
	/*
	 * Currently open to all but user access controls will be added later to restrict
	 * who can send modification requests.
	 */
	$canSendModificationRequest = !canEditReservation($rv_reservation_id) || !$canChangeStatuses;

	if ($userStatusId < 1) {
		return $api_instance->Error("Access denied.");
	}

	if ($userStatusId == 1 && !canAgentAccessReservation($rv_reservation_id, $userid)) {
		return $api_instance->Error("Access denied.");
	}

	if(!db_rv_reservation_exists($rv_reservation_id)) {
		return $api_instance->Error("Reservation does not exist.");
	}

	if ($canSendModificationRequest && $sendModificationRequest) {
		$body = '';
		if (!empty($customModificationText)) {
			$body .= $customModificationText . "\n\n";
		}

		$body .= "Status change requested for reservation $rv_reservation_id: ";
		$requestedStatus = (int) $rf_reservation_status_id;
		switch ($requestedStatus) {
			case 0:
				$body .= 'Quotation';
				break;
			case 10:
				$body .= 'Wait listed';
				break;
			case 15:
				$body .= 'Allocation';
				break;
			case 20:
				$body .= 'Provisional';
				break;
			case 25:
				$body .= 'In progress';
				break;
			case 30:
				$body .= 'Confirmed';
				break;
			case 40:
				$body .= 'Invoiced';
				break;
			case 50:
				$body .= 'Blocked';
				break;
			case 90:
				$body .= 'Cancelled';
				break;
			case 95:
				$body .= 'Deleted';
				break;
			case 97:
				$body .= 'Deleted allocation';
				break;
		}
		$body .= "\n";

		if ($requestedStatus == 90 && $rv_cancel_reason) {
			$body .= 'Cancellation reason: ' . $rv_cancel_reason;
		}

		$contactName = agentName(getAgent($userid));
		$principalEmail = $_SESSION['principal_email'];
		$userEmail = getContactDetails($userid)['email_address'];

		$to = $principalEmail;
		$from = $userEmail;
		$subject = "$contactName - Modification request for reservation $rv_reservation_id";
		if (!empty($customSubject)) {
			$subject = $customSubject;
		}
		$headers = "From: ".$contactName."<$from>\r\n";

		$error = email($from, $to, $subject, $headers, $body, 0, 0, $rv_reservation_id, $contactName);
		if (!$error) {
            $flagModRequest = "UPDATE rv_reservation
            SET rv_mod_request_yn = 1, rv_mod_request_date = NOW()
            WHERE rv_reservation_ix = '$rv_reservation_id'";
			$lDB->put($flagModRequest);
			if ($canChangeStatuses) {
				return ['success' => 'A modification request has been sent.'];
			} else {
				return ['success' => 'You do not have access to change statuses but a modification request has been sent.'];
			}
		} else {
			return $api_instance->Error("Unabled to send modification request");
		}
	}

	if(!canEditReservation($rv_reservation_id)) {
		return $api_instance->Error("Access denied.");
	}

	if($rf_reservation_status_id != "0" && $rf_reservation_status_id != "10" && $rf_reservation_status_id != "20" && $rf_reservation_status_id != "30" && $rf_reservation_status_id != "90") {
		return $api_instance->Error("Invalid reservation status");
	}

	if($rf_reservation_status_id != DB_RF_RESERVATION_STATUS_PROVISIONAL && $lDB->get("SELECT rf_reservation_status_id FROM rv_reservation WHERE rv_reservation_ix = '$rv_reservation_id'",4) == $rf_reservation_status_id) {
		return $api_instance->Error("Reservation status already set to requested status.");
	}

	if(!canChangeStatus($rv_reservation_id,$rf_reservation_status_id)) {
		return $api_instance->Error("You do not have the required access to change to this status.");
	}

	$statuses = allowStatuses($rv_reservation_id,$allowOverbooking, true); // 3rd param: ignore allocated rooms, to make OTA cancellations work

	if(!$statuses[$rf_reservation_status_id]['allow']) {
		return $api_instance->Error("Status change not allowed: " . join(", ", $statuses[$rf_reservation_status_id]['reasons']));
	}

	if($rf_reservation_status_id == DB_RF_RESERVATION_STATUS_PROVISIONAL && !$statuses[DB_RF_RESERVATION_STATUS_PROVISIONAL]['allow_date']) {
		return $api_instance->Error("Status change not allowed: " . join(", ", $statuses[$rf_reservation_status_id]['reasons']));
	}

	// Non-stock-holding status, remove any rooming information
	if (in_array($rf_reservation_status_id, array(0, 10, 90))) {
		removeRoomingInfo($rv_reservation_id);
	}

	db_rv_reservation_set_status($rv_reservation_id,$rf_reservation_status_id,$rv_provision_expiry_date,$rv_cancel_reason, false);

	if ($userStatusId == 1) {
		// Flag edit by agent
		$flagMod = "UPDATE rv_reservation
			SET rv_mod_yn = 1, rv_mod_date = NOW()
			WHERE rv_reservation_ix = '$rv_reservation_id'";
		$lDB->put($flagMod);
	}

	return true;
}

function api_rv_get_itinerary($rv_reservation_id) {
	global $api_instance;
	global $lDB;
	global $userStatusId;
	global $userid;

	if ($userStatusId < 1) {
		return $api_instance->Error("Access denied.");
	}

	if ($userStatusId == 1 && !canAgentAccessReservation($rv_reservation_id, $userid)) {
		return $api_instance->Error("Access denied.");
	}

	if(!db_rv_reservation_exists($rv_reservation_id)) {
		return $api_instance->Error("Reservation does not exist.");
	}

	$list = $lDB->get("
		SELECT
			rv_reservation_item.rv_reservation_item_ix AS 'id',
			rv_reservation_item.pr_business_id AS 'property',
			rv_reservation_item.ac_accomm_type_id AS 'accommodation',
			rv_reservation_item.rv_item_date_arrive AS 'arrive_date',
			rv_reservation_item.rv_item_date_depart AS 'depart_date',
			rv_reservation_item.rt_rate_type_id AS 'rate_type'
			/* rate typeinvoiced? allocations? variances? overridden? rate groups?*/
		FROM
			rv_reservation_item
		WHERE
			rv_reservation_item.rv_reservation_id = '$rv_reservation_id'
	",6);

	foreach($list as $key=>$item) {
		$rateGroups = $lDB->get("
			SELECT
				rv_res_item_rate_grp.rt_rate_group_id AS 'id',
				rv_res_item_rate_grp.rv_res_item_rate_grp_count AS 'qty'
			FROM
				rv_res_item_rate_grp
			WHERE
				rv_res_item_rate_grp.rv_reservation_item_id = '$item[id]'
		",6);
		$list[$key]['rate_groups'] = $rateGroups;
	}

	return $list;
}


//1.	Add a new method called rv_create_itinerary with parameters username, password, principal (bridge only), reservation id, and an array of items to add (format the same as parameter 4 of rv_create). This function should work in an all or nothing manner - in other words the stock object should be used to check that all itineraries are available and only process them if they are (again this code will be similar to rv_create and can probably be moved into a function shared by both methods). This function should just return a success or failure code. Only for internal users.
function api_rv_create_itinerary($rv_reservation_id,$items,$allowOverbooking=false) {
	global $api_instance;
	global $lDB;
	global $userStatusId;
	global $userid;

	if ($userStatusId < 1){
		return $api_instance->Error("Access denied.");
	}	

	if ($userStatusId == 1 && !canAgentAccessReservation($rv_reservation_id, $userid)) {
		return $api_instance->Error("Access denied.");
	}
	
	/*
	 * $items format (NEW):
	 *
	 * 0/'accommodation_type'
	 * 1/'date_arrive'
	 * 2/'date_depart'
	 * 3/array(array(),array())
	 * 4/override array
	 * 5/rate type
	 * 6/'use_allocations'
	 */	 	

	if (!db_rv_reservation_exists($rv_reservation_id)){
		return $api_instance->Error("Reservation does not exist.");
	} else {
		$reservation = $lDB->get("
			SELECT 
				rv_agent_id,
				rt_rate_type_id,
				rf_reservation_status_id
			FROM 
				rv_reservation 
			WHERE 
				rv_reservation_ix = '$rv_reservation_id'
		",1);
	}

	if(!is_array($items)) {
		return $api_instance->Error("Invalid itinerary specified.");
	}
	if(sizeof($items) < 1) {
		return $api_instance->Error("No itinerary specified.");
	}
	foreach($items as $key=>$item) {
		$newItem = processItem($item,$reservation['rt_rate_type_id'],$reservation['rv_agent_id']);
		if($newItem['error']) {
			return $api_instance->Error($newItem['error_message']);
		}	

		$allocations = array();
		if($reservation['rf_reservation_status_id'] != "0" && $reservation['rf_reservation_status_id'] != "10") {
			$dateArray = explode("-",$newItem['rv_item_date_depart']);
			$dateDepart = date("Y-m-d",mktime(0,0,0,$dateArray[1],$dateArray[2]-1,$dateArray[0]));
			
			$stock = new Stock($newItem['ac_accomm_type_id'],$newItem['rv_item_date_arrive'],$dateDepart,false,$reservation['rv_agent_id']);

			if($newItem['auto_allocate']) {
				$rawAllocations = $stock->ToReservationAllocation(false, true);
				foreach($rawAllocations as $day) {
					foreach($day as $allocation) {
						if(!$allocation['expired'] && $allocation['allocation_count'] > 0) {
							$allocations[] = $allocation['rv_reservation_ix'];
						}
					}
				}
				$allocations = array_unique($allocations);
			}			
		}
		$newItem['allocations'] = $allocations;

		$items[$key] = $newItem;
	}	 
	
	if(
		$reservation['rf_reservation_status_id'] != "0"
		&& $reservation['rf_reservation_status_id'] != "10"
		&& (!$allowOverbooking || db_sc_group_get_user_setting("sc_grp_res_ovr_overbooking_yn") != "1")
		&& !check_multi_item_availability($items)
	) {
		return $api_instance->Error("Not enough stock available.");
	}

	foreach($items as $item) {
		$itemDataRaw = convertItemToRaw($item['ac_accomm_type_id'],$item['rv_item_date_arrive'],$item['rv_item_date_depart'],$item['rateGroups']);

		$createItems = calCreateItem($rv_reservation_id, $itemDataRaw, $item['rt_rate_type_id'], "defaultOnly", $item['override'], false, "0", false, $item['allocations'], false, false, false, false, false, $item['rv_item_comm_perc']);
		if(sizeof($item['guests']) > 0) {
			processGuests($rv_reservation_id, $createItems, $item['guests']);
		}
	}

	if (!empty($createItems)) {
		ammendReservation($rv_reservation_id,"Itinerary (Add)");
		return true;
	} else {
		return false;
	}
	
}

function api_rv_can_edit_reservation($rv_reservation_id) {
	global $userStatusId;
	global $userid;

	if (!db_rv_reservation_exists($rv_reservation_id)){
		return $api_instance->Error("Reservation does not exist.");
	}

	if ($userStatusId == 1 && !canAgentAccessReservation($rv_reservation_id, $userid)) {
		return $api_instance->Error("Access denied.");
	}

	$canEditReservation = canEditReservation($rv_reservation_id);
	$canEditResDetails = canEditReservationDetails();
	$canEditItineraries = canEditItineraries();

	$result = [
		'can_edit' => $canEditReservation,
		'editable_items' => []
	];

	if ($canEditResDetails) {
		array_push($result['editable_items'], 'Res Details');
	}

	if ($canEditItineraries) {
		array_push($result['editable_items'], 'Itineraries');
	}

	$canChangeToHoldStock = db_sc_group_get_user_setting("sc_grp_res_status_unalloc_yn") == "1";
	$canChangeToNotHoldStock = db_sc_group_get_user_setting("sc_grp_res_status_alloc_yn") == "1";

	if($canChangeToHoldStock || $canChangeToNotHoldStock) {
		array_push($result['editable_items'], 'Status');
	}

	if (count($result['editable_items']) == 0) {
		$result['can_edit'] = false;
	}

	return $result;
}

function api_rv_update_itinerary(
	$rv_reservation_id,
	$oldItems,
	$newItems,
	$allowOverbooking=false,
	$sendModificationRequest=false,
	$customSubject='',
	$customModificationText=''
) {
	global $api_instance;
	global $userStatusId;
	global $userid;
	global $lDB;

	/*
	 * Currently open to all but user access controls will be added later to restrict
	 * who can send modification requests.
	 */
	$canSendModificationRequest = !canEditReservation($rv_reservation_id) || !canEditItineraries();

	if($userStatusId < 1) {
		return $api_instance->Error("Access denied.");
	}

	if ($userStatusId == 1 && !canAgentAccessReservation($rv_reservation_id, $userid)) {
		return $api_instance->Error("Access denied.");
	}

	if (!db_rv_reservation_exists($rv_reservation_id)){
		return $api_instance->Error("Reservation does not exist.");
	} else {
		$reservation = $lDB->get("
			SELECT 
				rv_agent_id,
				rt_rate_type_id,
				rf_reservation_status_id
			FROM 
				rv_reservation 
			WHERE 
				rv_reservation_ix = '$rv_reservation_id'
		",1);
	}

	if(!is_array($oldItems)) {
		return $api_instance->Error("Invalid old itineraries specified.");
	}
	foreach($oldItems as $rv_reservation_item_id) {
		if(!db_rv_reservation_item_exists($rv_reservation_item_id)) {
			return $api_instance->Error("Itinerary $rv_reservation_item_id does not exist.");
		}

		if($lDB->get("
			SELECT
				rv_reservation_item.rv_reservation_id
			FROM
				rv_reservation_item
			WHERE
				rv_reservation_item.rv_reservation_item_ix = '$rv_reservation_item_id'
		",4) != $rv_reservation_id) {
			return $api_instance->Error("Itinerary $rv_reservation_item_id does not belong to reservation $rv_reservation_id.");
		}

		if(!(canEditReservation($rv_reservation_id,true,2,$rv_reservation_item_id && canEditItineraries()))) {
			if (!$canSendModificationRequest || ($canSendModificationRequest && !$sendModificationRequest)) {
				return $api_instance->Error("Reservation edit access denied.");
			}
		}
	}

	if(!is_array($newItems)) {
		return $api_instance->Error("Invalid new itineraries specified.");
	}
	if(sizeof($newItems) < 1) {
		return $api_instance->Error("No itinerary specified.");
	}

	if ($canSendModificationRequest && $sendModificationRequest) {
		if (count($oldItems) != count($newItems)) {
			return $api_instance->Error("Number of new itineraries don't match the number of old itineraries.");
		}

		$diff = generateItineraryDifference($rv_reservation_id, $oldItems, $newItems);
		$body = '';
		if (!empty($customModificationText)) {
			$body .= $customModificationText . "\n\n";
		}

		$body .= "Itinerary changes requested for reservation $rv_reservation_id:\n\n";
		$body .= $diff;

		global $userid;
		$contactName = agentName(getAgent($userid));
		$principalEmail = $_SESSION['principal_email'];
		$userEmail = getContactDetails($userid)['email_address'];

		$to = $principalEmail;
		$from = $userEmail;
		$subject = "$contactName - Modification request for reservation $rv_reservation_id";
		if (!empty($customSubject)) {
			$subject = $customSubject;
		}
		$headers = "From: ".$contactName."<$from>\r\n";

		$error = email($from, $to, $subject, $headers, $body, 0, 0, $rv_reservation_id, $contactName);
		if (!$error) {
            $flagModRequest = "UPDATE rv_reservation
            SET rv_mod_request_yn = 1, rv_mod_request_date = NOW()
            WHERE rv_reservation_ix = '$rv_reservation_id'";
			$lDB->put($flagModRequest);

			if (canEditItineraries()) {
				return ['success' => 'A modification request has been sent.'];
			} else {
				return ['success' => 'You do not have access to edit itineraries but a modification request has been sent.'];
			}
		} else {
			return $api_instance->Error("Unabled to send modification request");
		}
	}

	foreach($newItems as $key=>$item) {
		$newItem = processItem($item,$reservation['rt_rate_type_id'],$reservation['rv_agent_id']);
		if($newItem['error']) {
			return $api_instance->Error($newItem['error_message']);
		}	

		$allocations = array();
		if($reservation['rf_reservation_status_id'] != "0" && $reservation['rf_reservation_status_id'] != "10") {
			$dateArray = explode("-",$newItem['rv_item_date_depart']);
			$dateDepart = date("Y-m-d",mktime(0,0,0,$dateArray[1],$dateArray[2]-1,$dateArray[0]));
			
			$stock = new Stock($newItem['ac_accomm_type_id'],$newItem['rv_item_date_arrive'],$dateDepart,false,$reservation['rv_agent_id']);

			if($newItem['auto_allocate']) {
				$rawAllocations = $stock->ToReservationAllocation();
				foreach($rawAllocations as $day) {
					foreach($day as $allocation) {
						if(!$allocation['expired'] && $allocation['allocation_count'] > 0) {
							$allocations[] = $allocation['rv_reservation_ix'];
						}
					}
				}
				$allocations = array_unique($allocations);
			}			
		}
		$newItem['allocations'] = $allocations;

		$newItems[$key] = $newItem;
	}	 

	if(
		$reservation['rf_reservation_status_id'] != "0"
		&& $reservation['rf_reservation_status_id'] != "10"
		&& (!$allowOverbooking || db_sc_group_get_user_setting("sc_grp_res_ovr_overbooking_yn") != "1")
		&& !check_multi_item_availability($newItems,$oldItems)
	) {
		return $api_instance->Error("Not enough stock available.");
	}

	$createdItems = array();
	foreach($newItems as $item) {
		$itemDataRaw = convertItemToRaw($item['ac_accomm_type_id'],$item['rv_item_date_arrive'],$item['rv_item_date_depart'],$item['rateGroups']);

		$newItemIds = calCreateItem($rv_reservation_id, $itemDataRaw, $item['rt_rate_type_id'], "defaultOnly", $item['override'], false, "0", false, $item['allocations'], false, false, false, false, false, $item['rv_item_comm_perc']);
		if(sizeof($item['guests']) > 0) {
			processGuests($rv_reservation_id, $newItemIds, $item['guests']);
		}
		$createdItems = array_merge($createdItems,$newItemIds);
	}

	if(sizeof($createdItems) < 1) {
		return $api_instance->Error("Itinerary update failed.");
	}

	foreach($oldItems as $rv_reservation_item_id) {
		calDeleteItem($rv_reservation_item_id,$rv_reservation_id);
	}

	ammendReservation($rv_reservation_id,"Itinerary (Edit)");

	if ($userStatusId == 1) {
		// Flag edit by agent
		$flagMod = "UPDATE rv_reservation
            SET rv_mod_yn = 1, rv_mod_date = NOW()
            WHERE rv_reservation_ix = '$rv_reservation_id'";
			$lDB->put($flagMod);
	}

	return true;
}

/**
 * Generates a diff showing the difference between old itinerary items and new ones.
 *
 * @param string $resId The id of the reservation.
 * @param array $oldItems The old itinerary items.
 * @param array $newItems The new itinerary items.
 * @return string The itinerary diff.
 */
function generateItineraryDifference($resId, $oldItems, $newItems) {
	global $lDB;

	$nItems = count($oldItems);
	$diff = '';

	for ($i = 0; $i<$nItems; $i++) {
		$sql = "
			SELECT
				rv_reservation_item.rv_item_date_arrive,
				rv_reservation_item.rv_item_date_depart,
				ac_accomm_type.ac_accomm_desc,
				pr_persona.pr_name_last,
				rv_reservation_item.rv_item_nights,
				rv_reservation_item.rv_item_accomm_count,
				rv_reservation_item.rv_item_adult_count,
				rv_reservation_item.rv_item_child_count
			FROM rv_reservation_item
			INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = rv_reservation_item.ac_accomm_type_id
			INNER JOIN pr_persona ON pr_persona.pr_persona_ix = ac_accomm_type.pr_business_id
			WHERE rv_reservation_item.rv_reservation_item_ix = '".$oldItems[$i]."'
		";
		$itinerary = $lDB->get($sql, 1);
		$diff .= '-------------------------------------------------------------'. "\n";
		$diff .= 'Itinerary id: ' . $oldItems[$i] ."\n";
		$diff .= 'Arrive date: ' . $itinerary['rv_item_date_arrive'] . "\n";
		$diff .= 'Depart date: ' . $itinerary['rv_item_date_depart'] . "\n";
		$diff .= 'Property: ' . $itinerary['pr_name_last'] . "\n";
		$diff .= 'Accommodation: ' . $itinerary['ac_accomm_desc'] . "\n";
		$diff .= 'Nights: ' . $itinerary['rv_item_nights'] . "\n";
		$diff .= 'Rooms: ' . $itinerary['rv_item_accomm_count'] . "\n";
		$diff .= 'Adult: ' . $itinerary['rv_item_adult_count'] . "\n";
		$diff .= 'Child: ' . $itinerary['rv_item_child_count'] . "\n\n";
		
		$diff .= '** Changes Requested **' . "\n";

		$sql = "
			SELECT
				ac_accomm_type.ac_accomm_desc,
				pr_persona.pr_name_last
			FROM ac_accomm_type
			INNER JOIN pr_persona ON pr_persona.pr_persona_ix = ac_accomm_type.pr_business_id
			WHERE ac_accomm_type_ix = '" . $newItems[$i][0] ."'
		";
		$newLocation = $lDB->get($sql, 1);

		if ($itinerary['rv_item_date_arrive'] !== $newItems[$i][1]) {
			$diff .= "Arrive date: " . $newItems[$i][1] . "\n";
		}
		if ($itinerary['rv_item_date_depart'] !== $newItems[$i][2]) {
			$diff .= "Depart date: " . $newItems[$i][2] . "\n";
		}
		if ($itinerary['pr_name_last'] !== $newLocation['pr_name_last']) {
			$diff .= 'Property: ' . $newLocation['pr_name_last'] . "\n";
		}
		if ($itinerary['ac_accomm_desc'] !== $newLocation['ac_accomm_desc']) {
			$diff .= 'Accommodation: ' . $newLocation['ac_accomm_desc'] . "\n";
		}

		$rooms = 0;
		$adults = 0;
		$children = 0;

		foreach ($newItems[$i][3] as $rateGroup) {
			$type = $lDB->get("
				SELECT rt_rate_group_class
				FROM rt_rate_group
				WHERE rt_rate_group_ix = '". $rateGroup[0] ."'
			", 4);
			if ($type == 0) {
				$rooms += $rateGroup[1];
			} else if($type == 1) {
				$adults += $rateGroup[1];
			} else {
				$children += $rateGroup[1];
			}
		}

		if ($rooms == 0) {
			// Can't have no rooms
			$rooms = 1;
		}

		if ($itinerary['rv_item_accomm_count'] != $rooms) {
			$diff .= 'Rooms: ' . $rooms . "\n";
		}
		if ($itinerary['rv_item_adult_count'] != $adults) {
			$diff .= 'Adult: ' . $adults . "\n";
		}
		if ($itinerary['rv_item_child_count'] != $children) {
			$diff .= 'Child: ' . $children . "\n";
		}

		$diff .= '-------------------------------------------------------------'. "\n\n\n";
	}
	return $diff;
}

function api_rv_delete_itinerary(
	$rv_reservation_id,
	$rv_reservation_item_id,
	$sendModificationRequest=false,
	$customSubject='',
	$customBody=''
) {
	global $api_instance;
	global $userStatusId;
	global $userid;
	global $lDB;

	if ($userStatusId < 1){
		return $api_instance->Error("Access denied.");
	}	

	if ($userStatusId == 1 && !canAgentAccessReservation($rv_reservation_id, $userid)) {
		return $api_instance->Error("Access denied.");
	}

	if (!db_rv_reservation_exists($rv_reservation_id)){
		return $api_instance->Error("Reservation does not exist.");
	}

	if(!db_rv_reservation_item_exists($rv_reservation_item_id)) {
		return $api_instance->Error("Itinerary does not exist.");
	}

	if($rv_reservation_id != $lDB->get("
		SELECT rv_reservation_id FROM rv_reservation_item WHERE rv_reservation_item.rv_reservation_item_ix = '$rv_reservation_item_id'
	",4)) {
		return $api_instance->Error("Itinerary $rv_reservation_item_id does not belong to reservation $rv_reservation_id.");
	}		

	/*
	 * Currently open to all but user access controls will be added later to restrict
	 * who can send modification requests.
	 */
	$canSendModificationRequest = !canEditReservation($rv_reservation_id) || !canEditItineraries();

	if ($canSendModificationRequest && $sendModificationRequest) {
		$body = '';
		
		if (!empty($customBody)) {
			$body .= $customBody . "\n\n";
		}

		$body .= "Deletion requested for reservation $rv_reservation_id:\n\n";
		$diff = '';
		$sql = "
			SELECT
				rv_reservation_item.rv_item_date_arrive,
				rv_reservation_item.rv_item_date_depart,
				ac_accomm_type.ac_accomm_desc,
				pr_persona.pr_name_last,
				rv_reservation_item.rv_item_nights,
				rv_reservation_item.rv_item_accomm_count,
				rv_reservation_item.rv_item_adult_count,
				rv_reservation_item.rv_item_child_count
			FROM rv_reservation_item
			INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = rv_reservation_item.ac_accomm_type_id
			INNER JOIN pr_persona ON pr_persona.pr_persona_ix = ac_accomm_type.pr_business_id
			WHERE rv_reservation_item.rv_reservation_item_ix = '".$rv_reservation_item_id."'
		";
		$itinerary = $lDB->get($sql, 1);
		$diff .= '-------------------------------------------------------------'. "\n";
		$diff .= 'Itinerary id: ' . $rv_reservation_item_id ."\n";
		$diff .= 'Arrive date: ' . $itinerary['rv_item_date_arrive'] . "\n";
		$diff .= 'Depart date: ' . $itinerary['rv_item_date_depart'] . "\n";
		$diff .= 'Property: ' . $itinerary['pr_name_last'] . "\n";
		$diff .= 'Accommodation: ' . $itinerary['ac_accomm_desc'] . "\n";
		$diff .= 'Nights: ' . $itinerary['rv_item_nights'] . "\n";
		$diff .= 'Rooms: ' . $itinerary['rv_item_accomm_count'] . "\n";
		$diff .= 'Adult: ' . $itinerary['rv_item_adult_count'] . "\n";
		$diff .= 'Child: ' . $itinerary['rv_item_child_count'] . "\n\n";
		$body .= $diff;

		global $userid;
		$contactName = agentName(getAgent($userid));
		$principalEmail = $_SESSION['principal_email'];
		$userEmail = getContactDetails($userid)['email_address'];

		$to = $principalEmail;
		$from = $userEmail;
		$subject = "$contactName - Deletion of itinerary request for reservation $rv_reservation_id";
		if (!empty($customSubject)) {
			$subject = $customSubject;
		}
		$headers = "From: ".$contactName."<$from>\r\n";

		$error = email($from, $to, $subject, $headers, $body, 0, 0, $rv_reservation_id, $contactName);
		if (!$error) {
            $flagModRequest = "UPDATE rv_reservation
            SET rv_mod_request_yn = 1, rv_mod_request_date = NOW()
            WHERE rv_reservation_ix = '$rv_reservation_id'";
			$lDB->put($flagModRequest);

			if (canEditItineraries()) {
				return ['success' => 'A modification request has been sent.'];
			} else {
				return ['success' => 'You do not have access to edit itineraries but a modification request has been sent.'];
			}
		} else {
			return $api_instance->Error("Unabled to send modification request");
		}
	}

	if(!(canEditReservation($rv_reservation_id,true,2,$rv_reservation_item_id) && canEditItineraries())) {
		return $api_instance->Error("Reservation edit access denied.");
	}

	$count = $lDB->get("SELECT COUNT(*) FROM rv_reservation_item WHERE rv_reservation_item.rv_reservation_id = '$rv_reservation_id'",4);
	if($count < 2) {
		return $api_instance->Error("Deletion of the last itinerary on a reservation is not allowed.");
	}

	calDeleteItem($rv_reservation_item_id,$rv_reservation_id);

	if ($userStatusId == 1) {
		// Flag edit by agent
		$flagMod = "UPDATE rv_reservation
            SET rv_mod_yn = 1, rv_mod_date = NOW()
            WHERE rv_reservation_ix = '$rv_reservation_id'";
			$lDB->put($flagMod);
	}

	return true;
}

function api_rf_get_booking_info() {
	global $lDB;

	$terms = $lDB->get("SELECT rf_booking_info FROM rf_custom",4);

	$search = array (
		"'<script[^>]*?>.*?</script>'si",  // Strip out javascript
		"'<[\/\!]*?[^<>]*?>'si",           // Strip out html tags
		"'([\r\n])[\s]+'",                 // Strip out white space
		"'&(quot|#34);'i",                 // Replace html entities
		"'&(amp|#38);'i",
		"'&(lt|#60);'i",
		"'&(gt|#62);'i",
		"'&(nbsp|#160);'i",
		"'&(iexcl|#161);'i",
		"'&(cent|#162);'i",
		"'&(pound|#163);'i",
		"'&(copy|#169);'i"
	);
	$replace = array (
		"",
		"",
		"\\1",
		"\"",
		"&",
		"<",
		">",
		" ",
		chr(161),
		chr(162),
		chr(163),
		chr(169)
	);

	//TODO: Replace with str_ireplace in PHP5
	$terms =
			str_replace("</div>","\n",
			str_replace("</DIV>","\n",
			str_replace("</p>","\n",
			str_replace("</P>","\n",
			str_replace("<br>","\n",
			str_replace("<BR>","\n",
		urldecode($terms)))))));
	$terms = preg_replace ($search, $replace, $terms);	
	$terms = preg_replace_callback(                                  
		"'&#(\d+);'",
		function($matches) {
			return chr($matches[1]);
		},
		$terms
	);

	return $terms;
}

function api_rv_request_cancel($rv_reservation_id, $reference=false, $reason=false, $from=false, $to=false) {
	global $api_instance;
	global $lDB;

	if (!db_rv_reservation_exists($rv_reservation_id)){
		return $api_instance->Error("Reservation does not exist.");
	}

	$result = requestCancelEmail($rv_reservation_id,false,$reference,$reason,$from,$to);
	ammendReservation($rv_reservation_id,"Cancellation email for reservation: $result[message]");

	if($result['error']) {
		return $api_instance->Error("Error sending email: $result[message]");
	}

	return true;
}

function api_rv_request_cancel_itinerary($rv_reservation_id, $rv_reservation_item_id, $reference=false, $reason=false, $from=false, $to=false) {
	global $api_instance;
	global $lDB;

	if (!db_rv_reservation_exists($rv_reservation_id)){
		return $api_instance->Error("Reservation does not exist.");
	}

	if(!db_rv_reservation_item_exists($rv_reservation_item_id)) {
		return $api_instance->Error("Itinerary does not exist.");
	}

	if($rv_reservation_id != $lDB->get("
		SELECT rv_reservation_id FROM rv_reservation_item WHERE rv_reservation_item.rv_reservation_item_ix = '$rv_reservation_item_id'
	",4)) {
		return $api_instance->Error("Itinerary $rv_reservation_item_id does not belong to reservation $rv_reservation_id.");
	}		

	$result = requestCancelEmail($rv_reservation_id,$rv_reservation_item_id,$reference,$reason,$from,$to);
	ammendReservation($rv_reservation_id,"Cancellation email for itinerary ($rv_reservation_item_id): $result[message]");

	if($result['error']) {
		return $api_instance->Error("Error sending email: $result[message]");
	}
	return true;
}

function requestCancelEmail($rv_reservation_id, $rv_reservation_item_id=false, $reference=false, $reason=false, $from=false, $to=false) {
	global $lDB;

	if(empty($reference)) {
		$reference = $lDB->get("SELECT rv_agent_ref FROM rv_reservation WHERE rv_reservation_ix = '$rv_reservation_id'",4);
	}

	if(empty($reference)) {
		$reference = $rv_reservation_id;
	}

	if(empty($reason)) {
		$reason = "No reason provided";
	}

	$emailTemplate = join("",file(__DIR__ . "/../request_cancel_email.txt"));
	$itemTemplate = join("",file(__DIR__ . "/../request_cancel_email_item.txt"));

	if(empty($rv_reservation_item_id)) {
		$where = "
			rv_reservation_item.rv_reservation_id = '$rv_reservation_id'	
		";
	} else {
		$where = "
			rv_reservation_item.rv_reservation_item_ix = '$rv_reservation_item_id'
		";
	}

	$rawItems = $lDB->get("
		SELECT
			rv_reservation_item.rv_item_date_arrive,
			rv_reservation_item.rv_item_date_depart,
			rv_reservation_item.rv_item_accomm_count,
			rv_reservation_item.rv_item_adult_count,
			rv_reservation_item.rv_item_child_count,
			pr_persona.pr_name_last AS pr_business_name,
			ac_accomm_type.ac_accomm_desc
		FROM
			rv_reservation_item
			INNER JOIN pr_persona ON pr_persona.pr_persona_ix = rv_reservation_item.pr_business_id
			INNER JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = rv_reservation_item.ac_accomm_type_id
		WHERE
			$where
		ORDER BY 
			rv_reservation_item.rv_item_date_arrive,
			rv_reservation_item.rv_item_date_depart,
			pr_persona.pr_name_last,
			ac_accomm_type.ac_accomm_desc
	",2);
	$items = "";
	foreach($rawItems as $item) {
		$item['rv_item_date_arrive_fmt'] = chng_date($item['rv_item_date_arrive'],"-");
		$item['rv_item_date_depart_fmt'] = chng_date($item['rv_item_date_depart'],"-");
		$items .= showpage($item,$itemTemplate);
	}

	$emailInfo = array_merge(array(
		'reference'=>$reference,
		'reason'=>$reason,
		'now'=>date("Y-m-d H:i:s T"),
		'items'=>$items
	),$lDB->get("
		SELECT
			rv_reservation.rv_reservation_ix,
			rv_reservation.rv_res_name,
			TRIM(CONCAT(pr_persona.pr_name_first,' ',pr_persona.pr_name_last)) AS pr_agent_name
		FROM
			rv_reservation
			LEFT JOIN pr_persona ON pr_persona.pr_persona_ix = rv_reservation.rv_agent_id
		WHERE
			rv_reservation.rv_reservation_ix = '$rv_reservation_id'
	",1));

	if(empty($emailInfo['pr_agent_name'])) {
		$emailInfo['pr_agent_name'] = "Agent";
	}

	$principalId = $lDB->get("SELECT rf_principal_id FROM rf_default", 4);
	$principalEmail = $lDB->get("SELECT pr_email FROM pr_persona WHERE pr_persona_ix = '$principalId'",4);
	if(empty($principalEmail)) {
		$principalEmail = "support@resrequest.com";
	}
	$emailInfo['pr_name'] = getContactFullName($principalId);

	$fromName = "";

	if(empty($from)) {
		$from = $principalEmail;
		$fromName = $emailInfo['pr_name'];
	}

	if(strpos($from,"@") === false) {
		return array(
			'error'=>true,
			'message'=>"Invalid from address \"$from\""
		);
	}

	if(!empty($fromName)) {
		$headers = "From: ".$emailInfo['pr_name']."<$from>\r\n";
	} else {
		$headers = "From: $from\r\n";
	}

	if(empty($to)) {
		$to = array();
	}

	if(!is_array($to)) {
		$to = preg_split("/[,;]/",$to);
	}

	$to[] = $principalEmail;
	$to[] = "res_cancel@resrequest.com";

	foreach($to as $email) {
		if(strpos($email,"@") === false) {
			return array(
				'error'=>true,
				'message'=>"Invalid to address \"$email\""
			);
		}
	}

	$to = join(",",array_unique($to));

	$subject = "ResRequest $rv_reservation_id | $emailInfo[rv_res_name] | CANCEL";
	$body = showpage($emailInfo,$emailTemplate);

	$error = email($from, $to, $subject, $headers, $body, 0, 0, $rv_reservation_id, $fromName);
	if (!$error) {
		return array(
			'error'=>false,
			'message'=>"Successfully sent"
		);
	} else {
		return array(
			'error'=>true,
			'message'=>"Failed sending"
		);
	}
}


function api_rv_update_tripsheet($rv_reservation_id,$pr_persona_id,$changes) {
	global $api_instance;
	global $lDB;
	global $userStatusId;

	if ($userStatusId < 2 || $lDB->isMaster == "0") {
		return $api_instance->Error("Access denied.");
	}

	if(!db_rv_reservation_exists($rv_reservation_id)) {
		return $api_instance->Error("Unknown reservation");
	}

	if(!db_pr_persona_exists($pr_persona_id)) {
		return $api_instance->Error("Unknown contact");
	}

	$properties = $lDB->get("
		SELECT DISTINCT
			rv_reservation_item.pr_business_id
		FROM
			rv_reservation_item
		WHERE
			rv_reservation_item.rv_reservation_id = '".$lDB->escape($rv_reservation_id)."'
	",3);
	$properties[] = "0";

	$rf_obj_trips = [];
	foreach($changes as $rf_obj_trip_id=>$value) {
		if(!db_rf_obj_trip_exists($rf_obj_trip_id)) {
			return $api_instance->Error("Unknown field id $rf_obj_trip_id");
		}
		$rf_obj_trips[$rf_obj_trip_id] = $lDB->get("
			SELECT
				rf_obj_trip.rf_obj_trip_type_ind,
				rf_obj_trip.rf_obj_trip_inactive_yn
			FROM
				rf_obj_trip
			WHERE
				rf_obj_trip.rf_obj_trip_ix = '".$lDB->escape($rf_obj_trip_id)."'
		",1);
		$commonProperties = array_intersect($properties, $lDB->get("
			SELECT
				rf_obj_trip_prop.pr_business_id
			FROM
				rf_obj_trip_prop
			WHERE
				rf_obj_trip_prop.rf_obj_trip_id = '".$lDB->escape($rf_obj_trip_id)."'
		",3));
		if(sizeof($commonProperties) == 0) {
			return $api_instance->Error("Reservation specified does not include the property required for field $rf_obj_trip_id");
		}
		$rf_obj_trips[$rf_obj_trip_id]['pr_business_id'] = array_shift($commonProperties);
		$rf_obj_trip = $rf_obj_trips[$rf_obj_trip_id];
		if($rf_obj_trip['rf_obj_trip_inactive_yn'] != "0") {
			return $api_instance->Error("Unknown field id $rf_obj_trip_id");
		}
		switch($rf_obj_trip['rf_obj_trip_type_ind']) {
		case DB_RF_OBJ_TRIP_TEXTAREA:
		case DB_RF_OBJ_TRIP_TEXTBOX:
			// no validation needed
			break;
		case DB_RF_OBJ_TRIP_NUMERIC:
			if(!empty($value) && !is_numeric($value)) {
				return $api_instance->Error("Invalid value $value for numeric field $rf_obj_trip_id");
			}
			break;
		case DB_RF_OBJ_TRIP_RADIO:
		case DB_RF_OBJ_TRIP_SELECTBOX:
			if($value !== "") {
				$check = $lDB->get("
					SELECT
						COUNT(*)
					FROM
						rf_obj_trip_data
					WHERE
						rf_obj_trip_data.rf_obj_trip_id = '".$lDB->escape($rf_obj_trip_id)."'
						AND rf_obj_trip_data.rf_obj_trip_data_content = '".$lDB->escape($value)."'
				",4);
				if($check == 0) {
					$api_instance->Error("Invalid value $value for field $rf_obj_trip_id");
				}
			}
			break;
		case DB_RF_OBJ_TRIP_CHECKBOX:
			if(!is_array($value)) {
				$value = [$value];
			}
			$value = array_unique(array_filter($value,function($item) {
				if(!empty(trim($item))) {
					return true;
				}
				return false;
			}));
			$changes[$rf_obj_trip_id] = $value;
			foreach($value as $rf_obj_trip_data_id) {
				if(!db_rf_obj_trip_data_exists($rf_obj_trip_data_id,$rf_obj_trip_id)) {
					return $api_instance->Error("Invalid value $rf_obj_trip_data_id for field $rf_obj_trip_id");
				}
			}
			break;
		}
	}

	$audit = new AuditTrail($pr_persona_id,TYPE_PERSONA, ['rv_reservation_id'=>$rv_reservation_id]);
	foreach($changes as $rf_obj_trip_id=>$value) {
		$rf_obj_trip = $rf_obj_trips[$rf_obj_trip_id];
		if($rf_obj_trip['rf_obj_trip_type_ind'] == DB_RF_OBJ_TRIP_CHECKBOX) {
			$rv_trip_obj_datas = $lDB->get("
				SELECT
					rv_trip_obj_data_ix,
					rf_obj_trip_data_id
				FROM
					rv_trip_obj_data
				WHERE
					rv_trip_obj_data.rv_reservation_id = '".$lDB->escape($rv_reservation_id)."'
					AND pr_guest_id = '".$lDB->escape($pr_persona_id)."'
					AND rf_obj_trip_id = '".$lDB->escape($rf_obj_trip_id)."'
			",6);
			$rf_obj_trip_data_ids = [];
			foreach ($rv_trip_obj_datas as $rv_trip_obj_data) {
				$rv_trip_obj_data_id = $rv_trip_obj_data['rv_trip_obj_data_ix'];
				$rf_obj_trip_data_ids[] = $rf_obj_trip_data_id = $rv_trip_obj_data['rf_obj_trip_data_id'];
				// Delete only items that no longer exist
				if(!in_array($rf_obj_trip_data_id, $value)) {
					$all = db_rv_trip_obj_data_get_all($rv_trip_obj_data_id);
					$audit->addDetail($all['id'], "", $rv_trip_obj_data_id, "", DB_AD_PERSONA_DETAIL_ACTION_DELETE, "tripsheet", "rv_trip_obj_data.", $rf_obj_trip_id, $all['value'], "");
					db_rv_trip_obj_data_delete($rv_trip_obj_data_id);
				}
			}
			foreach($value as $item) {
				// Add only new items
				if(!in_array($item, $rf_obj_trip_data_ids)) {
					$rv_trip_obj_data_id = db_rv_trip_obj_data_insert($rf_obj_trip_id, $pr_persona_id, $rv_reservation_id, $rf_obj_trip['pr_business_id'], $item, $rf_obj_trip['rf_obj_trip_type_ind']);
					$all = db_rv_trip_obj_data_get_all($rv_trip_obj_data_id);

					$audit->addDetail("", $all['id'], "", $rv_trip_obj_data_id, DB_AD_PERSONA_DETAIL_ACTION_ADD, "tripsheet", "rv_trip_obj_data.", $rf_obj_trip_id, "", $all['value']);
				}
			}
		} else {
			$newAll = db_rf_obj_trip_get_all($rf_obj_trip_id, $value);
			$rv_trip_obj_data_id = db_rv_trip_obj_data_select_by_id($rf_obj_trip_id, $rv_reservation_id, $pr_persona_id, $rf_obj_trip['pr_business_id']);
			if($rv_trip_obj_data_id != '0') {
				$oldAll = db_rv_trip_obj_data_get_all($rv_trip_obj_data_id);
				$action = DB_AD_PERSONA_DETAIL_ACTION_EDIT;
			} else {
				$oldAll = ['id'=>"", 'value'=>""];
				$action = DB_AD_PERSONA_DETAIL_ACTION_ADD;
			}
			$rv_trip_obj_data_id = db_rv_trip_obj_data_insert($rf_obj_trip_id, $pr_persona_id, $rv_reservation_id, $rf_obj_trip['pr_business_id'], $value, $rf_obj_trip['rf_obj_trip_type_ind']);
			$audit->addDetail($oldAll['id'], $newAll['id'], $rv_trip_obj_data_id, $rv_trip_obj_data_id, $action, "tripsheet", "rv_trip_obj_data.", $rf_obj_trip_id, $oldAll['value'], $newAll['value']);
		}
	}
	$audit->save("Tripsheet details (Edit)");

	return true;
}
