<?php

require_once(__DIR__ . "/rpc/xmlrpc.php");
require_once(__DIR__ . "/rpc/xmlrpcs.php");
require_once(__DIR__ . "/rpc/jsonrpc.php");
require_once(__DIR__ . "/rpc/jsonrpcs.php");
require_once(__DIR__ . "/rpc/xmlrpc_wrappers.php");

function api_wrapper($m="") {
	return $GLOBALS['api_instance']->Wrapper($m);
}

abstract class APIAbstract
{
	public $Mode;
	protected $Map;
	protected $BaseSignatures;
	protected $Errors;
	protected $Methods;
  public $uniqueId;
  public $bridgeUsername;
  public $principalUsername;
  public $environmentCode;
  public $principalId;
  public $method;
  public $headers = [];

  const CALL_TYPE_REQUEST = 1;
  const CALL_TYPE_RESPONSE = 2;

	/**
	 * Constructor
	 * 
	 */
	public function __construct()
	{
		global $api_instance;
		$this->Map = array();
		if(!isset($this->BaseSignatures)) {
			$this->BaseSignatures = array(
				array("struct","string","string")
			);
		}
		$this->Errors = array(
			"unknown_error"=>array(
				'code'=>800,
				'message'=>"Unknown error occured."
			),
			"invalid_login"=>array(
				'code'=>801,
				'message'=>"Invalid username or password."
			),
			"invalid_method_name"=>array(
				'code'=>802,
				'message'=>"Invalid method name."
			),
			"invalid_method_return"=>array(
				'code'=>803,
				'message'=>"Invalid method result."
			),
			"access_denied"=>array(
				'code'=>804,
				'message'=>"Access denied."
			),
			"multicall_optimisation"=>array(
				'code'=>805,
				'message'=>"Multicall optimisation."
			)
		);
		$this->Methods = array(
			'stock',
			'rate',
			'reservation',
			'contact',
			'default',
			'extra',
			'payment',
			'pos',
			'system'
		);
		foreach($this->Methods as $method) {
			$this->LoadMethod($method);
		}
		$api_instance = $this; // Workaround for the wrapper
	}

	public function Register($list)
	{
		foreach($list as $name=>$detail) {
			if(!array_key_exists('doc',$detail)) $detail['doc'] = "";
			if(!array_key_exists('signature',$detail)) $detail['signature'] = false;
			if(!array_key_exists('access',$detail)) $detail['access'] = false;
			if(!array_key_exists('forceFunction',$detail)) $detail['forceFunction'] = false;
			if(!array_key_exists('noPrincipal',$detail)) $detail['noPrincipal'] = false;

			if($detail['signature'] !== false && !is_array($detail['signature'])) {
				$detail['signature'] = $this->Signature(explode(",",$detail['signature']));
			}
			$this->Add($name,$detail['doc'],$detail['signature'],$detail['access'],$detail['forceFunction'],$detail['noPrincipal']);
		}
	}

	private function Signature()
	{
		$params = func_get_args();
		if(isset($params[0]) && is_array($params[0])) {
			$params = $params[0];
		}

		$bases = $this->BaseSignatures;
		$signatures = array();
		$optional = false;
		foreach($params as $param) {
			if($param[0] == "?") {
				$optional = true;
				$param = ltrim($param,"?");
			}
			if($optional) {
				foreach($bases as $base) {
					$signatures[] = $base;
				}
			}
			$types = explode("+",$param);
			$newBases = array();
			foreach($types as $type) {
				foreach($bases as $key=>$base) {
					$base[] = $type;
					$newBases[] = $base;
				}
			}
			$bases = $newBases;
		}
		foreach($bases as $base) {
			$signatures[] = $base;
		}

		return $signatures;
	}

	public function Add($name, $doc="", $signature=false, $access=false, $forceFunction=false, $noPrincipal=false)
	{
		if(array_key_exists($name,$this->Map)) {
			die("Duplicate function declaration");
		}

		if($signature === false) {
			$signature = $this->Signature();
		}

		$this->Map[$name] = array(
			'function'=>"APIAbstract::StaticWrapper",
			'signature'=>$signature,
			'docstring'=>$doc,
			'no_principal'=>$noPrincipal,
			'real_function'=>"api_" . $name,
			'force_function'=>$forceFunction
		);

		if($access !== false) {
			$this->Map[$name]['access'] = $access;
		}
	}

	protected function LoadMethod($method) {
		include(__DIR__ . '/api.' . $method . '.php');
	}

	public function Service() {
		set_time_limit(300);

		if(isset($_SERVER['CONTENT_TYPE']) && $_SERVER['CONTENT_TYPE'] == "text/xml") {
			$s=new xmlrpc_server($this->Map,false);
		} else {
			$s=new jsonrpc_server($this->Map,false);
		}
		
		$s->response_charset_encoding = "auto";
		$s->service();
	}


	public function StandardError($name, $extra="") {
		if(array_key_exists($name,$this->Errors)) {
			$error = $this->Errors[$name];
		} else {
			$error = $this->Errors['unknown_error'];
		}
		if(!empty($extra)) {
			$error['message'] .= " " . $extra;
		}

		return $this->Error($error['message'], $error['code']);
	}

  public function Error($errorMessage="A general error occurred.", $errorNumber=800) 
  {
    return new xmlrpcresp(0, $errorNumber, $errorMessage);
	}

	/**
	 * Construct the name of the log file.
   *
	 * @return string
	 */
	public function logFilename()
	{
		global $config;
		$directory = __DIR__;

		if (array_key_exists('logs', $config) && array_key_exists('directory', $config['logs'])) {
			$directory = $config['logs']['directory'];
		}

		if (!is_dir($directory)) {
      try {
        mkdir($directory, 0777, true);
      } catch (Exception $e) {
        $directory = __DIR__;
      }
		}

    $fileNameStart = strtolower(get_class($this)); // "Bridge" or "API"
    $fileName = "{$directory}/{$fileNameStart}_" . date("Y-m-d") . ".log";
		return $fileName;
	}

	/**
	 * Write call details or message to log file
	 *
   * @param string $batch The unique ID of the log entry. 
   *  Defaults to uniqid(). Useful to group request + response pairs
   * @param string bridgeUsername Bridge user name
   * @param string principalUsername User name of the principal user
   * @param string principalId Id of the principal
   * @param string environmentCode Code of the environment requested
   * @param string method The method being called/returned
   * @param int $callType Accepts self::CALL_TYPE_REQUEST or
   *  self::CALL_TYPE_RESPONSE
   * @param array $payload The paramers passed to the call or payload
   *  received from it
   * @param string $message Log a specific string to the log file
	 * @return void
	 */
  public function log(
    string $batch = null,
    string $bridgeUsername = "",
    string $principalUsername = "",
    string $principalId = "",
    string $environmentCode = "",
    string $method = "",
    int $callType = null,
    $payload = [],
    string $message = null
  ) {
    if ($callType || $message) {
      $structure = [
        'timestamp' => "[" . date("Y-m-d H:i:s") . "]",
        'remoteAddress' => '',
        'protocol' => '',
        'auth' => '',
        'method' => '',
        'batch' => $batch ?? uniqid(),
        'callType' => '',
        'payload' => '',
        'message' => $message ?? ""
      ];
      $serverProtocol = $this->isHttps() ? 'https' : 'http';
      $dataProtocol = $this->isJsonRPC() ? 'json' : 'xml';
      $structure['remoteAddress'] = $_SERVER['REMOTE_ADDR'];
      $structure['protocol'] = $serverProtocol . '+' . $dataProtocol;
      $structure['method'] = $method;
      if (is_array($payload)) {
        $structure['payload'] = json_encode($payload);
      } else {
        $structure['payload'] = $payload;
      }
      $structure['callType'] = $callType !== null ? $this->getCallTypeDescription($callType) : null;
      $structure['auth'] = "{$bridgeUsername} ({$principalId} {$principalUsername} {$environmentCode})";

      $fh = @fopen($this->logFilename(),"a");
      @flock($fh, LOCK_EX);
      @fwrite($fh, join(' | ', array_values($structure))."\n");
      @flock($fh, LOCK_UN);
      @fclose($fh);
    }
  }

	/**
	 * Whether Https is being used.
	 *
	 * @return boolean
	 */
	public function isHttps()
	{
		return (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
			|| $_SERVER['SERVER_PORT'] == 443;
	}

	/**
	 * Whether Json RPC was used or not.
	 *
	 * @return boolean
	 */
	public function isJsonRPC()
	{
		$isXml = isset($_SERVER['CONTENT_TYPE']) && $_SERVER['CONTENT_TYPE'] == "text/xml";
		return !$isXml;	
	}

	/**
	 * Get the type of call
	 *
	 * @return string
	 */
  public function getCallTypeDescription(int $callType = null)
	{
    switch ($callType) {
      case self::CALL_TYPE_REQUEST:
        return strtolower(get_class($this))." request";
        break;
      case self::CALL_TYPE_RESPONSE:
        return strtolower(get_class($this))." response";
        break;
      default:
        return "";
        break;
    }
	}

	/**
	 * Get the hashcode to verify call is being sent from bridge
	 *
	 * @return string
	 */
  public function getHashCode() {
    return md5("bbVwC242ajmTYQzU" . date("Y-m-d"));
  }

	static function StaticWrapper($m="") {
		return $GLOBALS['api_instance']->Wrapper($m);
	}

	abstract public function Wrapper($m="");
}
