<?php

/**
 * The base class for financial export items (E.g. invoice exports).
 */
class FinancialExport
{
    /**
     * The database connection.
     *
     * @var MySQLDB
     */
    protected $lDB;

    /**
     * A list of fields to be retrieved from the database. Each entry is
     * a string representing a field to retrieve. The field can be aliased
     * or use functions to obtain it.
     *
     * @var array
     */
    protected $fields;

    /**
     * The main table from which data will be retrieved from.
     *
     * @var string
     */
    protected $table;

    /**
     * A list of Additional tables to include via joins. Each entry is
     * a string that specifies a join on a table.
     *
     * @var array
     */
    protected $tableJoins;

    /**
     * A list of filters to be applied to the query. Each entry is a
     * string that specifies a criteria for selection.
     *
     * @var array
     */
    protected $filters;

    /**
     * A sorting clause.
     *
     * @var string
     */
    protected $orderBy;

    /**
     * A grouping clause.
     *
     * @var string
     */
    protected $groupBy;

    /**
     * A list of column headings for the export.
     *
     * @var array
     */
    protected $header;

    /**
     * Whether or not the header row should be displayed.
     *
     * @var boolean
     */
    protected $showHeader;

    /**
     * The list of fields that should be used for the data. An empty string
     * in one of the entries means that the column will be left empty.
     *
     * @var array
     */
    protected $dataFields;

    /**
     * Whether a blank row should be added after the header.
     *
     * @var boolean
     */
    protected $addBlankRowAfterHeader;

    /**
     * Whether a blank row should be added at the end of the file.
     *
     * @var boolean
     */
    protected $addBlankRowAtEnd;

    /**
     * Instantiates a new financial export.
     *
     * @param MySQLDB $lDB
     */
    public function __construct($lDB)
    {
        $this->lDB = $lDB;
        $this->fields = [];
        $this->table = '';
        $this->tableJoins = [];
        $this->filters = [];
        $this->orderBy = '';
        $this->groupBy = '';
        $this->header = [];
        $this->showHeader = true;
        $this->dataFields = [];
        $this->addBlankRowAfterHeader = false;
        $this->addBlankRowAtEnd = false;
    }

    /**
     * Add a new field to the list of fields.
     *
     * @param string $field
     * @return void
     */
    public function addField($field)
    {
        array_push($this->fields, $field);
    }

    /**
     * Add a list of new fields to the existing fields list.
     *
     * @param array $fields
     * @return void
     */
    public function addFields($fields)
    {
        $this->fields = array_merge($this->fields, $fields);
    }

    /**
     * Sets the main table to retrieve data from.
     *
     * @param string $table
     * @return void
     */
    public function setTable($table)
    {
        $this->table = $table;
    }

    /**
     * Add a new table to be included via a join.
     *
     * @param string $join
     * @return void
     */
    public function addJoin($join)
    {
        array_push($this->tableJoins, $join);
    }

    /**
     * Add a list of tables to be included via joins.
     *
     * @param array $joins
     * @return void
     */
    public function addJoins($joins)
    {
        $this->tableJoins = array_merge($this->tableJoins, $joins);
    }

    /**
     * Add a new filter to the list of filters.
     *
     * @param string filter
     * @return void
     */
    public function addFilter($filter)
    {
        array_push($this->filters, $filter);
    }

    /**
     * Add a list of filters to the existing filters.
     *
     * @param array $filters
     * @return void
     */
    public function addFilters($filters)
    {
        array_merge($this->filters, $filters);
    }

    /**
     * Sets the sorting clause.
     *
     * @param string $orderBy
     * @return void
     */
    public function setOrderBy($orderBy) {
        $this->orderBy = $orderBy;
    }

    /**
     * Sets the group by clause.
     *
     * @param string $groupBy
     * @return void
     */
    public function setGroupBy($groupBy) {
        $this->groupBy = $groupBy;
    }

    /**
     * Set the header row for the export.
     *
     * @param array $headerRow
     * @return void
     */
    public function setHeader($header)
    {
        $this->header = $header;
    }

    /**
     * Set whether or not the header should be displayed.
     *
     * @param boolean $showHeader
     * @return void
     */
    public function setShowHeader($showHeader)
    {
        $this->showHeader = $showHeader;
    }

    /**
     * Set the data fields to use for the export. An empty string entry
     * means that the column will be left empty.
     *
     * @param array $dataFields
     * @return void
     */
    public function setDataFields($dataFields)
    {
        $this->dataFields = $dataFields;
    }

    /**
     * Set whether a blank row should be added after the header.
     *
     * @param boolean $addBlankRowAfterHeader
     * @return void
     */
    public function setAddBlankRowAfterHeader($addBlankRowAfterHeader)
    {
        $this->addBlankRowAfterHeader = $addBlankRowAfterHeader;
    }

    /**
     * Set whether a blank row should be added at the end.
     *
     * @param boolean $addBlankRowAtEnd
     * @return void
     */
    public function setAddBlankRowAtEnd($addBlankRowAtEnd)
    {
        $this->addBlankRowAtEnd = $addBlankRowAtEnd;
    }

    /**
     * Generate the SQL query.
     *
     * @return string
     */
    public function buildQuery()
    {
        $sql = "SELECT\n\t";
        $fieldAdded = false;

        foreach ($this->fields as $field) {
            if ($fieldAdded) {
                $sql .= ", ";
            }
            
            $sql .= $field;
            $fieldAdded = true;
        }

        if (!$fieldAdded) {
            $sql .= '*';
        }

        $sql .= "\nFROM " . $this->table . "\n";

        foreach ($this->tableJoins as $join) {
            $sql .= $join . "\n";
        }

        $sql .= "WHERE 1\n";

        foreach ($this->filters as $filter) {
            $sql .= "AND " . $filter . "\n";
        }

        if ($this->orderBy) {
            $sql .= $this->orderBy . "\n";
        }

        if ($this->groupBy) {
            $sql .= "GROUP BY " . $this->groupBy . "\n";
        }

        return $sql;
    }

    /**
     * Runs the export.
     *
     * @return array
     */
    public function export()
    {
        $export = [];

        $data = $this->lDB->get($this->buildQuery(), 2);
        $this->processData($data);

        if ($this->showHeader) {
            array_push($export, $this->header);

            if ($this->addBlankRowAfterHeader) {
                array_push($export, []);
            }
        }

        foreach ($data as $row) {
            $exportRow = [];
            foreach ($this->dataFields as $field) {
                if (empty($field)) {
                    array_push($exportRow, '');
                } else {
                    array_push($exportRow, $row[$field]);
                }
            }
            array_push($export, $exportRow);
        }

        if ($this->addBlankRowAtEnd) {
            array_push($export, []);
        }

        return $export;
    }

    /**
     * Place to initialise the export.
     *
     * @return void
     */
    protected function init()
    {
        // Initialise export
    }

    /**
     * Place to build up the header.
     *
     * @return void
     */
    protected function buildHeader()
    {
        // Set headers
    }

    /**
     * Place to build up the fields.
     *
     * @return void
     */
    protected function buildFields()
    {
        // Add fields
    }

    /**
     * Place to build up the tables and joins.
     *
     * @return void
     */
    protected function buildTables()
    {
        // Set table and add joins
    }

    /**
     * Place to build up the filters.
     *
     * @return void
     */
    protected function buildFilters()
    {
        // Add filters
    }

    /**
     * Place to build up the data fields.
     *
     * @return void
     */
    protected function buildDataFields()
    {
        // Set data fields
    }

    /**
     * Process the data retrieved from the database before it is used in the export.
     * Can be overriden by subclasses to perform transformation of the data.
     *
     * @param array $data
     * @return void
     */
    protected function processData(& $data)
    {
        // Process data
    }
}