<?php

/**
 * The base class for financial interfaces.
 */
class FinancialInterface
{
    /**
     * The connection to the database.
     *
     * @var MySQLDB
     */
    protected $lDB;

    /**
     * The details of the invoicing unit that the financial interface is for.
     * Contains the keys: id, name, currency_id, currency_symbol.
     * 
     * @var array
     */
    protected $invoicingUnit;

    /**
     * An array containing the supported configuration options.
     * Each entry consists of the keys: id and default.
     *
     * @var array
     */
    protected $supportedOptions;

    /**
     * An array containing the actual options for the interface.
     * The keys are the ids of the supported options and the values are
     * either the defaults or the ones loaded from the database.
     *
     * @var array
     */
    protected $activeOptions;

    /**
     * Create an instance of the Financial Interface.
     *
     * @param MySQLD $lDB The connection to the database.
     * @param string $invoicingUnit The id of the invoicing unit.
     */
    public function __construct($lDB, $invoicingUnit)
    {
        $this->lDB = $lDB;
        $this->invoicingUnit = [
            'id' => $invoicingUnit,
            'name' => null,
            'currency_id' => null,
            'currency_symbol' => null
        ];

        $this->initInterface($invoicingUnit);
    }

    /**
     * Initialise the interface.
     *
     * @param string $invoicingUnit
     * @return void
     */
    private function initInterface($invoicingUnit)
    {
        $this->loadInvoicingUnit($invoicingUnit);
        $this->loadInterfaceOptions();
    }

    /**
     * Retrieve the details about the invoicing unit if it exists.
     *
     * @param string $invoicingUnit The id of the invoicing unit.
     * @return void
     */
    private function loadInvoicingUnit($invoicingUnit)
    {
        $sql = "
            SELECT
                pr_persona.pr_name_last AS invoicing_unit_name,
                pr_business.pr_bus_home_curr_id AS currency_id,
                rf_currency.rf_currency_symbol AS currency_symbol
            FROM pr_business
            INNER JOIN pr_persona ON pr_persona.pr_persona_ix = pr_business.pr_business_id
            INNER JOIN rf_currency ON rf_currency.rf_currency_ix = pr_business.pr_bus_home_curr_id
            WHERE pr_business.pr_business_id = '$invoicingUnit'
        ";

        $data = $this->lDB->get($sql, 1);

        if ($data) {
            // The invoicing unit exists.
            $this->invoicingUnit['name'] = $data['invoicing_unit_name'];
            $this->invoicingUnit['currency_id'] = $data['currency_id'];
            $this->invoicingUnit['currency_symbol'] = $data['currency_symbol'];
        } else {
            // The invoicing unit does not exist.
            throw new Exception('The invoicing unit does not exist');
        }
    }

    /**
     * Retrieve the options that have been configured for the interface.
     *
     * @return void
     */
    private function loadInterfaceOptions()
    {
        $options = [];

        if (!$this->supportedOptions) {
            return;
        }

        foreach ($this->supportedOptions as $option) {
            $this->activeOptions[$option['id']] = $option['default'];
            array_push($options, "'{$option['id']}'");
        }

        $optionsFilter = implode(',', $options);

        $sql = "SELECT
                    fn_option_data.fn_option_id,
                    fn_option_data.fn_option_data_contents
                FROM fn_option_data
                WHERE fn_option_id IN ($optionsFilter)
                AND pr_business_id = '{$this->invoicingUnit['id']}'
        ";

        $options = $this->lDB->get($sql, 2);

        foreach ($options as $option) {
            $this->activeOptions[$option['fn_option_id']] = $option['fn_option_data_contents'];
        }
    }

    /**
     * Retrieve the id of the invoicing unit.
     *
     * @return string
     */
    public function getInvoicingUnit()
    {
        return $this->invoicingUnit['id'];
    }

    /**
     * Retrieve the name of the invoicing unit.
     *
     * @return string
     */
    public function getInvoicingUnitName()
    {
        return $this->invoicingUnit['name'];
    }

    /**
     * Retrieve the id of the invoicing unit currency.
     *
     * @return string
     */
    public function getCurrency()
    {
        return $this->invoicingUnit['currency'];
    }

    /**
     * Retrieve the symbol of the invoicing unit currency.
     *
     * @return string
     */
    public function getCurrencySymbol()
    {
        return $this->invoicingUnit['currency_symbol'];
    }

    /**
     * Retrieve the configuration options for the financial interface.
     *
     * @return void
     */
    public function getInterfaceOptions()
    {
        return $this->activeOptions;
    }
}