<?php

 /**
  * class.fiscalator.php - Fiscalisation.
  */

require_once(__DIR__ . '/db.fn_invoice.php');

class Fiscalator {

    public $fiscalatorUrl = "";
    public $fiscalatorSystemId = 0;
    public $fiscalatorUrlOptions = array(
        "production" => "https://fiscalator.resrequest.com",
        "test" => "https://jaco-test.resrequest.net/fiscalator",
        "regression" => "https://regression.resrequest.net/fiscalator",
        "vagrant" => "http://fiscalator.vagrant",
        "rrdev" => "http://fiscalator.rrdev"
    );
    public $fiscalDocNoField = ['', 'INVNO'];
    public $fisDebug = false;

    public function __construct()
    {
        $determinedSystem = false;
        $serverName = $_SERVER['SERVER_NAME'];
        $hostname = gethostname();

        // Checks for local development environments against the server name first, then the hostname for droplets and servers.
        // If no match is found, it defaults to production.

        if (strpos($serverName, "vagrant") !== false) {
            $this->fiscalatorUrl = $this->fiscalatorUrlOptions['vagrant'];
            $this->fisDebug = true;
            $determinedSystem = true;
        } elseif (strpos($serverName, "rrdev") !== false) {
            $this->fiscalatorUrl = $this->fiscalatorUrlOptions['rrdev'];
            $this->fisDebug = true;
            $determinedSystem = true;
        }

        if (!$determinedSystem) {
            if (strpos($hostname, "jaco-test") !== false) {
                $this->fiscalatorUrl = $this->fiscalatorUrlOptions['test'];
                $this->fisDebug = true;
                $determinedSystem = true;
            } elseif (strpos($hostname, "testingbase") !== false) {
                $this->fiscalatorUrl = $this->fiscalatorUrlOptions['regression'];
                $determinedSystem = true;
            }
        }

        if (!$determinedSystem) {
            $this->fiscalatorUrl = $this->fiscalatorUrlOptions['production'];
        }
    }

    function uploadInvoice($invoiceList, $delay=false, $resubmit=false)
    {
        // Upload multiple invoices. Accepts: single invoice ID, or array of invoice ID's
        global $lDB;
        if (empty($invoiceList)) {          // Oh-oh, nothing here, reject it!
            return false;
        }

        if (!is_array($invoiceList)) {      // Turn single value into array
            $invoiceList = array($invoiceList);
        }

        $invoiceUnits = array();
        $results = array();
        $errors = array();
        $delay = count($invoiceList) > 1 ? true : $delay;

        foreach($invoiceList as $fn_invoice_id) {
            $fiscalatorKey = $this->getFiscalatorKeyByInvoice($fn_invoice_id);
            if (empty($fiscalatorKey)) {
                $errors[] = $fn_invoice_id . ": No Fiscalator account set up for invoicing unit";
                continue;
            }
            $accountDetails = json_decode(
                $this->fiscalatorApiCall(
                    $fiscalatorKey,
                    "GET",
                    "account"
                ),
                true
            );
            $this->fiscalatorSystemId = $accountDetails['business']['systemId'];

            if (!$resubmit) {
                $invoiceDetails = $this->getInvoiceComponentsExtras($fn_invoice_id);
            }
            else {
                $invoiceData = $lDB->get("
                    SELECT
                        fn_api_post_rr
                    FROM
                        fn_invoice_fiscal_api
                    WHERE
                        fn_invoice_id = '" . $fn_invoice_id . "'
                    ORDER BY
                        fn_api_time_post DESC
                    LIMIT 1
                ", 4);
                if ($invoiceData != "") {
                    // Resubmit data found. Decode JSON.
                    $invoiceDetails = json_decode($invoiceData, true);
                }
                else {
                    $errors[] = $fn_invoice_id . ": No Invoice found";
                    continue;
                }
                if (empty($invoiceDetails)) {
                    // $invoiceDetails contains invalid JSON. E.g. truncated, data too long for db field.
                    // Rebuild invoice from scratch.
                    $invoiceDetails = $this->getInvoiceComponentsExtras($fn_invoice_id);
                }
            }

            if (empty($invoiceDetails)) {
                $errors[] = $fn_invoice_id . ": No Invoice found";
                continue;
            }

            if (
                !empty($invoiceDetails) &&
                ( !isset($invoiceDetails['items']) || empty($invoiceDetails['items']) )
            ) {
                $errors[] = $fn_invoice_id . ": No Invoice items found";
                continue;
            }
            if ($this->fiscalatorSystemId == 2) {
                // This is a mind-bender of note. TIMS only. (SystemId 2) Read carefully!
                // This is repeated for credit notes as well.
                // Tims rejects negative amounts we need to allow for that.
                // Tims keeps a record of invoices and linked credits.
                // All negative amounts need to be converted to credit notes and linked to an existing, fiscalised invoice,
                // with the same customer and tax code
                // We first try to consolidate any negative amounts in an invoice by tax code in the same invoice
                // The linked invoice can only be linked to once, if there is a valid qualifying invoice item to link to
                //     with a large enough amount to credit against
                // The only information stored in Enterprise is a list of invoice items that have been linked to
                //     credits and the relevant credit invoice id.
                //     This list must be checked and updated each time to make sure
                // So:
                // 1. find item with positive amount and same tax code that matches the negatives
                // 2. if none, consolidate all items with the same tax code and combine them all into one positive item
                // 3. if the result of 2 is negative or the invoice has a negative total,
                //    retrieve a fiscalised invoice from the current reservation,
                //    to the same customer,
                //    with a positive amount larger than or equal to the negative amount we need to process
                //    the same tax code,
                //    that has not been credited against previously
                //    then Yay, submit a credit note to fiscalator,
                // If all the above fails, submit to fiscalator and receive a Tims error and the user must fix the folios
                
                $newItemsArray = [];

                $negativeItems = $this->findNegatives($invoiceDetails['items']);
                if (count($negativeItems) > 0) {
                    $this->fisDebug("Try to consolidate");
                    $newItemsArray = $this->consolidateAmountsByTaxCode($invoiceDetails['items'], $negativeItems);
                    
                    if ($newItemsArray != false && count($newItemsArray) < count($invoiceDetails['items']))
                    {
                        unset($invoiceDetails['items']);
                        $invoiceDetails['items'] = $newItemsArray;
                    }
                    else {
                        $this->fisDebug("Inv: Try to match");
                        $otherInvoiceIdList = [];
                        $otherInvoiceIdList = $this->otherInvoicesOnRes($invoiceDetails);
                        $matchResults = $this->matchInvoiceItems($invoiceDetails, $otherInvoiceIdList);

                        $newInvoiceDetails = !empty($matchResults['newInvoiceDetails']) ? $matchResults['newInvoiceDetails'] : [];
                        $linkList = !empty($matchResults['linkList']) ? $matchResults['linkList'] : [];

                        if(!empty($newInvoiceDetails) && count($invoiceDetails['items']) > 0 && !empty($linkList)) {
                            $invoiceDetails = $newInvoiceDetails;

                            $this->updateInvoiceStatus($fn_invoice_id, 5); // status 5 = pending credit upload
                            $this->createFiscalisedInvoiceRecord($fn_invoice_id, $invoiceDetails);
                    
                            $endpoint = $delay ? "invoice/credit/delay" : "invoice/credit";
                            $initiateCallTime = date("Y-m-d H:i:s");
                            $this->logFnFiscalApi(array($invoiceDetails), array(), $initiateCallTime, "");
                    
                            $callResult = json_decode(
                                $this->fiscalatorApiCall($fiscalatorKey, "POST", $endpoint, array($invoiceDetails)),
                                true
                            );

                            $endCallTime = date("Y-m-d H:i:s");
                    
                            $this->logFnFiscalApi(array($invoiceDetails), $callResult, $initiateCallTime, $endCallTime);
                            $this->updateFiscalisedInvoiceRecord($callResult, 5, $fn_invoice_id);
                            
                            $fiscal_data = $this->getFiscalData($fn_invoice_id);
                            if (
                                !empty($callResult)
                                && isset($fiscal_data['fiscalDocFields']['InvoiceNum'])
                                && !$newItemsArray
                            ) {
                                // We have a valid, fiscalised credit note, add the items to the link table
                                $this->fisDebug("Check FiscalData for new Credit note " . $fn_invoice_id);
                                $creditLink = $this->updateCreditLink($invoiceDetails['invoice_id'], $linkList);

                                $returnResult = array(
                                    "results" => $callResult,
                                    "errors" => $errors
                                );

                                return $returnResult;
                            }
                            else {
                                // Could not consolidate or match.
                                // Invoice totals are too small or don't have relevant tax codes, item amounts couldn't match. Error status.
                                return array(
                                    "errors" => "No valid invoices to link to"
                                );
                            }
                        }
                        else {
                            $this->fisDebug("Inv: No joy. Just carry on and get negative error");
                        }
                    }
                }
            }

            if (!isset($invoiceUnits[$fiscalatorKey])) {
                $invoiceUnits[$fiscalatorKey] = array();
            }

            $invoiceUnits[$fiscalatorKey][] = $invoiceDetails;
            $this->updateInvoiceStatus($fn_invoice_id, 1); // status 1 = pending
            $this->createFiscalisedInvoiceRecord($fn_invoice_id, $invoiceDetails); 
        }

        $endpoint = $delay ? "invoice/invoice/delay" : "invoice/invoice";
        foreach ($invoiceUnits as $fiscalatorKey => $invoices) {
            $initiateCallTime = date("Y-m-d H:i:s");
            $this->logFnFiscalApi($invoices, array(), $initiateCallTime, "");

            $callResult = json_decode(
                $this->fiscalatorApiCall($fiscalatorKey, "POST", $endpoint, $invoices),
                true
            );

            $endCallTime = date("Y-m-d H:i:s");
            $callResult = $this->findResponseIdNames($callResult);     // Error in response message with ID's? Try to replace with names

            $results[] = $callResult;
            $this->logFnFiscalApi($invoices, $callResult, $initiateCallTime, $endCallTime);
            $this->updateFiscalisedInvoiceRecord($callResult, 1, $fn_invoice_id);
        }
        
        $returnResult = array(
            "results" => $results,
            "errors" => $errors
        );

        return $returnResult;
    }

    function resubmitCreditInvoice($invoiceId)
    {
        // Re-uses previously submitted credit info
        $this->fisDebug("function resubmitCreditInvoice");
        global $lDB;
        $errors = array();

        $fiscalatorKey = $this->getFiscalatorKeyByInvoice($invoiceId);
        if (empty($fiscalatorKey)) {
            return array(
                "errors" => $invoiceId . ": No Fiscalator account set up for invoicing unit"
            );
        }
        $accountDetails = json_decode(
            $this->fiscalatorApiCall(
                $fiscalatorKey,
                "GET",
                "account"
            ),
            true
        );
        $this->fiscalatorSystemId = $accountDetails['business']['systemId'];

        $invoiceDetails = json_decode($lDB->get("
            SELECT
                fn_api_post_rr
            FROM
                fn_invoice_fiscal_api
            WHERE
                fn_invoice_id = '" . $invoiceId . "'
            ORDER BY
                fn_api_time_post DESC
            LIMIT 1
        ", 4), true);

        if (
            !empty($invoiceDetails) &&
            ( !isset($invoiceDetails['original_invoice_id']) || empty($invoiceDetails['original_invoice_id']) ) 
            ) {
            return array(
                "results" => "",
                "errors" => "Original invoice not found"
            );
        }

        if (
            !empty($invoiceDetails) &&
            ( !isset($invoiceDetails['items']) || empty($invoiceDetails['items']) )
        ) {
            return array(
                "results" => "",
                "errors" => $invoiceDetails['original_invoice_id'] . ": No Invoice items found"
            );
        } 
        
        $originalInvoiceId = $invoiceDetails['original_invoice_id'];
        
        $this->fisDebug("CN Resubmit: FiscalInvoiceId for " . $originalInvoiceId);
        
        return $this->uploadCreditInvoice($originalInvoiceId, $invoiceId, $invoiceDetails, false);                
    }

    function uploadCreditInvoice($originalInvoiceId, $invoiceId, $compExtraList, $delay=false)
    {
        // Upload single invoice credit. Accepts: single new invoice ID, and single original invoice ID
        $this->fisDebug('function uploadCreditInvoice()');

        global $lDB;

        if (empty($invoiceId) || empty($originalInvoiceId)) {          // Oh-oh, nothing here, reject it!
            return false;
        }

        $errors = array();

        $fiscalatorKey = $this->getFiscalatorKeyByInvoice($originalInvoiceId);
        if (empty($fiscalatorKey)) {
            return array(
                "results" => "",
                "errors" => $originalInvoiceId . ": No Fiscalator account set up for invoicing unit"
            );
        }

        $accountDetails = json_decode(
            $this->fiscalatorApiCall(
                $fiscalatorKey,
                "GET",
                "account"
            ),
            true
        );
        $this->fiscalatorSystemId = $accountDetails['business']['systemId'];

        // Get original invoice data
        if (!empty($compExtraList)) {
            // We have some components and extras already, use these instead and return
            $invoiceDetails = $compExtraList;
        } else {
            // Fetch from DB
            $invoiceDetails = $this->getInvoiceComponentsExtras($originalInvoiceId);
        }
        if (empty($invoiceDetails)) {
            return array(
                "results" => "",
                "errors" => $originalInvoiceId . ": No Invoice found"
            );
        }

        // Manipulate data to fix ID's for the credit record
        $invoiceDetails['invoice_id'] = $invoiceId;
        $invoiceDetails['original_invoice_id'] = $originalInvoiceId;

        if (
            !empty($invoiceDetails) &&
            ( !isset($invoiceDetails['items']) || empty($invoiceDetails['items']) )
        ) {
            return array(
                "results" => "",
                "errors" => $originalInvoiceId . ": No Invoice items found"
            );
        }
        // Need to send fiscal invoice doc number to credit for TIMS. Enterprise shouldn't know which integrator is being used.
        // Fiscalator will get the correct field from the array.
        $fiscal_data = $this->getFiscalData($originalInvoiceId);
        
        $invoiceDetails['fiscal_data'] = (isset($fiscal_data['fiscalDocFields']) && count($fiscal_data['fiscalDocFields']) == 1) ? $fiscal_data['fiscalDocFields'] : "";

        if ($this->fiscalatorSystemId == 2) {
            // Tims rejects negative amounts we need to allow for that, so we have to bring Tims knowledge into Enterprise.
            $this->fisDebug("Tims identified - CN");
            $newItemsArray = [];
            $otherInvoiceIdList = [];

            $negativeItems = $this->findNegatives($invoiceDetails['items']);
            if ($negativeItems) {
                // If this credit note is a reversal of an invoice with negative items Tims does not accept negative values
                // attempt item consolidation by tax code
                $this->fisDebug("Try consolidate negatives in credit note");
                $newItemsArray = $this->consolidateAmountsByTaxCode($invoiceDetails['items'], $negativeItems);
                if (count($newItemsArray) < count($invoiceDetails['items'])) {
                    unset($invoiceDetails['items']);
                    $invoiceDetails['items'] = $newItemsArray;
                }
                else { 
                    $this->fisDebug("CN: Try to match");

                    $otherInvoiceIdList = $this->otherInvoicesOnRes($invoiceDetails);
                    $matchResults = $this->matchInvoiceItems($invoiceDetails, $otherInvoiceIdList);
                    $newInvoiceDetails = !empty($matchResults['newInvoiceDetails']) ? $matchResults['newInvoiceDetails'] : [];
                    $linkList = !empty($matchResults['linkList']) ? $matchResults['linkList'] : []; // Do we need this here?
                    if(!empty($newInvoiceDetails) && count($invoiceDetails['items']) > 0) {
                        $this->fisDebug("We have details to credit");
                        $invoiceDetails = $newInvoiceDetails;
                    }
                }
            }
        }

        $this->updateInvoiceStatus($invoiceId, 5); // status 5 = pending credit upload
        $this->createFiscalisedInvoiceRecord($invoiceId, $invoiceDetails); 

        $endpoint = $delay ? "invoice/credit/delay" : "invoice/credit";
        $initiateCallTime = date("Y-m-d H:i:s");
        $this->logFnFiscalApi(array($invoiceDetails), array(), $initiateCallTime, "");

        $callResult = json_decode(
            $this->fiscalatorApiCall($fiscalatorKey, "POST", $endpoint, array($invoiceDetails)),
            true
        );

        $endCallTime = date("Y-m-d H:i:s");

        $this->logFnFiscalApi(array($invoiceDetails), $callResult, $initiateCallTime, $endCallTime);
        $this->updateFiscalisedInvoiceRecord($callResult, 5, $invoiceId);

        $fiscal_data = $this->getFiscalData($invoiceId);
        if (
            !empty($callResult) 
            && isset($fiscal_data['fiscalDocFields']['InvoiceNum'])
            && !$newItemsArray
            && !empty($linkList)
        ) { 
            // We have a valid, fiscalised credit note, add the items to the link table
            $this->fisDebug("Update credit link for new Credit note " . $invoiceId);
            $creditLink = $this->updateCreditLink($invoiceDetails['invoice_id'], $linkList);
        }

        $returnResult = array(
            "results" => $callResult,
            "errors" => $errors
        );
        return $returnResult;
    }

    function getInvoiceComponentsExtras($fn_invoice_id)
    {
        global $lDB;

        $invoiceDetails = $lDB->get("
            SELECT
                fn_invoice.fn_invoice_ix as invoice_id,
                fn_invoice.fn_inv_to_addr_line1,
                fn_invoice.fn_inv_to_addr_line2,
                fn_invoice.fn_inv_to_addr_line3,
                fn_invoice.fn_inv_to_city,
                fn_invoice.fn_inv_to_country,
                fn_invoice.fn_inv_to_post_code,
                fn_invoice.fn_inv_be_email as email,
                fn_invoice.fn_inv_be_phone as phone,
                fn_invoice.fn_inv_from as invoice_from,
                IF(fn_invoice.fn_inv_to!='', fn_invoice.fn_inv_to, 'Cash') as invoice_to,
                fn_folio.fn_folio_to_id as invoice_to_id,
                fn_invoice.fn_inv_amt_payable as amount,
                fn_folio.rf_currency_id as invoice_currency,
                pr_business.pr_bus_home_curr_id as gl_currency,
                fn_invoice.fn_inv_exch_rate,
                pr_business.contact_tin_statutory_key,
                fn_invoice.pr_business_id
            FROM
                fn_invoice
                LEFT JOIN fn_folio ON fn_folio.fn_invoice_id = fn_invoice.fn_invoice_ix
                LEFT JOIN pr_business ON pr_business.pr_business_id = fn_invoice.pr_business_id
            WHERE
                fn_invoice.fn_invoice_ix = '" . $fn_invoice_id . "'
        ", 6);
        if (empty($invoiceDetails)) {
            return array();
        }
        $invoiceDetails = $invoiceDetails[0];

        $address = array(
            $invoiceDetails['fn_inv_to_addr_line1'],
            $invoiceDetails['fn_inv_to_addr_line2'],
            $invoiceDetails['fn_inv_to_addr_line3'],
            $invoiceDetails['fn_inv_to_city'],
            $invoiceDetails['fn_inv_to_country'],
            $invoiceDetails['fn_inv_to_post_code'],
        );
        $invoiceDetails['address'] = join(", ", array_filter($address));

        $buyerTin = "";

        if (
            !empty($invoiceDetails['invoice_to_id']) &&
            $invoiceDetails['invoice_to_id'] != "0" &&
            !empty($invoiceDetails['contact_tin_statutory_key']) &&
            in_array($invoiceDetails['contact_tin_statutory_key'], array("1", "2", "3"))
        ) {
            $buyerTin = $lDB->get(
                "
                    SELECT
                        pr_statutory_" . $invoiceDetails['contact_tin_statutory_key'] . "
                    FROM
                        pr_persona
                    WHERE
                        pr_persona_ix = '" . $invoiceDetails['invoice_to_id'] . "'
                ",
                4
            );
        }

        $invoiceDetails['buyerTin'] = $buyerTin;

        if (
            !empty($invoiceDetails['invoice_to_id']) &&
            $invoiceDetails['invoice_to_id'] != "0"
        ) {
            $buyerDetails = $lDB->get(
                "
                    SELECT
                        pr_email,
                        pr_org_yn,
                        pr_phys_country_id,
                        pr_corr_country_id
                    FROM
                        pr_persona
                    WHERE
                        pr_persona_ix = '" . $invoiceDetails['invoice_to_id'] . "'
                ",
                1
            );
            if (!empty($buyerDetails)) {
                $invoiceDetails['email']  = $buyerDetails['pr_email'];
                $invoiceDetails['buyerIndividual']  = $buyerDetails['pr_org_yn'] == "0" ? "1" : "0";
                $invoiceDetails['buyerPhysicalCountry']  = $buyerDetails['pr_phys_country_id'];
                $invoiceDetails['buyerPostalCountry']  = $buyerDetails['pr_corr_country_id'];
            }
        }

        $invoiceItems = $lDB->get("
            SELECT
                fn_invoice_item_ix
            FROM
                fn_invoice_item
            WHERE
                fn_invoice_id = '" . $fn_invoice_id . "'
        ", 3);

        $invoiceComponents = $GLOBALS['lDB']->get("
            SELECT
                rv_res_item_comp_ix
            FROM
                rv_res_item_comp
            WHERE
                fn_invoice_item_id IN ('" . join("','", $invoiceItems) . "')
        ", 3);

        $invoiceExtras = $GLOBALS['lDB']->get("
            SELECT
                rv_extra_ix
            FROM
                rv_extra
            WHERE
                fn_invoice_item_id  IN ('" . join("','", $invoiceItems) . "')
        ", 3);

        $invoiceItemComponents = $lDB->get("
            SELECT
                CONCAT('CM_', rt_component.rt_component_ix) as product_code,
                '' as product_category_code,
                (rv_res_item_comp.rv_item_comp_amt_nett - rv_res_item_comp.rv_item_comp_amt_tax) as unit_nett_amount,
                1 as qty,
                (
                    SELECT
                        GROUP_CONCAT(rv_res_item_comp_tax.rf_tax_rate_id ORDER BY rv_res_item_comp_tax.rv_res_item_comp_tax_ix SEPARATOR ',')
                    FROM
                        rv_res_item_comp_tax
                    WHERE
                        rv_res_item_comp_tax.rv_res_item_comp_id = rv_res_item_comp.rv_res_item_comp_ix
                ) as tax_code,
                '" . $invoiceDetails['invoice_currency'] . "' as currency,
                rt_component.rt_component_desc as name,
                'component' as type,
                CONCAT('CM_', rv_res_item_comp.rv_res_item_comp_ix) as 'fs_inv_item_id'
            FROM
                rv_res_item_comp
                LEFT JOIN rt_component ON rt_component.rt_component_ix = rv_res_item_comp.rt_component_id
                LEFT JOIN fn_invoice_item ON fn_invoice_item.fn_invoice_item_ix = rv_res_item_comp.fn_invoice_item_id
            WHERE
                rv_res_item_comp.rv_res_item_comp_ix IN ('" . join("','", $invoiceComponents) . "')
            ORDER BY
                unit_nett_amount
            ", 6);

        $invoiceItemExtras = $lDB->get("
            SELECT
                CONCAT('EX_', rv_extra.ac_extra_id) as product_code,
                CONCAT('EC_', ac_extra.ac_extra_category_id) as product_category_code,
                (rv_extra.rv_extra_amt_nett - rv_extra.rv_extra_amt_tax) / rv_extra.rv_extra_units as unit_nett_amount,
                rv_extra.rv_extra_units as qty,
                (
                    SELECT
                        GROUP_CONCAT(rv_extra_tax.rf_tax_rate_id ORDER BY rv_extra_tax.rv_extra_tax_ix SEPARATOR ',')
                    FROM
                        rv_extra_tax
                    WHERE
                        rv_extra_tax.rv_extra_id = rv_extra.rv_extra_ix
                ) as tax_code,
                rf_currency.rf_currency_ix as currency,
                ac_extra.ac_ext_desc as name,
                'extra' as type,
                CONCAT('EX_', rv_extra.rv_extra_ix) as 'fs_inv_item_id'
            FROM
                rv_extra
                LEFT JOIN rf_currency ON rf_currency.rf_currency_ix = rv_extra.rf_currency_id
                LEFT JOIN ac_extra ON ac_extra.ac_extra_ix = rv_extra.ac_extra_id
            WHERE
                rv_extra.rv_extra_ix IN ('" . join("','", $invoiceExtras) . "')
            ORDER BY
                unit_nett_amount
        ", 6);

        $invoiceDetails['items'] = array_merge($invoiceItemComponents, $invoiceItemExtras);

        $taxRates = $GLOBALS['lDB']->get(
            "
                SELECT
                    rf_tax_rate_ix,
                    rf_tax_rate_vat
                FROM
                    rf_tax_rate
            ",
            2
        );
        $taxRateVatStatuses = array();
        foreach ($taxRates as $taxRate) {
            $taxRateVatStatuses[$taxRate['rf_tax_rate_ix']] = $taxRate['rf_tax_rate_vat'];
        }

        foreach ($invoiceDetails['items'] as $compKey => $item) {
            $taxIDs = explode(',', $item['tax_code']);
            if (count($taxIDs) == 1) {
                $chosenTaxID = $taxIDs[0];
            }
            $chosenTaxID = "";
            foreach ($taxIDs as $taxID) {
                $taxCommodity = $this->taxCommodityMapped($taxID, $invoiceDetails['pr_business_id']);
                if (!empty($taxCommodity)) {        // we have a mapped commodity
                    if (
                        in_array(
                            substr($taxCommodity, 0, 3),
                            ["CM_", "EX_", "EC_"]
                        )
                    ) {
                        $taxCommodity = substr($taxCommodity, 3);
                    }
                    $commodityItemComponent = $this->getComponent($taxCommodity, $invoiceDetails['invoice_currency']);
                    $commodityItemExtra = $this->getExtra($taxCommodity);

                    if (!empty($commodityItemComponent)) {
                        $commodityItem = $commodityItemComponent;
                        $taxPercentage = $lDB->get("SELECT rv_res_item_comp_tax_perc FROM rv_res_item_comp_tax WHERE rf_tax_rate_id = '" . $taxID . "'", 4);
                    } else if (!empty($commodityItemExtra)) {
                        $commodityItem = $commodityItemExtra;
                        $taxPercentage = $lDB->get("SELECT rv_extra_tax_perc FROM rv_extra_tax WHERE rf_tax_rate_id = '" . $taxID . "'", 4);
                    } else {
                        continue;
                    }

                    $commodityTaxes = explode(',', $commodityItem['tax_code']);
                    if (count($commodityTaxes) > 1) {
                        $commodityItem['tax_code'] = $commodityTaxes[0];    // use first tax ID
                    }
                    $commodityItem['unit_nett_amount'] = ($item['unit_nett_amount'] / 100) * $taxPercentage * $item['qty'];
                    $invoiceDetails['items'][] = $commodityItem;
                } else {
                    if (
                        empty($chosenTaxID) &&
                        in_array($taxRateVatStatuses[$taxID], array("0", "1"))
                    ) {      // use first VAT tax ID
                        $chosenTaxID = $taxID;
                    }
                }
            }
            if (!empty($chosenTaxID)) {
                $invoiceDetails['items'][$compKey]['tax_code'] = $chosenTaxID;
            }
        }

        return $invoiceDetails;
    }

    function taxCommodityMapped($taxId, $pr_business_id)
    {
        return $GLOBALS['lDB']->get(
            "
                SELECT
                    fs_commodity_id
                FROM
                    fs_tax_commodity
                WHERE
                    pr_business_id = '" . $pr_business_id . "'
                    AND rf_tax_rate_id = '" . $taxId . "'
            ",
            4
        );
    }

    function getComponent($id, $currency)
    {
        global $lDB;

        return $lDB->get("
            SELECT
                CONCAT('CM_', rt_component.rt_component_ix) as product_code,
                '' as product_category_code,
                (rv_res_item_comp.rv_item_comp_amt_nett - rv_res_item_comp.rv_item_comp_amt_tax) as unit_nett_amount,
                1 as qty,
                (
                    SELECT
                        GROUP_CONCAT(rv_res_item_comp_tax.rf_tax_rate_id ORDER BY rv_res_item_comp_tax.rv_res_item_comp_tax_ix SEPARATOR ',')
                    FROM
                        rv_res_item_comp_tax
                    WHERE
                        rv_res_item_comp_tax.rv_res_item_comp_id = rv_res_item_comp.rv_res_item_comp_ix
                ) as tax_code,
                '" . $currency . "' as currency,
                rt_component.rt_component_desc as name,
                'component' as type
            FROM
                rv_res_item_comp
                LEFT JOIN rt_component ON rt_component.rt_component_ix = rv_res_item_comp.rt_component_id
                LEFT JOIN fn_invoice_item ON fn_invoice_item.fn_invoice_item_ix = rv_res_item_comp.fn_invoice_item_id
            WHERE
                rv_res_item_comp.rt_component_id = '" . $id . "'
        ", 1);
    }

    function getExtra($id)
    {
        global $lDB;

        return $lDB->get("
            SELECT
                CONCAT('EX_', ac_extra.ac_extra_ix) as product_code,
                ac_extra.ac_extra_category_id as product_category_code,
                1 as qty,
                IF(
                    ac_extra.rf_tax_ind=10,
                    ac_extra.rf_tax_id,
                    (
                        SELECT
                            GROUP_CONCAT(rt_tax_group_item.rf_tax_rate_id ORDER BY rt_tax_group_item.rf_tax_rate_id SEPARATOR ',')
                        FROM
                            rt_tax_group_item
                        WHERE
                            rt_tax_group_item.rt_tax_group_id = ac_extra.rf_tax_id
                    )
                ) as tax_code,
                ac_extra.rf_currency_id as currency,
                ac_extra.ac_ext_desc as name,
                'extra' as type
            FROM
                ac_extra
            WHERE
                ac_extra_ix = '" . $id . "'
        ", 1);
    }

    function getFiscalatorKeyByInvoice($fn_invoice_id)
    {
        $fiscalatorKey = $GLOBALS['lDB']->get(
            "
                SELECT
                    pr_business.fiscalator_key
                FROM
                    fn_invoice
                    LEFT JOIN pr_business ON pr_business.pr_business_id = fn_invoice.pr_business_id
                WHERE
                    fn_invoice.fn_invoice_ix = '" . $fn_invoice_id . "'
            ",
            4
        );
        return $fiscalatorKey;
    }

    function getFiscalatorKeyByInvoiceUnit($pr_business_id)
    {
        $fiscalatorKey = $GLOBALS['lDB']->get(
            "
                SELECT
                    fiscalator_key
                FROM
                    pr_business
                WHERE
                    pr_business_id = '" . $pr_business_id . "'
            ",
            4
        );
        return $fiscalatorKey;
    }

    function updateInvoiceStatus($fn_invoice_id, $status)
    {
        global $lDB;

        return $lDB->put("
            UPDATE
                fn_invoice 
            SET
                fn_inv_fiscal_status_ind = '" . $status . "' 
            WHERE
                fn_invoice_ix = '" . $fn_invoice_id . "'
        ");
    }

    function getInvoiceStatus($fn_invoice_id)
    {
        global $lDB;
        if (empty($fn_invoice_id)) {
            $errors[] = $fn_invoice_id . ": No Invoice items found";
            return $errors;
        }
        $status = $lDB->get("
            SELECT
                fn_inv_fiscal_status_ind
            FROM
                fn_invoice
            WHERE
                fn_invoice_ix = '" . $fn_invoice_id . "'
        ", 4);
        return !empty($status) ? $status : false;
    }

    function logFnFiscalApi($invoices, $callResult=array(), $initiateCallTime, $endCallTime="")
    {
        // log of transaction communication with Fiscalator
        global $lDB;

        $endCallTime = empty($endCallTime) ? $initiateCallTime : $endCallTime;

        foreach ($invoices as $key => $invoice) {

            $result = $lDB->put("
                INSERT INTO fn_invoice_fiscal_api
                (
                    fn_invoice_fiscal_api_id,
                    fn_invoice_id,
                    fn_api_post_rr,
                    fn_api_post_fs,
                    fn_api_time_post,
                    fn_api_time_response
                ) VALUES (
                    (SELECT GET_UUID()),
                    '" . $lDB->escape($invoice['invoice_id']) . "',
                    '" . $lDB->escape(json_encode($invoice)) . "',
                    '" . $lDB->escape(json_encode($callResult)) . "',
                    '" . $lDB->escape($initiateCallTime) . "',
                    '" . $lDB->escape($endCallTime) . "'
                )
            ");
        }
        return $result;
    }

    function createFiscalisedInvoiceRecord($fn_invoice_id, $invoiceDetails)
    {
        global $lDB;

        if ($this->existsFiscalisedInvoiceRecord($fn_invoice_id)) { 
            return true; 
        }
        
        return $lDB->put("
            INSERT INTO fs_invoice
            (
                fs_invoice_db,
                fs_invoice_ix,
                fn_invoice_id,
                fs_data,
                fs_time_created
            ) VALUES (
                '" . $lDB->escape($GLOBALS['dbcode']) . "',
                (SELECT GET_UUID()),
                '" . $lDB->escape($fn_invoice_id) . "',
                '" . $lDB->escape("Submitted: " . json_encode($invoiceDetails)) . "',
                now()
            )
        ");
    }

    function existsFiscalisedInvoiceRecord($fn_invoice_id)
    {
        global $lDB;
        $records = $lDB->get("
            SELECT
                count(fn_invoice_id)
            FROM
                fs_invoice
            WHERE
                fn_invoice_id = '$fn_invoice_id'
        ", 4);
        return $records == 0 ? false : true;
    }

    public function updateFiscalisedInvoiceRecord($callResult=[], $fn_inv_fiscal_status_ind="", $fn_invoice_id="")
    {
        global $lDB;

        if (empty($callResult)) {
            return false;
        }
        $result = array();

        $callResult = json_decode(json_encode($callResult), true);
        if (
            !empty($callResult)
            && (
                !isset($callResult['transactions'])
                || (
                    empty($callResult['transactions'])
                    || is_null($callResult['transactions'])
                    || $callResult['transactions'][0] == null
                )
            )
            && $fn_invoice_id != ""
        )
        {
            // No transactions found for invoice, but it has already been submitted
            switch ($fn_inv_fiscal_status_ind) {
                case 1:
                    $this->updateInvoiceStatus($fn_invoice_id, 4); // status: Pending Invoice
                    break;
                case 5:
                    $this->updateInvoiceStatus($fn_invoice_id, 8); // status: Pending Credit note
                    break;
                default:
                    break;
            }
            
            return false;
        }

        foreach ($callResult['transactions'] as $invoice) {
            if (empty($invoice)) {
                continue;
            }
            $invoice = json_decode(json_encode($invoice), true);
            $fiscalInvNoUpdateSql = "";
            $fiscalInvDocNo = null;
            $fiscalSearchNo = "";

            $invoiceDetails = $lDB->get("
                SELECT
                    fn_inv_fiscal_status_ind as status,
                    rv_reservation_id as resId
                FROM
                    fn_invoice
                WHERE
                    fn_invoice_ix = '" . $invoice['invoiceId'] . "'
            ", 1);

            $fiscalatorStatuses = array(
                "invoiceVerified"   => 2,
                "invoiceError"      => 4,
                "error"             => 4,
                "creditPending"     => 6,
                "creditApproved"    => 7,
                "creditError"       => 8
            );

            if ($invoiceDetails['status'] != $fiscalatorStatuses[ $invoice['statusDesc'] ]) {
                $this->updateInvoiceStatus(
                    $invoice['invoiceId'],
                    $fiscalatorStatuses[ $invoice['statusDesc'] ]
                );
                
                // Audit trail
                switch ($fiscalatorStatuses[ $invoice['statusDesc'] ]) {
                    case 2:
                        ammendReservation($invoiceDetails['resId'], "Invoice fiscalisation verified: " . $invoice['invoiceId']);
                        break;
                    case 7:
                        ammendReservation($invoiceDetails['resId'], "Credit note fiscalisation approved: " . $invoice['invoiceId']);
                        break;
                }
            }

            $fiscalDocExcludes = array("qrcode", "approvestatus", "issuedt");
            $fiscalDocArray = array();
            if (isset($invoice['fiscalDocFields'])) {
                // Iterate through fiscal document fields, add them to DB for searching purposes
                foreach ($invoice['fiscalDocFields'] as $key => $fiscalDocField) {
                    if (!in_array(strtolower($key), $fiscalDocExcludes)) {
                        if (!empty(trim($fiscalDocField))) {
                            $fiscalDocArray[] = trim($fiscalDocField);
                        }
                    }
                }
            }

            $result[] = $lDB->put("
                UPDATE
                    fs_invoice
                SET 
                    fs_data = '" . $lDB->escape(json_encode($invoice)) . "',
                    fs_time_changed = now(),
                    fn_invoice_doc = '" . $lDB->escape(join( ',', $fiscalDocArray )) . "'
                WHERE
                    fn_invoice_id = '" . $invoice['invoiceId'] . "'
            ");
        };

        return $result;
    }

    function getFiscalData($invoiceId)
    {
        global $lDB;
        
        if(empty($invoiceId)) {
            return false;
        }

        $fs_data = $lDB->get("
            SELECT 
                fs_data
            FROM
                fs_invoice
            WHERE
                fn_invoice_id = '" . $invoiceId . "'
        ", 4);

        return json_decode($fs_data, true);
    }

    function triggerProcessInvoices($invoiceList)
    {
        if(empty($invoiceList)) {
            return false;
        }

        $callResult = "";
        $endpoint   = "";

        if(!is_array($invoiceList)) {
            $callResult = json_decode(
                $this->fiscalatorApiCall($this->getFiscalatorKeyByInvoice($invoiceList), "POST", $endpoint, array($invoiceList)),
                true
            );
        } else {
            $newInvoiceList = array();
            foreach($invoiceList as $key => $invoice) {
                $callResult = json_decode(
                    $this->fiscalatorApiCall($this->getFiscalatorKeyByInvoice($invoiceList), "POST", $endpoint, array($invoiceList)),
                    true
                );
            }
        }

        return $callResult;
    }

    function queryInvoice($fn_invoice_id)
    {
        if (empty($fn_invoice_id)) {
            $errors[] = $fn_invoice_id . ": No Invoice items found";
            return $errors;
        }
        $fn_inv_fiscal_status_ind = $this->getInvoiceStatus($fn_invoice_id);

        $fiscalatorKey = $this->getFiscalatorKeyByInvoice($fn_invoice_id);

        $endpoint = "invoice/";

        $callResult = json_decode(
            $this->fiscalatorApiCall($fiscalatorKey, "GET", $endpoint . $fn_invoice_id),
            true
        );
        $this->updateFiscalisedInvoiceRecord($callResult, $fn_inv_fiscal_status_ind, $fn_invoice_id);
        
        $returnResult = array(
            "results" => $callResult,
            "errors" => ""
        );

        return $returnResult;
    }

    function queryCreditApproval($fn_invoice_id)
    {
        // Front end check to prevent reversals when linked and fiscalised

        $this->fisDebug("function queryCreditApproval");
        if (empty($fn_invoice_id)) {
            $errors[] = $fn_invoice_id . ": No Invoice items found";
            return $errors;
        }
        $fn_inv_fiscal_status_ind = $this->getInvoiceStatus($fn_invoice_id);

        $fiscalatorKey = $this->getFiscalatorKeyByInvoice($fn_invoice_id);

        $endpoint = "querycreditapproval/";

        $callResult = json_decode(
            $this->fiscalatorApiCall($fiscalatorKey, "GET", $endpoint . $fn_invoice_id),
            true
        );
        $this->updateFiscalisedInvoiceRecord($callResult, $fn_inv_fiscal_status_ind, $fn_invoice_id);
        
        $returnResult = array(
            "results" => $callResult,
            "errors" => ""
        );

        return $returnResult;
    }

    function fiscalatorApiCall($fiscalatorKey, $method, $endpoint, $data=array())
    {

        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $this->removeSlashes($this->fiscalatorUrl) . "/" . $this->removeSlashes($endpoint));
        curl_setopt($curl, CURLOPT_POST, (strtolower($method) == "post" ? true : false));
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($curl, CURLOPT_HTTPHEADER, array(
            "key: " . $fiscalatorKey,
        ));
        if (!empty($data)) {
            curl_setopt($curl, CURLOPT_POSTFIELDS, [
                "data" => json_encode($data)
            ]);
        }
        $result = curl_exec($curl);
        curl_close($curl);
        return $result;
    }

    function removeSlashes($string)
    {
        if (substr($string, 0, 1) == "/") {
            $string = substr($string, 1);
        }
        if (substr($string, -1) == "/") {
            $string = substr($string, 0, (strlen($string)-1));
        }
        return $string;
    }

    function findResponseIdNames($callResult)
    {
        global $lDB;

        // Try to find names to ID's cited in mapping error codes

        if (
            is_array($callResult) &&
            isset($callResult['transactions']) &&
            !empty($callResult['transactions'])
        ) {
            foreach ($callResult['transactions'] as $tranKey => $tran) {
                if (isset($tran['responseMessage']) && !empty($tran['responseMessage'])) {
                    if (substr($tran['responseMessage'], 0, 24) == "Tax rate is not mapped: ") {
                        $names = array();
                        $ids = explode(',', substr($tran['responseMessage'], 38));
                        foreach ($ids as $id) {
                            $name = $lDB->get("
                                SELECT 
                                    rf_tax_rate_desc as name
                                FROM
                                    rf_tax_rate
                                WHERE
                                    rf_tax_rate_ix = '" . $id . "'
                            ", 4);
                            $names[] = !empty($name) ? $name : $id;
                        }
                        $callResult['transactions'][$tranKey]['responseMessage'] = substr($tran['responseMessage'], 0, 24) . join(', ', $names);
                    }

                    if (substr($tran['responseMessage'], 0, 29) == "Currency code is not mapped: ") {
                        $names = array();
                        $id = substr($tran['responseMessage'], 29);
                        $name = $lDB->get("
                            SELECT 
                                rf_currency_name as name
                            FROM
                                rf_currency
                            WHERE
                                rf_currency_ix = '" . $id . "'
                        ", 4);
                        $name = !empty($name) ? $name : $id;
                        $callResult['transactions'][$tranKey]['responseMessage'] = substr($tran['responseMessage'], 0, 29) . $name;
                    }

                    if (substr($tran['responseMessage'], 0, 39) == "Item code is not mapped to an HS code: ") {
                        $names = array();
                        $ids = explode(',', substr($tran['responseMessage'], 39));
                        foreach ($ids as $id) {
                            // Try components
                            $searchId = $id;
                            $searchId = substr($id, 0, 3) == "CM_" ? substr($id, 3) : $searchId;
                            $searchId = substr($id, 0, 3) == "EX_" ? substr($id, 3) : $searchId;
                            $name = $lDB->get("
                                SELECT 
                                    rt_component_desc as name
                                FROM
                                    rt_component
                                WHERE
                                SUBSTRING(rt_component_ix, 1, 33) = '" . $searchId . "'
                            ", 4);
                            if (!empty($name)) {
                                $names[] = $name;
                            }
                            // Try Extras
                            $name = $lDB->get("
                                SELECT 
                                    ac_ext_desc as name
                                FROM
                                    ac_extra
                                WHERE
                                SUBSTRING(ac_extra_ix, 1, 33) = '" . $searchId . "'
                            ", 4);
                            if (!empty($name)) {
                                $names[] = $name;
                            }
                            if (empty($names)) {
                                $names[] = $searchId;
                            }
                        }
                        $callResult['transactions'][$tranKey]['responseMessage'] = substr($tran['responseMessage'], 0, 39) . join(' or ', $names);
                    }                    
                }
            }
        }
        return $callResult;
    }

    public function fisDebug($message) 
    {
        if ($this->fisDebug) {
            if (is_array($message)) {
                error_log(serialize($message));
            } else {
                error_log($message);
            }
        }
    }

// Tims specific stuff

/**
  * Maybe extract to separate file to be included by class.fiscalator.php - Fiscalisation.
  * specifically for Tims
  */

    function findNegatives($items) 
    {
        $negativeItems = [];
        foreach($items as $key => $item) {
            if ($item['unit_nett_amount'] < 0) {
                $negativeItems[$key] = $item;
            }
        }
        return $negativeItems;
    }

    function consolidateAmountsByTaxCode($allItems, $negativeItems) 
    {
        $this->fisDebug("function consolidateAmountsByTaxCode()");
        $itemList = $allItems;
        $newItemList = [];
        $largestPositiveAmount = [];
        $originalItemListCount = count($itemList);
        foreach($negativeItems as $negativeItem) {
            $selectedTaxCode = $negativeItem['tax_code'];
            $consolidatedTotal = 0;
            $largestPositiveAmount[$selectedTaxCode] = 0;
            $matchedItems = [];
            $consolidatedItem = [];

            foreach($itemList as $key => $item) {
                if($item['tax_code'] == $selectedTaxCode) {
                    $consolidatedTotal += ($item['unit_nett_amount'] * $item['qty']);
                    if ($item['unit_nett_amount'] >= $largestPositiveAmount[$selectedTaxCode]) {
                        $largestPositiveAmount[$selectedTaxCode] = $item['unit_nett_amount'];
                        $consolidatedItem = $item;
                    }
                    $matchedItems[$key] = $itemList[$key];
                    unset($itemList[$key]);
                }
            }
            if($consolidatedTotal >= 0) {
                $consolidatedItem['unit_nett_amount'] = $consolidatedTotal;
                array_push($newItemList,$consolidatedItem);
            }
            else {
                foreach($matchedItems as $matchedItem) {
                    array_push($newItemList, $matchedItem);
                }
            }
        }
        if(!empty($itemList)) {
            foreach($itemList as $newItem) {
                array_push($newItemList, $newItem);
            }
        }
        if (count($newItemList) < $originalItemListCount) {
            return $newItemList;
        }
        else {
            return false;
        }
    }

    function otherInvoicesOnRes($selectedInvoiceDetails)
    {
        // Find invoices on reservation with matching customers
        $this->fisDebug("function otherInvoicesOnRes()");
        global $lDB;
        $fn_invoice_id = $selectedInvoiceDetails['invoice_id'];
        $selectedInvoiceReservation = $lDB->get("
            SELECT
                rv_reservation_id,
                pr_business_id,
                fn_inv_curr
            FROM
                fn_invoice
            WHERE
                fn_invoice_ix = '" . $fn_invoice_id . "'
        ", 1);
        /*
        * Cash sales sometimes get a '0' and sometimes an empty field in the fn_folio.fn_folio_to_id field.
        * This causes a little bit of pain when matching credits with existing invoices.
        * We need to take this into account.
        */
        $fn_folio_to_id = $lDB->get("
            SELECT
                fn_folio.fn_folio_to_id
            FROM
                fn_folio
            WHERE
                fn_folio.fn_invoice_id = '" . $fn_invoice_id . "'
        ",4);
        $new_fn_folio_to_id_filter = (empty($fn_folio_to_id) || $fn_folio_to_id == '0') ? "0', '" : $fn_folio_to_id;
        $otherInvoiceIdList = $lDB->get("
            SELECT
                fn_invoice.fn_invoice_ix
            FROM
                fn_invoice
                LEFT JOIN fn_folio on fn_folio.fn_folio_ix = fn_invoice.fn_folio_id
            WHERE
                fn_invoice.fn_inv_status_ind = 2
                AND
                fn_invoice.fn_inv_fiscal_status_ind = 2
                AND
                fn_invoice.fn_inv_amt_payable >= ABS(" . $selectedInvoiceDetails['amount'] . ")
                AND
                fn_invoice.pr_business_id = '" . $selectedInvoiceReservation['pr_business_id'] . "'
                AND
                fn_invoice.fn_inv_curr = TRIM('" . $selectedInvoiceReservation['fn_inv_curr'] . "')
                AND
                fn_invoice.fn_invoice_ix != '" . $fn_invoice_id . "'
                AND
                fn_folio.fn_folio_to_id IN ('" . $new_fn_folio_to_id_filter . "')
                AND
                fn_invoice.rv_reservation_id = '" . $selectedInvoiceReservation['rv_reservation_id'] . "'
            ORDER BY fn_invoice.fn_inv_amt_payable ASC
        ", 3);
        return $otherInvoiceIdList;

    }

    function queryInvoiceCreditLink($fn_invoice_id)
    {
        $this->fisDebug("function queryInvoiceCreditLink()");
        if (empty($fn_invoice_id)) {
            $errors[] = $fn_invoice_id . ": No Invoice items found";
            return $errors;
        }
        $invoiceDetails = $this->getInvoiceComponentsExtras($fn_invoice_id);
        $linkedInvoiceList = $this->getLinkedItemList($invoiceDetails['items']);
        $linkedCreditNoteList = $this->getLinkedCreditNotes($fn_invoice_id);
        $linkedItemList = array_merge($linkedInvoiceList, $linkedCreditNoteList);

        return $linkedItemList;
    }

    function getLinkedCreditNotes($fn_invoice_id)
    {
        $this->fisDebug("function getLinkedCreditNotes()");
        global $lDB;
        $linkedCreditNoteList =  $lDB->get("
            SELECT
                fs_inv_cr_link_inv_item_id
            FROM
                fs_invoice_credit_link
            WHERE
                fs_inv_cr_link_credit_id = '" . $fn_invoice_id . "'
        ",3);
        return $linkedCreditNoteList;
    }

    function getLinkedItemList($invoiceItemList)
    {
        $this->fisDebug("function getLinkedItemList()");
        global $lDB;
        if (!is_array($invoiceItemList)) {
            $invoiceItemList = array($invoiceItemList);
        }

        $LinkedItemList = $lDB->get("
            SELECT
                fs_inv_cr_link_inv_item_id
            FROM
                fs_invoice_credit_link
            WHERE
                fs_inv_cr_link_inv_item_id IN ('" . join("','", array_column($invoiceItemList, 'fs_inv_item_id')) . "')
        ",3);
        return $LinkedItemList;
    }

    private function matchInvoiceItems($invoiceDetails, $otherInvoiceIdList) 
    {
        $this->fisDebug("matchInvoiceItems()");
        if (empty($invoiceDetails['items'])) {
            $this->fisDebug("No invoice items found for matching.");
            return array(
                "errors" => "No invoice items found for matching."
            );
        }
        $selectedInvoiceItemCount = 0;
        foreach($invoiceDetails['items'] as $item) {
            if ($item['unit_nett_amount'] != 0) {   
                $selectedInvoiceItemCount ++;
            }
        }
        // Step through each credit note item and check for matching tax code with >= amount.
        // If all fit, then link, otherwise error state. Cannot link a credit note to multiple invoices.
        $linkList = [];

        foreach($otherInvoiceIdList as $otherInvoiceId) { 
            $otherInvoiceDetails = $this->getInvoiceComponentsExtras($otherInvoiceId);
            $headerTotals = ($otherInvoiceDetails['amount'] >= abs($invoiceDetails['amount']));
            $otherInvoiceItemCount = count($otherInvoiceDetails['items']);
            $linkedItemList = $this->getLinkedItemList($otherInvoiceDetails['items']);
            if (
                $headerTotals && 
                $selectedInvoiceItemCount <= $otherInvoiceItemCount
            ) 
            {

                $errorState = true;
                $newItems = [];
                foreach($invoiceDetails['items'] as $key => $selectedInvoiceItem) {
                    if ($selectedInvoiceItem['unit_nett_amount'] != 0) {
                        foreach($otherInvoiceDetails['items'] as $otherInvoiceItem) {
                            if (
                                (
                                    empty($linkedItemList) ||
                                    !in_array($otherInvoiceItem['fs_inv_item_id'], $linkedItemList)
                                )  
                                && !in_array($otherInvoiceItem['fs_inv_item_id'], $linkList)
                                && $selectedInvoiceItem['tax_code'] == $otherInvoiceItem['tax_code'] 
                                && abs($selectedInvoiceItem['unit_nett_amount'] * $selectedInvoiceItem['qty']) <= $otherInvoiceItem['unit_nett_amount'] * $otherInvoiceItem['qty']
                            ) {
                                $this->fisDebug("Valid, unlinked invoice item");
                                $newInvoiceItem = $selectedInvoiceItem;
                                $newInvoiceItem['name'] = $otherInvoiceItem['name'];
                                $newInvoiceItem['unit_nett_amount'] = abs($selectedInvoiceItem['unit_nett_amount']);
                                $invoiceDetails['items'][$key] = $newInvoiceItem;
                                array_push($newItems, $newInvoiceItem);
                                array_push($linkList, $otherInvoiceItem['fs_inv_item_id']);
                                continue 2;
                            }
                            else {
                                // $this->fisDebug("Invalid item - already linked, wrong tax code or amount too small");
                            }
                        }
                    }
                    else {
                        array_push($newItems, $selectedInvoiceItem);
                    }
                }
        
                if (count($newItems) == 0) {
                    $this->fisDebug("Could not match items or already has linked Credit");
                    $linkList = [];
                    $newItems = [];
                }
                else {
                    $this->fisDebug("Found matching Items");
                    // Change negative amounts to positive.
                    // Header is not submitted to Tims, but is sent to Fiscalator and 
                    // appears in the fiscal query tool - change it to positive anyway
                    $invoiceDetails['amount'] = abs($invoiceDetails['amount']);
                    $invoiceDetails['original_invoice_id'] = $otherInvoiceId;
                    $fiscal_data = $this->getFiscalData($otherInvoiceId);
                    $invoiceDetails['fiscal_data'] = (isset($fiscal_data['fiscalDocFields']) && count($fiscal_data['fiscalDocFields']) == 1) ? $fiscal_data['fiscalDocFields'] : "";
            
                    return array(
                        "newInvoiceDetails" => $invoiceDetails,
                        "linkList" => $linkList
                    );
                }
            }
        }
    }

    function updateCreditLink($invoiceId, $linkList) 
    {
        global $lDB;
        $linkTableCount = 0;
        foreach ($linkList as $linkItem) {
            $linking = $lDB->put("  
                INSERT INTO fs_invoice_credit_link
                    (
                        fs_invoice_credit_link_db, 
                        fs_inv_cr_link_inv_item_id, 
                        fs_inv_cr_link_credit_id, 
                        fs_time_created
                    )
                    VALUES(
                        '" . $lDB->escape($GLOBALS['dbcode']) . "',
                        '" . $lDB->escape($linkItem) . "',
                        '" . $lDB->escape($invoiceId) . "',
                        now()
                    )
            ");
            $linkTableCount ++; 
        }
        return $linkTableCount;
    }

}
