<?php

require_once(__DIR__ . '/class.financial.export.php');

/**
 * Export invoices in the format used by the Great Plains accounting system.
 */
class GreatPlainsInvoiceExport extends FinancialExport
{
    /**
     * The configuration settings for the Great Plains interface.
     *
     * @var array
     */
    protected $options;

    /**
     * The id of the batch the invoices should be pulled from.
     *
     * @var string
     */
    protected $batchId;

    /**
     * Create a new instance of a Great Plains invoice export.
     *
     * @param MySQLDB $lDB The connection to the database.
     * @param array $options The options for the Great Plains financial interface.
     * @param string $batchId The id of the batch that invoices belong to.
     */
    public function __construct($lDB, $options, $batchId)
    {
        parent::__construct($lDB);
        $this->options = $options;
        $this->batchId = $batchId;

        $this->init();
    }

    /**
     * Initialise the export.
     *
     * @return void
     */
    protected function init()
    {
        $this->buildHeader();
        $this->buildFields();
        $this->buildTables();
        $this->buildFilters();
        $this->buildDataFields();
    }

    protected function buildHeader()
    {
        $this->setHeader([
            'BATCH_ID',
            'DOC_DATE',
            'DOC_TYPE',
            'DEBTOR_ID',
            'DEBTOR_NAME',
            'RES_NO',
            'RES_ARR_DATE',
            'RES_DEP_DATE',
            'ITEM_NO',
            'ITEM_DESC',
            'QTY',
            'AMT_EXCL',
            'TAX_OPTION',
            'TAX_SCHEDULE',
            'VAT_AMT',
            'REV_CTR',
            'ADD_1',
            'ADD_2',
            'VOUCHER_NUMBER'
        ]);
    }

    protected function buildFields()
    {
        $this->addFields([
            'fn_tran.fn_batch_id AS batch_id',
            'fn_tran.fn_tran_date_ledger AS invoice_date',
            'fn_debtor.fn_debtor_code AS debtor_code',
            'debtor_persona.pr_name_last AS debtor_name_last',
            'debtor_persona.pr_name_first AS debtor_name_first',
            'fn_tran.rv_reservation_id AS res_number',
            'rv_reservation.rv_date_arrive AS arrival_date',
            'rv_reservation.rv_date_depart AS departure_date',
            'fn_ledger.fn_ledger_code AS gl_code',
            'fn_tran_item.fn_tran_item_note AS item_description',
            '1 AS quantity',
            'tax_ledger.fn_ledger_code AS tax_code',
            'fn_tran_item.fn_tran_item_amt',
            'fn_tran_item.fn_tran_item_amt_incl',
            'fn_cost_centre.fn_cost_centre_code AS property_code',
            'rv_reservation.rv_res_name AS res_name',
            'fn_tran.fn_tran_link_id AS invoice_number',
            'rv_reservation.rv_agent_ref AS agent_ref'
        ]);
    }

    protected function buildTables()
    {
        $this->setTable('fn_tran');
        $this->addJoins([
            'INNER JOIN fn_tran_item ON fn_tran_item.fn_tran_id = fn_tran.fn_tran_ix',
            'LEFT JOIN fn_ledger ON fn_ledger.fn_ledger_ix = fn_tran_item.fn_ledger_id',
            'LEFT JOIN fn_folio ON fn_folio.fn_folio_ix = fn_tran.fn_folio_id',
            'LEFT JOIN fn_cost_centre ON fn_cost_centre.fn_cost_centre_ix = fn_tran_item.fn_tran_cost_ctr1_id',
            'LEFT JOIN rf_tax_inv ON rf_tax_inv.rf_tax_inv_ix = fn_tran_item.rf_tax_inv_id',
            'LEFT JOIN fn_ledger AS tax_ledger ON tax_ledger.fn_ledger_ix = rf_tax_inv.fn_ledger_id',
            'LEFT JOIN pr_persona AS debtor_persona ON debtor_persona.pr_persona_ix = fn_folio.fn_folio_to_id',
            'INNER JOIN rv_reservation ON rv_reservation.rv_reservation_ix = fn_tran.rv_reservation_id',
            'LEFT JOIN fn_tran_item AS debtor_item ON debtor_item.fn_tran_id = fn_tran_item.fn_tran_id AND debtor_item.fn_tran_item_type_ind = 10',
            'LEFT JOIN fn_debtor ON fn_debtor.fn_debtor_ix = debtor_item.fn_debtor_id',
        ]);
    }

    protected function buildFilters()
    {
        if (!empty($this->batchId) && $this->batchId != 'all' && $this->batchId != '0') {
            $this->addFilter("
                fn_tran.fn_batch_id = '{$this->batchId}'
            ");     // From the specified batch only
        }

        $this->addFilter("
            fn_tran.fn_tran_link_ind = '".DB_FN_TRAN_LINK_INVOICE."'
        ");     // Invoices only

        $this->addFilter("
            fn_tran_item.fn_tran_item_type_ind <> " . DB_FN_TRAN_ITEM_TYPE_TAX . "
        ");     // No tax items

        $this->addFilter("
            fn_tran_item.fn_tran_item_type_ind <> " . DB_FN_TRAN_ITEM_TYPE_DEBTOR . "
        ");     // No tax items
    }

    protected function buildDataFields()
    {
        $this->setDataFields([
            'batch_id',
            'invoice_date',
            'document_type',
            'debtor_code',
            'debtor_name',
            'res_number',
            'arrival_date',
            'departure_date',
            'gl_code',
            'item_description',
            'quantity',
            'unit_price',
            'tax_options',
            'tax_code',
            'tax_amount',
            'property_code',
            'res_name',
            'invoice_number',
            'agent_ref'
        ]);
    }

    protected function processData(&$data)
    {
        foreach($data as &$row) {
            $row['invoice_date'] = date('d/m/Y', strtotime($row['invoice_date']));
            $row['document_type'] = 'Invoice';

            if ($row['debtor_name_first']) {
                $row['debtor_name'] = $row['debtor_name_first'] . ' ' . $row['debtor_name_last'];
            } else {
                $row['debtor_name'] = $row['debtor_name_last'];
            }

            $row['arrival_date'] = date('d/m/Y', strtotime($row['arrival_date']));
            $row['departure_date'] = date('d/m/Y', strtotime($row['departure_date']));

            $row['unit_price'] = $row['fn_tran_item_amt'] * -1;

            if ($row['tax_code']) {
                $row['tax_options'] = 'Taxable';
                $row['tax_amount'] = $row['fn_tran_item_amt_incl'] - $row['fn_tran_item_amt'];
                $row['tax_amount'] *= -1;
            } else {
                $row['tax_options'] = 'Non taxable';
                $row['tax_code'] = '';
                $row['tax_amount'] = 0;
            }

            fixAmtDisplay($row['unit_price']);
            fixAmtDisplay($row['tax_amount']);
        }
    }
}