<?php

 /**
  * class.inventory.php - Inventory Processing - TallOrder
  */
  
define("STATUS_SUCCESS",200);
define("STATUS_INVALID",400);

define("ITEM_TRAN_IND_POR", 1);
define("ITEM_TRAN_IND_SUPPLIER_RETURN", 2);
define("ITEM_TRAN_IND_COS", 3);
define("ITEM_TRAN_IND_ISSUE", 4);
define("ITEM_TRAN_IND_TFR", 5);
define("FN_ITEM_TRAN_IND_POR", 51);
define("FN_ITEM_TRAN_IND_RETURN", 52);
define("FN_ITEM_TRAN_IND_COS", 53);
define("FN_ITEM_TRAN_IND_ISSUE", 54);
define("FN_ITEM_TRAN_IND_TFR", 55);
define("FN_TRAN_ITEM_TYPE_SUPPLIER", 70);
define("FN_TRAN_ITEM_TYPE_INVENTORY", 71);
define("FN_TRAN_ITEM_TYPE_INPUT_VAT", 72);
define("FN_TRAN_ITEM_TYPE_EXPENSE", 73);

require_once(__DIR__ . '/db.pr_business.php');
require_once(__DIR__ . '/db.fn_tran.php');
require_once(__DIR__ . '/db.fn_tran_item.php');
require_once(__DIR__ . '/db.fn_ledger.php');
require_once(__DIR__ . '/functions.datetime.php');
require_once(__DIR__ . '/class.audit.php');
require_once(__DIR__ . '/class.newrelic.php');

class Inventory{

	private $tranUrls = array();

	public  $InventoryInvoiceUnits = array();
	public  $PropChildrenIds = array(); // all inventory child property ids
	public  $propertyList = array();
	public  $invoiceUnitChildProperties = array();
	public  $LocationList = array();

	public  $TranTypeData = array();	
	private $TranTypeDesc = ["", "Purchase Order", "Supplier Return", "Cost of Sales", "Stock Issue", "Stock Transfer"];
	public $FnTranTypeInd = ["", 51, 52, 53, 54, 55];

	private $POSAPICredentials;
	public  $pr_business_id;
	public  $currentUserId;
	public  $TenantId;
	public  $StoreId;
	private $lDB;
	private $first_tran_date;
	private $last_tran_date;
	private $errorLog = array();

	var $InventDebug = false;
	var $newRelic;

	public function __construct($pr_business_id = ''){
		$this->lDB = $GLOBALS['lDB'];
		$this->currentUserId = $_SESSION['userid'];
		$this->newRelic = new NewRelic();
		$this->fetchInventoryInvoiceUnits();
		$this->fetchLocations();
	}

	private function fetchInventoryInvoiceUnits() {
		$this->newRelic->record_transaction("Inventory -" . __FUNCTION__);

		$this->PropChildrenIds = array();
		$inventoryInvoiceUnits = $this->lDB->get("
			SELECT
				pr_business.pr_business_id,
				pr_persona.pr_name_last,
				pr_business.pr_bus_home_curr_id,
				pr_business.fn_ledger_id_input_vat,
				pr_business.pr_bus_post_inventory_yn,
				pr_business.pr_bus_post_in_po_yn,
				pr_business.pr_bus_api_url_po,
				pr_business.pr_bus_post_in_returns_yn,
				pr_business.pr_bus_api_url_returns,
				pr_business.pr_bus_post_in_cos_yn,
				pr_business.pr_bus_api_url_cos,
				pr_business.pr_bus_post_in_issues_yn,
				pr_business.pr_bus_api_url_issues,
				pr_business.pr_bus_post_in_trfs_yn,
				pr_business.pr_bus_api_url_transfers,
				pr_business.pr_bus_api_client,
				pr_business.pr_bus_api_token,
				pr_business.pr_bus_api_url_base
			FROM
				pr_business
				INNER JOIN pr_persona ON pr_persona.pr_persona_ix = pr_business.pr_business_id
			WHERE
					pr_bus_inactive_yn = '0' 
				AND 
				(
					pr_business.pr_bus_billing_yn = '1'
					OR 
					pr_business.pr_bus_billing_prop_yn = '1'
				)
				AND 
					pr_bus_post_inventory_yn = '1'
			ORDER BY 
				pr_business_sequence ASC
		",6);
		if (!empty($inventoryInvoiceUnits)) {
			foreach($inventoryInvoiceUnits as $inventoryInvoiceUnit) {
				$iUnitID = $inventoryInvoiceUnit['pr_business_id'];
				$this->InventoryInvoiceUnits[$iUnitID] = $inventoryInvoiceUnit;
				$this->PropChildrenIds = array_merge($this->PropChildrenIds, db_pr_business_get_properties($iUnitID));
				$this->invoiceUnitChildProperties[$iUnitID] = db_pr_business_get_properties($iUnitID);
			}
		};
		$newProp = array();
		foreach($this->PropChildrenIds as $property) {
			$newProp[$property] = $this->getPropertyName($property)[0];
		}
		asort($newProp);
		$this->propertyList = $newProp;
		$this->newRelic->stop_transaction();
	}

	public function getPropertyName($pr_persona_ix) {
		return $this->lDB->get("
			SELECT
				pr_persona.pr_name_last
			FROM
				pr_persona
			WHERE pr_persona_ix = '$pr_persona_ix'
		",3);
	}

	public function propertyOptions() {
		$properties = $this->lDB->get("
			SELECT
				pr_persona.pr_persona_ix, 
				pr_persona.pr_name_last
			FROM
				pr_persona
			WHERE pr_persona_ix IN ('" . 
				implode ( "', '", $this->PropChildrenIds) . 
			"')
			ORDER BY
				pr_persona.pr_name_last
		",6);
		$propertyOptions = "";
		foreach($properties as $property) {
			$propertyOptions[$property['pr_persona_ix']] = $property['pr_name_last'];
		}
		return $propertyOptions;
	}

	private function fetchLocations() {

		$list = $this->lDB->get("
		SELECT DISTINCT
			ac_stock.ac_stock_ix,
			ac_stock.ac_stock_db,
			ac_stock.fn_cost_centre_1_id,
			fn_cost_centre_1.fn_cost_centre_code AS fn_cost_centre_1_code,
			fn_cost_centre_1.fn_cost_centre_desc AS fn_cost_centre_1_desc,
			ac_stock.fn_cost_centre_2_id,
			fn_cost_centre_2.fn_cost_centre_code AS fn_cost_centre_2_code,
			fn_cost_centre_2.fn_cost_centre_desc AS fn_cost_centre_2_desc,
			ac_stock.pr_business_id,
			pr_persona.pr_name_last,
			ac_stock.ac_stock_key_pos,
			ac_stock.ac_stock_name_rr,
			ac_stock.ac_stock_inactive_yn,
			ac_stock.ac_stock_trf_yn
		FROM
			ac_stock
			LEFT JOIN pr_persona ON pr_persona.pr_persona_ix = ac_stock.pr_business_id 
			LEFT JOIN fn_cost_centre as fn_cost_centre_1 ON fn_cost_centre_1.fn_cost_centre_ix = ac_stock.fn_cost_centre_1_id
			LEFT JOIN fn_cost_centre as fn_cost_centre_2 ON fn_cost_centre_2.fn_cost_centre_ix = ac_stock.fn_cost_centre_2_id
		ORDER BY pr_persona.pr_name_last
		",6);
		$newLocationList = array();
		foreach($list as &$item) {
			$item['allow_edit'] = canEditDB($item['ac_stock_db']);
			$item['allow_delete'] = $item['allow_edit'];
			$newLocationList[$item['ac_stock_ix']] = $item;
		}
		unset($item);
		return $this->LocationList = $newLocationList;
	}

	public function listLocations() {
		$listHtml = "";
		$locationList = $this->fetchLocations();
		if (count($locationList)) {
			foreach($this->LocationList as $key => $locationItem) {
				$cc1Code = !empty($locationItem['fn_cost_centre_1_code']) ? $locationItem['fn_cost_centre_1_code'] . ": " . $locationItem['fn_cost_centre_1_desc'] : "";
				$cc2Code = !empty($locationItem['fn_cost_centre_2_code']) ? $locationItem['fn_cost_centre_2_code'] . ": " . $locationItem['fn_cost_centre_2_desc'] : "";
				$rowHtml = "
							<tr id='" . $key . "' onclick=\"selectLocation('" . $key . "')\" class='linkrow'>
								<td class='bb'>
									<input type='hidden' value='" . $locationItem['ac_stock_db'] . "' class='" . $locationItem['ac_stock_db'] . "'>
									<input type='hidden' value='" . $locationItem['pr_business_id'] . "' class='" . $locationItem['pr_business_id'] . "'>
									" . $locationItem['pr_name_last'] . "
									<input type='hidden' value='" . $locationItem['allow_edit'] . "' class='" . $locationItem['allow_edit'] . "'>
									<input type='hidden' value='" . $locationItem['allow_delete'] . "' class='" . $locationItem['allow_delete'] . "'>
								</td>
								<td class='bb ex_ac_stock_key_pos'>" . $locationItem['ac_stock_key_pos'] . "</td>
								<td class='bb ex_ac_stock_name_rr'>" . $locationItem['ac_stock_name_rr'] . "</td>
								<td class='bb'>$cc1Code</td>
								<td class='bb'>$cc2Code</td>
								<td class='uk-text-center bb'>";
					$rowHtml .= $locationItem['ac_stock_inactive_yn'] == 1 ? "X" : "";
					$rowHtml .= "</td>
							</tr>";
				$listHtml .= $rowHtml;
			}
		}
		return $listHtml;
	}

	public function selectLocation($ac_stock_ix) {
		$locationList = $this->fetchLocations();
		return $this->LocationList[$ac_stock_ix];
	}

	public function saveLocation($location) {
		if(checkJob(4001,2) < 10) {
			return false;
		}
		if(!is_array($location)) {
			return false;
		}
		
		$ac_stock_ix = $location[0];
		$pr_business_id = $location[1];
		$ac_stock_key_pos = $location[2];
		$ac_stock_name_rr = $location[3];

		if(
			!$pr_business_id
			|| !$ac_stock_key_pos
			|| !$ac_stock_name_rr
		) {
			return false;
		}

		if ($this->db_ac_stock_exists($ac_stock_ix)) {
			$this->db_ac_stock_update($ac_stock_ix, $ac_stock_key_pos, $ac_stock_name_rr);
		} else {
			$this->db_ac_stock_insert($pr_business_id, $ac_stock_key_pos, $ac_stock_name_rr);
		}
		return true;
	}

	public function db_ac_stock_exists($ac_stock_id, $pr_business_id="") {
		$check = false;
		if ($pr_business_id != "") {
			$check = $this->lDB->get("
				SELECT 
					ac_stock_ix 
				FROM 
					ac_stock 
				WHERE 
					ac_stock_ix = '" . $this->lDB->escape($ac_stock_id) . "' 
					AND 
					pr_business_id = '" . $this->lDB->escape($pr_business_id) . "' 
			",4);
		} else {
			$check = $this->lDB->get("
				SELECT 
					ac_stock_ix 
				FROM 
					ac_stock 
				WHERE 
					ac_stock_ix = '" . $this->lDB->escape($ac_stock_id) . "'
			",4);
		}
		return $check;
	}

	private function db_ac_stock_insert($pr_business_id, $ac_stock_key_pos, $ac_stock_name_rr) {
		global $dbcode;
	
		if(!db_pr_business_exists($pr_business_id)) {
			return false;
		}

		$hasDefault = $this->lDB->get("
			SELECT 
				pr_business.fn_cost_centre_1_id
			FROM
				pr_business
			WHERE
				pr_business.pr_business_id = '$pr_business_id'
		",4);

		$defaultPropCC = ($hasDefault) ?? "";

		$this->lDB->put("
			INSERT INTO ac_stock (
				ac_stock_db,
				ac_stock_key_pos,
				pr_business_id,
				ac_stock_name_rr,
				fn_cost_centre_1_id
			) VALUES (
				'" . $this->lDB->escape($dbcode) . "',
				'" . $this->lDB->escape($ac_stock_key_pos) . "',
				'" . $this->lDB->escape($pr_business_id) . "',
				'" . $this->lDB->escape($ac_stock_name_rr) . "',
				'" . $this->lDB->escape($defaultPropCC) . "'
			)
		");
		$ac_stock_id = $this->lDB->insert_id;
		return $ac_stock_id;
	}

	private function db_ac_stock_update($ac_stock_id, $ac_stock_key_pos, $ac_stock_name_rr) {

		if(!$this->db_ac_stock_exists($ac_stock_id)) {
			return false;
		}
	
		$this->lDB->put("
			UPDATE ac_stock SET
				ac_stock_key_pos = '" . $this->lDB->escape($ac_stock_key_pos) . "',
				ac_stock_name_rr = '" . $this->lDB->escape($ac_stock_name_rr) . "'
			WHERE
				ac_stock.ac_stock_ix = '" . $this->lDB->escape($ac_stock_id) . "'
		");
		return true;
	}

	public function db_ac_stock_update_cc($ac_stock_id, $fn_cost_centres = array()) {
		if(!$this->db_ac_stock_exists($ac_stock_id) || !is_array($fn_cost_centres) || count($fn_cost_centres) == 0) {
			return false;
		}
		$setFields = array();
		foreach ($fn_cost_centres as $fieldName => $ccId) {
			$ccIdValue = $this->lDB->escape($ccId) == 'None' ? "" : $this->lDB->escape($ccId);
			$setFields[] = $fieldName . " = '" . $ccIdValue . "'";
		}
		$setFieldsSql = join(', ', $setFields);
		$this->lDB->put("
			UPDATE 
				ac_stock 
			SET
				" . $setFieldsSql . "
			WHERE
				ac_stock.ac_stock_ix = '" . $this->lDB->escape($ac_stock_id) . "'
		");
		return true;
	}

	public function deleteLocation($ac_stock_id) {

		if(!$this->db_ac_stock_exists($ac_stock_id)) {
			return false;
		}

		$this->lDB->put("DELETE FROM ac_stock WHERE ac_stock_ix = '" . $this->lDB->escape($ac_stock_id) . "'");

	}

	public function getTranSettings($pr_business_id) {
		global $api_instance;

		if (db_pr_business_exists($pr_business_id)) {
			$this->pr_business_id = $pr_business_id;
			$results = "";
			if (!$results = $this->getActiveTranTypeData()) {
				$results = "There is an error with your GL Setup.";
			}
			return $results;
		}
	}

	private function getActiveTranTypeData() {

		$tranTypeData = $this->lDB->get("
			SELECT 
				pr_bus_post_inventory_yn,
				pr_bus_post_in_po_yn,
				pr_bus_post_in_returns_yn,
				pr_bus_post_in_cos_yn,
				pr_bus_post_in_issues_yn,
				pr_bus_post_in_trfs_yn
			FROM
				pr_business
			WHERE
				pr_business_id = '$this->pr_business_id'
		", 6);
		for ($i=1; $i<=5; $i++) {
			$tranTypeData[0][$i . '_last_date'] = $this->getLastTranDate($this->pr_business_id, $i);
		}
		return $tranTypeData[0]['pr_bus_post_inventory_yn'] == '1' ? $tranTypeData : false;
	}

	public function getLastTranDate($pr_business_id, $in_api_request_item_tran_ind) {
		$lastDate = $this->lDB->get("
			SELECT 
				in_api_request_end_invntry 
			FROM 
				in_api_request 
			WHERE 
				in_api_request_item_tran_ind = $in_api_request_item_tran_ind 
				AND pr_business_id = '$pr_business_id' 
			ORDER BY 
				in_api_request_end_invntry DESC 
				LIMIT 1;
		", 4);
		return $lastDate == '0000-00-00 00:00:00' || empty($lastDate) ? date("Y-m") . "-01 00:00:00" : $lastDate;
	}

	private function get_pos_api_credentials($pr_business_id) {
		$this->InventCredentials = array();
		$this->tranUrls = array();

		$this->InventCredentials = $this->lDB->get("
			SELECT 
				pr_bus_api_client,
				pr_bus_api_token
			FROM
				pr_business
			WHERE
				pr_business_id = '" . $this->pr_business_id . "'
		");

		$this->tranUrls = $this->lDB->get("
			SELECT 
				pr_bus_api_url_base AS '0',
				pr_bus_api_url_po AS '1',
				pr_bus_api_url_returns AS '2',
				pr_bus_api_url_cos AS '3',
				pr_bus_api_url_issues AS '4',
				pr_bus_api_url_transfers AS '5'
			FROM
				pr_business
			WHERE
				pr_business_id =  '" . $this->pr_business_id . "'
		");
	}

	public function pullTransactions($pr_business_id, $processTranList, $requestStartTime) {
		$this->newRelic->record_transaction("Inventory -" . __FUNCTION__);
			// 1 = POR, 2 = Return, 3=COS, 4 = Issue, 5 = Transfer
		$results = array();
		$in_api_request_start_invntry = $in_api_request_end = $in_api_request_end_invntry = '';
		$in_api_request_result = "400";
		$processTranList = json_decode($processTranList);
		if (empty($this->InventCredentials)) {$this->get_pos_api_credentials($pr_business_id);}

		foreach($processTranList as $tranTypeData) {

			$in_api_request_ix = $this->new_api_request($pr_business_id, $tranTypeData[0], $requestStartTime);
			$in_api_request_item_count = 0;
			$ad_Insert_id = "";
			$RawReturn = "";
			$apiTranTypeDesc = "";
			$data = array();
			if (!empty($tranTypeData)) {

				$RawReturn = json_decode($this->InventoryJsonCall($tranTypeData[0], $tranTypeData), true);
				$rawCount = !empty($RawReturn['total']) ? $RawReturn['total'] : '0';
				if (!isset($RawReturn['statusCode']) && $rawCount == "0" && gettype($RawReturn) == 'integer') {
					// TO api returns no status when Amazon Web Services  times out - +/-30secs.
					// Only the following timeout message:   {"message": "Endpoint request timed out"}
					// Sometimes AWS seems to need more time to wake the api up first after a long sojourn or siesta, and just 
					// refreshing the page and re-doing the api call works.

					$results[$tranTypeData[0]]['message'] = "There was a <span title='" . $RawReturn['message'] . "'>problem</span> with this api call. Please try again later. Error code: " . $RawReturn;
					return json_encode($results);
				}
				if (!empty($RawReturn['data'])) {
					$data = $RawReturn['data'];
					if (!empty($data['updatedAt'])) {
						array_multisort(array_column($data, 'updatedAt'), SORT_ASC, $data);
					}
					if (count($data) != $rawCount) {
						$results[$tranTypeData[0]]['message'] = "There was a <span title='" . $RawReturn['message'] . "'>problem</span> with this api call. Please try again later.";
						return json_encode($results);
					} else {
						$results[$tranTypeData[0]]['message'] = "Successful API call.";
					}
				}
				$in_api_request_result = $RawReturn["statusCode"];

				if ($RawReturn["statusCode"] != STATUS_SUCCESS) {
					$results[$tranTypeData[0]]['message'] = " <span title='" . $RawReturn["statusCode"] . " " . $RawReturn["error"] . " " . $RawReturn['message'] . "'>Invalid response received while processing " .  $this->TranTypeDesc[$tranTypeData[0]] . ".</span>";
				} else {
					$results[$tranTypeData[0]]['api']['count'] = $in_api_request_item_count = count($data);
					if (!$in_api_request_item_count) {
						$results[$tranTypeData[0]]['message'] = "There were no " . $this->TranTypeDesc[$tranTypeData[0]] . " records returned. " . $RawReturn["error"];
					} else {
						$results[$tranTypeData[0]]['in_api']['count'] = $this->generate_in_api_transactions($in_api_request_ix, $tranTypeData[0], $data);
						$results[$tranTypeData[0]]['in_api']['firstDate'] = $this->first_tran_date;
						$results[$tranTypeData[0]]['in_api']['lastDate'] = $this->last_tran_date;
					}
					$results[$tranTypeData[0]]['fn_transactions'] = $this->generate_fn_transactions($tranTypeData[0],  $pr_business_id, $in_api_request_ix);
				}
			}
			$new_api_request_update_result = $this->new_api_request_update($pr_business_id, $in_api_request_ix, $this->first_tran_date, $this->last_tran_date, $in_api_request_item_count, $in_api_request_result);
		}
		if(count($this->errorLog)) {array_merge($results,$this->errorLog);}
		$this->newRelic->stop_transaction();
		return json_encode($results);
	}

	private function InventoryJsonCall($action, $tranTypeData) {
		if (empty($this->POSAPICredentials)) {
			$this->get_pos_api_credentials($this->pr_business_id);
		}

		$url = "https://" . $this->tranUrls['0'] . $this->tranUrls[$action];

		$lastTransactionDate = $this->getLastTranDate($this->pr_business_id, $tranTypeData[0]);
		if ($lastTransactionDate != date('Y-m-01 00:00:00')) {
			$startDateTZ = convertDateTimeZone($lastTransactionDate, date('e'), 'UTC', 1, 'Y-m-d H:i:s');
			$startDate = $startDateTZ["datetime_custom"];
		}
		else {
			$startDate = $lastTransactionDate;
		}
		$startDate = date('Y-m-d H:i:s',strtotime('-5 minutes', strtotime($startDate)));
		$endDateTZ   = convertDateTimeZone(date('Y-m-d H:i:s'), date('e'), 'UTC', 1, 'Y-m-d H:i:s');
		$endDate   = $endDateTZ["datetime_custom"];

		$ch = curl_init();
		curl_setopt_array($ch, array(
			CURLOPT_URL => $url,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING => '',
			CURLOPT_MAXREDIRS => 10,
			CURLOPT_TIMEOUT => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST => 'POST',
			CURLOPT_POSTFIELDS =>'{
				"StartDate": "' . $startDate . '",
				"EndDate": "' . $endDate . '"
			}',
			CURLOPT_HTTPHEADER => array(
				'api-key: ' . $this->InventCredentials['pr_bus_api_client'] . '-' . $this->InventCredentials['pr_bus_api_token'],
				'Content-Type: application/json'
			),
		));

		$content = curl_exec( $ch );
		$err     = curl_errno( $ch );
		$errmsg  = curl_error( $ch );
		$header  = curl_getinfo( $ch );
		curl_close( $ch );

		if ($err != 0 && $err != '') {
			return('error:'.$errmsg);
		}

		if ($header['http_code'] != '200') {
			// Gateway timeout, server error, anything other than success.
			return($header['http_code']);
		}
		return($content);
	}

	private function new_api_request($pr_business_id, $in_api_request_item_tran_ind, $in_api_request_start) {
		global $dbcode;
	
		if($this->currentUserId === false) {
			$this->currentUserId = $GLOBALS['userid'];
		}
	
		$in_api_request_ix = $this->lDB->get("SELECT GET_UUID()", 4);
	
		$this->lDB->put("
			INSERT INTO in_api_request (
				in_api_request_db, 
				in_api_request_ix, 
				pr_business_id, 
				pr_user_request_id, 
				in_api_request_start, 
				in_api_request_item_tran_ind
			) VALUES (
				'".$this->lDB->escape($dbcode)."',
				'".$this->lDB->escape($in_api_request_ix)."',
				'".$this->lDB->escape($pr_business_id)."',
				'".$this->lDB->escape($this->currentUserId)."',
				'".$this->lDB->escape($in_api_request_start)."',
				'".$this->lDB->escape($in_api_request_item_tran_ind)."'
			)
		");
	
		return $in_api_request_ix;
	}

	private function new_api_request_update($pr_business_id, $in_api_request_ix, $in_api_request_start_invntry='', $in_api_request_end_invntry='', $in_api_request_item_count='0', $in_api_request_result) {

		if(!$this->db_api_request_exists($in_api_request_ix, $pr_business_id) ) {
			return false;
		}
		$in_api_request_end = date("Y-m-d H:i:s");
		// Start and end dates already converted.
		$requestStartInvent = $in_api_request_start_invntry;
		$requestEndInvent   = $in_api_request_end_invntry;

		return $this->lDB->put("
			UPDATE 
				in_api_request
			SET 
				in_api_request_start_invntry = '".trim($this->lDB->escape($requestStartInvent))."', 
				in_api_request_end = '"			 .trim($this->lDB->escape($in_api_request_end))."', 
				in_api_request_end_invntry = '"	 .trim($this->lDB->escape($requestEndInvent))."', 
				in_api_request_item_count = '"	 .trim($this->lDB->escape($in_api_request_item_count))."',
				in_api_request_result = '"	 	 .trim($this->lDB->escape($in_api_request_result))."'
			WHERE
				in_api_request_ix = '$in_api_request_ix'
		");
	}

	public function db_api_request_exists($in_api_request_ix, $pr_business_id) {
		if (!$in_api_request_ix || !$pr_business_id) {
			return false;
		}
		return $this->lDB->get("
			SELECT 
				in_api_request_ix
			FROM
				in_api_request
			WHERE
				pr_business_id = '" . $this->lDB->escape($pr_business_id) . "'
				AND
				in_api_request_ix = '" . $this->lDB->escape($in_api_request_ix) . "'
		", 4);
	}

	private function generate_in_api_transactions($in_api_request_ix, $tran_type_ind, $raw_tran_json) {
		// Collects results from api per transaction type, processes and inserts into "raw" tables - in_api_*
		global $dbcode;
		$this->first_tran_date  = "";
		$this->last_tran_date   = "";
		$in_api_header_ix   	= "";
		$tran_values      		= array();
		$tran_item_values 		= array();
		$header 		  		= false;
		$raw_tran_count   		= count($raw_tran_json);
		$in_api_header_fields 	= "in_api_tran_db, in_api_tran_id, in_api_tran_ix, in_api_request_id, in_api_api_id, in_api_tran_ref, in_api_tran_doc_ref, in_api_tran_location, in_api_tran_date, in_api_tran_voided, 
									in_api_tran_inv_amt, in_api_tran_vat_amt, in_api_tran_supp_code, in_api_tran_supp_name, in_api_tran_supp_address, in_api_tran_timestamp";
		$dateIndexName			= "date";
		$headerIdName 			= "id";
		$itemIdName   			= "id";
		$tranDocRefName		 	= "invoiceNo";
		$tranRefName			= "reference";
		$locationIdName   		= "sourceStoreId";
		$in_api_tran_inv_amt	= "total";
		$in_api_tran_vat_amt	= "totalTax";
		$in_api_item_price		= "totalCost";

		switch ($tran_type_ind) { 
			case ITEM_TRAN_IND_COS: 
				$dateIndexName 			= "dateClosed";
				$headerIdName  			= "tabId";
				$itemIdName    			= "lineId";
				$locationIdName 		= "storeId";
				$tranRefName			= "invoiceNo";
				$in_api_tran_vat_amt	= "taxTotal";	
				$in_api_tran_inv_amt	= "costTotal";
				break;
			case ITEM_TRAN_IND_ISSUE:
				$in_api_tran_inv_amt	= "totalExTax";
				break;
			case ITEM_TRAN_IND_POR: 
			case ITEM_TRAN_IND_SUPPLIER_RETURN: 
			case ITEM_TRAN_IND_TFR: 
			default:
				break;
		}

		$recs = 0;
		$tran_sql = "INSERT INTO in_api_tran (" . $in_api_header_fields . ") VALUES ";
		$tran_item_sql = "INSERT INTO in_api_item ( in_api_item_db, in_api_item_ix, in_api_header_id, in_api_item_type_ind, in_api_api_item_id, in_api_item_stock_code, in_api_item_invent_code, in_api_item_exp_code, in_api_item_qty, in_api_item_unit_price, in_api_item_price, in_api_item_voided) VALUES ";

		foreach($raw_tran_json as $raw_tran) {

			if ( $raw_tran === reset( $raw_tran_json ) ) {
				$this->first_tran_date = !empty($raw_tran['updatedAt']) ? $this->convertDate($raw_tran['updatedAt']) : $this->convertDate($raw_tran['dateClosed']);
				$this->last_tran_date  = !empty($raw_tran['updatedAt']) ? $this->convertDate($raw_tran['updatedAt']) : $this->convertDate($raw_tran['dateClosed']);
			} 
			else if( $raw_tran === end( $raw_tran_json ) && $raw_tran_count > 1) {
				$this->last_tran_date = !empty($raw_tran['updatedAt']) ? $this->convertDate($raw_tran['updatedAt']) : $this->convertDate($raw_tran['dateClosed']);
			}

			if ($this->db_in_api_tran_exists($tran_type_ind, $raw_tran[$headerIdName], $this->convertDate($raw_tran[$dateIndexName])) == 0) {
				/* Step over non-CoS duplicates only. This is raw data received from the api
				 * updatedAt date = dateClosed or date. 
				 * If transaction is changed - voids, etc., TO record is edited & updatedAt (and for COS, dateClosed) changes
				 * We insert these into the raw tables as well and post adjusting transactions for fn records
				 * Except trans voided at header level. Usually done before tab is closed - null items and voided flagged. 
				 */

				$in_api_header_ix = $this->lDB->get("SELECT GET_UUID()", 4);
				$voided           = $tran_type_ind == ITEM_TRAN_IND_COS ? $raw_tran['voided'] : 0;
				$taxAmount        = ($tran_type_ind == ITEM_TRAN_IND_POR || $tran_type_ind == ITEM_TRAN_IND_SUPPLIER_RETURN) ? $raw_tran[$in_api_tran_vat_amt] : 0;
				$tran_inv_amt     = $raw_tran[$in_api_tran_inv_amt];
				$supplierAddress  = isset($raw_tran['in_api_tran_supp_address']) ? $raw_tran['in_api_tran_supp_address'] : "";
				$supplierId       = isset($raw_tran['supplierCode']) ? $raw_tran['supplierCode'] : "";
				$supplierName     = isset($raw_tran['supplierName']) ? $raw_tran['supplierName'] : "";

				if ($tran_type_ind == ITEM_TRAN_IND_COS) {
					$existingApiHeaderData = $this->db_get_existing_api_tran_data($raw_tran[$headerIdName]);

					if(!count($existingApiHeaderData)) {
						if ($voided) { continue; }
					}
					else {
						// Transaction already posted. 
						// $existingApiHeaderData['in_api_tran_inv_amt'] is SUMmed to get "running total", so only 1 record returned.
						// Whatever the cost balance, we post the adjusting transaction. Voided headers should be zero.
						$tran_inv_amt = $tran_inv_amt - $existingApiHeaderData[0]['in_api_tran_inv_amt'];
					}
				}

				$tranDate = $this->convertDate($raw_tran[$dateIndexName]);
				$tran_values = "(
					'" . $this->lDB->escape($dbcode) . "',
					'0', 
					'" . $this->lDB->escape($in_api_header_ix) . "', 
					'" . $this->lDB->escape($in_api_request_ix) . "',  	
					'" . $this->lDB->escape($raw_tran[$headerIdName]) . "',	
					'" . $this->lDB->escape($raw_tran[$tranRefName]) . "',  	
					'" . $this->lDB->escape($raw_tran[$tranDocRefName]) . "',  	
					'" . $this->lDB->escape($raw_tran[$locationIdName]) . "',  	
					'" . $this->lDB->escape($tranDate) . "',
					'" . $this->lDB->escape($voided) . "',
					'" . $this->lDB->escape($tran_inv_amt) . "',
					'" . $this->lDB->escape($taxAmount) . "',
					'" . $this->lDB->escape($supplierId) . "', 
					'" . $this->lDB->escape($supplierName) . "',
					'" . $this->lDB->escape($supplierAddress) . "',	
					'" . $this->lDB->escape(date("Y-m-d H:i:s")) . "'
				)";

				$header = $this->lDB->put($tran_sql . $tran_values);
				if ($header) {
					$recs++;

					if (count($raw_tran['items'])) {
						$tran_item_values = array();
						foreach ($raw_tran['items'] as $raw_tran_item) {
							$voidedItem = "0";
							$itemPrice = $raw_tran_item[$in_api_item_price];
							$unitCost  = $raw_tran_item['unitCost'];

							if ($tran_type_ind == ITEM_TRAN_IND_COS) {
								$voidedItem = $raw_tran_item['voided'];
			
								$existingItemData = $this->db_get_existing_api_item_data($raw_tran_item[$itemIdName]);

								if (count($existingItemData)) {
									if ($voidedItem) {
										if ($existingItemData[0]['in_api_item_voided']) {
											$itemPrice = 0;
											$unitCost  = 0;
										} 
										else {
											$itemPrice = -($existingItemData[0]['in_api_item_price']);
											$unitCost  = -($existingItemData[0]['in_api_item_unit_price']);
										}
									} 
									else {continue;}
								} 
								else {
									if ($voidedItem) { $itemPrice = 0; }
								}
							}

							$in_api_item_ix = $this->lDB->get("SELECT GET_UUID()", 4);
							array_push($tran_item_values, "(
								'" . $this->lDB->escape($dbcode) . "',	
								'" . $this->lDB->escape($in_api_item_ix) . "',  
								'" . $this->lDB->escape($in_api_header_ix) 	. "',	
								'" . $this->lDB->escape($tran_type_ind)	. "',	
								'" . $this->lDB->escape($raw_tran_item[$itemIdName]) . "',  	
								'" . $this->lDB->escape($raw_tran_item['sku']) . "',  	
								'" . $this->lDB->escape($raw_tran_item['inventoryAccountCode']) . "', 
								'" . $this->lDB->escape($raw_tran_item['costOfSalesAccountCode']) . "', 
								'" . $this->lDB->escape($raw_tran_item['quantity']) . "', 
								'" . $this->lDB->escape($unitCost) . "', 
								'" . $this->lDB->escape($itemPrice) . "',
								$voidedItem
								)"
							);
						}
						$this->lDB->put($tran_item_sql . join(",", $tran_item_values));
					}
				}
			}
		}
		return $recs;
	}

	public function db_in_api_tran_exists($tran_type_ind, $headerId, $updatedAt) {
		if ($tran_type_ind != ITEM_TRAN_IND_COS) {
			return $this->lDB->get( "SELECT COUNT(*) FROM in_api_tran WHERE in_api_api_id = '$headerId'",4);
		}
		else {
			return $this->lDB->get("SELECT COUNT(*) FROM in_api_tran WHERE in_api_api_id = '$headerId' AND in_api_tran_date = '$updatedAt'",4);
		}
	}

	public function generate_fn_transactions($tran_type_ind, $pr_business_id, $in_api_request_ix) {
		global $dbcode;

		$in_api_header_list			= array();
		$in_api_header_list_count	= 0;
		$in_api_item_list			= array();
		$in_api_header_ix_name		= 'in_api_tran_ix';
		$transactions				= array();
		$fn_tran_count				= 0;
		$in_api_result = array();
		$sql = "
			SELECT
				in_api_tran.in_api_tran_ix,
				in_api_tran.in_api_tran_db AS in_api_header_db,
				in_api_tran.in_api_tran_date AS in_api_header_date,
				in_api_tran.in_api_api_id,
				in_api_tran.in_api_tran_inv_amt AS in_api_header_inv_amt,
				(in_api_tran.in_api_tran_inv_amt - in_api_tran.in_api_tran_vat_amt) AS fn_tran_amt_excl,
				(in_api_tran.in_api_tran_inv_amt * 1) AS fn_tran_amt_source,
				in_api_tran.in_api_tran_ref,
				in_api_tran.in_api_tran_doc_ref,
				1 as exch_rate,
				NULL as batch_id,
				'" . $pr_business_id . "' AS pr_business_id,
				'" . $this->InventoryInvoiceUnits[$pr_business_id]['pr_bus_home_curr_id'] . "' AS rf_currency_id,
				in_api_tran.in_api_tran_ix AS fn_tran_link_id,
				'0' AS fn_tran_inv_yn,
				" . $this->FnTranTypeInd[$tran_type_ind] . " AS fn_tran_link_ind,
				now() AS ad_create_date,
				'" . date("e") . "' AS ad_create_date_tz,
				'" . $this->currentUserId . "' AS ad_create_user_id,
				'0' AS fn_tran_trf_yn,
				in_api_tran.in_api_tran_ref AS fn_tran_item_note, 
				in_api_tran.in_api_tran_location,
				in_api_tran.in_api_tran_date,
				in_api_tran.in_api_request_id,
				in_api_tran.in_api_tran_voided,
				
				in_api_tran.in_api_tran_supp_code, 
				in_api_tran.in_api_tran_supp_name, 
				in_api_tran.in_api_tran_vat_amt AS in_api_tran_vat_amt, 

				ac_stock.fn_cost_centre_1_id, 
				ac_stock.fn_cost_centre_2_id,
				ac_stock.ac_stock_name_rr 
			FROM 
				in_api_tran
				LEFT JOIN ac_stock ON ac_stock.ac_stock_key_pos = in_api_tran.in_api_tran_location
			WHERE 
				in_api_tran.in_api_request_id = '$in_api_request_ix'
		";
		$in_api_header_list = $this->lDB->get($sql,6);

		if (count($in_api_header_list)) {
			$transactionItems = array();
			$fn_tran_count = 0;

			foreach($in_api_header_list as $transactionData) {

				$fn_tran_ix = $transactionData['in_api_header_inv_amt'] == 0 ? '' : $this->db_insert_fn_tran($transactionData);
				if ($fn_tran_ix) {
					$fn_tran_count ++;
					$transactionItems = array();
					$transactionItem  = array();

					$transactionItems = $this->lDB->get("
						SELECT 
							sum(in_api_item.in_api_item_price) AS fn_tran_item_amt_incl, 
							(sum(in_api_item.in_api_item_price) * " . $transactionData['exch_rate'] . ") AS fn_tran_item_amt_source_incl,
							in_api_item.in_api_item_invent_code,
							in_api_item.in_api_item_exp_code 
						FROM 
							in_api_item 
						WHERE  
							in_api_header_id = '" . $transactionData['in_api_tran_ix'] . "' 
						GROUP BY 
							in_api_item_invent_code
					",6);
					switch ($tran_type_ind) {
						case ITEM_TRAN_IND_COS:
							$transactionItems = $this->lDB->get("
								SELECT 
									in_api_item.in_api_api_item_id, 
									in_api_item.in_api_item_qty, 
									in_api_item.in_api_item_unit_price, 
									in_api_item.in_api_item_price AS fn_tran_item_amt_incl, 
									(in_api_item.in_api_item_price * " . $transactionData['exch_rate'] . ") AS fn_tran_item_amt_source_incl,
									in_api_item.in_api_item_invent_code,
									in_api_item.in_api_item_exp_code, 
									in_api_item.in_api_item_voided
								FROM 
									in_api_item 
								WHERE  
									in_api_header_id = '" . $transactionData['in_api_tran_ix'] . "' 
							",6);

							foreach($transactionItems as $transactionItem) {
								$this->db_insert_fn_tran_item('71', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, -1); // CR
								$this->db_insert_fn_tran_item('73', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, 1); // DR
							}
							break;
						case ITEM_TRAN_IND_ISSUE:
							foreach($transactionItems as $transactionItem) {
								$this->db_insert_fn_tran_item('71', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, -1); // CR
								$this->db_insert_fn_tran_item('73', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, 1); // DR
							}
							break;
						case ITEM_TRAN_IND_POR:
							$this->db_insert_fn_tran_item('70', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, -1); // CR
							$this->db_insert_fn_tran_item('72', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, 1); // DR
							foreach($transactionItems as $transactionItem) {
								$this->db_insert_fn_tran_item('71', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, 1); // DR
							}
							break;
						case ITEM_TRAN_IND_SUPPLIER_RETURN:
							$this->db_insert_fn_tran_item('70', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, 1); // DR
							$this->db_insert_fn_tran_item('72', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, -1); // CR
							foreach($transactionItems as $transactionItem) {
								$this->db_insert_fn_tran_item('71', $this->FnTranTypeInd[$tran_type_ind], $fn_tran_ix, $transactionData, $transactionItem, -1); // CR
							}
							break;
						case ITEM_TRAN_IND_TFR:
							break;
						default:
							break;
					}
				}
			}
		}
		$in_api_result['count'] = $fn_tran_count;
		return $in_api_result;
	}

	public function db_get_existing_api_tran_data($tabId) {
		return $this->lDB->get("
			SELECT 
				in_api_tran.in_api_tran_db,
				in_api_tran.in_api_tran_db,
				in_api_tran.in_api_tran_id,
				in_api_tran.in_api_tran_ix,
				in_api_tran.in_api_request_id,
				in_api_tran.in_api_api_id,
				in_api_tran.in_api_tran_ref,
				in_api_tran.in_api_tran_doc_ref,
				in_api_tran.in_api_tran_location,
				SUM(in_api_tran.in_api_tran_inv_amt) AS in_api_tran_inv_amt,
				in_api_tran.in_api_tran_vat_amt,
				in_api_tran.in_api_tran_supp_code,
				in_api_tran.in_api_tran_supp_name,
				in_api_tran.in_api_tran_supp_address,
				in_api_tran.in_api_tran_timestamp,
				in_api_tran.in_api_tran_voided,
				in_api_tran.in_api_tran_date
			FROM
				in_api_tran
			WHERE
				in_api_api_id = '" . $tabId . "'
		", 6);
	}

	public function db_get_existing_api_item_data($filterBy) {
		return $this->lDB->get("
			SELECT 
				in_api_item.in_api_item_db,
				in_api_item.in_api_item_id,
				in_api_item.in_api_item_ix,
				in_api_item.in_api_header_id,
				in_api_item.in_api_api_item_id,
				in_api_item.in_api_item_type_ind,
				in_api_item.in_api_item_stock_code,
				in_api_item.in_api_item_invent_code,
				in_api_item.in_api_item_exp_code,
				in_api_item.in_api_item_qty,
				in_api_item.in_api_item_unit_price,
				in_api_item.in_api_item_price,
				in_api_item.in_api_item_voided
			FROM
				in_api_item
				LEFT JOIN in_api_tran ON in_api_tran.in_api_tran_ix = in_api_item.in_api_header_id  
			WHERE
				in_api_api_item_id  = '" . $filterBy . "'
			ORDER BY in_api_tran_date DESC LIMIT 1
		", 6);
	}

	public function db_inv_fn_tran_exists($fn_tran_link_ind, $fn_tran_link_id) {
		if ($fn_tran_link_ind != FN_ITEM_TRAN_IND_COS) {
			return $this->lDB->get("SELECT COUNT(*) FROM fn_tran WHERE fn_tran_link_id = '$fn_tran_link_id'",4);
		} else {
			return 0;
		}
	}

	public function in_api_tran_has_items($in_api_tran_ix) {
		return $this->lDB->get("SELECT COUNT(*) FROM in_api_item WHERE in_api_header_id = '$in_api_tran_ix'",4);
	}

	public function fn_tran_items_exist($in_api_api_item_id) {
		return $this->lDB->get("SELECT COUNT(*) FROM in_api_item WHERE in_api_api_item_id = '$in_api_api_item_id'",4) > 1 ?? '0';
	}

	public function db_insert_fn_tran($transactionData) {
		global $dbcode;

		if (!$this->db_inv_fn_tran_exists($transactionData['fn_tran_link_ind'], $transactionData['fn_tran_link_id'])) { 
			// No duplication of any transactions allowed, except COS, for POS voiding when tab is re-opened.

			$fn_tran_amt_incl	= $transactionData['in_api_header_inv_amt'];
			$fn_tran_amt_excl	= $transactionData['fn_tran_amt_excl'];
			$fn_tran_amt_source	= $transactionData['fn_tran_amt_source'];
			$fn_tran_detail 	= $transactionData['ac_stock_name_rr'];

			$fn_tran_ix = $this->lDB->get("SELECT GET_UUID()", 4);

			$tran_type_ind = $transactionData['fn_tran_link_ind'];

			switch ($transactionData['fn_tran_link_ind']) {
					case FN_ITEM_TRAN_IND_POR:
					case FN_ITEM_TRAN_IND_RETURN:
						$fn_tran_detail = $transactionData['ac_stock_name_rr'] . '; ' . $transactionData['in_api_tran_supp_name'] . '; ' . $transactionData['in_api_tran_doc_ref'];
						break;
					case FN_ITEM_TRAN_IND_ISSUE:
						$fn_tran_detail = $transactionData['ac_stock_name_rr'] . '; ' . $transactionData['in_api_tran_doc_ref'];
						break;										
					case FN_ITEM_TRAN_IND_COS:
					case FN_ITEM_TRAN_IND_TFR:
						break;
			}

			$fn_tran_insert_sql = "
				INSERT INTO fn_tran
				(
					fn_tran_ix,	
					fn_tran_id,	
					fn_tran_db,	
					fn_tran_date_ledger,
					fn_tran_exch_rate,
					fn_tran_amt_excl,
					fn_tran_amt_incl,
					fn_tran_amt_source, 
					fn_tran_detail, 
					fn_batch_id, 
					pr_business_id,
					rf_currency_id,
					fn_tran_link_id,
					fn_tran_inventory_doc,
					fn_tran_inv_yn,
					fn_tran_link_ind,
					ad_create_date,
					ad_create_date_tz,
					ad_create_user_id,
					fn_tran_trf_yn
				)
				VALUES 
				(
					'" . $this->lDB->escape($fn_tran_ix) . "',  
					'0',  
					'" . $this->lDB->escape($transactionData['in_api_header_db']) . "', 
					'" . $this->lDB->escape($transactionData['in_api_header_date']) . "', 
					'" . $this->lDB->escape($transactionData['exch_rate']) . "', 
					'" . $this->lDB->escape($fn_tran_amt_excl) . "', 
					'" . $this->lDB->escape($fn_tran_amt_incl) . "', 
					'" . $this->lDB->escape($fn_tran_amt_source) . "', 
					'" . $this->lDB->escape($fn_tran_detail) . "', 
					'" . $this->lDB->escape($transactionData['batch_id']) . "',  
					'" . $this->lDB->escape($transactionData['pr_business_id']) . "',  
					'" . $this->lDB->escape($transactionData['rf_currency_id']) . "',  
					'" . $this->lDB->escape($transactionData['fn_tran_link_id']) . "',  
					'" . $this->lDB->escape($transactionData['in_api_tran_ref']) . "',  
					'" . $this->lDB->escape($transactionData['fn_tran_inv_yn']) . "',  
					'" . $this->lDB->escape($transactionData['fn_tran_link_ind']) . "',  
					'" . $this->lDB->escape($transactionData['ad_create_date']) . "',  
					'" . $this->lDB->escape($transactionData['ad_create_date_tz']) . "',  
					'" . $this->lDB->escape($transactionData['ad_create_user_id']) . "',  
					'" . $this->lDB->escape($transactionData['fn_tran_trf_yn']) . "'  
				)";
			return $this->lDB->put($fn_tran_insert_sql) ? $fn_tran_ix : false ;
		}
	}

	public function db_insert_fn_tran_item($fn_tran_item_type_ind, $fn_tran_type_ind, $fn_tran_ix, $transactionData, $transactionItemData, $direction) {

		global $dbcode;
		/* 
		 * Transaction item records:
		 * - to the supplier, full value of the invoice - PO (CR), Return(DR). fn_tran_item_type_ind = '70'
		 * - debit the Input Vat account - Only PO.	fn_tran_item_type_ind = '72'
		 * - to each affected inventory account rolled up and summed (grouped) by inventory account - PO(DR), Return, COS, Issue(CR). fn_tran_item_type_ind = '71'
		 * - to debit each affected expense account - COS & issues. fn_tran_item_type_ind = '73'
		 * return is reverse of PO
		 * Issue & COS - credit inventory, debit expense account 
		 * direction: -1 CR; 1 DR
		 * Voided CoS Items have 
		 * $voidDirection always '1' except when items exist and have been voided on POS, then '-1' to reverse
		 */

		$fn_tran_item_ix = '';
		$fn_tran_item_amt_incl = 0;
		$fn_tran_item_amt_source_incl = 0;
		$fn_ledger_id = "";
		$supplierCode = "";
		$fn_tran_item_note = "";
		switch ($fn_tran_item_type_ind) {
			case FN_TRAN_ITEM_TYPE_SUPPLIER:
				$fn_tran_item_amt_incl 		  = $transactionData['in_api_header_inv_amt'] * $direction;
				$fn_tran_item_amt_source_incl = $transactionData['in_api_header_inv_amt'] * $direction;
				$supplierCode = $transactionData['in_api_tran_supp_code'];
				break;
			case FN_TRAN_ITEM_TYPE_INVENTORY:
				$fn_tran_item_amt_incl 			= $transactionItemData['fn_tran_item_amt_incl'] * $direction;
				$fn_tran_item_amt_source_incl 	= $transactionItemData['fn_tran_item_amt_source_incl'] * $direction;
				if (!empty($transactionItemData['in_api_item_invent_code'])) {
					$fn_ledger_id	= $this->lDB->get("
						SELECT 
							fn_ledger.fn_ledger_ix
						FROM 
							fn_ledger
						WHERE 
							fn_ledger.fn_ledger_code = '" . $transactionItemData['in_api_item_invent_code'] . "'
					",4);
					if (empty($fn_ledger_id)) { // auto generate new GL Code from api code
						$fn_ledger_id = $this->db_fn_ledger_auto_create($transactionItemData['in_api_item_invent_code'], $transactionData['pr_business_id']);
					}
				}

				break;
			case FN_TRAN_ITEM_TYPE_INPUT_VAT:
				$fn_tran_item_amt_incl 		  = $transactionData['in_api_tran_vat_amt'] * $direction;
				$fn_tran_item_amt_source_incl = $transactionData['in_api_tran_vat_amt'] * $direction;
				$fn_ledger_id	= $this->lDB->get("
					SELECT 
						pr_business.fn_ledger_id_input_vat
					FROM 
						pr_business
					WHERE 
						pr_business.pr_business_id = '" . $transactionData['pr_business_id'] . "'
				",4);

				break;
			case FN_TRAN_ITEM_TYPE_EXPENSE:
				$fn_tran_item_amt_incl 			= $transactionItemData['fn_tran_item_amt_incl'] * $direction;
				$fn_tran_item_amt_source_incl 	= $transactionItemData['fn_tran_item_amt_source_incl'] * $direction;
				$default_fn_ledger_id	= $this->lDB->get("
					SELECT 
						fn_ledger.fn_ledger_ix
					FROM 
						fn_ledger
					WHERE 
						fn_ledger.fn_ledger_code = '" . $transactionItemData['in_api_item_exp_code'] . "'
				",4);
				if (empty($default_fn_ledger_id) && !empty($transactionItemData['in_api_item_exp_code'])) { // auto generate new GL Code from api code
					$default_fn_ledger_id = $this->db_fn_ledger_auto_create($transactionItemData['in_api_item_exp_code'], $transactionData['pr_business_id']);
				}
				$fn_ledger_id = $fn_tran_type_ind == '54' ? $this->hasOverride($transactionData['fn_cost_centre_2_id'], $default_fn_ledger_id) : $default_fn_ledger_id;
				break;
		}

		if ($fn_tran_item_amt_incl != 0) { 
			// We are not interested in posting with a zero 
			// - except if POS sends a qty X price miscalculation then we change the totals to zero to give a GL balance error in the report

			$fn_tran_item_ix = $this->lDB->get("SELECT GET_UUID()", 4);
			if ($fn_tran_type_ind == DB_FN_TRAN_LINK_COS) {
				// PHP float calculation issue necessitates the use of round() - PHP bcmath module not installed on our servers
				$calcTotal = round($transactionItemData['in_api_item_qty'] * $transactionItemData['in_api_item_unit_price'], 2); 
				$cosTotalCost = $transactionItemData['fn_tran_item_amt_incl'];
				if ($calcTotal != $cosTotalCost) {
					$fn_tran_item_amt_incl = 0;
					$fn_tran_item_amt_source_incl = 0;
					$fn_tran_item_amt_incl = 0;
					$fn_tran_item_amt_source_incl = 0;
				}
			}

			$fn_tran_item = $this->lDB->put("
				INSERT INTO fn_tran_item (
					fn_tran_item_db,
					fn_tran_item_id, 
					fn_tran_item_ix, 
					fn_tran_item_amt, 
					fn_tran_item_amt_incl, 
					fn_tran_item_amt_source, 
					fn_tran_item_amt_source_incl, 
					fn_tran_item_type_ind, 
					fn_tran_item_note, 
					fn_tran_cost_ctr1_id, 
					fn_tran_cost_ctr2_id, 
					fn_ledger_id, 
					fn_supplier_code,
					fn_tran_id, 
					rf_currency_id, 
					ad_create_date, 
					ad_create_date_tz,	
					ad_create_user_id,
					fn_tran_item_trf_yn
				)
				VALUES (
					'" . $dbcode . "',  
					'0',  
					'" . $this->lDB->escape($fn_tran_item_ix) . "',
					'" . $fn_tran_item_amt_incl . "',  
					'" . $fn_tran_item_amt_source_incl . "',  
					'" . $fn_tran_item_amt_incl . "',  
					'" . $fn_tran_item_amt_source_incl . "',  
					'" . $this->lDB->escape($fn_tran_item_type_ind) . "',  
					'" . $this->lDB->escape($transactionData['in_api_tran_doc_ref']) . "',  
					'" . $this->lDB->escape($transactionData['fn_cost_centre_1_id']) . "',  
					'" . $this->lDB->escape($transactionData['fn_cost_centre_2_id']) . "',  
					'" . $this->lDB->escape($fn_ledger_id) . "',  
					'" . $this->lDB->escape($supplierCode) . "',  
					'" . $this->lDB->escape($fn_tran_ix) . "',  
					'" . $this->lDB->escape($transactionData['rf_currency_id']) . "',  
					'" . $this->lDB->escape($transactionData['ad_create_date']) . "',  
					'" . $this->lDB->escape($transactionData['ad_create_date_tz']) . "',  
					'" . $this->lDB->escape($transactionData['ad_create_user_id']) . "',  
					'0'  
				)
			");
		}
		return $fn_tran_item_ix;
	}

	private function hasOverride($fn_cost_centre_2_id, $default_fn_ledger_id) {

		$override_id = $this->lDB->get("
			SELECT 
				ac_stock_oride.fn_ledger_oride_id 
			FROM 
				ac_stock_oride
			WHERE 
				ac_stock_oride.fn_cost_centre_2_id = '$fn_cost_centre_2_id'
				AND
				ac_stock_oride.fn_ledger_default_id = '$default_fn_ledger_id'
		",4);

		return !empty($override_id) ? $override_id : $default_fn_ledger_id;
	}

	private function convertDate($dateString) {
		// Dates received as strings in UTC need to be converted to system timezone to be inserted correctly into the db.
		$newDate = new DateTime($dateString, new DateTimeZone('UTC'));
		$newDate->setTimezone(new DateTimeZone(date("e")));
		return $newDate->format('Y-m-d H:i:s');
	}

	private function db_fn_ledger_auto_create($fn_ledger_code, $pr_business_id) {
		$fn_ledger_desc = 'API-AUTO-GEN-';
		$exists = $GLOBALS['lDB']->get("
			SELECT
				COUNT(fn_ledger.fn_ledger_ix)
			FROM
				fn_ledger
			WHERE
				fn_ledger.pr_business_id = '$pr_business_id'
				AND (
					fn_ledger.fn_ledger_desc LIKE '$fn_ledger_desc%'
				)
		",4);
		return db_fn_ledger_insert($fn_ledger_code, $fn_ledger_desc . $exists++, $pr_business_id);
	}
}
