<?php

 /**
  * class.minlos.php MinLOS
  */

use MaglLegacyApplication\Application\MaglLegacy;

require_once(__DIR__ . '/db.sc_group.php');
require_once(__DIR__ . '/class.audit.php');
require_once(__DIR__ . '/class.newrelic.php');
require_once(__DIR__ . '/class.mysqldb.php');
require_once(__DIR__ . '/inc.setup.php');


class MinLOS {
	private $conn;

	private $accommTypeId;
	private $startDate;
	private $endDate;
	private $rateTypeId;
	private $agentId;
	private $rules;

	public function __construct($accommTypeId, $startDate="0000-00-00", $endDate="0000-00-00", $rateTypeId = false, $agentId = null
	) {
		$application = MaglLegacy::getInstance()->getApplication();
		$this->conn = $application->getServiceManager()->get('EnterpriseEntityManager')->getConnection();

		$this->accommTypeId = $accommTypeId;
		$this->startDate = new \DateTime($startDate);
		$this->endDate = new \DateTime($endDate);
		$this->rateTypeId = $rateTypeId;
		$this->agentId = $agentId;

		// Override agent ID for public user
		$publicUserId = $this->conn->fetchColumn("SELECT pr_persona.pr_persona_ix FROM pr_persona WHERE pr_sys_code = 7");
		if(!empty($publicUserId) && $publicUserId == $agentId) {
			$this->agentId = 0;
		}
	}

	/**
	 * Returns ML rules that meet the criteria
	 *
	 * @return array $rules
	 */
	function getRules(DateTime $startDate = null, DateTime $endDate = null) {
		global $lDB;

		if ($startDate === null && $endDate === null) {
			$startDate = $this->startDate;
			$endDate = $this->endDate;
		}

		$rules = false;
		if ($this->rules === null) {
			$periods = $lDB->get("
				SELECT
					rt_period.rt_period_ix
				FROM
					rt_period
					LEFT JOIN rt_period_dates ON rt_period_dates.rt_period_id = rt_period.rt_period_ix
				WHERE
					(
						rt_period_dates.rt_period_from >= '" . $startDate->format('Y-m-d') . "' 
						AND rt_period_dates.rt_period_from <= '" . $endDate->format('Y-m-d') . "' 
					) OR (
						rt_period_dates.rt_period_to >= '" . $startDate->format('Y-m-d') . "' 
						AND rt_period_dates.rt_period_to <= '" . $endDate->format('Y-m-d') . "' 
					) OR (
						rt_period_dates.rt_period_from <= '" . $startDate->format('Y-m-d') . "' 
						AND rt_period_dates.rt_period_to >= '" . $endDate->format('Y-m-d') . "' 
					)
				ORDER BY 
					rt_period_dates.rt_period_from,
					rt_period_dates.rt_period_to
			", 3);

			if ($periods === false) {
				$periods = [];
			}
			$periodIds = implode("','", $periods);

			$rules = $lDB->get("
				SELECT DISTINCT
					ml_rule.ml_rule_ix,
					ml_rule.ml_min_los,
					ml_rule.ml_lapse_days,
					ml_rule.ml_rule_desc,
					ml_rule.ml_allocations_yn,
					ml_rule.ml_days_of_week,

					ml_rate_type.rt_rate_type_id,
					ml_period.rt_period_id,
					ml_accomm.ac_accomm_type_id,
					ml_agent.ml_agent_ix
				FROM
					ml_rule
					LEFT JOIN ml_rate_type ON ml_rate_type.ml_rule_id = ml_rule.ml_rule_ix
					LEFT JOIN ml_period ON ml_period.ml_rule_id = ml_rule.ml_rule_ix
					LEFT JOIN ml_accomm ON ml_accomm.ml_rule_id = ml_rule.ml_rule_ix
					LEFT JOIN ml_agent ON ml_agent.ml_rule_id = ml_rule.ml_rule_ix
				WHERE
					(
						ml_rate_type.rt_rate_type_id IS NULL
						OR ml_rate_type.rt_rate_type_id = '" . $this->rateTypeId . "' 
					)
					AND (
						ml_period.rt_period_id IS NULL
						OR ml_period.rt_period_id IN ('" . $periodIds . "')
					)
					AND (
						ml_accomm.ac_accomm_type_id IS NULL
						OR ml_accomm.ac_accomm_type_id = '" . $this->accommTypeId . "'
					)
					AND (
						ml_agent.ml_agent_ix IS NULL
						OR ml_agent.pr_agent_id = '" . $this->agentId . "'
					)
				ORDER BY 
					ml_rule.ml_min_los DESC,
					ml_rule.ml_lapse_days
			", 2);

			if ($startDate === null && $endDate === null) {
				$this->rules = $rules;
			}
		} else {
			$rules = $this->rules;
		}


		$tmpEndDate = clone $endDate;

		$datePeriod = new \DatePeriod($startDate, new \DateInterval('P1D'), $tmpEndDate);
		$daysOfWeek = []; // The numerical days of the week encompassed in the date range
		foreach ($datePeriod as $day) {
			if (!in_array($day->format('w'), $daysOfWeek)) {
				array_push($daysOfWeek, $day->format('w'));
			}
		}

		if (empty($daysOfWeek)) {
			$daysOfWeek[] = $startDate->format('w');
		}

		$applicableRules = [];
		foreach ($rules as $rule) {
			if ($rule['ml_days_of_week'] == null) {
				$ruleDaysOfWeek = [];
			} else {
				$ruleDaysOfWeek = explode(',', $rule['ml_days_of_week']);
			}

			$rule['ml_days_of_week'] = $ruleDaysOfWeek;
			$intersectingDaysOfWeek = array_intersect($daysOfWeek, $ruleDaysOfWeek);

			// Date range overlaps minLOS day of week
			if (empty($ruleDaysOfWeek) || !empty($intersectingDaysOfWeek)) {
				$applicableRules[] = $rule;
			}
		}

		return $applicableRules;
	}

	function getRule(DateTime $startDate = null, DateTime $endDate = null) {
		$rules = $this->getRules($startDate, $endDate);

		if (empty($rules)) {
			return false;
		}

		return $rules[0];
	}

	function getMinLOS() {
		$rule = $this->getRule();

		if (empty($rule)) {
			return false;
		}

		return intval($rule['ml_min_los']);
	}

	function canBook() {
		$minLos = $this->getMinLOS();

		if ($minLos === false) {
			return true;
		}

		$rule = $this->getRule();
		$lapseDays = $rule['ml_lapse_days'];

		$today = new \DateTime();
		$difference = $today->diff($this->startDate)->d;

		if (!empty($lapseDays) && $difference <= $lapseDays) {
			return true;
		}

		$nights = intval($this->endDate->diff($this->startDate)->format('%a'));

		return $nights >= $minLos;
	}

	function getMinLOSByDay() {
		global $lDB;

		$rules = [];
		$result = [];

		$periodRule = $this->getRule();
		$canBook = $this->canBook();

		$tmpEndDate = clone $this->endDate;
		$tmpEndDate->modify('+1 day');

		$datePeriod = new \DatePeriod($this->startDate, new \DateInterval('P1D'), $tmpEndDate);
		foreach ($datePeriod as $day) {
			$rule = $this->getRule($day, $day);

			$day = $day->format('Y-m-d');

			$result[$day] = [
				'minLos' => null,
				'ruleId' => null
			];

			if ($rule === false) {
				$result[$day]['minLos'] = 0;
				$result[$day]['ruleId'] = false;
			} else {
				$result[$day]['minLos'] = intval($rule['ml_min_los']);
				$result[$day]['ruleId'] = $rule['ml_rule_ix'];

				if (!in_array($rule['ml_rule_ix'], array_column($rules, 'ml_rule_ix'))) {
					$rules[] = $rule;
				}
			}
		}

		$result = [
			'days' => $result,
			'rules' => $rules,
			'rule' => $periodRule,
			'canBook' => $canBook
		];

		return $result;
	}
}