<?php

require_once(__DIR__ . "/db.sc_group.php");
require_once(__DIR__ . "/functions.reservation.itinerary.php");

/**
 * class.rate.standard.php - New standard rates calculation class
 */

class Rate extends AbstractRate
{

	var $RateCompGrpAmounts;
	var $TaxAmounts;
	var $OrideAccess;
	var $OrideLevel;
	
	function __construct($accommTypeId,$arriveDate=false,$departDate=false,$rateGroups=false,$rateTypeId=false,$optionals=false,$override=false,$reservationId=false,$commission=false,$exchange=false,$deduct=false)
	{
		$this->SetPeriodSplitDefault();
		$this->SetAccommTypeId($accommTypeId);
		$this->SetArriveDate($arriveDate);
		$this->SetDepartDate($departDate);
		$this->SetNights();
		$this->SetOverride($override);
		$this->SetReservationId($reservationId,$rateTypeId,$commission,$exchange,$deduct);
		$this->SetOptionals($optionals);
		$this->SetRateGroups($rateGroups);
		$this->SetUnits();
	}	
	
	function BuildPeriod()
	{
		global $lDB;

		/* If TBA is requested don't look up the period or rate, return a single empty period */
		if(
			is_array($this->Override)
			&& array_key_exists("level", $this->Override)
			&& $this->Override['level'] == 3
			&& array_key_exists("amounts", $this->Override)
			&& array_key_exists("stay", $this->Override['amounts'])
			&& strtolower($this->Override['amounts']['stay']) == "tba"
		) {
			$days = dateSubtract($this->EndDate,$this->ArriveDate) + 1;
			$this->Periods = array(array(
				'rt_period_ix'=>"",
				'rt_period_date_from'=>$this->ArriveDate,
				'rt_period_date_to'=>$this->EndDate,
				'ac_prd_date_depart'=>$this->DepartDate,
				'ac_prd_days'=>$days,
				'rt_rate_ix'=>"",
				'rt_rate_comm_yn'=>"",
				'rt_rate_overide_level_ind'=>"",
				'rt_rate_oride_comm_yn'=>"",
				'rt_rate_ssupp_comm_yn'=>"",
				'rt_rate_waive_checks_yn'=>"",
				'rt_pax_rate_group_yn'=>"",
				'rt_rate_round_ind'=>"",
				'rt_rate_round_dir'=>"",
				'rt_rate_round_scale'=>"",
				'rateGroupAmts'=>[]
			));

			return;
		}

		$periods = $lDB->get("
			SELECT
				rt_period.rt_period_ix AS rt_period_ix,
				rt_period_dates.rt_period_from AS rt_period_date_from,
				rt_period_dates.rt_period_to AS rt_period_date_to,
				rt_rate.rt_rate_ix AS rt_rate_ix,
				rt_rate.rt_rate_comm_yn,
				rt_rate.rt_rate_overide_level_ind,
				rt_rate.rt_rate_oride_comm_yn,
				rt_rate.rt_rate_ssupp_comm_yn,
				rt_rate.rt_rate_waive_checks_yn,
				rt_rate.rt_pax_rate_group_yn,
				rt_rate.rt_rate_round_ind,
				rt_rate.rt_rate_round_dir,
				rt_rate.rt_rate_round_scale
			FROM
				rt_rate
				INNER JOIN rt_period ON rt_period.rt_period_ix = rt_rate.rt_period_id
				INNER JOIN rt_period_dates ON rt_period_dates.rt_period_id = rt_period.rt_period_ix
			WHERE
				rt_rate.ac_accomm_type_id = '".$this->AccommTypeId."'
				AND rt_rate.rt_rate_type_id = '".$this->RateTypeId."'
				AND ( 
					(
						rt_period_dates.rt_period_from >= '".$this->ArriveDate."'
						AND rt_period_dates.rt_period_from <= '".$this->EndDate."'
					) OR (
						rt_period_dates.rt_period_to >= '".$this->ArriveDate."'
						AND rt_period_dates.rt_period_to <= '".$this->EndDate."'
					) OR (
						rt_period_dates.rt_period_from <= '".$this->ArriveDate."'
						AND rt_period_dates.rt_period_to >= '".$this->EndDate."'
					)
				)
			ORDER BY 
				rt_period_dates.rt_period_from,
				rt_period_dates.rt_period_to
		",2);
		
		if($this->PeriodSplit == 1) {
			$days = dateSubtract($this->EndDate,$this->ArriveDate) + 1;
			
			if(!empty($this->Override) && !$periods[0]['rt_rate_oride_comm_yn']) {
				$periods[0]['rt_rate_comm_yn'] = "0";
			}
			$this->Periods = array(array(
				'rt_period_ix'=>isset($periods[0]) ? $periods[0]['rt_period_ix'] : "",
				'rt_period_date_from'=>$this->ArriveDate,
				'rt_period_date_to'=>$this->EndDate,
				'ac_prd_date_depart'=>$this->DepartDate,
				'ac_prd_days'=>$days,
				'rt_rate_ix'=>isset($periods[0]) ? $periods[0]['rt_rate_ix'] : "",
				'rt_rate_comm_yn'=>isset($periods[0]) ? $periods[0]['rt_rate_comm_yn'] : "",
				'rt_rate_overide_level_ind'=>isset($periods[0]) ? $periods[0]['rt_rate_overide_level_ind'] : "",
				'rt_rate_oride_comm_yn'=>isset($periods[0]) ? $periods[0]['rt_rate_oride_comm_yn'] : "",
				'rt_rate_ssupp_comm_yn'=>isset($periods[0]) ? $periods[0]['rt_rate_ssupp_comm_yn'] : "",
				'rt_rate_waive_checks_yn'=>isset($periods[0]) ? $periods[0]['rt_rate_waive_checks_yn'] : "",
				'rt_pax_rate_group_yn'=>isset($periods[0]) ? $periods[0]['rt_pax_rate_group_yn'] : "",
				'rt_rate_round_ind'=>isset($periods[0]) ? $periods[0]['rt_rate_round_ind'] : "",
				'rt_rate_round_dir'=>isset($periods[0]) ? $periods[0]['rt_rate_round_dir'] : "",
				'rt_rate_round_scale'=>isset($periods[0]) ? $periods[0]['rt_rate_round_scale'] : "",
				'rateGroupAmts'=>isset($periods[0]) ? getRateGroupAmts($periods[0]['rt_rate_ix']) : array()
			));
		} else {
			$arriveArray = explode("-",$this->ArriveDate);
			$days = dateSubtract($this->EndDate,$this->ArriveDate)+1;
			$daysArray = array();
			for($count=0;$count<$days;$count++) {
				$daysArray[date("Y-m-d",round(mktime(12,0,0,$arriveArray[1],$arriveArray[2]+$count,$arriveArray[0])))] = 0;
			}

			foreach($periods as $key=>$item) {
				if($item['rt_period_date_from'] < $this->ArriveDate) {
					$item['rt_period_date_from'] = $this->ArriveDate;
				}
				if($item['rt_period_date_to'] > $this->EndDate) {
					$item['rt_period_date_to'] = $this->EndDate;
				}
				$days = dateSubtract($item['rt_period_date_to'],$item['rt_period_date_from'])+1;

				$dateArray = explode("-",$item['rt_period_date_to']);
				$item['ac_prd_date_depart'] = date("Y-m-d",mktime(0,0,0,$dateArray[1],$dateArray[2]+1,$dateArray[0]));

				$arriveArray = explode("-",$item['rt_period_date_from']);
				for($count=0;$count<$days;$count++) {
					$daysArray[date("Y-m-d",round(mktime(12,0,0,$arriveArray[1],$arriveArray[2]+$count,$arriveArray[0])))]++;
				}
				
				if(!empty($this->Override) && !$item['rt_rate_oride_comm_yn']) {
					$item['rt_rate_comm_yn'] = "0";
				}
				array_push($this->Periods,array(
					'rt_period_ix'=>$item['rt_period_ix'],
					'rt_period_date_from'=>$item['rt_period_date_from'],
					'rt_period_date_to'=>$item['rt_period_date_to'],
					'ac_prd_date_depart'=>$item['ac_prd_date_depart'],
					'ac_prd_days'=>$days,
					'rt_rate_ix'=>$item['rt_rate_ix'],
					'rt_rate_comm_yn'=>$item['rt_rate_comm_yn'],	
					'rt_rate_overide_level_ind'=>$item['rt_rate_overide_level_ind'],
					'rt_rate_oride_comm_yn'=>$item['rt_rate_oride_comm_yn'],	
					'rt_rate_ssupp_comm_yn' => $item['rt_rate_ssupp_comm_yn'],
					'rt_rate_waive_checks_yn'=>$periods[0]['rt_rate_waive_checks_yn'],
					'rt_pax_rate_group_yn'=>$periods[0]['rt_pax_rate_group_yn'],
					'rt_rate_round_ind'=>$periods[0]['rt_rate_round_ind'],
					'rt_rate_round_dir'=>$periods[0]['rt_rate_round_dir'],
					'rt_rate_round_scale'=>$periods[0]['rt_rate_round_scale'],
					'rateGroupAmts'=>getRateGroupAmts($item['rt_rate_ix'])
				));
			}
			$newPeriodDays = array();
			$newPeriod = false;
			foreach($daysArray as $date=>$count) {
				if($count == 0) {
					$newPeriod = true;
					$newPeriodDays[] = $date;
				}
				if($count > 0 && $newPeriod) {
					$fromDate = $newPeriodDays[0];
					$toDate = $newPeriodDays[sizeof($newPeriodDays)-1];

					$days = dateSubtract($toDate,$fromDate)+1;

					$dateArray = explode("-",$toDate);
					$departDate = date("Y-m-d",mktime(0,0,0,$dateArray[1],$dateArray[2]+1,$dateArray[0]));

					array_push($this->Periods,array(
						'rt_period_ix'=>false,
						'rt_period_date_from'=>$fromDate,
						'rt_period_date_to'=>$toDate,
						'ac_prd_date_depart'=>$departDate,
						'ac_prd_days'=>$days,
						'rt_rate_ix'=>false,
						'rt_rate_comm_yn'=>false,
						'rt_rate_overide_level_ind'=>false,
						'rt_rate_oride_comm_yn'=>false,
						'rt_rate_ssupp_comm_yn'=>false,
						'rt_rate_waive_checks_yn'=>false,
						'rt_pax_rate_group_yn'=>false,
						'rt_rate_round_ind'=>false,
						'rt_rate_round_dir'=>false,
						'rt_rate_round_scale'=>false,
						'rateGroupAmts'=>array()
					));

					$newPeriod = false;
					$newPeriodDays = array();
				}
			}
			if($newPeriod) {
				$fromDate = $newPeriodDays[0];
				$toDate = $newPeriodDays[sizeof($newPeriodDays)-1];

				$days = dateSubtract($toDate,$fromDate)+1;

				$dateArray = explode("-",$toDate);
				$departDate = date("Y-m-d",mktime(0,0,0,$dateArray[1],$dateArray[2]+1,$dateArray[0]));

				array_push($this->Periods,array(
					'rt_period_ix'=>false,
					'rt_period_date_from'=>$fromDate,
					'rt_period_date_to'=>$toDate,
					'ac_prd_date_depart'=>$departDate,
					'ac_prd_days'=>$days,
					'rt_rate_ix'=>false,
					'rt_rate_comm_yn'=>false,
					'rt_rate_overide_level_ind'=>false,
					'rt_rate_oride_comm_yn'=>false,
					'rt_rate_ssupp_comm_yn'=>false,
					'rt_rate_waive_checks_yn'=>false,
					'rt_pax_rate_group_yn'=>false,
					'rt_rate_round_ind'=>false,
					'rt_rate_round_dir'=>false,
					'rt_rate_round_scale'=>false,					
					'rateGroupAmts'=>array()
				));
			}
		}

		// Reduce all periods to a single day
		if($this->PeriodSplitSingleDay) {
			foreach($this->Periods as $key=>$period) {
				$arriveDate = $period['rt_period_date_from'];
				$dateArray = explode("-",$arriveDate);
				$departDate = date("Y-m-d",mktime(0,0,0,$dateArray[1],$dateArray[2]+1,$dateArray[0]));

				$period['rt_period_date_to_full'] = $period['rt_period_date_to'];
				$period['ac_prd_date_depart_full'] = $period['ac_prd_date_depart'];
				$period['ac_prd_days_full'] = $period['ac_prd_days'];

				$period['rt_period_date_to'] = $arriveDate;
				$period['ac_prd_date_depart'] = $departDate;
				$period['ac_prd_days'] = 1;

				$this->Periods[$key] = $period;
			}
		}
	}

	function GetComponent($rt_rate_id,$rt_rate_comm_yn,$rt_rate_oride_comm_yn,$rt_rate_ssupp_comm_yn)
	{
		global $lDB;		
		
		$components = $lDB->get("
			SELECT 
				rt_rate_component.rt_rate_component_ix,
				rt_rate_component.rt_comp_comm_yn,		
				rt_rate_component.rt_comp_freq_ind,	
				rt_rate_component.rt_comp_freq_factor,	
				rt_rate_component.rt_comp_itin_yn,
				rt_rate_component.rt_comp_los_yn,
				rt_rate_component.rt_comp_opt_yn AS ac_acc_cmp_option_yn,
				rt_rate_component.rt_comp_round_ind,
				rt_rate_component.rt_comp_separate_yn,
				rt_rate_component.rt_component_id,
				rt_rate_component.rt_tax_id,
				rt_rate_component.rt_tax_ind,
				'0' AS rt_component_sys_code,
				'0' AS rv_item_comp_var
			FROM
				rt_rate_component
			WHERE
				rt_rate_component.rt_rate_id = '$rt_rate_id'
		",2);
		
		$perStayEnabled = $GLOBALS['lDB']->get("SELECT rf_per_stay_yn FROM rf_default", 4);
		if($perStayEnabled) {
			$perStayComp = $GLOBALS['lDB']->get("
				SELECT 
					rt_component.rt_component_ix AS rt_component_id,
					rt_component.rt_component_abbrv ,
					'1' AS ac_acc_cmp_option_yn,
					'0' AS rt_comp_comm_yn,
					'0' AS rt_comp_freq_ind,
					'1' AS rt_comp_freq_factor,
					'0' AS rt_comp_separate_yn,
					rt_component.rt_component_sys_code,
					'0' AS rv_item_comp_var
				FROM
					rt_component
				WHERE
					rt_component.rt_component_sys_code = 1
			",1);
			array_push($components,$perStayComp);
		}

		$newComponents = array();
		
		foreach($components as $compKey=>$compItem) {
			if(
				$compItem['ac_acc_cmp_option_yn'] == 0 
				|| ($compItem['ac_acc_cmp_option_yn'] == 1 && in_array($compItem['rt_component_id'],$this->Optionals))
			) {

				$compCommDB = $compItem['rt_comp_comm_yn']; // commission flag to save (for use with specials)
				$compComm = 0; // commission flag to use in calculation

				if($rt_rate_comm_yn) { // allow commission where enabled
					$compComm = $compItem['rt_comp_comm_yn'];
				}
				
				array_push($newComponents,array(
					'ac_acc_cmp_type'=>"standard",
					'rt_component_id'=>$compItem['rt_component_id'],
					'comm_ind'=>$compComm,
					'comm_ind_db'=>$compCommDB,
					'rt_rate_ssupp_comm_yn'=>$rt_rate_ssupp_comm_yn,
					'rt_comp_freq_ind'=>$compItem['rt_comp_freq_ind'],
					'rt_comp_freq_factor'=>$compItem['rt_comp_freq_factor'],
					'ac_acc_cmp_option_yn'=>$compItem['ac_acc_cmp_option_yn'],
					'rt_component_separate_yn'=>$compItem['rt_comp_separate_yn'],
					'rt_tax_id'=>$compItem['rt_tax_id'],
					'rt_tax_ind'=>$compItem['rt_tax_ind'],
					'rt_component_sys_code'=>$compItem['rt_component_sys_code'],
					'rv_item_comp_var'=>$compItem['rv_item_comp_var']
				));
			}
		}
		return $newComponents;
	}
	
	function BuildComponent()
	{
		global $lDB;
		
		foreach($this->Periods as $key=>$item) {
			$components = array();
			if(rt_rate_exists_by_id($item['rt_rate_ix'])) {
				$components = $this->GetComponent($item['rt_rate_ix'],$item['rt_rate_comm_yn'],$item['rt_rate_oride_comm_yn'],$item['rt_rate_ssupp_comm_yn']);
			}
			$this->Periods[$key]['baseComponents'] = $components;
			$this->Periods[$key]['components'] = array();
		}
	}
	
	function BuildCharge()
	{
		$firstPeriod = true;
		$perStayEnabled = $GLOBALS['lDB']->get("SELECT rf_per_stay_yn FROM rf_default", 4);
		$perStayAutoEnabled = $GLOBALS['lDB']->get("SELECT rf_per_stay_auto_yn FROM rf_default", 4);	
		$noPerStayComp = $GLOBALS['lDB']->get("SELECT rt_component_ix FROM rt_component WHERE rt_component_sys_code = 1", 4);
		
		foreach($this->Periods as $key=>$item) {
			$totals = array(
				'rv_item_amt_gross'=>false,
				'rv_item_amt_nett'=>false,
				'rv_item_amt_payable'=>false,
				'rv_item_amt_comm'=>false,
				'rv_item_amt_comm_calc'=>false,
				'rv_item_amt_tax'=>false
			);
			
			$period =& $this->Periods[$key];
			$baseComponents =& $period['baseComponents'];
			$components =& $period['components'];
			
			$noPerStayCompExists = false;
			foreach($baseComponents as $compKey=>$compItem) {
				if($compItem['rt_component_id'] == $noPerStayComp) {
					$noPerStayCompExists = true;
				}
			}

			if((!$firstPeriod && $perStayEnabled == 1) || ($noPerStayCompExists)) {
				$ignorePerStays = true;
			} else {
				$ignorePerStays = false;
			}
			
			// Use rate's override level?
			if($this->Override['level'] == "") {
				$override_level = $item['rt_rate_overide_level_ind'];
			} else {
				$override_level = $this->Override['level'];
			}

			$rateGroupAmts = array();
			if($item['rt_pax_rate_group_yn']) {
				// [START] Pax Only		
				// If we're dealing with "pax only", we need to do some trickery here to simulate rate group
				// amounts based on the pax .  We create a "fabricated" array of rate group amounts.
				$unitGroup = $GLOBALS['lDB']->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1", 4);

				// Build more readable / accessible array of amounts
				foreach($item['rateGroupAmts'] as $key=>$rateGroupAmt) {
					$paxAmounts[$rateGroupAmt['rt_rate_grp_amt_qty']] = $rateGroupAmt['rateGroupTotal'];
				}

				if($this->People != 0) {
					if($this->People == 1) {
						$per_person_total = $paxAmounts[1];
					} else {
						$per_person_total = $paxAmounts[$this->People] / $this->People;
					}
				} else {
					$per_person_total = 0;
				}
	

				foreach($this->RateGroups as $rateGroup) {
					if($rateGroup['qty'] > 0 && $rateGroup['rt_rate_group_ix'] != $unitGroup) {
						$rateGroupAmts[] = array(
							'rt_rate_grp_amt_qty' => ($rateGroup['qty'] == 1 && $this->People > 1) ? 0 : $rateGroup['qty'],
							'rateGroupTotal' => $per_person_total * $rateGroup['qty'],
							'rt_rate_group_ix' => $rateGroup['rt_rate_group_ix']
						);
					}
				}	
				// [END] Pax Only
			}  else {
				$rateGroupAmts = $item['rateGroupAmts'];	
			}

			if(count($rateGroupAmts) > 0) {
				$this->RecalculateAllAmounts($item['rt_rate_ix'], $rateGroupAmts, $override_level, $item['ac_prd_days'], $ignorePerStays);

				foreach($baseComponents as $compKey=>$compItem) {
					if($compItem['rt_comp_freq_ind'] == 2 && !$firstPeriod && $perStayAutoEnabled == 1 && !$noPerStayCompExists) {
						$perStayComp = $GLOBALS['lDB']->get("
							SELECT 
								'standard' AS 'ac_acc_cmp_type',
								rt_component.rt_component_ix AS rt_component_id,
								rt_component.rt_component_abbrv ,
								'1' AS ac_acc_cmp_option_yn,
								'0' AS rt_comp_comm_yn,
								'0' AS rt_comp_freq_ind,
								'1' AS rt_comp_freq_factor,
								'0' AS rt_comp_separate_yn,
								rt_component.rt_component_sys_code,
								'0' AS rv_item_comp_var
							FROM
								rt_component
							WHERE
								rt_component.rt_component_sys_code = 1
						",1);
						$perStayComp['tax_detail'] = array();
						$components[$compKey] = $perStayComp;
						continue;
					}

					if($compItem['rt_comp_freq_ind'] == 2 && $noPerStayCompExists) {
						continue;
					}
					
					$multiplier = $this->GetMultiplier($compItem,$item['ac_prd_days']);

					$compItem = $this->CalculateComponent($compItem, $multiplier);

					$baseComponents[$compKey] = array_merge($baseComponents[$compKey],$compItem);

				}

				$components = $this->CalculateTotals($baseComponents,$item['ac_prd_days'],$item['rt_rate_comm_yn']);

				// [START] Rounding
				if($period['rt_rate_round_ind'] > 0 && $period['rt_rate_round_scale'] > 0) {
					if($period['rt_rate_round_ind'] == 1) { // Rounding per itinerary
						$total_nett = 0;
						$total_comm = 0;
						foreach($period['components'] as $component) {
							$total_nett += $component['rv_item_comp_amt_nett'];
							$total_comm += $component['rv_item_comp_amt_comm'];
						}

						$rounded_nett = $this->CalculateRoundedAmount($total_nett,$period['rt_rate_round_dir'],$period['rt_rate_round_scale']);
						$comm_diff = $total_nett - $rounded_nett;
						$comm_diff_rg_total = 0;
						$largestComponent = "";
						$largestRateGroups = array();
						$largetsComponentRatio = 0;
						foreach($period['components'] as $comp_key => $component) {
							if($component['rv_item_comp_amt_comm'] != 0) {
								$comp_ratio = $component['rv_item_comp_amt_comm'] / $total_comm;
								
								if($comp_ratio > $largetsComponentRatio) {
									$largetsComponentRatio = $comp_ratio;
									$largestComponent = $comp_key;
								}

								$add_comp_comm = $comm_diff * $comp_ratio;
								$largestRateGroupRatio = 0;
								foreach($component['rate_groups'] as $rg_key => $rate_group) {
									if($rate_group['comm'] != 0) {
										$rate_group_ratio = $rate_group['comm'] /  $component['rv_item_comp_amt_comm'];
								
										if($rate_group_ratio > $largestRateGroupRatio) {
											$largestRateGroupRatio = $rate_group_ratio;
											$largestRateGroups[$comp_key] = $rg_key;
										}
								
										$add_rate_group_comm = $add_comp_comm * $rate_group_ratio;
										// Additional commission required to cater for rounding
										$baseComponents[$comp_key]['rate_groups'][$rg_key]['add_comm'] = round($add_rate_group_comm,2);
										$comm_diff_rg_total += round($add_rate_group_comm,2);
									}
										
								}
							}
						}

						// Cater for possible difference from rounding at the
						// rate group level, by adding the difference to the
						// largest component's largest rate group (wait, what?!?!)
						if(!empty($largestComponent) && $comm_diff_rg_total != $comm_diff) {
							$round_diff = $comm_diff - $comm_diff_rg_total;
							$baseComponents[$largestComponent]['rate_groups'][$largestRateGroups[$largestComponent]]['add_comm'] += $round_diff;
						}

					}

					if($period['rt_rate_round_ind'] == 2) { // Rounding per person
						$unitGroup = $GLOBALS['lDB']->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1", 4);
						$rateGroupTotals = [];
						foreach($period['components'] as $component) {
							foreach($component['rate_groups'] as $rt_rate_group_id=>$rate_group) {
								if(!array_key_exists($rt_rate_group_id,$rateGroupTotals)) {
									$rateGroupTotals[$rt_rate_group_id] = [
										'qty'=>$rate_group['qty'],
										'nett'=>0,
										'comm'=>0
									];
								}
								$rateGroupTotals[$rt_rate_group_id]['nett'] += $rate_group['nett'];
								$rateGroupTotals[$rt_rate_group_id]['comm'] += $rate_group['comm'];
							}
						}
						foreach($rateGroupTotals as $rt_rate_group_id=>$rateGroupTotal) {
							$total_nett = $rateGroupTotal['nett'];
							$total_comm = $rateGroupTotal['comm'];

							if($total_comm == 0) {
								continue;
							}

							$nett = $total_nett / $this->Nights / $rateGroupTotal['qty'];
							if($rt_rate_group_id != $unitGroup) {
								$nett /= $this->Units;
							}
							$rounded_nett = $this->CalculateRoundedAmount($nett, $period['rt_rate_round_dir'], $period['rt_rate_round_scale']) * $this->Nights * $rateGroupTotal['qty'];
							if($rt_rate_group_id != $unitGroup) {
								$rounded_nett *= $this->Units;
							}
							$comm_diff = $total_nett - $rounded_nett;
							$comm_diff_rg_total = 0;

							$largestRatio = 0;
							$largestComponent = false;

							foreach($period['components'] as $comp_key=>$component) {
								if(isset($component['rate_groups'][$rt_rate_group_id])) {
									$rate_group = $component['rate_groups'][$rt_rate_group_id];
									$comp_ratio = $rate_group['comm'] / $total_comm;

									if($comp_ratio > $largestRatio) {
										$largestRatio = $comp_ratio;
										$largestComponent = $comp_key;
									}

									$add_comm = round($comm_diff * $comp_ratio,2);
									$baseComponents[$comp_key]['rate_groups'][$rt_rate_group_id]['add_comm'] = $add_comm;
									$comm_diff_rg_total += $add_comm;

								}
							}

							if(!empty($largestComponent) && $comm_diff_rg_total != $comm_diff) {
								$round_diff = $comm_diff - $comm_diff_rg_total;
								$baseComponents[$largestComponent]['rate_groups'][$rt_rate_group_id]['add_comm'] += $round_diff;
							}
						}
					}

					$components = $this->CalculateTotals($baseComponents,$item['ac_prd_days'],$item['rt_rate_comm_yn']);
				}
				// [END] Rounding

				// Itinerary Totals
				foreach($components as $component) {
					$totals['rv_item_amt_gross'] += $component['rv_item_comp_amt_gross'];
					$totals['rv_item_amt_nett'] += $component['rv_item_comp_amt_nett'];
					$totals['rv_item_amt_payable'] += $component['rv_item_comp_amt_payable'];
					$totals['rv_item_amt_comm'] += $component['rv_item_comp_amt_comm'];
					$totals['rv_item_amt_comm_calc'] += $component['rv_item_comp_amt_comm_calc'];
					$totals['rv_item_amt_tax'] += $component['rv_item_comp_amt_tax'];
				}

				fixAmtDisplay($totals['rv_item_amt_gross']);
				fixAmtDisplay($totals['rv_item_amt_nett']);
				fixAmtDisplay($totals['rv_item_amt_payable']);
				fixAmtDisplay($totals['rv_item_amt_comm']);
				fixAmtDisplay($totals['rv_item_amt_comm_calc']);
				fixAmtDisplay($totals['rv_item_amt_tax']);
				
				// Beware all ye who enter here.  Some crazy shit takes place inside this loop!
				$waiveValues = $GLOBALS['lDB']->get("SELECT rt_rate_waive_checks_yn FROM rt_rate WHERE rt_rate_ix = '".$item['rt_rate_ix']."'",4);
				if($waiveValues == "1") {
					$nonUnitTotal = 0;
					$unitGroup = $GLOBALS['lDB']->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1", 4);
					$unitComp = "";
					$rateCompSplits = getRateComponentSplits($item['rt_rate_ix'], $unitGroup);
					foreach($rateCompSplits as $rateCompSplit) {
						if($rateCompSplit['rt_rate_comp_perc'] == 100) {
							$unitComp = $rateCompSplit['rt_component_id'];
						}
					}
						
					$unitCompKey = "";
					foreach($components as $key => $component) {
						if($component['rt_component_id'] != $unitComp) {
							$components[$key]['rate_groups'] = array();				// clear the rate groups
							$nonUnitTotal += $component['rv_item_comp_rate'];		// add to the total to deduct from unit component
						} else {
							$unitCompKey = $key;
						}
					}
		
					if($unitCompKey !== "") {
						$components[$unitCompKey]['rv_item_comp_rate'] -= $nonUnitTotal;
					}
				}				

			} else {
				// Rate is TBA
			}
	
			$period = array_merge($period,$totals);

			unset($period);
			unset($baseComponents);
			unset($components);

			$firstPeriod = false;
		}
		
	}

	// Works out the totals for a component, including the individual calculated commission for that component.
	// NB: The TOTAL commission could differ and this is calculated afterwards in CalculateTotals.
	function GetComponentTotals($component=false,$days=false,$commYN=false)
	{
	
		$totals = array();
	
		if(is_array($component) && (!isset($component['rt_component_sys_code']) || $component['rt_component_sys_code'] != "1")) {
			$multiplier = $this->GetMultiplier($component,$days);

			$base = $component['rv_item_comp_rate'];
			$baseComm = $component['rv_item_comp_rate_comm'];

			$variance = $component['rv_item_comp_var'] * $multiplier;
			
			$gross = ($base * $this->Exchange) + $variance;
			$grossComm = ($baseComm * $this->Exchange) + $variance;

			$commcalc = 0;

			if($commYN && $component['comm_ind'] != 0) {
				$commcalc = round($grossComm * ($this->Commission / 100),2);
			}

			
			
			// Rate Group Amounts
			if(is_array($component['rate_groups'])) {
				foreach($component['rate_groups'] as $key => $rateGroup) {

					$rg_base = $rateGroup['rate'];
					$rg_base_comm = $rateGroup['rate_comm'];

					$sp_discount = $rateGroup['rv_item_comp_rate_grp_amt_sp_discount'];
					
					$rg_variance = $rateGroup['rv_item_comp_rate_grp_amt_var'] * $multiplier;
					
					$rg_gross = ($rg_base * $this->Exchange) + $rg_variance - $sp_discount;
					$rg_gross_comm = ($rg_base_comm * $this->Exchange) + $rg_variance - $sp_discount;

					$rg_commcalc = 0;
					if($commYN && $component['comm_ind'] != 0) {
						$rg_commcalc = round($rg_gross_comm * ($this->Commission / 100),2);
					}					
					
					// Add any additional commission for rounding
					if(isset($rateGroup['add_comm'])) {
						$rg_commcalc += $rateGroup['add_comm'];
					}

					$rateGroups[$key]['rate'] = round($rg_base,2);
					$rateGroups[$key]['rate_comm'] = round($rg_base_comm,2);
					$rateGroups[$key]['gross'] = round($rg_gross,2);
					$rateGroups[$key]['comm_calc'] = round($rg_commcalc,2);

					$component['rate_groups'][$key] = array_merge($component['rate_groups'][$key],$rateGroups[$key]);	
				}
				
				
			}
			
			
		} else {
			$amount = false;
			$gross = false;
			$nett = false;
			$payable = false;
			$tax = false;
		}

		$totals = array(
			'rate_groups'=>$component['rate_groups'],
		);
		return $totals;
	
	}
	
	function SetRateGroups($usedRateGroups=false)
	{

		global $lDB;
		$diffs = 0;
		$checksum = "";
		
		if(!$usedRateGroups) {
			$usedRateGroups = array();
		}
		
		$this->BuildPeriod();
		foreach($this->Periods as $key => $period) {
			$this->Periods[$key]['RateGroups'] = $lDB->get("
				SELECT DISTINCT
					rt_rate_group.rt_rate_group_ix,
					rt_rate_group.rt_rate_group_desc,
					rt_rate_group.rt_rate_group_abbrv,
					rt_rate_group.rt_rate_group_sys_code,
					rt_rate_group.rt_rate_group_class,
					rt_rate_comp_grp.rt_rate_comp_grp_seq
				FROM
					rt_rate_group
					INNER JOIN rt_rate_comp_grp ON rt_rate_comp_grp.rt_rate_group_id = rt_rate_group.rt_rate_group_ix
					INNER JOIN rt_rate_component ON rt_rate_component.rt_rate_component_ix = rt_rate_comp_grp.rt_rate_component_id
					INNER JOIN rt_rate ON rt_rate.rt_rate_ix = rt_rate_component.rt_rate_id
				WHERE
					rt_rate.ac_accomm_type_id  = '$this->AccommTypeId'
					AND rt_rate.rt_rate_type_id   = '$this->RateTypeId'
					AND rt_rate.rt_period_id    = '".$period['rt_period_ix']."'
				ORDER BY
					CASE rt_rate_comp_grp.rt_rate_comp_grp_seq WHEN 0 THEN 1 ELSE 0 END,
					rt_rate_comp_grp.rt_rate_comp_grp_seq,
					rt_rate_group.rt_rate_group_desc						
			",2);
			
			if($checksum != serialize(array_flatten($this->Periods[$key]['RateGroups']))) {
				$diffs++;
			}
			
			$checksum = serialize(array_flatten($this->Periods[$key]['RateGroups']));
		}

		if($diffs > 1 || empty($this->Periods)) {
			$rateGroups = array();
		} else {
			$rateGroups = $this->Periods[0]['RateGroups'];
		}
		

		// If we STILL have no rate groups, we supply a default set consisting of the Unit record and any defaults
		if(count($rateGroups) == 0) {
			$rateGroups = $lDB->get("
				SELECT 
					rt_rate_group.rt_rate_group_ix,
					rt_rate_group.rt_rate_group_desc,
					rt_rate_group.rt_rate_group_abbrv,
					rt_rate_group.rt_rate_group_sys_code,
					rt_rate_group.rt_rate_group_class
				FROM
					rt_rate_group
				WHERE
					rt_rate_group.rt_rate_group_sys_code = 1 OR rt_rate_group.rt_rate_group_default_yn = 1
			",2);
		}

		// Check if we already have a qty for the allowable rate groups.
		foreach($rateGroups as $key => $rateGroup) {
			$rateGroups[$key]['qty'] = 0;
			foreach($usedRateGroups as $usedRateGroup) {
				if (strtoupper($rateGroup['rt_rate_group_ix']) == strtoupper($usedRateGroup['rt_rate_group_ix'])) {
					$rateGroups[$key]['qty'] = isset($usedRateGroup['qty'])?$usedRateGroup['qty']:0;
				}
			}
		}
		
		$this->RateGroups = $rateGroups;
		$this->BuildPeople();
		$this->ResetData();
	}	

	function RecalculateAllAmounts($rateID, $rateGroupAmts, $overrideLevel=false, $days=false, $ignorePerStays=false)
	{
		global $lDB;

		$maxCapacity = getAccommTypeMaxCapacity($this->AccommTypeId);	
		$waiveValues = $lDB->get("SELECT rt_rate_waive_checks_yn FROM rt_rate WHERE rt_rate_ix = '".$rateID."'",4);
		$unitGroup = $lDB->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1", 4);
		
		// If we have more units than MaxCapacity, we need to project a total for that number of rooms and add it to the array
		if($this->Units > $maxCapacity) {
			$count = $this->Units;

			foreach($this->RateGroups as $rateGroup) {
				$maxCapTotal = 0;
				$avgQuantity = 0;
				$calculatedAmount = 0;
				
				foreach($rateGroupAmts as $key=>$rateGroupAmt) {
					if($rateGroupAmt['rt_rate_grp_amt_qty'] == $maxCapacity && $rateGroupAmt['rt_rate_group_ix'] == $rateGroup['rt_rate_group_ix']) {
						$maxCapTotal = $rateGroupAmt['rateGroupTotal'];
					}
				}		
				if($maxCapacity != 0) {
					$avgQuantity = $maxCapTotal / $maxCapacity;
				} else {
					$avgQuantity = 0;
				}
				$calculatedAmount = $avgQuantity * $this->Units;
				
				$rateGroupAmts[] = array(
					'rt_rate_group_ix' => $rateGroup['rt_rate_group_ix'],
					'rt_rate_grp_amt_qty' => $this->Units,
					'rateGroupTotal' => $calculatedAmount
				);
			}
		} else {
			$count = $maxCapacity;
		}

		// Build more readable / accessible array of amounts
		foreach($rateGroupAmts as $key=>$rateGroupAmt) {
			$rateGroupAmtsSort[$rateGroupAmt['rt_rate_group_ix']][$rateGroupAmt['rt_rate_grp_amt_qty']] = $rateGroupAmt['rateGroupTotal'];
		}			

		// recalculate all amounts for the groups that we have values for
		// Separating this into two distinct sections since Family Unit is a very unique case
		if($waiveValues == "1") { // Family Unit
			for($i=0;$i<=$count; $i++) {
				foreach($this->RateGroups as $rateGroup) {
					foreach($rateGroupAmts as $rateGroupAmt) {
						if($rateGroupAmt['rt_rate_group_ix'] == $rateGroup['rt_rate_group_ix'] && $rateGroupAmt['rt_rate_grp_amt_qty'] == $i) {
							if(!empty($this->Override) && $this->Override['amounts']['rate_group'][$unitGroup] != "" && $rateGroup['rt_rate_group_ix'] == $unitGroup) {
								// We assume that override level is PER unit and therefore the override amount is the Group Total here
								$groupTotal = $this->Override['amounts']['rate_group'][$unitGroup];
							} else if($rateGroup['rt_rate_group_ix'] == $unitGroup) {
								$groupTotal = $rateGroupAmt['rateGroupTotal'];
							} else {
								$groupTotal = 0;
							}
							$this->CalculateSingleAmount($rateID, $rateGroup['rt_rate_group_ix'], $groupTotal, $i, $days, $ignorePerStays);
						}
					}
				}
			}
		} else {	// Normal mode
			for($i=0;$i<=$count; $i++) {
				foreach($this->RateGroups as $rateGroup) {
					foreach($rateGroupAmts as $rateGroupAmt) {
						if($rateGroupAmt['rt_rate_group_ix'] == $rateGroup['rt_rate_group_ix'] && $rateGroupAmt['rt_rate_grp_amt_qty'] == $i) {
							if(is_numeric($rateGroupAmt['rateGroupTotal'])) {
								if($this->Override) {
									$groupTotal = $this->GetOverrideGroupTotal($overrideLevel,$i,$rateGroup,$rateGroupAmtsSort);
								} else {
									$groupTotal = $rateGroupAmt['rateGroupTotal'];
								}
								$this->CalculateSingleAmount($rateID, $rateGroup['rt_rate_group_ix'], $groupTotal, $i, $days, $ignorePerStays);
							}
						}
					}
				}
			}
		}
	}

	function GetOverrideGroupTotal($overrideLevel,$numPax,$rateGroup,$rateGroupAmts)
	{
		$unitGroup = $GLOBALS['lDB']->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1", 4);
		$groupTotal = 0;

		$amountTotal = 0;
		// Work out REGULAR amount total
		foreach($this->RateGroups as $item) {
			if($item['qty'] > 0) {
				if($item['rt_rate_group_ix'] == $unitGroup) {
					if(isset($rateGroupAmts[$item['rt_rate_group_ix']])) {
						if (isset($rateGroupAmts[$item['rt_rate_group_ix']][$item['qty']])) {
							$amountTotal += $rateGroupAmts[$item['rt_rate_group_ix']][$item['qty']];
						}						
					}
				} else {
					if($item['qty'] == 1 && $this->People > 1) {
						$amountTotal += $rateGroupAmts[$item['rt_rate_group_ix']][0] * $this->Units;
					} else {
					    if(isset($rateGroupAmts[$item['rt_rate_group_ix']][$item['qty']])) {
					        $amountTotal += $rateGroupAmts[$item['rt_rate_group_ix']][$item['qty']] * $this->Units;
                        }
					}
				}
			}
		}

		if($amountTotal != 0) {
			switch ($overrideLevel) {
				case 1: // per rate group
					if(isset($this->Override['amounts']['rate_group'][$rateGroup['rt_rate_group_ix']]) && $this->Override['amounts']['rate_group'][$rateGroup['rt_rate_group_ix']] != "") {
						$groupTotal = $this->Override['amounts']['rate_group'][$rateGroup['rt_rate_group_ix']] * $rateGroup['qty'];
					} else {
						if($rateGroup['qty'] == 1 && $this->People > 1) {
							$groupTotal = $rateGroupAmts[$rateGroup['rt_rate_group_ix']][0];
						} elseif($rateGroup['qty'] != 0) {
							$groupTotal = $rateGroupAmts[$rateGroup['rt_rate_group_ix']][$rateGroup['qty']];
						}					
					}
					break;

				case 2:  // per unit
					if($this->Units != 0) {
						if($rateGroup['rt_rate_group_ix'] == $unitGroup) {
							if($this->Override['amounts']['rate_group'][$unitGroup] != "") {
								$groupTotal = $this->Override['amounts']['rate_group'][$unitGroup] * $this->Units;
							} else {
								$groupTotal = 0;
							}
						} else {
							$groupTotal = 0;
						}						
						
					} else {
						$groupTotal = 0;
					}
					break;					

				case 3: // per stay
					$overrideTotalPerNight = $this->Override['amounts']['stay'] / $this->Nights;
					
					if($rateGroup['rt_rate_group_ix'] == $unitGroup) {
						$groupTotal = ($rateGroupAmts[$rateGroup['rt_rate_group_ix']][$rateGroup['qty']] / $amountTotal) * $overrideTotalPerNight;
					} else {
						if($rateGroup['qty'] == 1 && $this->People > 1) {
							$groupTotal = ((($rateGroupAmts[$rateGroup['rt_rate_group_ix']][0] * $this->Units) / $amountTotal) * $overrideTotalPerNight) / $this->Units;
						} else {
							if($rateGroup['qty'] != 0) {
								$groupTotal = ((($rateGroupAmts[$rateGroup['rt_rate_group_ix']][$rateGroup['qty']] * $this->Units) / $amountTotal) * $overrideTotalPerNight) / $this->Units;
							} else {
								$groupTotal = 0;
							}
						}
					}	

					//  If negative rooms were captured, reverse the polarity of the override total since with per stay, we disregard the number of units
					if($this->Units < 0) {
						$groupTotal = $groupTotal * -1;
					}

					break;
			}
		}
		else {
			switch ($overrideLevel) {
				case 1: // per rate group
					if($this->Units != 0) {
						
						if(isset($this->Override['amounts']['rate_group'][$rateGroup['rt_rate_group_ix']])) {
							$groupTotal = $this->Override['amounts']['rate_group'][$rateGroup['rt_rate_group_ix']] * $rateGroup['qty'];
						} else {
							$groupTotal = 0;
						}						
						
					} else {
						$groupTotal = 0;
					}
					break;
				case 2:  // per unit
				
				
					if($this->Units != 0) {
						if($rateGroup['rt_rate_group_ix'] == $unitGroup) {
							if($this->Override['amounts']['rate_group'][$unitGroup] != "") {
								$groupTotal = $this->Override['amounts']['rate_group'][$unitGroup] * $this->Units;
							} else {
								$groupTotal = 0;
							}
						} else {
							$groupTotal = 0;
						}						
						
					} else {
						$groupTotal = 0;
					}
					break;				

				case 3: // per stay
					$overrideTotalPerNight = $this->Override['amounts']['stay'] / $this->Nights;
			
					if($rateGroup['rt_rate_group_ix'] == $unitGroup && $this->Units != 0) {
						$groupTotal = ((1 * ($rateGroup['qty'] == 0 ? 1 : $rateGroup['qty'])) / (1 * $this->Units)) * $overrideTotalPerNight;
					} elseif($this->Units > 0 && $this->People > 0) {
						$groupTotal = ((((1 * ($rateGroup['qty'] == 0 ? 1 : $rateGroup['qty'])) * $this->Units) / (1 * $this->People * $this->Units)) * $overrideTotalPerNight) / $this->Units;
					} else {
						$groupTotal = 0;
					}	

					//  If negative rooms were captured, reverse the polarity of the override total since with per stay, we disregard the number of units
					if($this->Units < 0) {
						$groupTotal = $groupTotal * -1;
					}	
									
					break;				
			}

		}

		return $groupTotal;
	
	}
	
	function CalculateSingleAmount($rateID, $groupID, $groupTotal, $qty, $days, $ignorePerStays=false)
	{
		$calcTotal = $groupTotal;
		$rateCompSplits = getRateComponentSplits($rateID, $groupID);
		$runningTotal = 0;
		foreach($rateCompSplits as $key => $rateCompSplit) {
			$this->RateCompGrpAmounts[$rateCompSplit['rt_component_id']][$groupID]['commYN'] = $rateCompSplit['rt_rate_comp_grp_comm_yn'];		
		
			if($ignorePerStays && $rateCompSplit['rt_comp_freq_ind'] == 2) { continue; }
			if($rateCompSplit['rt_rate_comp_val_ind'] == 1) { // Amt
				if($rateCompSplit['rt_comp_opt_yn'] == 0) {	
					$amount = round($rateCompSplit['rt_rate_comp_amt'] * ($qty == 0 ? 1 : $qty),2);
					if($days != 0) {
						if($rateCompSplit['rt_comp_freq_ind'] == 2) {
							$amount = $amount / $days;
						} else {
							$amount =  ($amount / $days) * floor($days/$rateCompSplit['rt_comp_freq_factor']);
						}
					} else {
						$amount = 0;
					}

					$this->RateCompGrpAmounts[$rateCompSplit['rt_component_id']][$groupID][$qty] = $amount;
					$calcTotal = $calcTotal - $amount;
					$runningTotal += $amount;
				} else if($rateCompSplit['rt_comp_opt_yn'] == 1 && in_array($rateCompSplit['rt_component_id'],$this->Optionals)) {	// optional
					$amount = round($rateCompSplit['rt_rate_comp_amt'] * ($qty == 0 ? 1 : $qty),2);
					
					if($rateCompSplit['rt_comp_freq_ind'] == 2) {
						$amount = $amount / $days;
					}
					
					$this->RateCompGrpAmounts[$rateCompSplit['rt_component_id']][$groupID][$qty] = $amount;
				}
			} else {	// Perc
				if($key == count($rateCompSplits)-1 && $groupTotal > 0 && $rateCompSplit['rt_rate_comp_perc'] > 0) {	// last perc element
					$amount = $groupTotal - $runningTotal;
					$this->RateCompGrpAmounts[$rateCompSplit['rt_component_id']][$groupID][$qty] = $amount;
				} else {
					$amount = round($calcTotal * ($rateCompSplit['rt_rate_comp_perc'] / 100),2);
					$this->RateCompGrpAmounts[$rateCompSplit['rt_component_id']][$groupID][$qty] = $amount;
					$runningTotal += $amount;
				}
			}
		}
	}	

	function CalculateComponent($component, $multiplier)
	{
		$componentId = $component['rt_component_id'];
		$unitGroup = $GLOBALS['lDB']->get("SELECT rt_rate_group_ix FROM rt_rate_group WHERE rt_rate_group_sys_code = 1", 4);

		$componentRateGroups = [];
		$componentAmount = 0;
		$componentAmountCommissionable = 0;
		$roundedComponentAmount = 0;
		$roundedComponentAmountCommissionable = 0;

		// Get last rate group in array with a qty
		foreach($this->RateGroups as $key => $rateGroup) {
			if($rateGroup['qty'] > 0) {
				$last = $key;
			}
		}

		foreach($this->RateGroups as $key => $rateGroup) {
			$rateGroupId = $rateGroup['rt_rate_group_ix'];
			$isUnit = $rateGroupId == $unitGroup;

			$amount = 0;
			$amountCommissionable = 0;

			if(
				$rateGroup['qty'] == 1
				&& (
					$isUnit
					|| (!$isUnit && $this->People > 1)
				)
			) {
				// Use 1 sharing value (first value on Amount tab) for:
				// - For single units
				// - For single pax rates group where the room is shared with other rate groups
				$quantity = 0;
			} else {
				$quantity = $rateGroup['qty'] ;
			}

			if($rateGroup['qty'] > 0) {
				if(isset($this->RateCompGrpAmounts[$componentId][$rateGroupId])) {
					if(!isset($this->RateCompGrpAmounts[$componentId][$rateGroupId][$quantity])) {
						$this->RateCompGrpAmounts[$componentId][$rateGroupId][$quantity] = 0;
					}
					$amount = $this->RateCompGrpAmounts[$componentId][$rateGroupId][$quantity];	
					if($this->RateCompGrpAmounts[$componentId][$rateGroupId]['commYN'] == 1) {
						// If Single Supplement is not commissionable,
						// we store the value for 1 Sharing for commission calcs later
						if($component['rt_rate_ssupp_comm_yn'] == 0 && $this->People == 1) {
							if(!isset($this->RateCompGrpAmounts[$componentId][$rateGroupId][0])) {
								$this->RateCompGrpAmounts[$componentId][$rateGroupId][0] = 0;
							}
							$amountCommissionable = $this->RateCompGrpAmounts[$componentId][$rateGroupId][0];
						} else {
							$amountCommissionable = $amount;
						}
					}
					if($isUnit) {
						// TODO: Since the component amount gets multiplied by
						//       the number of units at the end of this process,
						//       we need to take this into account at this point 
						//       by dividing the Units rate group amount by the 
						//       number of Units.
						//       In future, GetMultiplier should be called for
						//       each Rate Group, rather than for each component
						//       (if possible).
						$amount /= $this->Units;
						$amountCommissionable /= $this->Units;
					}
				}

				$rate = $amount * $multiplier;
				$rateCommissionable = $amountCommissionable * $multiplier;

				$componentAmount += $rate;
				$componentAmountCommissionable += $rateCommissionable;

				$roundedRate = round($rate, 2);
				$roundedRateCommissionable = round($rateCommissionable, 2);


				// Provision for rounding differences between rate group amounts.
				// If this is the last rate group, ensure that there won't be any 
				// rounding issues by forcing the amount to be what we would expect
				if($key != $last) {
					$componentRateGroups[$rateGroupId] = [
						'qty'=>$rateGroup['qty'],
						'rv_item_comp_rate_grp_amt_var'=>0,
						'rv_item_comp_rate_grp_amt_sp_discount'=>0,
						'rate'=>$roundedRate,
						'rate_comm'=>$roundedRateCommissionable
					];
				} else {
					$componentRateGroups[$rateGroupId] = [
						'qty'=>$rateGroup['qty'],
						'rv_item_comp_rate_grp_amt_var'=>0,
						'rv_item_comp_rate_grp_amt_sp_discount'=>0,
						'rate'=>round(round($componentAmount, 2) - $roundedComponentAmount, 2),
						'rate_comm'=>round(round($componentAmountCommissionable, 2) - $roundedComponentAmountCommissionable, 2)
					];
				}

				$roundedComponentAmount += $roundedRate;
				$roundedComponentAmountCommissionable += $roundedRateCommissionable;

			}
		}

		return array_merge($component, [
			'rate_groups'=>$componentRateGroups,
			'rv_item_comp_rate'=>round($componentAmount, 2),
			'rv_item_comp_rate_comm'=>round($componentAmountCommissionable, 2)
		]);
	}
	
	function CalculateComponentTax(&$component, $payable)
	{
		global $lDB;
		$taxAmounts = array();
		
		
		$taxAmounts[$component['rt_component_id']] = array();
		$tax = 0;
		
		switch($component['rt_tax_ind']) {
		case 10:		// Tax Rate
			$taxRate = $lDB->get("
				SELECT 
					rf_tax_rate.rf_tax_rate_ix,
					rf_tax_rate.rf_tax_rate_perc 
				FROM
					rf_tax_rate
				WHERE
					rf_tax_rate.rf_tax_rate_ix = '".$component['rt_tax_id']."';
			",1);
			
			$taxAmounts[$component['rt_component_id']][$taxRate['rf_tax_rate_ix']]['perc'] = $taxRate['rf_tax_rate_perc'];
			$taxAmounts[$component['rt_component_id']][$taxRate['rf_tax_rate_ix']]['amount'] = $tax = $payable / (100+$taxRate['rf_tax_rate_perc']) * $taxRate['rf_tax_rate_perc'];
			
			break;
			
		case 20:		// Tax Group

			$taxRates = $lDB->get("
				SELECT 
					rf_tax_rate.rf_tax_rate_ix,
					rf_tax_rate.rf_tax_rate_perc,
					rt_tax_group_item.rt_tax_base_ind
				FROM
					rf_tax_rate
					INNER JOIN rt_tax_group_item ON rt_tax_group_item.rf_tax_rate_id = rf_tax_rate.rf_tax_rate_ix
				WHERE
					rt_tax_group_item.rt_tax_group_id = '".$component['rt_tax_id']."'
				ORDER BY 
					rt_tax_group_item.rt_tax_base_ind
			",2);		
			
			//--------START OLD SYSTEM WITH SEQUENCES--------------
			
			// $taxableAmt = $payable;
			// $taxInd = 0;
			// $groupAmount = 0;
			// $taxRateByInd = array();
			// foreach($taxRates as $key=>$taxRate) {
				// $taxRateByInd[$taxRate['rt_tax_base_ind']]['accummTaxPerc'] += $taxRate['rf_tax_rate_perc'];
			// }
			// foreach($taxRates as $key=>$taxRate) {
				// $taxInd = $taxRate['rt_tax_base_ind'];
				// $amount = $taxableAmt / (100+$taxRateByInd[$taxInd]['accummTaxPerc']) * $taxRate['rf_tax_rate_perc'];
				// $taxAmounts[$component['rt_component_id']][$taxRate['rf_tax_rate_ix']]['perc'] = $taxRate['rf_tax_rate_perc'];
				// $taxAmounts[$component['rt_component_id']][$taxRate['rf_tax_rate_ix']]['amount'] += $amount;
				
				// if ($taxInd != $taxRates[$key+1]['rt_tax_base_ind']) {
					// $groupAmount += $amount;
					// $taxableAmt = $taxableAmt - $groupAmount;
					// $groupAmount = 0;
				// } else {
					// $groupAmount += $amount;
				// }
			// }
			// break;			
			
			//--------END OLD SYSTEM WITH SEQUENCES--------------	
			
			// Step 1 : Work out amount ex ALL TAXES
			$percSum = 0;
			$taxableAmt = $payable;
			foreach($taxRates as $key=>$taxRate) {
				$percSum += $taxRate['rf_tax_rate_perc'];
			}
			$amtExTax = $taxableAmt / (100 + $percSum) * 100;
			
			foreach($taxRates as $key=>$taxRate) {

				$taxAmounts[$component['rt_component_id']][$taxRate['rf_tax_rate_ix']]['perc'] = $taxRate['rf_tax_rate_perc'];
				$taxAmounts[$component['rt_component_id']][$taxRate['rf_tax_rate_ix']]['amount'] = $amtExTax * ($taxRate['rf_tax_rate_perc'] / 100);
				
			}				
			
		}		
	
		return $taxAmounts;
	}
	
	function BuildPeople()
	{
		$this->People = 0;
		foreach($this->RateGroups as $rateGroup) {
			if($rateGroup['rt_rate_group_sys_code'] != 1) {
				$this->People += isset($rateGroup['qty']) ? $rateGroup['qty'] : 0;
			}
		}
	}	

	function GetMultiplier($component,$days)
	{
		$unitDays = $this->Units * $days;
		$rt_comp_freq_ind = isset($component['rt_comp_freq_ind'])?$component['rt_comp_freq_ind']:"";
		switch($rt_comp_freq_ind) {
		case 1:
			return $unitDays;
			break;
		case 2:
			return $unitDays;
			break;
		default:
			return 1;
			break;
		}		
	}
	
	function SetUnits()
	{
		foreach($this->RateGroups as $rateGroup) {
			if($rateGroup['rt_rate_group_sys_code'] == 1) {
				$this->Units = isset($rateGroup['qty']) ? $rateGroup['qty'] : 0;
			}
		}
	}

	function SetOptionals($sOptionals)
	{
		if($sOptionals === false && db_sc_group_get_user_setting("sc_grp_def_optionals_yn") == "1") {
			$defaultOptionals = getDefaultOptionalComponents($this->AccommTypeId,$this->RateTypeId,$this->ArriveDate,$this->DepartDate);
			$sOptionals = array();
			foreach($defaultOptionals as $optional) {
				$sOptionals[] = $optional['rt_component_ix'];
			}
		}
		if(!is_array($sOptionals)) {
			$sOptionals = array();
		}

		$this->Optionals = array();
		foreach($sOptionals as $rt_component_id) {
			if(db_rt_component_exists($rt_component_id)) {
				array_push($this->Optionals,$rt_component_id);
			}
		}
	}

	function SetPeriodSplitDefault()
	{
		global $lDB;

		$this->PeriodSplitDefault = $lDB->get("SELECT rf_period_split FROM rf_default",4);

		// Incude all days in the period based on the chosen date range
		// This is overridden in the API by the rt_get_rate_total_range call to
		// only return a single day per period
		$this->PeriodSplitSingleDay = false;
	}
	
	function CalculateTotals($componentArray, $days, $commYN)
	{
	
		$newComponentArray = array();
		foreach($componentArray as $key => $component) {
			$newComponentArray[$key] = array_merge($component,$this->GetComponentTotals($component,$days,$commYN));
		}	

		// Commission
		$calcCompRateGroupTotal = 0;
		$calcCompRateGroupTotals = array();
		$deductCompRateGroupTotals = array();
		$deductCompKeys = array();
		$deductAllCompKey = "";		
		foreach($newComponentArray as $key => $component) {
			// Init comm field
			$newComponentArray[$key]['rv_item_comp_amt_comm'] = 0;
			foreach($component['rate_groups'] as $rgKey => $rateGroup) {
				$newComponentArray[$key]['rate_groups'][$rgKey]['comm'] = 0;
			}
			
			if($component['comm_ind'] == 1) {
				foreach($component['rate_groups'] as $rgKey => $rateGroup) {
					if(!isset($calcCompRateGroupTotals[$rgKey])) {
						$calcCompRateGroupTotals[$rgKey] = 0;
					}
					if(isset($rateGroup['comm_calc'])) {
						$calcCompRateGroupTotals[$rgKey] += $rateGroup['comm_calc'];
						$calcCompRateGroupTotal += $rateGroup['comm_calc'];
					}
				}				
			}				
			if($component['comm_ind'] == 2) {
				$deductCompKeys[] = $key;
				foreach($component['rate_groups'] as $rgKey => $rateGroup) {
					if(!isset($deductCompRateGroupTotals[$rgKey])) {
						$deductCompRateGroupTotals[$rgKey] = 0;
					}
					if(isset($rateGroup['comm_calc'])) {
						$deductCompRateGroupTotals[$rgKey] += $rateGroup['comm_calc'];
						$newComponentArray[$key]['rate_groups'][$rgKey]['comm'] += $rateGroup['comm_calc'];
					}
					
				}				
			}			
			if($component['comm_ind'] == 3) {
				$deductAllCompKey = $key;
				foreach($component['rate_groups'] as $rgKey => $rateGroup) {
					if(isset($rateGroup['comm_calc'])) {
						$newComponentArray[$key]['rate_groups'][$rgKey]['comm'] += $rateGroup['comm_calc'];
					}
				}
			}		
		}

		if(sizeof($calcCompRateGroupTotals) > 0) {
			if($deductAllCompKey !== "") {
				foreach($newComponentArray[$deductAllCompKey]['rate_groups'] as $rgKey => $rateGroup) {
					$newComponentArray[$deductAllCompKey]['rate_groups'][$rgKey]['comm'] += $calcCompRateGroupTotals[$rgKey];
				}
			} else {
				if(is_array($deductCompKeys) && !empty($deductCompKeys)) {
					$largestDeductComponentKey = "";
					$largestDeductComponentRateGroupKey = "";
					$largestDeductComponentRateGroupTotal = 0;
					$totalExtraComponentComm = 0;
					foreach($deductCompKeys as $deductCompKey) {
						foreach($newComponentArray[$deductCompKey]['rate_groups'] as $rgKey => $rateGroup) {
							if($deductCompRateGroupTotals[$rgKey] != 0) {

								$newComponentArray[$deductCompKey]['rate_groups'][$rgKey]['comm'] += round($calcCompRateGroupTotals[$rgKey] * ($rateGroup['comm'] / $deductCompRateGroupTotals[$rgKey]),2);
								if($newComponentArray[$deductCompKey]['rate_groups'][$rgKey]['comm'] > $largestDeductComponentRateGroupTotal || $largestDeductComponentRateGroupTotal == 0) {
									 $largestDeductComponentRateGroupTotal = $newComponentArray[$deductCompKey]['rate_groups'][$rgKey]['comm'];
									 $largestDeductComponentRateGroupKey = $rgKey;
									 $largestDeductComponentKey = $deductCompKey;
								}
								$totalExtraComponentComm += round($calcCompRateGroupTotals[$rgKey] * ($rateGroup['comm'] / $deductCompRateGroupTotals[$rgKey]),2);
							}
						}
					}
					
					// Cater for possiblity of 1 cent difference after apportioning commission
					if($calcCompRateGroupTotal != $totalExtraComponentComm) {
						$diff = $totalExtraComponentComm - $calcCompRateGroupTotal;
						// Remove it from first component's first rate group
						if(!isset($newComponentArray[$largestDeductComponentKey]['rate_groups'][$largestDeductComponentRateGroupKey]['comm'])) {
							$newComponentArray[$largestDeductComponentKey]['rate_groups'][$largestDeductComponentRateGroupKey]['comm'] = 0;
						}
						$newComponentArray[$largestDeductComponentKey]['rate_groups'][$largestDeductComponentRateGroupKey]['comm'] -= $diff;
					}					
					
				}
			}
		}

		//Update Nett and Payable amounts based on new commission amounts
		$compTaxPercSum = 0;
		foreach($newComponentArray as $key => $component) {
			$newComponentArray[$key]['tax_detail'] = array();
			foreach($component['rate_groups'] as $rgKey => $rateGroup) {
				$newComponentArray[$key]['rate_groups'][$rgKey]['gross'] = isset($newComponentArray[$key]['rate_groups'][$rgKey]['gross'])?$newComponentArray[$key]['rate_groups'][$rgKey]['gross']:"";

				$gross = $newComponentArray[$key]['rate_groups'][$rgKey]['gross'];

				$newComponentArray[$key]['rate_groups'][$rgKey]['nett'] = $gross - $newComponentArray[$key]['rate_groups'][$rgKey]['comm'];

				if($this->Deduct == "gross") {
					$newComponentArray[$key]['rate_groups'][$rgKey]['payable'] = $gross;
				} else {
					$newComponentArray[$key]['rate_groups'][$rgKey]['payable'] = $newComponentArray[$key]['rate_groups'][$rgKey]['nett'];
				}			
				$temp_array = $this->CalculateComponentTax($component, $newComponentArray[$key]['rate_groups'][$rgKey]['payable']);
				$rg_tax = 0;
				$compTaxPercSum = 0; // Reset at this level as only the last iteration is used
				foreach($temp_array[$component['rt_component_id']] as $tax_rate_id => $rg_taxAmount) {
					$rg_tax += $rg_taxAmount['amount'];
					$compTaxPercSum += $rg_taxAmount['perc'];
					// Component Tax Detail
					if(!isset($newComponentArray[$key]['tax_detail'][$tax_rate_id])) {
						$newComponentArray[$key]['tax_detail'][$tax_rate_id] = array();
						$newComponentArray[$key]['tax_detail'][$tax_rate_id]['amount'] = 0;
					}
					$newComponentArray[$key]['tax_detail'][$tax_rate_id]['amount'] += $rg_taxAmount['amount'];
					$newComponentArray[$key]['tax_detail'][$tax_rate_id]['perc'] = $rg_taxAmount['perc'];
				}
				$newComponentArray[$key]['rate_groups'][$rgKey]['tax'] = round($rg_tax,6);
			}
			// Update component "total tax percentage" field
			$newComponentArray[$key]['rf_tax_rate_perc'] = $compTaxPercSum;
		}
		
		//Sum up rate groups per component 
		foreach($newComponentArray as $key => $component) {
			$newComponentArray[$key]['rv_item_comp_amt_gross'] = 0;
			$newComponentArray[$key]['rv_item_comp_amt_comm'] = 0;
			$newComponentArray[$key]['rv_item_comp_amt_comm_calc'] = 0;
			$newComponentArray[$key]['rv_item_comp_amt_nett'] = 0;
			$newComponentArray[$key]['rv_item_comp_amt_payable'] = 0;
			$newComponentArray[$key]['rv_item_comp_amt_tax'] = 0;
			$newComponentArray[$key]['rv_item_comp_amt_sp_discount'] = 0;
			
			foreach($component['rate_groups'] as $rgKey => $rateGroup) {
				$newComponentArray[$key]['rv_item_comp_amt_gross'] += $rateGroup['gross'];
				$newComponentArray[$key]['rv_item_comp_amt_comm'] += $rateGroup['comm'];
				$newComponentArray[$key]['rv_item_comp_amt_comm_calc'] += $rateGroup['comm_calc'];
				$newComponentArray[$key]['rv_item_comp_amt_nett'] += $rateGroup['nett'];
				$newComponentArray[$key]['rv_item_comp_amt_payable'] += $rateGroup['payable'];
				$newComponentArray[$key]['rv_item_comp_amt_tax'] += $rateGroup['tax'];
				$newComponentArray[$key]['rv_item_comp_amt_sp_discount'] += $rateGroup['rv_item_comp_rate_grp_amt_sp_discount'];
			}
		}
		return ($newComponentArray);
	}

	function CalculateDailyTotals() {
		$output = array();

		foreach($this->Periods as $period) {
			$dayAmount = 0;
			foreach($period['components'] as $component) {
				foreach($component['rate_groups'] as $rate_group_id=>$rate_group) {
					if($period['ac_prd_days'] != 0) {
						$dayAmount += $rate_group['payable'] / $period['ac_prd_days'];
					}
				}
			}

			$item = array(
				"start"=>$period['rt_period_date_from'],
				"end"=>$period['rt_period_date_to'],
				"total"=>db_round($dayAmount,2)
			);
			if($this->PeriodSplitSingleDay) {
				$item['end'] = $period['rt_period_date_to_full'];
			}
			$output[] = $item;
		}
		return $output;
	}

	function CalculateRoundedAmount($amount, $direction, $scale)
	{
		switch($scale) {
		case 0;
			return $amount;
		case 1;
			$factor = 1;
			break;
		case 2:
			$factor = 10;
			break;
		case 3:
			$factor = 100;
			break;
		case 4:
			$factor = 1000;
			break;			
		}

		switch($direction) {
		case 0; // Default
			$amount = (int) round($amount, -($scale-1));
			break;
		case 1: // Up
			$amount = ((int) (ceil($amount/$factor))) * $factor;
			break;
		case 2: // Down
			$amount = ((int) ($amount/$factor)) * $factor;
			break;
		}

		return $amount;
	}

}
