<?php

require_once(__DIR__ . '/class.report.php');

class ReportParam {
	var $Name;
	var $Type;
	var $Arg;
	var $Report;
	var $Scripts;
	var $Profile;

	function __construct($name,$profile=true) {
		$this->Name = $name;
		$this->Type = "unknown";
		$this->Arg = false;
		$this->Scripts = array();
		$this->Profile = $profile;
	}

	function GetArg() {
		$this->Arg = $this->Report->ArgAlloc();
	}

	function Init() {
		if (isset($_POST['url']) && !empty($_POST['url'])) {
			$args = explode("+",$_POST['url']);
			$this->Value = isset($args[$this->Arg]) ? $args[$this->Arg] : null;
		} else {
			$this->Value = isset($_SERVER['argv'][$this->Arg]) ? $_SERVER['argv'][$this->Arg] : null;
		}
	}

	function RenderAll() {
		return $this->RenderStart() . $this->Render() . $this->RenderEnd();
	}

	function RenderStart() {
		return join("",file(__DIR__ . "/../../../public/html/rep_template_param_start.htm"));
	}

	function RenderEnd() {
		return join("",file(__DIR__ . "/../../../public/html/rep_template_param_end.htm"));
	}

	function Render() {
		return "";
	}

	function Overlib($overlib) {
		if(empty($overlib)) {
			return "";
		}

		if(is_string($overlib)) {
			$overlib = array('message'=>$overlib);
		}

		if(!is_array($overlib)) {
			return "";
		}

		$overlibParams = "";
		foreach($overlib as $key=>$item) {
			if($key == "message") {
				continue;
			}
			$overlibParams .= ", " . mb_strtoupper($key) . ", " . $item;
		}

		return "onmouseover=\"overlib('".t_encodeOverlib($overlib['message'])."'".$overlibParams.");\" onmouseout=\"nd();\"";
	}

	function AddScript($script) {
		$this->Scripts[] = $script;
	}

	function RenderScripts() {
		foreach($this->Scripts as $k=>$i) {
			if(!preg_match("/\\.js$/",$i)) {
				$this->Scripts[$k] .= ".js";
			}
		}
		return $this->Scripts;
	}

	function GetNames() {
		return array($this->Name);
	}

	function &GetParam($name) {
		if($this->Name == $name) {
			return $this;
		} else {
			$false = false;
			return $false;		// function needs to return a valid variable
		}
	}
}

class ReportParamEmpty extends ReportParam {
	function __construct() {
		parent::__construct("",false);
	}

	function GetArg() {
		return false;
	}

	function Init() {
		return;
	}

	function Render() {
		return join("",file(__DIR__ . "/../../../public/html/rep_template_filter_empty.htm"));
	}
}

class ReportParamHeading extends ReportParam {
	function __construct($label) {
		parent::__construct("",false);
		$this->Label = $label;
	}

	function GetArg() {
		return false;
	}

	function Init() {
		$this->Heading = array(
			'label'=>$this->Label
		);
		return;
	}

	function Render() {
		return showpage($this->Heading, join("", file(__DIR__ . "/../../../public/html/rep_template_filter_heading.htm")));
	}
}

class ReportParamList extends ReportParam {
	var $Params;

	function __construct($name,$params=array(),$profile=true) {
		parent::__construct($name,$profile);
		$this->Params = array();
		foreach($params as $param) {
			$this->AddParam($param);
		}
	}

	function AddParam($param) {
		$this->Params[] = $param;
	}

	function GetArg() {
		parent::GetArg();
		foreach($this->Params as $k=>$param) {
			$this->Params[$k]->Report =& $this->Report;
			$this->Params[$k]->GetArg();
		}
	}

	function Init() {
		parent::Init();
		foreach($this->Params as $k=>$param) {
			$this->Params[$k]->Init();
		}
	}

	function Render() {
		$html = "";
		foreach($this->Params as $k=>$param) {
			$html .= $this->Params[$k]->RenderAll();
		}
		return $html;
	}

	function RenderScripts() {
		$scripts = array();
		foreach($this->Params as $k=>$param) {
			$scripts = array_merge($scripts,$this->Params[$k]->RenderScripts());
		}
		return array_merge($this->Scripts,$scripts);
	}

	function GetNames() {
		$names = array($this->Name);
		foreach($this->Params as $param) {
			$names = array_merge($names,$param->GetNames());
		}
		return $names;
	}

	function &GetParam($name) {
		if($this->Name == $name) {
			return $this;
		} else {
			foreach($this->Params as $k=>$i) {
				$param =& $this->Params[$k]->GetParam($name);
				if($param !== false) {
					return $param;
				}
			}
			$false = false;
			return $false;		// function needs to return a valid variable
		}
	}

}

class ReportParamSection extends ReportParamList {
	var $Section;

	function __construct($name,$label,$params,$default=false,$overrideDef=array(),$profile=true) {
		parent::__construct($name,$params,$profile);
		$this->Type = "section";
		$this->Label = $label;
		$this->Default = $default;
		$this->Overrides = $overrideDef;
	}

	function Init() {
		parent::Init();

		if(!isset($this->Value)) {
			if($this->Default) {
				$this->Value = "show";
			} else {
				$this->Value = "hide";
			}
		}

		if($this->Value == "show") {
			$labelSuffix = "&lt;&lt;";
			$class = "";
		} else {
			$labelSuffix = "&gt;&gt;";
			$class = "collapse";
		}
		$this->Section = array_merge(array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'labelSuffix'=>$labelSuffix,
			'arg'=>$this->Arg,
			'default'=>$this->Default,
			'state'=>$this->Value,
			'class'=>$class
		),$this->Overrides);
		$this->AddScript("rep_template_filter_section.js");
	}

	function Render() {
		if(!isset($this->Section['filterLabelWidth'])) {
			$this->Section['filterLabelWidth'] = $this->Report->FilterWidths['label'];
		}
		if(!isset($this->Section['filterButtonWidth'])) {
			$this->Section['filterButtonWidth'] = $this->Report->FilterWidths['button'];
		}

		$html = showpage($this->Section,join("",file(__DIR__ . "/../../../public/html/rep_template_filter_section_start.htm")));
		$html .= parent::Render();
		$html .= join("",file(__DIR__ . "/../../../public/html/rep_template_filter_section_end.htm"));


		return $html;
	}
}

class ReportParamFilterCheckArea extends ReportParamList {
	var $Filter;

	function __construct($name, $label, $params, $default=false, $profile=true) {
		parent::__construct($name,$params,$profile);
		$this->Type = "filterCheckArea";
		$this->Label = $label;
		$this->Default = $default;
	}

	function Init() {
		parent::Init();

		if(!isset($this->Value)) {
			$this->Value = $this->Default;
		}

		$this->Filter = array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'checked'=>($this->Value == "1"?"checked":"")
		);
		$this->AddScript("rep_template_filter_check_area_start.js");
	}

	function Render() {
		$this->Filter['filterLabelWidth'] = $this->Report->FilterWidths['label'];
		$this->Filter['filterButtonWidth'] = $this->Report->FilterWidths['button'];
		$html = showpage($this->Filter,join("",file(__DIR__ . "/../../../public/html/rep_template_filter_check_area_start.htm")));
		$html .= parent::Render();
		$html .= join("",file(__DIR__ . "/../../../public/html/rep_template_filter_check_area_end.htm"));

		return $html;
	}

}

class ReportParamRender extends ReportParam {
	function __construct($name,$profile=false) {
		parent::__construct($name,$profile);
		$this->Type = "render";
	}

	function Init() {
		parent::Init();
		list($this->Mode,$this->Type) = decodeRenderOption($this->Value);
	}

	function RenderAll() {
		return str_replace("!arg!",$this->Arg,join("",file(__DIR__ . "/../../../public/html/rep_template_render.htm")));
	}
}

class ReportParamFilter extends ReportParam {
	var $Filter;
	var $Prefix;
	var $Alias;
	var $Default;
	var $Overrides;

	function __construct($prefix,$alias=false,$default=false,$overrideDef=array(),$profile=true) {
		if($alias === false) {
			$name = $prefix;
		} else {
			$name = $alias;
		}
		parent::__construct($name,$profile);
		$this->Type = "filter";
		$this->Prefix = $prefix;
		$this->Alias = $alias;
		$this->Default = $default;
		$this->Overrides = $overrideDef;
	}

	function Init() {
		global $repFilters;
		global $lDB;

		parent::Init();
		$filterDef = array_merge($repFilters[$this->Prefix],$this->Overrides);
		if(isset($filterDef['includeList']) && $filterDef['includeList'] !== false && !is_array($filterDef['includeList'])) {
			$filterDef['includeList'] = $lDB->get($filterDef['includeList'],3);
		}

		if(!isset($this->Value) && $this->Default !== false) {
			$this->Value = $this->Default;
		}
		$itemList = $this->Value;
		$filter = useFilter($this->Prefix,$itemList,$filterDef['selectName'],$filterDef['whereName'], $filterDef['includeList'], $filterDef['emptyLabel'], $filterDef['emptyWhere'], $filterDef['emptyNull']);
		$filter['name'] = $this->Prefix;
		$filter['arg'] = $this->Arg;
		if(!isset($filterDef['type'])) {
			$filter['type'] = "standard";
			$filter['filterFunction'] = "openFilter";
		} else if($filterDef['type'] == "accomm") {
			$filter['type'] = "accomm";
			$filter['filterFunction'] = "openAccommFilter";
		}
		if($filterDef['label'] === false) {
			$filter['label'] = $this->Prefix;
		} else {
			$filter['label'] = $filterDef['label'];
		}
		if($this->Alias === false) {
			$filter['filterName'] = $this->Prefix;
		} else {
			$filter['filterName'] = $this->Alias;
		}
		if(isset($filterDef['job'])) {
			$filter['job'] = $filterDef['job'];
		} else {
			$filter['job'] = "false";
		}
		if(isset($filterDef['transAfter'])) {
			$filter['transAfter'] = $filterDef['transAfter'];
		} else {
			$filter['transAfter'] = "false";
		}
		$filter['whereIds'] = "('" . join("','",explode(":",$filter['ids'])) . "')";
		$this->Filter = $filter;
        $this->FilterDef = $filterDef;
	}

	function SQL($template=false) {
		if($template === false) {
			return $this->Filter['where'];
		} else {
			if(!empty($this->Filter['ids'])) {
				return showpage($this->Filter,$template);
			} else {
				return "";
			}
		}
	}

	function Render() {
		$filter = "";
		$filterHTML = join("",file(__DIR__ . "/../../../public/html/rep_template_filter.htm"));
		switch($this->Filter['type']) {
		case "standard":
			$this->AddScript("rep_template_filter.js");
			$filter .= showpage($this->Filter,$filterHTML);
			break;
		case "accomm":
			$this->AddScript("rep_template_filter_accomm.js");
			$filter .= showpage($this->Filter,$filterHTML);
			break;
		}
		return $filter;
	}
}

class ReportParamFilterExtra extends ReportParamFilter {

	function __construct($prefix=false,$alias=false,$default=false,$overrideDef=array(),$profile=true) {
		if($prefix === false) {
			$prefix = "extra";
		}
		parent::__construct($prefix,$alias,$default,$overrideDef,$profile);
	}

	function Init() {
		parent::Init();

		$this->Filter['type'] = "extra";
		$this->Filter['filterFunction'] = "openExtraFilter";
	}

	function Render() {
		global $lDB;

		$filterHTML = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_extra.htm"));
		$filterRowHTML = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_extra_rows.htm"));
		$this->AddScript("rep_template_filter_extra.js");

		$extras = $lDB->get("
			SELECT
				ac_extra.ac_extra_ix,
				ac_extra.ac_ext_desc,
				ac_extra_category.ac_extra_category_ix,
				ac_extra_category.ac_extra_cat_desc
			FROM
				ac_extra
				INNER JOIN ac_extra_category ON ac_extra_category.ac_extra_category_ix = ac_extra.ac_extra_category_id
			WHERE
				ac_extra.ac_extra_ix IN ('".join("','",explode(":",$this->Value))."')
			ORDER BY
				ac_extra_category.ac_extra_cat_desc,
				ac_extra.ac_ext_desc
		",2);
		$categories = array();
		foreach($extras as $extra) {
			if(!array_key_exists($extra['ac_extra_category_ix'],$categories)) {
				$categories[$extra['ac_extra_category_ix']] = array(
					'ac_extra_category_id'=>$extra['ac_extra_category_ix'],
					'ac_extra_cat_desc'=>$extra['ac_extra_cat_desc'],
					'extras'=>array()
				);
			}
			$categories[$extra['ac_extra_category_ix']]['extras'][$extra['ac_extra_ix']] = array(
				'ac_extra_id'=>$extra['ac_extra_ix'],
				'ac_ext_desc'=>$extra['ac_ext_desc']
			);
		}

		$filterRows = "";
		foreach($categories as $category) {
			$item = array(
				'filterName'=>$this->Filter['filterName'],
				'ac_extra_category_id'=>$category['ac_extra_category_id'],
				'ac_extra_cat_desc'=>$category['ac_extra_cat_desc']
			);
			$extras = array();
			$extraIds = array();
			foreach($category['extras'] as $extra) {
				$extras[] = $extra['ac_ext_desc'];
				$extraIds[] = $extra['ac_extra_id'];
			}
			$item['ids'] = join(":",$extraIds);
			$item['extras'] = join(",",$extras);
			$filterRows .= showpage($item,$filterRowHTML);
		}
		$this->Filter['filterRows'] = $filterRows;

		if(sizeof($categories) > 0) {
			$this->Filter['allClass'] = "collapse printCollapse";
		} else {
			$this->Filter['allClass'] = "expand";
		}

		$filter = showpage($this->Filter,$filterHTML);
		return $filter;
	}
}

class ReportParamFilterDateRange extends ReportParam {
	var $Filter;

	function __construct($name,$label,$default=false,$fields=array('from'=>"rv_reservation.rv_date_arrive",'to'=>"rv_reservation.rv_date_depart"),$profile=false) {
		parent::__construct($name,$profile);
		$this->Type = "filterDateRange";
		$this->Label = $label;
		$this->Fields = $fields;
		$this->Default = $default;
	}

	function Init() {
		parent::Init();

		if(!isset($this->Value) && $this->Default !== false) {
			$this->Value = $this->Default;
		}

		$name = isset($name) ? $name : "";
		$dates = useDateRange($name,$this->Value);
		$this->Filter = array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'fromDate'=>$dates['from']['dateArray'][0] . "-" . $dates['from']['dateArray'][1] . "-" . $dates['from']['dateArray'][2],
			'fromY'=>$dates['from']['dateArray'][0],
			'fromM'=>$dates['from']['dateArray'][1],
			'fromD'=>$dates['from']['dateArray'][2],
			'toDate'=>$dates['to']['dateArray'][0] . "-" . $dates['to']['dateArray'][1] . "-" . $dates['to']['dateArray'][2],
			'toY'=>$dates['to']['dateArray'][0],
			'toM'=>$dates['to']['dateArray'][1],
			'toD'=>$dates['to']['dateArray'][2]
		);
		$this->SQL = "
			AND (
				(
					".$this->Fields['from']." >= '".$this->Filter['fromDate']."'
					AND ".$this->Fields['from']." <= '".$this->Filter['toDate']."'
				)
				OR
				(
					".$this->Fields['to']." >= '".$this->Filter['fromDate']."'
					AND ".$this->Fields['to']." <= '".$this->Filter['toDate']."'
				)
				OR
				(
					".$this->Fields['from']." < '".$this->Filter['fromDate']."'
					AND ".$this->Fields['to']." > '".$this->Filter['toDate']."'
				)
			)
		";
	}

	function SQL($template=false) {
		if($template === false) {
			$template = "
				AND (
					(
						rv_reservation.rv_date_arrive >= '!fromDate!'
						AND rv_reservation.rv_date_arrive <= '!toDate!'
					) OR (
						rv_reservation.rv_date_depart >= '!fromDate!'
						AND rv_reservation.rv_date_depart <= '!toDate!'
					) OR (
						rv_reservation.rv_date_arrive < '!fromDate!'
						AND rv_reservation.rv_date_depart > '!toDate!'
					)
				)
			";
		}
		return showpage($this->Filter,$template);
	}

	function Render() {
		$this->AddScript("rep_template_filter_date_range.js");
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_date_range.htm"));
		if($this->Filter['label'] !== false) {
			$html = "<td>!label!</td>\n".$html;
			$this->Filter['colspan'] = "2";
		} else {
			$this->Filter['colspan'] = "3";
		}
		return showpage($this->Filter,$html);
	}
}

class ReportParamFilterDateRangeStatus extends ReportParam {
	var $Filter;

	function __construct($name,$label,$default=false,$profile=false) {
		parent::__construct($name,$profile);
		$this->Type = "filterDateRangeStatus";
		$this->Label = $label;
		$this->Default = $default;
	}

	function Init() {
		parent::Init();

		if(!isset($this->Value) && $this->Default !== false) {
			$this->Value = $this->Default;
		}

		list($statusType, $fromDate, $toDate) = array_pad( explode(":",$this->Value), 3, '' );

		$dates = useDateRange($this->Name,$fromDate . ":" . $toDate);
		$this->Filter = array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'statusType'=>$statusType,
			'fromDate'=>$dates['from']['dateArray'][0] . "-" . $dates['from']['dateArray'][1] . "-" . $dates['from']['dateArray'][2],
			'fromY'=>$dates['from']['dateArray'][0],
			'fromM'=>$dates['from']['dateArray'][1],
			'fromD'=>$dates['from']['dateArray'][2],
			'toDate'=>$dates['to']['dateArray'][0] . "-" . $dates['to']['dateArray'][1] . "-" . $dates['to']['dateArray'][2],
			'toY'=>$dates['to']['dateArray'][0],
			'toM'=>$dates['to']['dateArray'][1],
			'toD'=>$dates['to']['dateArray'][2]
		);
	}

	function Render() {
		$this->AddScript("rep_template_filter_date_range.js");
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_date_range_status.htm"));
		if($this->Filter['label'] !== false) {
			$html = "<td>!label!</td>\n".$html;
			$this->Filter['colspan'] = "2";
		} else {
			$this->Filter['colspan'] = "3";
		}

		$this->Filter['statusDateType'] = "";
		$this->Filter['statusDateAllChecked'] = "";
		$this->Filter['statusDateCancelChecked'] = "";
		$this->Filter['statusDateConfirmChecked'] = "";
		$this->Filter['statusDateDeleteChecked'] = "";
		$this->Filter['statusDateProvisionalChecked'] = "";
		$this->Filter['statusDateQuoteChecked'] = "";
		$this->Filter['statusDateWaitlistChecked'] = "";

		switch($this->Filter['statusType']) {
		case "all":
		default:
			$this->Filter['statusDateType'] = "All status date changes";
			$this->Filter['statusDateAllChecked'] = "selected";
			break;
		case "cancel":
			$this->Filter['statusDateType'] = "Reservation cancellation date";
			$this->Filter['statusDateCancelChecked'] = "selected";
			break;
		case "confirm":
			$this->Filter['statusDateType'] = "Reservation confirmation date";
			$this->Filter['statusDateConfirmChecked'] = "selected";
			break;
		case "delete":
			$this->Filter['statusDateType'] = "Reservation deleted date";
			$this->Filter['statusDateDeleteChecked'] = "selected";
			break;
		case "provisional":
			$this->Filter['statusDateType'] = "Reservation provisional date";
			$this->Filter['statusDateProvisionalChecked'] = "selected";
			break;
		case "quote":
			$this->Filter['statusDateType'] = "Reservation quotation date";
			$this->Filter['statusDateQuoteChecked'] = "selected";
			break;
		case "waitlist":
			$this->Filter['statusDateType'] = "Reservation waitlist date";
			$this->Filter['statusDateWaitlistChecked'] = "selected";
			break;
		}
		return showpage($this->Filter,$html);
	}

	function SQL($template=false) {
		if($template === false) {
			switch($this->Filter['statusType']) {
			case "all":
				$template = "
					AND (
						(
							rv_reservation.rv_cancel_date >= '!fromDate!'
							AND rv_reservation.rv_cancel_date <= '!toDate!'
						) OR (
							rv_reservation.rv_confirmation_date >= '!fromDate!'
							AND rv_reservation.rv_confirmation_date <= '!toDate!'
						) OR (
							rv_reservation.rv_date_deleted >= '!fromDate!'
							AND rv_reservation.rv_date_deleted <= '!toDate!'
						) OR (
							rv_reservation.rv_prov_date >= '!fromDate!'
							AND rv_reservation.rv_prov_date <= '!toDate!'
						) OR (
							rv_reservation.rv_date_quote >= '!fromDate!'
							AND rv_reservation.rv_date_quote <= '!toDate!'
						) OR (
							rv_reservation.rv_wait_list_date >= '!fromDate! 00:00:00'
							AND rv_reservation.rv_wait_list_date <= '!toDate! 23:59:59'
						)
					)
				";
				break;
			case "cancel":
				$template = "
					AND rv_reservation.rv_cancel_date >= '!fromDate!'
					AND rv_reservation.rv_cancel_date <= '!toDate!'
				";
				break;
			case "confirm":
				$template = "
					AND rv_reservation.rv_confirmation_date >= '!fromDate!'
					AND rv_reservation.rv_confirmation_date <= '!toDate!'
				";
				break;
			case "delete":
				$template = "
					AND rv_reservation.rv_date_deleted >= '!fromDate!'
					AND rv_reservation.rv_date_deleted <= '!toDate!'
				";
				break;
			case "provisional":
				$template = "
					AND rv_reservation.rv_prov_date >= '!fromDate!'
					AND rv_reservation.rv_prov_date <= '!toDate!'
				";
				break;
			case "quote":
				$template = "
					AND rv_reservation.rv_date_quote >= '!fromDate!'
					AND rv_reservation.rv_date_quote <= '!toDate!'
				";
				break;
			case "waitlist":
				$template = "
					AND rv_reservation.rv_wait_list_date >= '!fromDate! 00:00:00'
					AND rv_reservation.rv_wait_list_date <= '!toDate! 23:59:59'
				";
				break;
			}
		}
		return showpage($this->Filter,$template);
	}

}

class ReportParamFilterDateRangeTravel extends ReportParam {
	var $Filter;

	function __construct($name,$label,$default=false,$profile=false) {
		parent::__construct($name,$profile);
		$this->Type = "filterDateRangeTravel";
		$this->Label = $label;
		$this->Default = $default;
	}

	function Init() {
		parent::Init();

		if(!isset($this->Value) && $this->Default !== false) {
			$this->Value = $this->Default;
		}

		if ($this->Value != "") {
			list($travelType, $fromDate, $toDate) = explode(":",$this->Value);
		} else {
			$travelType = "";
			$fromDate = "";
			$toDate = "";
		}
		$dates = useDateRange($this->Name,$fromDate . ":" . $toDate);
		$this->Filter = array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'travelType'=>$travelType,
			'fromDate'=>$dates['from']['dateArray'][0] . "-" . $dates['from']['dateArray'][1] . "-" . $dates['from']['dateArray'][2],
			'fromY'=>$dates['from']['dateArray'][0],
			'fromM'=>$dates['from']['dateArray'][1],
			'fromD'=>$dates['from']['dateArray'][2],
			'toDate'=>$dates['to']['dateArray'][0] . "-" . $dates['to']['dateArray'][1] . "-" . $dates['to']['dateArray'][2],
			'toY'=>$dates['to']['dateArray'][0],
			'toM'=>$dates['to']['dateArray'][1],
			'toD'=>$dates['to']['dateArray'][2]
		);
	}

	function Render() {
		$this->AddScript("rep_template_filter_date_range.js");
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_date_range_travel.htm"));
		if($this->Filter['label'] !== false) {
			$html = "<td>!label!</td>\n".$html;
			$this->Filter['colspan'] = "2";
		} else {
			$this->Filter['colspan'] = "3";
		}

		$this->Filter['travelDateType'] = "";
		$this->Filter['travelDateAllChecked'] = "";
		$this->Filter['travelDateArriveChecked'] = "";
		$this->Filter['travelDateDepartChecked'] = "";
		$this->Filter['travelDateInhouseChecked'] = "";

		switch($this->Filter['travelType']) {
		case "all";
		default:
			$this->Filter['travelDateType'] = "All";
			$this->Filter['travelDateAllChecked'] = "selected";
			break;
		case "arrive":
			$this->Filter['travelDateType'] = "Arrival";
			$this->Filter['travelDateArriveChecked'] = "selected";
			break;
		case "depart":
			$this->Filter['travelDateType'] = "Departure";
			$this->Filter['travelDateDepartChecked'] = "selected";
			break;
		case "inhouse":
			$this->Filter['travelDateType'] = "In house";
			$this->Filter['travelDateInhouseChecked'] = "selected";
			break;
		}
		return showpage($this->Filter,$html);
	}

	function SQL($template=false) {
		if($template === false) {
			switch($this->Filter['travelType']) {
			case "all":
				$template = "
					AND (
						(
							rv_reservation.rv_date_arrive >= '!fromDate!'
							AND rv_reservation.rv_date_arrive <= '!toDate!'
						) OR (
							rv_reservation.rv_date_depart >= '!fromDate!'
							AND rv_reservation.rv_date_depart <= '!toDate!'
						) OR (
							rv_reservation.rv_date_arrive < '!fromDate!'
							AND rv_reservation.rv_date_depart > '!toDate!'
						)
					)
				";
				break;
			case "arrive":
				$template = "
					AND rv_reservation.rv_date_arrive >= '!fromDate!'
					AND rv_reservation.rv_date_arrive <= '!toDate!'
				";
				break;
			case "depart":
				$template = "
					AND rv_reservation.rv_date_depart >= '!fromDate!'
					AND rv_reservation.rv_date_depart <= '!toDate!'
				";
				break;
			case "inhouse":
				$template = "
					AND (
						(
							rv_reservation.rv_date_arrive >= '!fromDate!'
							AND rv_reservation.rv_date_arrive <= '!toDate!'
						) OR (
							rv_reservation.rv_date_depart >= '!fromDate!'
							AND rv_reservation.rv_date_depart < '!toDate!'
						) OR (
							rv_reservation.rv_date_arrive < '!fromDate!'
							AND rv_reservation.rv_date_depart > '!toDate!'
						)
					)
				";
				break;
			}
		}
		return showpage($this->Filter,$template);
	}

}

class ReportParamFilterDateRangeRadio extends ReportParam {
	var $Filter;

	function __construct($name,$label,$default=false,$fields=array('from'=>"rv_reservation.rv_date_arrive",'to'=>"rv_reservation.rv_date_depart"),$overrideDef=array(),$profile=true) {
		parent::__construct($name,$profile);
		$this->Type = "filterDateRangeRadio";
		$this->Label = $label;
		$this->Fields = $fields;
		$this->Default = $default;
		$this->Overrides = is_array($overrideDef)?$overrideDef:array();
	}

	function Init() {
		parent::Init();

		if(!isset($this->Value) && $this->Default !== false) {
			$this->Value = $this->Default;
		}

		if ($this->Value != "") {
			list($dateRangeType, $fromDate, $toDate) = explode(":",$this->Value);
		} else {
			$dateRangeType = "today";
			$fromDate = date("Y-m-d");
			$toDate = date("Y-m-d");
			$this->Value = $dateRangeType . ":" . $fromDate . ":" . $toDate;
		}
		$dates = useDateRange($this->Name,$fromDate . ":" . $toDate);
		$this->Filter = array_merge(array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'value'=>$this->Value,
			'dateRangeType'=>$dateRangeType,
			'today'=>date("d M Y"),
			'fromDate'=>$dates['from']['dateArray'][0] . "-" . $dates['from']['dateArray'][1] . "-" . $dates['from']['dateArray'][2],
			'fromY'=>$dates['from']['dateArray'][0],
			'fromM'=>$dates['from']['dateArray'][1],
			'fromD'=>$dates['from']['dateArray'][2],
			'toDate'=>$dates['to']['dateArray'][0] . "-" . $dates['to']['dateArray'][1] . "-" . $dates['to']['dateArray'][2],
			'toY'=>$dates['to']['dateArray'][0],
			'toM'=>$dates['to']['dateArray'][1],
			'toD'=>$dates['to']['dateArray'][2]
		),$this->Overrides);
		$this->SQL = "
			AND (
				(
					".$this->Fields['from']." >= '".$this->Filter['fromDate']."'
					AND ".$this->Fields['from']." <= '".$this->Filter['toDate']."'
				)
				OR
				(
					".$this->Fields['to']." >= '".$this->Filter['fromDate']."'
					AND ".$this->Fields['to']." <= '".$this->Filter['toDate']."'
				)
				OR
				(
					".$this->Fields['from']." < '".$this->Filter['fromDate']."'
					AND ".$this->Fields['to']." > '".$this->Filter['toDate']."'
				)
			)
		";
	}

	function SQL($template=false) {
		if($template === false) {
			$template = "
				AND (
					(
						rv_reservation.rv_date_arrive >= '!fromDate!'
						AND rv_reservation.rv_date_arrive <= '!toDate!'
					) OR (
						rv_reservation.rv_date_depart >= '!fromDate!'
						AND rv_reservation.rv_date_depart <= '!toDate!'
					) OR (
						rv_reservation.rv_date_arrive < '!fromDate!'
						AND rv_reservation.rv_date_depart > '!toDate!'
					)
				)
			";
		}
		return showpage($this->Filter,$template);
	}

	function Render() {
		$this->AddScript("rep_template_filter_date_range.js");
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_date_range_radio.htm"));
		if($this->Filter['label'] !== false) {
			$html = "<td>!label!</td>\n".$html;
			$this->Filter['colspan'] = "2";
		} else {
			$this->Filter['colspan'] = "3";
		}

		$this->Filter['dateRangeTypeAllChecked'] = "";
		$this->Filter['dateRangeTypeTodayChecked'] = "";
		$this->Filter['dateRangeTypeCustomChecked'] = "";

		switch($this->Filter['dateRangeType']) {
		case "all";
		default:
			$this->Filter['dateType'] = "All";
			$this->Filter['dateRangeTypeAllChecked'] = "checked";
			break;
		case "today":
			$this->Filter['dateType'] = "Today";
			$this->Filter['dateRangeTypeTodayChecked'] = "checked";
			break;
		case "custom":
			$this->Filter['dateType'] = "Custom";
			$this->Filter['dateRangeTypeCustomChecked'] = "checked";
			break;
		}

		return showpage($this->Filter,$html);
	}
}

class ReportParamFilterCheck extends ReportParam {
	var $Filter;

	function __construct($name, $label, $default=false, $overrideDef=array(), $profile=true) {
		parent::__construct($name,$profile);
		$this->Type = "filterCheck";
		$this->Label = $label;
		$this->Default = $default;
		$this->Overrides = is_array($overrideDef)?$overrideDef:array();
	}

	function Init() {
		parent::Init();
		if(!isset($this->Value)) {
			$this->Value = $this->Default;
		}
		$this->Filter = array_merge(array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'checked'=>($this->Value == "1"?"checked":""),
			'onclick'=>""
		),$this->Overrides);
	}

	function Render() {
		$attr = array();
		if(isset($this->Filter['onclick']) && !empty($this->Filter['onclick'])) {
			$attr[] = "onclick=\"" . $this->Filter['onclick'] . "\"";
		}

		$this->Filter['attr'] = join(" ",$attr);

		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_check.htm"));
		$this->Filter['colspan'] = "3";
		return showpage($this->Filter,$html);
	}
}

class ReportParamFilterCheckDiffersBy extends ReportParam {
	var $Filter;

	function __construct($name, $label, $default=false, $profile=true) {
		parent::__construct($name,$profile);
		$this->Type = "filterCheckDiffersBy";
		$this->Label = $label;
		$this->Default = $default;
	}

	function Init() {
		parent::Init();

		if(!isset($this->Value) && $this->Default !== false) {
			$this->Value = $this->Default;
		}

		if ($this->Value != "" && count(explode(":",$this->Value)) == 3) {
			list($checked, $diffType, $diffAmount) = explode(":",$this->Value);
		} else {
			$checked = "";
			$diffType = "";
			$diffAmount = "";
		}
		if(empty($diffType)) {
			$diffType = "more";
		}
		if(empty($diffAmount)) {
			$diffAmount = "0";
		}
		$this->Filter = array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'checked'=>($checked == "1"?"checked":""),
			'diffType'=>$diffType,
			'diffAmount'=>$diffAmount
		);
		$this->AddScript("rep_template_filter_check_differs_by.js");
	}

	function Render() {
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_check_differs_by.htm"));
		$this->Filter['colspan'] = "3";
		$this->Filter['diffTypeMore'] = "";
		$this->Filter['diffTypeLess'] = "";
		if($this->Filter['diffType'] == "more") {
			$this->Filter['diffTypeMore'] = "selected";
		} else {
			$this->Filter['diffTypeLess'] = "selected";
		}
		return showpage($this->Filter,$html);
	}
}

class ReportParamFilterRadioGroup extends ReportParam {
	var $Filter;
	var $Options;
	var $Width;
	var $Default;

	function __construct($name, $label, $options=array(), $default=false, $overrideDef=array(), $profile=true) {
		parent::__construct($name,$profile);
		$this->Type = "filterSelect";
		$this->Label = $label;
		$this->Options = $options;
		$this->Default = $default;
		$this->Overrides = is_array($overrideDef)?$overrideDef:array();
	}

	function Init() {
		parent::Init();
		if($this->Default !== false && empty($this->Value)) {
			$this->Value = $this->Default;
		}
		$this->Filter = array_merge(array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'value'=>$this->Value,
			'overlib'=>""
		),$this->Overrides);
	}

	function Render() {
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_radio_group.htm"));
		$optionHTML = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_radio_group_option.htm"));
		$options = "";
		foreach($this->Options as $value=>$desc) {
			$selected = "";
			if($value == $this->Value) {
				$selected = "checked";
			}
			$options .= showpage(array('value'=>$value, 'desc'=>$desc, 'selected'=>$selected, 'name'=>$this->Name),$optionHTML);
		}
		$this->Filter['overlib'] = $this->Overlib($this->Filter['overlib']);
		$this->Filter['options'] = $options;
		$this->Filter['colspan'] = "2";
		return showpage($this->Filter,$html);
	}
}

class ReportParamFilterSelect extends ReportParam {
	var $Filter;
	var $Options;
	var $Width;
	var $Default;

	function __construct($name, $label, $options=array(), $default=false, $overrideDef=array(), $profile=true) {
		parent::__construct($name,$profile);
		$this->Type = "filterSelect";
		$this->Label = $label;
		$this->Options = $options;
		$this->Default = $default;
		$this->Overrides = is_array($overrideDef)?$overrideDef:array();
	}

	function Init() {
		parent::Init();
		if($this->Default !== false && empty($this->Value)) {
			$this->Value = $this->Default;
		}
		$this->Filter = array_merge(array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'value'=>$this->Value,
			'overlib'=>""
		),$this->Overrides);
	}

	function Render() {
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_select.htm"));
		$optionHTML = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_select_option.htm"));
		$options = "";
		foreach($this->Options as $value=>$desc) {
			$selected = "";
			if($value == $this->Value) {
				$selected = "selected";
			}
			$options .= showpage(array('value'=>$value, 'desc'=>$desc, 'selected'=>$selected),$optionHTML);
		}
		$this->Filter['overlib'] = $this->Overlib($this->Filter['overlib']);
		$this->Filter['options'] = $options;
		$this->Filter['colspan'] = "2";
		return showpage($this->Filter,$html);
	}
}

class ReportParamFilterLookup extends ReportParam {
	var $Filter;
	var $Overrides;
	var $Width;
	var $Default;
	var $Type;
	var $Label;

	function __construct($name, $label, $formId, $default=false, $overrideDef=array(), $profile=true) {
		parent::__construct($name,$profile);
		$this->Type = "filterLookup";
		$this->Label = $label;
		$this->FormId = $formId;
		$this->Default = $default;
		$this->Overrides = is_array($overrideDef)?$overrideDef:array();
	}

	function Init() {
		parent::Init();
		$this->AddScript("rep_template_builder.js");
		$this->AddScript("ajax.js");
		$this->AddScript("rep_template_filter_lookup.js");
		if($this->Default !== false && empty($this->Value)) {
			$this->Value = $this->Default;
		}
		$this->Filter = array_merge(array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'value'=>str_replace("\\r\\n", "\r\n",$this->Value),
			'formId'=>$this->FormId,
			'overlib'=>"",
			'javascript'=>""
		),$this->Overrides);
	}

	function Render() {
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_lookup.htm"));
		$this->Filter['overlib'] = $this->Overlib($this->Filter['overlib']);
		$this->Filter['colspan'] = "2";
		$this->Filter['placeholderText'] = $this->Overrides['placeholderText'];
		return showpage($this->Filter,$html);
	}

	public static function getLookupElement($form, $items) {
		if(!isset($GLOBALS[$form])) {
			$GLOBALS[$form] = new Form($form, "", "");
		}

		$lookupHTML = Join("",file(__DIR__ . "/../../../public/html/rep_template_filter_lookup_item.htm"));
		$lookupRowHTML = Join("",file(__DIR__ . "/../../../public/html/rep_template_filter_lookup_item_rows.htm"));
		$colWidth = 290;
		$columnNames = (is_array($items) && count($items) > 0) ? array_keys($items[0]) : array();
		$columnNames = (!empty($columnNames)) ? $columnNames : array("id", "");
		$columnHeadings = "";
		foreach ($columnNames as $columnName) {
			if ($columnName != "id") {
				$columnHeadings .= "<td class='outLt' style='width: 100%;'>$columnName</td>";
			}
		}
		$itemFilters = array(
			'outerRateWidth'=>$colWidth,
			'innerRateWidth'=>$colWidth-20,
			'columnHeadings'=>$columnHeadings
		);
		$lookupHTML = showpage($itemFilters,$lookupHTML);
		$lookupRowHTML = showpage($itemFilters,$lookupRowHTML);

		$rows = "";
		$rateList = array();
		$maxLen = 75;
		$items = (!empty($items)) ? $items : array(array('id'=>"", ''=>"No results found"));
		foreach($items as $item) {
			foreach ($columnNames as $columnName) {
				$value = str_replace("\r\n", "\\r\\n",$item[$columnName]);
				if ($columnName != "id") {
					$filter['rowClass'] = "txtg";
					$rowDescription = substr($value, 0, $maxLen - 3);
					if (strlen($value) > $maxLen) {
						$rowDescription .= '...';
					}
					$filter['rowDescription'] = str_replace("\\r\\n", "" ,$rowDescription);

					if ($value != "No results found" && strlen($value) > $maxLen) {
						$filter['rowIcon'] = "<img src=\"/resource/Resrequest/Application/public/img/button_note.gif\" onmouseout=\"nd()\" onmouseover=\"overlib('" . t_encodeOverlib($value) . "',LEFT,WIDTH,400)\" />";
					} else {
						$filter['rowIcon'] = "";
					}
					$rows .= showpage($filter,$lookupRowHTML);
				} else {
					// This value is set to a javascript variable, which doesn't support
					// multiline string literals, so escape appropriately
					$filter['rowValue'] = $value;
				}
			}
		}
		$output = str_replace("!rows!",$rows,$lookupHTML);

		echo $output;
	}
}

class ReportParamFilterText extends ReportParam {
	var $Filter;
	var $Width;
	var $Default;

	function __construct($name, $label, $default=false, $overrideDef=array(), $profile=true) {
		parent::__construct($name,$profile);
		$this->Type = "filterText";
		$this->Label = $label;
		$this->Default = $default;
		$this->Overrides = is_array($overrideDef)?$overrideDef:array();
	}

	function Init() {
		parent::Init();
		if($this->Default !== false && empty($this->Value)) {
			$this->Value = $this->Default;
		}
		$this->Filter = array_merge(array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'value'=>$this->Value,
			'width'=>"100px",
			'overlib'=>""
		),$this->Overrides);
	}

	function Render() {
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_text.htm"));
		$this->Filter['overlib'] = $this->Overlib($this->Filter['overlib']);
		$this->Filter['placeholderText'] = $this->Overrides['placeholderText'];
		$this->Filter['colspan'] = "2";
		$this->Filter['value'] = $this->Value;
		return showpage($this->Filter,$html);
	}
}

class ReportParamFilterNumber extends ReportParam {
	var $Filter;

	function __construct($name, $label, $default=false, $overrideDef=array(), $profile=true) {
		parent::__construct($name, $profile);
		$this->Type = "filterNumber";
		$this->Label = $label;
		$this->Default = $default;
		$this->Overrides = is_array($overrideDef)?$overrideDef:array();
	}

	function Init() {
		parent::Init();

		if(!isset($this->Value) && $this->Default !== false) {
			$this->Value = $this->Default;
		}

		if ($this->Value != "" && count(explode(":",$this->Value)) == 2) {
			list($number, $operator) = explode(":",$this->Value);
		} elseif($this->Value != "") {
			$number = $this->Value;
			$operator = "";
		} else {
			$number = "";
			$operator = "";
		}
		if(empty($operator)) {
			$operator = ">";
		}
		$this->Filter = array(
			'name'=>$this->Name,
			'label'=>$this->Label,
			'arg'=>$this->Arg,
			'width'=>"100px",
			'overlib'=>"",
			'number'=>$number,
			'operator'=>$operator
		);
	}

	function Render() {
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_filter_number.htm"));
		$this->Filter['overlib'] = $this->Overlib($this->Filter['overlib']);
		$this->Filter['placeholderText'] = $this->Overrides['placeholderText'];
		$this->Filter['colspan'] = "2";
		$this->Filter['operatorLess'] = "";
		$this->Filter['operatorEqual'] = "";
		$this->Filter['operatorMore'] = "";
		switch($this->Filter['operator']) {
		case "<":
			$this->Filter['operatorLess'] = "selected";
			break;
		case "<=":
			$this->Filter['operatorLessEqual'] = "selected";
			break;
		case "=":
			$this->Filter['operatorEqual'] = "selected";
			break;
		case ">=":
			$this->Filter['operatorMoreEqual'] = "selected";
			break;
		case ">":
			$this->Filter['operatorMore'] = "selected";
			break;
		}
		return showpage($this->Filter,$html);
	}
}

class ReportParamFilterServer extends ReportParamFilter {

    function __construct($prefix=false,$alias=false,$default=false,$overrideDef=array(),$profile=true) {
        if($prefix === false) {
            $prefix = "server";
        }
        parent::__construct($prefix,$alias,$default,$overrideDef,$profile);
    }

    function Init() {
        global $lDB;
        parent::Init();

        if(empty($this->Filter['ids'])) {
            return;
        }

        // Replace 'Res' with reservation environments
        $envs = array_unique(explode(":", $this->Filter['ids']));
        if (in_array("Res", $envs)) {
            $newEnvs = array();
            foreach ($envs as $environment) {
                switch($environment) {
                case "Res":
                    // Res, replace with all reservation environments
                    $resEnviroments = $lDB->get("SELECT rf_db_code FROM rf_database WHERE rf_db_env_type_ind = '2'",3);
                    foreach ($resEnviroments as $environment) {
                        $newEnvs[] = $environment;
                    }
                    break;
                case "":
                    // Empty, ignore
                    break;
                default:
                    // Anything other than Res, add as is
                    $newEnvs[] = $environment;
                    break;
                }
            }
            $this->Filter['where'] = " AND " . $this->FilterDef['whereName'] . " IN ('" . join("','", $newEnvs) . "')";
            $this->Filter['whereIds'] = "('" . join("','", $newEnvs) . "')";
        }
    }
}

class ReportParamBuilder extends ReportParam {
	function __construct($name, $label=false, $items, $cols=false, $default=false,$overrideDef=array(), $profile=true) {
		parent::__construct($name,$profile);
		$this->Type = "builder";
		$this->Label = $label;
		$this->Items = array();
		foreach($items as $item) {
			$this->Items[$item['name']] = $item;
		}
		if($cols === false) {
			$cols = array(
				array('name'=>"label",'label'=>"Label",'type'=>"text",'width'=>"110",'overlib'=>"The ResRequest database field name can be changed for the report display for example 'Reservation number' can be changed to show as 'No.'"),
				array('name'=>"width",'label'=>"Width",'type'=>"text",'width'=>"35",'overlib'=>"This specifies the width allocated to the column."),
				array('name'=>"sort_order",'label'=>"Sort",'type'=>"sort",'width'=>"40",'overlib'=>"This specifies the sorting order of this column."),
				array('name'=>"sort_direction",'label'=>"A / D",'type'=>"select",'width'=>"40",'overlib'=>"This specifies if the sort is ascending or descending for this column.",'items'=>array('a'=>"Ascending",'d'=>"Descending")),
				array('name'=>"header",'label'=>"Group Header",'type'=>"text",'width'=>"75",'overlib'=>"This is an optional field. Records sharing the same group header are grouped under the shared 'group header' column. For example: when the Group header is: 'PAX details' for fields Pax, Adults and Child, then the results columns for Pax, Adults and Child will be grouped under a main heading: 'PAX details'"),
				array('name'=>"align",'label'=>"Align",'type'=>"select",'width'=>"73",'overlib'=>"Align Determines the vertical orientation of the field i.e.: left / centre / right",'items'=>array('l'=>"Left",'c'=>"Center",'r'=>"Right"))
			);
		}
		$this->Cols = array();
		foreach($cols as $col) {
			$this->Cols[$col['name']] = $col;
		}
		$this->Default = $default;
		$this->Overrides = $overrideDef;
	}

	function Init() {
		parent::Init();
		$this->AddScript("rep_template_builder.js");
		$this->Value = unserialize(stripslashes($this->Value));

		$this->Filter = array(
			'availableWidth'=>"100",
			'buttonWidth'=>"32",
			'activeWidth'=>"475"
		);
		$this->Filter = array_merge($this->Filter,$this->Overrides);
		foreach($this->Items as $k=>$i) {
			$item =& $this->Items[$k];
			if(!isset($item['name']) || empty($item['name'])) die("Error: Empty name on report builder " . $this->Name);
			if(!isset($item['active']) || empty($item['active'])) $item['active'] = "0";
			if(!isset($item['order'])) $item['order'] = "0";
			if(!isset($item['label']) || empty($item['label'])) $item['label'] = $item['name'];

			$item['cols'] = $this->Cols;

			unset($itemValue);
			if(is_array($this->Value) && isset($this->Value[$item['name']])) {
				$itemValue =& $this->Value[$item['name']];
				$item['active'] = $itemValue['active'];
				$item['order'] = $itemValue['order'];
			} else {
				$itemValue = array();
			}

			foreach($item['cols'] as $colName=>$col) {
				if(isset($itemValue['cols'][$colName])) {
					$item['cols'][$colName]['value'] = $itemValue['cols'][$colName]['value'];
				} elseif(isset($item[$colName])) {
					$item['cols'][$colName]['value'] = $item[$colName];
				} else {
					$item['cols'][$colName]['value'] = "";
				}
			}
		}
	}

	function Render() {
		$html = join("",file(__DIR__ . "/../../../public/html/rep_template_builder.htm"));
		$this->Filter['name'] = $this->Name;
		$this->Filter['arg']= $this->Arg;
		if($this->Label !== false) {
			$html = "<td>!label!</td>\n".$html;
			$this->Filter['label'] = $this->Label;
			$this->Filter['colspan'] = "2";
		} else {
			$this->Filter['colspan'] = "3";
		}
		$items = array();

		$cols = "";
		$headers = "";
		foreach($this->Cols as $col) {
			$cols .= "<col style=\"width: $col[width]px\" />\n";
			$headers .= "
				<td class=\"outLt\"><label onmouseover=\"overlib('".t_encodeOverlib($col['overlib'])."', WIDTH, 300);\" onmouseout=\"nd();\">$col[label]</label></td>
			";

		}
		$this->Filter['cols'] = $cols;
		$this->Filter['headers'] = $headers;
		$this->Filter['header_colspan'] = sizeof($this->Cols) + 1;
		$this->Filter['items'] = json_encode($this->Items);

		return showpage($this->Filter,$html);
	}

	function FieldSetValue($field,$key,$value) {
		if(array_key_exists($field,$this->Items)) {
			$this->Items[$field][$key] = $value;
		}
	}

	function FieldIsActive($field) {
		if(array_key_exists($field,$this->Items) && array_key_exists("active",$this->Items[$field])) {
			return $this->Items[$field]['active'] == 1;
		} else {
			return false;
		}
	}

	function FieldHasPropertyValue($property, $value, $active=true) {
		$fields = $this->FieldGetPropertyValue($property, $value);
		return (count($fields) > 0);
	}

	function FieldGetPropertyValue($property, $value, $active=true) {
		$returnFields = array();
		$value = is_array($value) ? $value : array($value);
		foreach ($this->Items as $item) {
			if (isset($item[$property])) {
				$hasValue = false;
				$checkActive = $active ? $item['active'] : true;
				if ($checkActive) {
					foreach ($value as $v) {
						$hasValue = (in_array($v, $item[$property])) ? true : false;
						if ($hasValue) {
							break;
						}
					}
				}
				if ($hasValue) {
					$returnFields[] = $item;
				}
			}
		}
		return $returnFields;
	}
}
