<?php

require_once(__DIR__ . '/class.financial.export.php');

/**
 * Export invoices and payments in the journal format used by Sage Evolution.
 */
class SageEvolutionJournalExport extends FinancialExport
{
    /**
     * The configuration settings for the Sage Evolution interface.
     *
     * @var array
     */
    protected $options;

    /**
     * The id of the batch the invoices and payments should be pulled from.
     *
     * @var string
     */
    protected $batchId;

    /**
     * The invoicing unit to export for.
     *
     * @var string
     */
    protected $invoicingUnit;

    /**
     * Create a new instance of a Sage Evolution journal export.
     *
     * @param MySQLDB $lDB The connection to the database.
     * @param array $options The options for the Sage Evolution financial interface.
     * @param string $batchId The id of the batch that invoices and payments belong to.
     * @param string $invoicingUnit The business entity.
     */
    public function __construct($lDB, $options, $batchId, $invoicingUnit)
    {
        parent::__construct($lDB);
        $this->options = $options;
        $this->batchId = $batchId;
        $this->invoicingUnit = $invoicingUnit;
        $this->setShowHeader(false);
        $this->init();
    }

    /**
     * Initialise the export.
     *
     * @return void
     */
    protected function init()
    {
        $this->buildHeader();
        $this->buildFields();
        $this->buildTables();
        $this->buildFilters();
        $this->buildDataFields();
    }

    protected function buildHeader()
    {
        $this->setHeader([
            'Tran date',
            'Desc',
            'Ref',
            'Amount',
            'Use tax',
            'Tax type',
            'Tax account',
            'Tax amount',
            'Project',
            'Gl code',
            'Debit'
        ]);
    }

    protected function buildFields()
    {
        $this->addFields([
            'fn_tran.fn_tran_date_ledger AS transaction_date',
            'fn_tran.fn_tran_link_id',
            'fn_tran.fn_tran_link_ind',
            'fn_tran.fn_tran_detail',
            'fn_tran.rv_reservation_id',
            'fn_tran_item.fn_tran_item_note',
            'fn_tran_item.fn_tran_item_amt_incl',
            'fn_tran_item.fn_tran_item_amt_source_incl',
            'fn_tran_item.rv_payment_item_id',
            'tax_ledger.fn_ledger_code AS tax_type',
            'fn_tran_item.fn_tran_item_amt_source',
            'fn_cost_centre.fn_cost_centre_code AS cost_centre1_code',
            'fn_tran_item.fn_tran_item_type_ind',
            'fn_debtor.fn_debtor_code',
            'fn_ledger.fn_ledger_code',
            'debtor_persona.pr_name_last as debtor_name_last',
            'debtor_persona.pr_name_first as debtor_name_first',
            'rv_reservation.rv_res_name'
        ]);
    }

    protected function buildTables()
    {
        $this->setTable('fn_tran');
        $this->addJoins([
            'INNER JOIN fn_tran_item ON fn_tran_item.fn_tran_id = fn_tran.fn_tran_ix',
            'LEFT JOIN fn_ledger ON fn_ledger.fn_ledger_ix = fn_tran_item.fn_ledger_id',
            'LEFT JOIN fn_debtor ON fn_debtor.fn_debtor_ix = fn_tran_item.fn_debtor_id',
            'LEFT JOIN fn_folio ON fn_folio.fn_folio_ix = fn_tran.fn_folio_id',
            'LEFT JOIN fn_cost_centre ON fn_cost_centre.fn_cost_centre_ix = fn_tran_item.fn_tran_cost_ctr1_id',
            'LEFT JOIN rf_tax_inv ON rf_tax_inv.rf_tax_inv_ix = fn_tran_item.rf_tax_inv_id',
            'LEFT JOIN fn_ledger AS tax_ledger ON tax_ledger.fn_ledger_ix = rf_tax_inv.fn_ledger_id',
            'LEFT JOIN pr_persona AS debtor_persona ON debtor_persona.pr_persona_ix = fn_folio.fn_folio_to_id',
            'INNER JOIN rv_reservation ON rv_reservation.rv_reservation_ix = fn_tran.rv_reservation_id'
        ]);
    }

    protected function buildFilters()
    {
        if (!empty($this->batchId) && $this->batchId != 'all' && $this->batchId != '0') {
            $this->addFilter("
                fn_tran.fn_batch_id = '{$this->batchId}'
            ");     // From the specified batch only
        }

        $this->addFilter("
            fn_tran_item.fn_tran_item_type_ind <> " . DB_FN_TRAN_ITEM_TYPE_TAX . "
        ");
    }

    protected function buildDataFields()
    {
        $this->setDataFields([
            'transaction_date',
            'description',
            'reference',
            'amount',
            'use_tax',
            'tax_type',
            'tax_account',
            'tax_amount',
            'project',
            'gl_code',
            'is_debit'
        ]);
    }

    protected function processData(&$data)
    {
        $businessId = $this->invoicingUnit['id'];
        $sql = "
            SELECT fn_ledger.fn_ledger_code
            FROM pr_business
            INNER JOIN fn_ledger ON fn_ledger.fn_ledger_ix = pr_business.fn_ledger_id_tax
            WHERE pr_business.pr_business_id = '$businessId' 
        ";
        $taxControlAccount = $this->lDB->get($sql, 4);

        foreach($data as &$row) {
            $row['transaction_date'] = date('Y/m/d', strtotime($row['transaction_date']));
            $row['project'] = '';

            if ($row['tax_type']) {
                $row['tax_account'] = $taxControlAccount;
            } else {
                $row['tax_account'] = '';
            }
            $tranDetails = explode(';', $row['fn_tran_detail']);

            if ($row['fn_tran_link_ind'] == (string) DB_FN_TRAN_LINK_INVOICE) {
                // Processing an invoice
                switch ($this->options[FinancialOptions::OPTION_REFERENCE_FORMAT]) {
                    case FinancialOptions::REFERENCE_FORMAT_LINE_ITEM:
                        switch ($row['fn_tran_item_type_ind']) {
                            case '10':
                                $row['reference'] = 'Debtor';
                                break;
                            case '20':
                                $row['reference'] = 'Tax';
                                break;
                            case '30':
                                $row['reference'] = 'Accommodation';
                                break;
                            case '40':
                                $row['reference'] = 'Extra';
                                break;
                            case '50':
                                $row['reference'] = 'Deposit';
                                break;
                            case '60':
                                $row['reference'] = 'Bank';
                                break;
                            default:
                                $row['reference'] = 'Unknown';
                                break;
                        }
                        break;
                    case FinancialOptions::REFERENCE_FORMAT_RES_NUM:
                        $row['reference'] = $row['rv_reservation_id'];
                        break;
                    case FinancialOptions::REFERENCE_FORMAT_LINK:
                        $row['reference'] = $row['fn_tran_link_id'];
                        break;
                    case FinancialOptions::REFERENCE_FORMAT_DEBTOR_CODE:
                        $row['reference'] = $row['fn_debtor_code'];
                        break;
                    case FinancialOptions::REFERENCE_FORMAT_DOC_NUM:
                        $row['reference'] = $row['fn_tran_link_id'];      // Invoice can't have a payment number
                        break;
                    default:
                        $row['reference'] = $row['rv_reservation_id'];
                        break;
                }

                $description = '';
                $descriptionFields = explode(',', $this->options[FinancialOptions::OPTION_MEMO_FORMAT_INVOICE]);

                foreach ($descriptionFields as $descriptionField) {
                    if ($description) {
                        $description .= ';';        // Item separator
                    }
                    switch ($descriptionField) {
                        case '1':   // Transaction type
                            $description .= $tranDetails[0];
                            break;
                        case '2':   // Source doc id
                            $description .= $tranDetails[1];
                            break;
                        case '3':   // Res no
                            $description .= $row['rv_reservation_id'];
                            break;
                        case '4':   // Res name
                            $description .= $row['rv_res_name'];
                            break;
                        case '5':   // Debtor name
                            $description .= $tranDetails[3];
                            break;
                        case '6':   // Source curr
                            $description .= $tranDetails[4];
                            break;
                        case '7':   // Exchange rate
                            $description .= $tranDetails[5];
                            break;
                        case '20':  // Revenue item (fn_tran_item_note for invoices)
                            $description .= $row['fn_tran_item_note'];
                            break;
                        default:    // Use blank for unknown sequencing value
                            $description .= '';
                            break;
                    }
                }
                $row['description'] = $description;
                $row['amount'] = $row['fn_tran_item_amt_source_incl'];
                $row['tax_amount'] = $row['fn_tran_item_amt_source_incl'] - $row['fn_tran_item_amt_source'];

                if ($row['fn_tran_item_type_ind'] == (string) DB_FN_TRAN_ITEM_TYPE_DEBTOR) {
                    $row['gl_code'] = $row['fn_debtor_code'];
                    $row['use_tax'] = 'N';

                    if ($row['debtor_name_first']) {
                        $row['description'] = $row['debtor_name_first'] . ' ' . $row['debtor_name_last'];
                    } else {
                        $row['description'] = $row['debtor_name_last'];
                    }
                } else {
                    $row['use_tax'] = 'Y';
                    $row['gl_code'] = $row['fn_ledger_code'] . $row['cost_centre1_code'];
                }

                if ($row['amount'] < 0) {
                    $row['is_debit'] = 'N';
                    $row['amount'] *= -1;
                } else {
                    $row['is_debit'] = 'Y';
                }

                if ($row['tax_amount'] < 0) {
                    $row['tax_amount'] *= -1;
                }

                fixAmtDisplay($row['amount']);
                fixAmtDisplay($row['tax_amount']);

                if (!$row['tax_type']) {
                    $row['tax_amount'] = '';
                }

            } else {
                // Processing a payment, deposit, or reversal.
                $row['use_tax'] = 'N';
                $isPayment = $row['fn_tran_link_ind'] == (string) DB_FN_TRAN_LINK_PAYMENT;
                $itemNote = explode(';', $row['fn_tran_item_note']);
                $paymentNote = '';

                if (count($itemNote) > 0) {
                    $paymentNote = $itemNote[0];
                }

                $description = '';
                $descriptionFields = explode(',', $this->options[FinancialOptions::OPTION_MEMO_FORMAT_PAYMENT]);
                $paymentReference = '';
                if (count($itemNote) > 1) {
                    $paymentReference = $itemNote[1];
                }

                foreach ($descriptionFields as $descriptionField) {
                    if ($description) {
                        $description .= ';';        // Item separator
                    }
                    switch ($descriptionField) {
                        case '1':   // Transaction type
                            $description .= $tranDetails[0];
                            break;
                        case '2':   // Source doc id
                            $description .= $tranDetails[1];
                            break;
                        case '3':   // Payment item key
                            $description .= $row['fn_tran_link_id'];
                            break;
                        case '4':   // Res no
                            $description .= $row['rv_reservation_id'];      // fn_tran_detail contains more than just res no
                            break;
                        case '5':   // Res name
                            $description .= $row['rv_res_name'];
                            break;
                        case '6':   // Debtor name
                            $description .= $isPayment ? $tranDetails[4] : $tranDetails[3];
                            break;
                        case '7':   // Source currency
                            $description .= $isPayment ? $tranDetails[5] : $tranDetails[4];
                            break;
                        case '8':   // Exchange rate
                            $description .= $isPayment ? $tranDetails[6] : $tranDetails[5];
                            break;
                        case '9':   // Bank/CP abbreviation
                            $description .= $isPayment ? $tranDetails[7] : $tranDetails[6];
                            break;
                        case '20':  // Payment note
                            $description .= $itemNote[0];
                            break;
                        case '21':  // Payment reference
                            $description .= $paymentReference;
                            break;
                        default:    // Use blank for unknown sequencing value
                            $description .= '';
                            break;
                    }
                }

                $row['description'] = $description;
                $row['reference'] = $row['rv_reservation_id'];

                if ($row['fn_tran_item_amt_incl'] < 0) {
                    $row['amount'] = -1 * $row['fn_tran_item_amt_incl'];
                    $row['is_debit'] = 'N';
                } else {
                    $row['amount'] = $row['fn_tran_item_amt_incl'];
                    $row['is_debit'] = 'Y';
                }

                if ($row['fn_tran_item_type_ind'] == (string) DB_FN_TRAN_ITEM_TYPE_DEBTOR) {
                    $row['gl_code'] = $row['fn_debtor_code'];

                    if ($row['debtor_name_first']) {
                        $row['description'] = $row['debtor_name_first'] . ' ' . $row['debtor_name_last'];
                    } else {
                        $row['description'] = $row['debtor_name_last'];
                    }
                } else {
                    $row['gl_code'] = $row['fn_ledger_code'];
                }

                fixAmtDisplay($row['amount']);
                $row['tax_type'] = '';
                $row['tax_account'] = '';
                $row['tax_amount'] = '';
            }
        }
    }
}