<?php

require_once(__DIR__ . '/class.financial.export.php');

class SunSystemsTransactionExport extends FinancialExport
{
    /**
     * The configuration settings for the interface.
     *
     * @var array
     */
    protected $options;

    /**
     * The id of the batch the invoices should be pulled from.
     *
     * @var string
     */
    protected $batchId;

    /**
     * The invoicing unit home currency.
     *
     * @var string
     */
    protected $currency;

    /**
     * Create a new instance of the export.
     *
     * @param MySQLDB $lDB The connection to the database.
     * @param array $options The options for the financial interface.
     * @param string $batchId The id of the batch that invoices belong to.
     * @param string $currency The business entity's home currency.
     */
    public function __construct($lDB, $options, $batchId, $currency)
    {
        parent::__construct($lDB);
        $this->options = $options;
        $this->batchId = $batchId;
        $this->currency = $currency;

        $this->init();
    }

    /**
     * Initialise the export.
     *
     * @return void
     */
    protected function init()
    {
        $this->buildHeader();
        $this->buildFields();
        $this->buildTables();
        $this->buildFilters();
        $this->buildDataFields();
    }

    protected function buildHeader()
    {
        $this->setHeader([
            'AccntCode',
            'Period',
            'Date',
            'Reference',
            '',
            'Amount',
            '',
            'Jtype',
            'Opera Dept Name',
            'Cur',
            'OtherAmt',
            'T0=Property',
            'T1=Dept',
        ]);
    }

    protected function buildFields()
    {
        $this->addFields([
            'fn_ledger.fn_ledger_code AS gl_code',
            'fn_debtor.fn_debtor_code AS debtor_code',
            'fn_tran.fn_tran_date_ledger AS transaction_date',
            'fn_tran.fn_tran_link_ind AS fn_tran_link_ind',
            'fn_tran.fn_tran_detail',
            'fn_tran.fn_tran_link_id',
            'fn_tran_item.fn_tran_item_amt AS gl_amount',
            'fn_tran_item.fn_tran_item_amt_source AS source_amount',
            'fn_tran_item.fn_tran_item_note AS revenue_desc',
            'fn_tran_item.fn_tran_item_type_ind AS item_type_ind',
            'fn_tran_item.fn_supplier_code',
            'fn_batch.fn_batch_name AS batch_name',
            'rf_currency.rf_currency_symbol AS currency',
            'rf_source_currency.rf_currency_symbol AS source_currency',
            'fn_cost_centre1.fn_cost_centre_code AS revenue_centre1',
            'fn_cost_centre2.fn_cost_centre_code AS revenue_centre2',
            'fn_cost_centre2.fn_cost_centre_desc AS location',
            'tax_ledger.fn_ledger_code AS tax_code',
            'rf_tax_rate.rf_tax_rate_desc AS tax_rate',
            'rt_tax_group.rt_tax_group_desc AS tax_group',
            'fn_tran.fn_tran_inventory_doc AS tran_doc_ref'
        ]);
    }

    protected function buildTables()
    {
        $this->setTable('fn_tran');
        $this->addJoins([
            'INNER JOIN fn_tran_item ON fn_tran_item.fn_tran_id = fn_tran.fn_tran_ix',
            'LEFT JOIN fn_batch ON fn_batch.fn_batch_ix = fn_tran.fn_batch_id',
            'LEFT JOIN fn_ledger ON fn_ledger.fn_ledger_ix = fn_tran_item.fn_ledger_id',
            'LEFT JOIN fn_debtor ON fn_debtor.fn_debtor_ix = fn_tran_item.fn_debtor_id',            
            'LEFT JOIN rf_currency ON rf_currency.rf_currency_ix = fn_tran.rf_currency_id',
            'LEFT JOIN rf_currency AS rf_source_currency ON rf_source_currency.rf_currency_ix = fn_tran_item.rf_currency_id',
            'LEFT JOIN fn_cost_centre AS fn_cost_centre1 ON fn_cost_centre1.fn_cost_centre_ix = fn_tran_item.fn_tran_cost_ctr1_id',
            'LEFT JOIN fn_cost_centre AS fn_cost_centre2 ON fn_cost_centre2.fn_cost_centre_ix = fn_tran_item.fn_tran_cost_ctr2_id',
            'LEFT JOIN rf_tax_inv ON rf_tax_inv.rf_tax_inv_ix = fn_tran_item.rf_tax_inv_id',
            'LEFT JOIN rf_tax_rate ON rf_tax_rate.rf_tax_rate_ix = rf_tax_inv.rf_tax_id',
            'LEFT JOIN rt_tax_group ON rt_tax_group.rt_tax_group_ix = rf_tax_inv.rf_tax_id',
            'LEFT JOIN fn_ledger AS tax_ledger ON tax_ledger.fn_ledger_ix = rf_tax_inv.fn_ledger_id',
        ]);
    }

    protected function buildFilters()
    {
        if (!empty($this->batchId) && $this->batchId != 'all' && $this->batchId != '0') {
            $this->addFilter("
                fn_tran.fn_batch_id = '{$this->batchId}'
                AND fn_tran_item.fn_tran_item_amt != '0'
            ");     // From the specified batch only
        }
    }

    protected function buildDataFields()
    {
        $this->setDataFields([
            'gl_code',
            'acc_period',
            'transaction_date',
            'reference',
            'constant1',
            'gl_amount',
            'debit_credit',
            'constant2',
            'revenue_desc',
            'source_currency',
            'source_amount',
            'revenue_centre1',
            'revenue_centre2',
        ]);
    }

    protected function processData(&$data)
    {
        $newData = [];
        $constant2 = array('10' => 'RRQ-PM','20' => 'RRQ-DP', '30' => 'RRQ-RD', '40' => 'RRQ-IN', '51' => 'RRQ-PR', '52' => 'RRQ-RT', '53' => 'RRQ-CS', '54' => 'RRQ-SI');

        foreach($data as &$row) {
            $row['transaction_date'] = date('d-m-Y', strtotime($row['transaction_date']));
            $row['acc_period'] = $this->accountingPeriod($row['transaction_date']);
            $tranDetails = explode(';', $row['fn_tran_detail']);
 
            switch ($row['fn_tran_link_ind']) {
                case '40':
                    $row['reference'] = $row['fn_tran_link_id'];
                    break;
                case '51':
                    $row['reference'] = $tranDetails[2] . '; ' . $row['tran_doc_ref'];
                    $row['revenue_desc'] = $tranDetails[0];
                    break;
                case '52';
                case '53';
                case '54';
                    $row['reference'] = $row['tran_doc_ref'];
                    $row['revenue_desc'] = $row['location'];
                    break;
                default:
                    $row['reference'] = $tranDetails[1];
                    $row['revenue_desc'] = '';
                    break;
            }
            if (!$row['gl_code'] && $row['debtor_code']) {
                $row['gl_code'] = $row['debtor_code'];
            }
            if (!$row['gl_code'] && $row['fn_supplier_code']) {
                $row['gl_code'] = $row['fn_supplier_code'];
            }

            if ($row['item_type_ind'] == 20) {
                $row['revenue_desc'] = ($row['tax_rate']) ?? $row['tax_group'];
                $row['revenue_centre2'] = '';
            }
            if ( in_array($row['item_type_ind'], [70, 71, 72])) {
                $row['revenue_centre2'] = '';
            }

            $row['constant1'] = 'L';
            $row['constant2'] = $constant2[$row['fn_tran_link_ind']];
            $row['currency'] = $this->currency;

            $keyFields = [
                'gl_code',
                'transaction_date',
                'revenue_desc',
                'currency',
                'revenue_centre1',
                'revenue_centre2'
            ];
            $this->ensureValueOrEmptyString($row, $keyFields);

            $key = $this->rowKey(
                $row['gl_code'],
                $row['transaction_date'],
                $row['revenue_desc'],
                $row['currency'],
                $row['revenue_centre1'],
                $row['revenue_centre2']
            );

            if (!array_key_exists($key, $newData)) {
                array_push($newData, $row);
            } else {
                $newData[$key]['gl_amount'] += $row['gl_amount'];
            }
        }

        foreach ($newData as &$row) {
            if ($row['gl_amount'] < 0) {
                $row['gl_amount'] *= -1;
                $row['debit_credit'] = 'C';
            } else {
                $row['debit_credit'] = 'D';
            }
            fixAmtDisplay($row['gl_amount']);
            if ($row['source_amount'] < 0) {
                $row['source_amount'] *= -1;
            }
            fixAmtDisplay($row['source_amount']);
        }

        $data = $newData;
    }

    private function yearAndMonth($date)
    {
        $year = date('Y', strtotime($date));
        $month = date('m', strtotime($date));
        return [
            'year' => $year,
            'month' => $month
        ];
    }

    private function accountingPeriod($date)
    {
        $yearAndMonth = $this->yearAndMonth($date);
        return $yearAndMonth['year'] . '0' . $yearAndMonth['month'];
    }

    private function rowKey(
        string $glCode,
        string $transactionDate,
        string $revenueDescription,
        string $sourceCurrency,
        string $revenueCentre1,
        string $revenueCentre2
    ) {
        return $glCode
            . '_' . $transactionDate
            . '_' . $revenueDescription
            . '_' . $sourceCurrency
            . '_' . $revenueCentre1
            . '_' . $revenueCentre2;
    }

    /**
     * Ensure that values are either present or the empty string (avoid nulls).
     *
     * @param array $data
     * @param array $fields
     * @return void
     */
    private function ensureValueOrEmptyString(array &$data, array $fields)
    {
        foreach ($fields as $field) {
            if (empty($data[$field])) {
                $data[$field] = '';
            }
        }
    }
}