<?php

require_once(__DIR__ . '/class.financial.export.php');

/**
 * Export invoices in the format used by the Great Plains accounting system.
 */
class TwinfieldsInvoiceExport extends FinancialExport
{
    /**
     * The configuration settings for the Great Plains interface.
     *
     * @var array
     */
    protected $options;

    /**
     * The id of the batch the invoices should be pulled from.
     *
     * @var string
     */
    protected $batchId;

    /**
     * Create a new instance of a Twinfields invoice export.
     *
     * @param MySQLDB $lDB The connection to the database.
     * @param array $options The options for the Twinfields financial interface.
     * @param string $batchId The id of the batch that invoices belong to.
     */
    public function __construct($lDB, $options, $batchId)
    {
        parent::__construct($lDB);
        $this->options = $options;
        $this->batchId = $batchId;

        $this->init();
    }

    /**
     * Initialise the export.
     *
     * @return void
     */
    protected function init()
    {
        $this->buildHeader();
        $this->buildFields();
        $this->buildTables();
        $this->buildFilters();
        $this->buildDataFields();
    }

    protected function buildHeader()
    {
        switch ($this->options[FinancialOptions::OPTION_HEADER_LANGUAGE]) {
            case FinancialOptions::HEADER_LANGUAGE_NONE:
                $this->setShowHeader(false);
                break;
            case FinancialOptions::HEADER_LANGUAGE_ENGLISH:
                $this->setHeader([
                    'Code',
                    'Currency',
                    'Invoice date',
                    'Period',
                    'Invoice number',
                    'Due date',
                    'Number',
                    'Gl account',
                    'Rel/Cost centre',
                    'Prj/fixed asset',
                    'Amount',
                    'DebitCredit',
                    'Description',
                    'BTWcode'
                ]);
                break;
            case FinancialOptions::HEADER_LANGUAGE_DUTCH:
                $this->setHeader([
                    'Code',
                    'Valuta',
                    'Factuurdatum',
                    'Periode',
                    'Factuurnummer',
                    'Vervaldatum',
                    'Nummer',
                    'Grtboekrek',
                    'Rel/KPL',
                    'Prj/activa',
                    'Bedrag',
                    'debit_credit',
                    'Omschrijving',
                    'Btwcode'
                ]);
                break;
            default:
                $this->setShowHeader(false);
                break;
        }
        
    }

    protected function buildFields()
    {
        $this->addFields([
            'rf_currency.rf_currency_symbol AS currency',
            'fn_tran.fn_tran_date_ledger AS invoice_date',
            'fn_tran.fn_tran_link_id AS invoice_number',
            'fn_ledger.fn_ledger_code AS gl_account',
            'fn_cost_centre.fn_cost_centre_code AS cost_centre',
            'fn_tran_item.fn_tran_item_amt_source AS item_amount',
            'fn_tran_item.fn_tran_item_amt_source_incl AS item_amount_incl',
            'fn_tran_item.fn_tran_item_type_ind',
            'fn_debtor.fn_debtor_code AS debtor_code',
            'debtor_persona.pr_name_last As debtor_name_last',
            'debtor_persona.pr_name_first As debtor_name_first',
            'fn_tran_item.fn_tran_item_note'
        ]);
    }

    protected function buildTables()
    {
        $this->setTable('fn_tran');
        $this->addJoins([
            'INNER JOIN fn_tran_item ON fn_tran_item.fn_tran_id = fn_tran.fn_tran_ix',
            'LEFT JOIN fn_ledger ON fn_ledger.fn_ledger_ix = fn_tran_item.fn_ledger_id',
            'LEFT JOIN fn_folio ON fn_folio.fn_folio_ix = fn_tran.fn_folio_id',
            'LEFT JOIN fn_cost_centre ON fn_cost_centre.fn_cost_centre_ix = fn_tran_item.fn_tran_cost_ctr1_id',
            'LEFT JOIN pr_persona AS debtor_persona ON debtor_persona.pr_persona_ix = fn_folio.fn_folio_to_id',
            'LEFT JOIN fn_debtor ON fn_debtor.fn_debtor_ix = fn_tran_item.fn_debtor_id',
            'LEFT JOIN rf_currency ON rf_currency.rf_currency_ix = fn_tran.rf_currency_id'
        ]);
    }

    protected function buildFilters()
    {
        if (!empty($this->batchId) && $this->batchId != 'all' && $this->batchId != '0') {
            $this->addFilter("
                fn_tran.fn_batch_id = '{$this->batchId}'
            ");     // From the specified batch only
        }

        $this->addFilter("
            fn_tran.fn_tran_link_ind = '".DB_FN_TRAN_LINK_INVOICE."'
        ");     // Invoices only

        $this->setOrderBy("ORDER BY DATE(fn_tran_item.ad_create_date), fn_tran.fn_tran_link_id, fn_tran_item.fn_tran_item_type_ind");
    }

    protected function buildDataFields()
    {
        $this->setDataFields([
            'code',
            'currency',
            'invoice_date',
            'period',
            'invoice_number',
            'due_date',
            'number',
            'gl_account',
            'cost_centre',
            'fixed_asset',
            'amount',
            'debit_credit',
            'description',
            'btw_code'
        ]);
    }

    protected function processData(&$data)
    {
        foreach($data as &$row) {
            $row['code'] = $this->options[FinancialOptions::OPTION_TWINFIELDS_CODE];
            $row['invoice_date'] = date('d-m-Y', strtotime($row['invoice_date']));
            $row['period'] = date('Y/m', strtotime($row['invoice_date']));
            $row['due_date'] = $row['invoice_date'];
            $row['number'] = $row['invoice_number'];
            $row['fixed_asset'] = '';
            $row['btw_code'] = '';

            if ($row['fn_tran_item_type_ind'] == DB_FN_TRAN_ITEM_TYPE_DEBTOR) {
                $row['cost_centre'] = $row['debtor_code'];
                $row['amount'] = $row['item_amount_incl'];

                if ($row['debtor_name_first']) {
                    $row['description'] = $row['debtor_name_first'] . ' ' . $row['debtor_name_last'];
                } else {
                    $row['description'] = $row['debtor_name_last'];
                }
            } else {
                $row['amount'] = $row['item_amount'];
                $description = '';
                $descriptionFields = explode(',', $this->options[FinancialOptions::OPTION_MEMO_FORMAT_INVOICE]);

                foreach ($descriptionFields as $descriptionField) {
                    if ($description) {
                        $description .= ';';        // Item separator
                    }
                    switch ($descriptionField) {
                        case '1':   // Transaction type
                            $description .= $tranDetails[0];
                            break;
                        case '2':   // Source doc id
                            $description .= $tranDetails[1];
                            break;
                        case '3':   // Res no
                            $description .= $row['rv_reservation_id'];
                            break;
                        case '4':   // Res name
                            $description .= $row['rv_res_name'];
                            break;
                        case '5':   // Debtor name
                            $description .= $tranDetails[3];
                            break;
                        case '6':   // Source curr
                            $description .= $tranDetails[4];
                            break;
                        case '7':   // Exchange rate
                            $description .= $tranDetails[5];
                            break;
                        case '20':  // Revenue item (fn_tran_item_note for invoices)
                            $description .= $row['fn_tran_item_note'];
                            break;
                        default:    // Use blank for unknown sequencing value
                            $description .= '';
                            break;
                    }
                }
                $row['description'] = $description;
            }

            if (strlen($row['description']) > 40) {
                $row['description'] = substr($row['description'], 0, 40);
            }

            if ($row['amount'] < 0) {
                $row['amount'] *= -1;
                $row['debit_credit'] = 'credit';
            } else {
                $row['debit_credit'] = 'debit';
            }

            fixAmtDisplay($row['amount']);
        }
    }
}
