<?php

 /**
  * class.wetu.php - Wetu
  */

define("STATUS_SUCCESS",0);
define("STATUS_NO_DATA",10);
define("STATUS_MISSING_MAPPING",20);
define("STATUS_OVERLAPPING_ITINERARIES",30);
define("STATUS_ITINERARY_GAPS",40);
define("STATUS_ERROR",50);
define("GENERATION_STATUS_NOTGENERATED",0);
define("GENERATION_STATUS_OUTOFDATE",1);
define("GENERATION_STATUS_UPTODATE",2);
define("URL_WETU_API_JSON", "https://wetu.com/API/Pins/");
define("URL_WETU_API_XML", "https://wetu.com/Map/ItineraryServiceV7.asmx");

require_once(__DIR__ . '/db.sc_group.php');
require_once(__DIR__ . '/class.audit.php');
require_once(__DIR__ . '/class.newrelic.php');

class Wetu{

	var $WetuEnabled = false;
	var $WetuGenerationStatus = GENERATION_STATUS_NOTGENERATED;
	var $ReservationID = '';
	var $WetuID = '';
	var $WetuAuth = array();
	var $Reservation = array();
	var $ErrorItineraryOverlaps = array();
	var $ErrorMissingMapping = array();
	var $ErrorItineraryGaps = array();
	var $WetuCredentials = array();
	var $lDB = NULL;
	var $WetuDebug = false;
	var $newRelic;

	const WETU_STATUSES = array('0' => 'Quoted',
			'10' => 'AwaitingQuotation',
			'20' => 'Provisional',
			'30' => 'Booked',
			'90' => 'Cancelled',
			'95' => 'Cancelled'
		);

	public function __construct($reservationId, $lDB){
		$this->lDB = $lDB;
		$this->newRelic = new NewRelic();

		if ($reservationId != '' && !db_rv_reservation_exists($reservationId)) {
			return false;
		}
		$this->ReservationID = $reservationId;
		$this->WetuEnabled = db_sc_group_get_user_setting("sc_grp_res_wetu_yn") == 1 ? true : false;
		if ($this->WetuEnabled) {
			$this->GetWetuCredentials();
			if (empty($this->WetuCredentials)) {
				$this->WetuEnabled = false;
			}
		}
		if ($this->WetuEnabled) {
			$this->GetWetuGenerationStatus();
		}
	}

	public function BuildReservationDetails() {
		$this->GetReservation();
		$this->GetItineraries();
		$this->GetTravelExtras();
		$this->GetActivityExtras();
		$this->GetGuests();
		$this->GetContact();
	}

	private function GetReservation() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$this->Reservation = $this->lDB->get("
			SELECT
				rv_reservation.rv_res_name AS res_name,
				rv_reservation.rv_wetu_id AS wetu_id,
				rv_reservation.rv_date_arrive AS res_arrive,
				rv_reservation.rv_date_depart AS res_depart,
				rv_reservation.rv_amt_accomm_payable AS payable,
				rv_reservation.rf_reservation_status_id AS status,
				rf_currency.rf_currency_symbol AS currency,
				pr_persona.pr_name_first AS contact_name_first,
				pr_persona.pr_name_last AS contact_name_last,
				pr_persona.pr_email AS contact_email,
				pr_phone.pr_phone_number AS contact_tel
			FROM
				rv_reservation
				LEFT JOIN pr_persona ON pr_persona_ix = rv_reservation.rv_corr_persona_id
				LEFT JOIN pr_phone ON pr_phone.pr_persona_id = pr_persona.pr_persona_ix
				LEFT JOIN rf_currency ON rf_currency.rf_currency_ix = rv_reservation.rv_invoice_currency_id
			WHERE
				rv_reservation.rv_reservation_ix = '".$this->ReservationID."'
		",1);

        $resTotals = $this->lDB->get("
            SELECT
                rv_amt_accomm_payable,
                rv_amt_extra_payable,
                rv_amt_travel_payable
            FROM
                rv_reservation
            WHERE
                rv_reservation_ix = '" . $this->ReservationID . "'
        ",1);
        $this->Reservation['payable'] = $resTotals['rv_amt_accomm_payable'] + $resTotals['rv_amt_extra_payable'] + $resTotals['rv_amt_travel_payable'];
        fixAmtDisplay($this->Reservation['payable'],2);

		// WETU status names
		$this->Reservation['status'] = self::WETU_STATUSES[$this->Reservation['status']];
		$this->newRelic->stop_transaction();
	}

	private function GetItineraries() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$this->Reservation['Itineraries'] = $this->lDB->get("
			SELECT
				rv_reservation_item.rv_reservation_item_ix AS id,
				rv_reservation_item.rv_item_date_arrive AS arrive,
				rv_reservation_item.rv_item_date_depart AS depart,
				rv_reservation_item.rv_item_nights AS nights,
				(rv_reservation_item.rv_item_adult_count * rv_reservation_item.rv_item_accomm_count) AS adults,
				(rv_reservation_item.rv_item_child_count * rv_reservation_item.rv_item_accomm_count) AS children,
				rv_reservation_item.rv_item_accomm_count AS accomm_count,
				pr_business.pr_bus_wetu_id AS wetu_prop_id,
				pr_business.pr_bus_wetu_exclude AS wetu_prop_exclude,
				ac_accomm_type.ac_accomm_wetu_id AS wetu_accomm_id,
				ac_accomm_type.ac_accomm_wetu_exclude AS wetu_accomm_exclude,
				ac_accomm_type.ac_accomm_desc AS accomm_name,
				pr_persona.pr_name_last AS prop_name
			FROM
				rv_reservation_item
				LEFT JOIN pr_business ON pr_business.pr_business_id = rv_reservation_item.pr_business_id
				LEFT JOIN ac_accomm_type ON ac_accomm_type.ac_accomm_type_ix = rv_reservation_item.ac_accomm_type_id
				LEFT JOIN rv_reservation ON rv_reservation.rv_reservation_ix = '".$this->ReservationID."'
				LEFT JOIN pr_persona ON pr_persona_ix = pr_business.pr_business_id
			WHERE
				rv_reservation_item.rv_reservation_id = '".$this->ReservationID."'
				AND pr_business.pr_bus_wetu_exclude =! 1
				AND ac_accomm_type.ac_accomm_wetu_exclude != 1
			ORDER BY
				rv_reservation_item.rv_item_date_arrive
		",2);

		$this->Reservation['Itineraries'] = array_merge($this->Reservation['Itineraries'], $this->lDB->get("
			SELECT
				rv_extra.rv_extra_ix AS id,
				rv_extra.rv_extra_date_serv AS arrive,
				rv_extra.rv_extra_date_depart AS depart,
				rv_extra.rv_extra_nights AS nights,
				(rv_extra.rv_extra_adult_count * rv_extra.rv_extra_units) AS adults,
				(rv_extra.rv_extra_child_count * rv_extra.rv_extra_units) AS children,
				rv_extra.rv_extra_units AS accomm_count,
				ac_extra.ac_ext_desc AS prop_name,
				ac_extra.ac_ext_desc AS accomm_name,
				'Accommodation Extra' AS itinerary_type,
				ac_extra.ac_ext_wetu_prop_id AS wetu_prop_id,
				ac_extra.ac_ext_wetu_accomm_id AS wetu_accomm_id,
				ac_extra.ac_ext_wetu_exclude AS wetu_prop_accomm_exclude,
				rv_extra.rv_extra_note AS note,
				ac_extra_category.ac_extra_cat_desc as category
			FROM
				rv_extra
				LEFT JOIN ac_extra ON ac_extra.ac_extra_ix = rv_extra.ac_extra_id
				LEFT JOIN ac_extra_category ON ac_extra_category.ac_extra_category_ix = ac_extra.ac_extra_category_id
			WHERE
				rv_extra.rv_reservation_id = '".$this->ReservationID."'
				AND rv_extra.rv_extra_void_ind = 0
				AND ac_extra_category.ac_extra_cat_ind = 3
				AND ac_extra.ac_ext_wetu_exclude != 1
			ORDER BY
				rv_extra.rv_extra_date_serv
		",2) );

		// Find and merge duplicate itineraries (identical prop, accomm and dates, but different pax or room count)
		$newItins = array();
		foreach ($this->Reservation['Itineraries'] as $itin) {
			$newItins[$itin['id']] = $itin;
		}

		foreach ($this->Reservation['Itineraries'] as $keyMain => $itinMain) {
			foreach ($this->Reservation['Itineraries'] as $keyCompare => $itinCompare) {
				if ($itinMain['id'] != $itinCompare['id']) {
					if (
						$itinMain['arrive'] == $itinCompare['arrive'] &&
						$itinMain['depart'] == $itinCompare['depart'] &&
						$itinMain['wetu_prop_id'] == $itinCompare['wetu_prop_id'] &&
						$itinMain['wetu_accomm_id'] == $itinCompare['wetu_accomm_id'] &&
						$itinMain['nights'] == $itinCompare['nights']
					) {
						$id = $this->Reservation['Itineraries'][$keyMain]['id'];
						if (isset($newItins[$id])) {
							$newItins[$id]['adults'] += $this->Reservation['Itineraries'][$keyCompare]['adults'];
							$newItins[$id]['children'] += $this->Reservation['Itineraries'][$keyCompare]['children'];
							$newItins[$id]['accomm_count'] += $this->Reservation['Itineraries'][$keyCompare]['accomm_count'];
							unset($newItins[$itinCompare['id']]);
						}
					}
				}
			}
		}
		$this->Reservation['Itineraries'] = $newItins;

		$this->newRelic->stop_transaction();
	}

	private function GetTravelExtras() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$this->Reservation['TravelExtras'] = $this->lDB->get("
			SELECT
				rv_extra.rv_extra_ix AS rv_extra_id,
				rv_extra.rv_extra_date_serv AS service_date,
				rv_extra.rv_extra_units AS accomm_count,
				pr_business.pr_bus_wetu_id AS wetu_prop_id,
				ac_extra.ac_ext_desc AS description,
				rv_extra.rv_extra_note AS note,
				rv_extra.rv_extra_travel_etd AS time_departure,
				rv_extra.rv_extra_travel_eta AS time_arrival,
				rv_extra.rv_extra_ref AS reference,
				CONCAT(pr_persona.pr_name_first, ' ', pr_persona.pr_name_last) AS supplier,
                ac_extra.ac_ext_wetu_prop_id AS wetu_origin_id,
                ac_extra.ac_ext_wetu_accomm_id AS wetu_destination_id
			FROM
				rv_extra
				LEFT JOIN pr_business ON pr_business.pr_business_id = rv_extra.pr_business_id
				LEFT JOIN ac_extra ON ac_extra.ac_extra_ix = rv_extra.ac_extra_id
				LEFT JOIN ac_extra_category ON ac_extra_category.ac_extra_category_ix = ac_extra.ac_extra_category_id
				LEFT JOIN pr_persona ON pr_persona.pr_persona_ix = ac_extra.pr_supplier_id
			WHERE
				rv_extra.rv_reservation_id = '".$this->ReservationID."'
				AND rv_extra.rv_extra_void_ind = 0
				AND ac_extra_category.ac_extra_cat_ind = 2
		",2);
		$this->newRelic->stop_transaction();
	}

	private function GetActivityExtras() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$this->Reservation['ActivityExtras'] = $this->lDB->get("
			SELECT
				rv_extra.rv_extra_ix AS rv_extra_id,
				rv_extra.rv_extra_date_serv AS service_date,
				ac_extra.ac_ext_desc AS description,
				rv_extra.rv_extra_note AS note,
                ac_extra.ac_ext_wetu_prop_id AS wetu_activity_id,
                ac_extra.ac_ext_wetu_accomm_id AS wetu_supplier_id
			FROM
				rv_extra
				LEFT JOIN ac_extra ON ac_extra.ac_extra_ix = rv_extra.ac_extra_id
				LEFT JOIN ac_extra_category ON ac_extra_category.ac_extra_category_ix = ac_extra.ac_extra_category_id
			WHERE
				rv_extra.rv_reservation_id = '".$this->ReservationID."'
				AND rv_extra.rv_extra_void_ind = 0
				AND (
					ac_extra_category.ac_extra_cat_ind = 0
					OR ac_extra_category.ac_extra_cat_ind = 1
				)
		",2);
		$this->newRelic->stop_transaction();
	}

	private function GetGuests() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$this->Reservation['Guests'] = $this->lDB->get("
			SELECT
				pr_persona.pr_name_first AS guest_name_first,
				pr_persona.pr_name_last AS guest_name_last,
				rf_title.rf_title_desc AS guest_title
			FROM
				rv_reservation_guest
				LEFT JOIN pr_persona ON pr_persona.pr_persona_ix = rv_reservation_guest.pr_guest_id
				LEFT JOIN pr_guest ON pr_guest.pr_guest_id = rv_reservation_guest.pr_guest_id
				LEFT JOIN rf_title ON rf_title.rf_title_ix = pr_persona.pr_title_id
			WHERE
				rv_reservation_guest.rv_reservation_id = '".$this->ReservationID."'
				AND pr_guest.pr_guest_inactive_yn = 0
		",2);
		$this->newRelic->stop_transaction();
	}

	private function GetContact() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$this->Reservation['Contact'] = $this->lDB->get("
			SELECT
				pr_persona.pr_name_first AS contact_name_first,
				pr_persona.pr_name_last AS contact_name_last,
				pr_persona.pr_email AS contact_email,
				pr_phone.pr_phone_number AS contact_tel
			FROM
				rv_reservation
				INNER JOIN pr_persona ON pr_persona.pr_persona_ix = rv_reservation.rv_corr_persona_id
				LEFT JOIN pr_phone ON pr_phone.pr_persona_id = pr_persona.pr_persona_ix AND pr_phone.pr_default_yn = 1
			WHERE
				rv_reservation.rv_reservation_ix = '".$this->ReservationID."'
		",1);
		$this->newRelic->stop_transaction();
	}

	private function GetWetuCredentials() {
		$this->WetuCredentials = $this->lDB->get("
			SELECT
				rf_sys_wetu_username AS username,
				rf_sys_wetu_password AS password,
				rf_sys_wetu_api_key AS api_key
			FROM
				rf_system
			LIMIT 1
		",1);
	}

	private function GetWetuGenerationStatus() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$this->WetuID = $this->lDB->get("
			SELECT
				rv_wetu_id
			FROM
				rv_reservation
			WHERE
				rv_reservation.rv_reservation_ix = '".$this->ReservationID."'
		",4);

		if (empty($this->WetuID)) {
			$this->WetuGenerationStatus = GENERATION_STATUS_NOTGENERATED;
		} else {
			$lastGenerated = $this->lDB->get("
				SELECT
					rv_wetu_timestamp
				FROM
					rv_reservation
				WHERE
					rv_reservation.rv_reservation_ix = '".$this->ReservationID."'
			",4);

			$lastAuditChange = $this->lDB->get("
				SELECT
					ad_res_time
				FROM
					ad_reservation
				WHERE
					ad_reservation.rv_reservation_id = '".$this->ReservationID."'
					AND ad_res_form NOT LIKE '%wetu%'
				ORDER BY
					ad_reservation.ad_res_time DESC
				LIMIT 1
			",4);
			$this->WetuGenerationStatus = ( strtotime($lastGenerated) <= strtotime($lastAuditChange) ) ? GENERATION_STATUS_OUTOFDATE : GENERATION_STATUS_UPTODATE;
		}
		$this->newRelic->stop_transaction();
	}

	private function UpdateWetuID($rv_wetu_id) {
		$this->lDB->put("
			UPDATE
				rv_reservation
			SET
				rv_wetu_id = '".$this->lDB->escape($rv_wetu_id)."',
				rv_wetu_timestamp = NOW()
			WHERE
				rv_reservation.rv_reservation_ix = '".$this->ReservationID."'
		");
		ammendReservation($this->ReservationID,"Wetu Itinerary (Generate)");
	}

	public function GenerateButtontext() {
		if (!$this->WetuEnabled) {
			return '';
		}
		switch ($this->WetuGenerationStatus) {
		case GENERATION_STATUS_NOTGENERATED:
			// Not generated yet
			return '<tr class="wetuButton" onmouseover="this.style.backgroundColor=\'#ccffcc\';" onmouseout="this.style.backgroundColor=\'\';" onclick="generateWetu(\''.$this->ReservationID.'\');">
				<td width="15" align="center">
					<img alt="" class="imgLink" src="/resource/Resrequest/Application/public/img/doc_wetu.gif"/>
				</td>
				<td id="wetuButtonText">Generate and view Wetu</td>
			</tr>';
			break;
		case GENERATION_STATUS_OUTOFDATE:
			// Generated, but possibly out of date
			return '<tr class="wetuButton" onmouseover="this.style.backgroundColor=\'#ccffcc\';" onmouseout="this.style.backgroundColor=\'\';" onclick="viewWetuOutdated(\''.$this->WetuID.'\');">
				<td width="15" align="center">
					<img alt="" class="imgLink" src="/resource/Resrequest/Application/public/img/doc_wetu.gif"/>
				</td>
				<td id="wetuButtonText">View Wetu</td>
			</tr>
			<tr id="wetuButtonTextRegenerateContainer" class="wetuButton" onmouseover="this.style.backgroundColor=\'#ccffcc\';" onmouseout="this.style.backgroundColor=\'\';" onclick="regenerateWetu(\''.$this->ReservationID.'\');">
				<td width="15" align="center">
					<img alt="" class="imgLink" src="/resource/Resrequest/Application/public/img/doc_wetu.gif"/>
				</td>
				<td id="wetuButtonTextRegenerate">
					Re-Generate and view Wetu
					<input type="hidden" id="resID" value="'.$this->ReservationID.'">
				</td>
			</tr>';
			break;
		case GENERATION_STATUS_UPTODATE:
			// Up to date, View only
			return '<tr class="wetuButton" onmouseover="this.style.backgroundColor=\'#ccffcc\';" onmouseout="this.style.backgroundColor=\'\';" onclick="viewWetu(\''.$this->WetuID.'\');">
				<td width="15" align="center">
					<img alt="" class="imgLink" src="/resource/Resrequest/Application/public/img/doc_wetu.gif"/>
				</td>
				<td id="wetuButtonText">View Wetu</td>
			</tr>';
			break;
		}
	}

	public function SearchWetuProperty($searchString) {
		return ($this->WetuJsonCall('SearchWetuProperty', $searchString));
	}

	public function SearchWetuAccomm($propID) {
		return ($this->WetuJsonCall('SearchWetuAccomm', $propID));
	}

    public function SearchWetuActivity($searchString) {
        $results = ($this->WetuJsonCall('SearchWetuActivity', $searchString));
        $results = json_decode($results, true);
        $filteredResults = array();
        foreach ($results as $result) {
            if (isset($result['type']) && $result['type'] == "Activity") {
                $filteredResults[] = array("id"=>$result['id'], "name"=>$result['name']);
            }
        }
        return json_encode($filteredResults, true);
    }

    public function SearchWetuActivitySupplier($searchString) {
        $results = ($this->WetuJsonCall('SearchWetuActivity', $searchString));
        $results = json_decode($results, true);
        $filteredResults = array();
        foreach ($results as $result) {
            $filteredResults[] = array("id"=>$result['id'], "name"=>$result['name']);
        }
        return json_encode($filteredResults, true);
    }

    public function SearchWetuActivitySupplierActivity($id) {
        $results = ($this->WetuJsonCall('SearchWetuAccomm', $id));
        $results = json_decode($results, true);

        $filteredResults = array();
        if ($results[0]['type'] == "Activity") {
        	$options = $results[0]['options'];
        }
        if ($results[0]['type'] == "Accommodation") {
        	$options = $results[0]['activities'];
        }
        foreach ($options as $result) {
            $filteredResults[] = array("id"=>$result['id'], "name"=>$result['name']);
        }
        return json_encode($filteredResults, true);
    }

    public function SearchWetuTravel($searchString) {
        return ($this->WetuJsonCall('SearchWetuTravel', $searchString));
    }

    public function LookupName($searchString) {
        $result = $this->WetuJsonCall('LookupName', $searchString);
        $result = json_decode($result, true);
        if (isset($result[0]['name'])) {
            return array(
                "name" => $result[0]['name'],
                "category" => isset($result[0]['category']) ? $result[0]['category']: "",
                "type" => $result[0]['type'],
            );
        } else {
            return false;
        }
    }

    public function LookupAccommName($accommId, $propId) {
        $result = $this->WetuJsonCall('LookupName', $propId);
        $result = json_decode($result, true);

        if (stripos($result[0]['type'], "Mobile") !== false) {
            return array(
                "name" => $result[0]['name'] . " (" . $result[0]['type'] . ")"
            );
        }

        foreach ($result[0]['rooms'] as $room) {
        	if ($room['id'] == $accommId) {
                return array(
                    "name" => $result[0]['name'] . ": " . $room['name']
                );
        	}
        }
        return false;
    }

    public function LookupNameSupplier($activityId, $supplierId="") {
        $result = $this->WetuJsonCall('LookupName', $supplierId);
        $result = json_decode($result, true);
        if (!isset($result[0]['name'])) {
        	return false;
        }

        if ($result[0]['type'] == "Activity") {
        	$activities = $result[0]['options'];
        }
        if ($result[0]['type'] == "Accommodation") {
        	$activities = $result[0]['activities'];
        }
        foreach ($activities as $activity) {
        	if ($activity['id'] == $activityId) {
	            return array(
	                "name" => $activity['name'],
	                "category" => isset($activity['category']) ? $activity['category']: "",
	                "supplier" => $result[0]['name']
	            );
        	}
        }
        return false;
    }

	private function WetuJsonCall($action, $param) {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		if (empty($this->WetuCredentials)) {
			$this->GetWetuCredentials();
		}

		switch ($action) {
		case 'SearchWetuProperty':
			$url = URL_WETU_API_JSON.$this->WetuCredentials['api_key']."/Search/".urlencode($param)."?all=include";
			break;
		case 'SearchWetuAccomm':
			$url = URL_WETU_API_JSON.$this->WetuCredentials['api_key']."/Get?ids=".$param;
			break;
        case 'SearchWetuActivity':
            $url = URL_WETU_API_JSON.$this->WetuCredentials['api_key']."/Search/".urlencode($param)."?all=include";
            break;
        case 'SearchWetuTravel':
            $url = URL_WETU_API_JSON.$this->WetuCredentials['api_key']."/Search/".urlencode($param)."?all=include";
            break;
        case 'LookupName':
            $url = URL_WETU_API_JSON.$this->WetuCredentials['api_key']."/Get?ids=".$param;
            break;
		}

		$user_agent='Mozilla/5.0 (Windows NT 6.1; rv:8.0) Gecko/20100101 Firefox/8.0';

		$options = array(
			CURLOPT_CUSTOMREQUEST  =>"GET",        //set request type post or get
			CURLOPT_POST           =>false,        //set to GET
			CURLOPT_USERAGENT      => $user_agent, //set user agent
			CURLOPT_RETURNTRANSFER => true,     // return web page
			CURLOPT_HEADER         => false,    // don't return headers
			CURLOPT_FOLLOWLOCATION => true,     // follow redirects
			CURLOPT_ENCODING       => "",       // handle all encodings
			CURLOPT_AUTOREFERER    => true,     // set referer on redirect
			CURLOPT_CONNECTTIMEOUT => 120,      // timeout on connect
			CURLOPT_TIMEOUT        => 120,      // timeout on response
			CURLOPT_MAXREDIRS      => 10,       // stop after 10 redirects
		);

		$ch      = curl_init( $url );
		curl_setopt_array( $ch, $options );
		$content = curl_exec( $ch );
		$err     = curl_errno( $ch );
		$errmsg  = curl_error( $ch );
		$header  = curl_getinfo( $ch );
		curl_close( $ch );
		$this->newRelic->stop_transaction();

		if ($err != 0 && $err != '') {
			return('error:'.$errmsg);
		}

		return($content);
	}

	private function WetuXmlCall($action) {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$url = URL_WETU_API_XML;

		$headers = array(
			"Content-type: text/xml;charset=\"utf-8\"",
			"Accept: text/xml",
			"Cache-Control: no-cache",
			"Pragma: no-cache",
			"SOAPAction: http://wetu.com/Map/ItineraryServiceV7/".$action,
			"Content-length: ".strlen($this->WetuXmlString),
		);

		$ch = curl_init();
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 1);
		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_ANY);
		curl_setopt($ch, CURLOPT_TIMEOUT, 10);
		curl_setopt($ch, CURLOPT_POST, true);
		curl_setopt($ch, CURLOPT_POSTFIELDS, $this->WetuXmlString);
		curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

		$response = curl_exec($ch); 
		$err     = curl_errno( $ch );
		$errmsg  = curl_error( $ch );
		$header  = curl_getinfo( $ch );
		curl_close($ch);

		$response = str_replace(array("<soap:Body>", "</soap:Body>"), "", $response);
		$response = str_replace("soap:Fault","Fault",$response);
		$parsedResponse = simplexml_load_string($response);
		$this->newRelic->stop_transaction();

		return $parsedResponse;
	}

	public function SendWetuItinerary() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$result = $this->GenerateXML();

		// Determine any errors, else action the call and return result
		if (!empty($this->ErrorMissingMapping)) {
			$this->ErrorMissingMapping = array_unique($this->ErrorMissingMapping);
			$this->newRelic->stop_transaction();
			return( array('status' => STATUS_MISSING_MAPPING, 'desc' => implode("<br>", $this->ErrorMissingMapping)) );
		}
		if (!empty($this->ErrorItineraryOverlaps)) {
			$this->ErrorItineraryOverlaps = array_unique($this->ErrorItineraryOverlaps);
			$this->newRelic->stop_transaction();
			return( array('status' => STATUS_OVERLAPPING_ITINERARIES, 'desc' => implode("<br>", $this->ErrorItineraryOverlaps)) );
		}
		if (!empty($this->ErrorItineraryGaps)) {
			$this->ErrorItineraryGaps = array_unique($this->ErrorItineraryGaps);
			$this->newRelic->stop_transaction();
			return( array('status' => STATUS_ITINERARY_GAPS, 'desc' => implode("<br>", $this->ErrorItineraryGaps)) );
		}
		$result = $this->WetuXmlCall("AuthenticateAndSaveItinerary");

		if (!empty($result)) {
			if (isset($result->Fault->faultstring)) {
				$faultString = (string)$result->Fault->faultstring;
				return( array('status' => STATUS_ERROR, 'desc' => $faultString) );
			} else {
				$wetuIdentifier = $result->AuthenticateAndSaveItineraryResponse->AuthenticateAndSaveItineraryResult->Identifier;
				$this->Reservation['wetu_id'] = $wetuIdentifier;
				$this->UpdateWetuID($wetuIdentifier);
				$this->newRelic->stop_transaction();
				return( array('status' => STATUS_SUCCESS, 'desc' => $wetuIdentifier) );
			}
		} else {
			$this->newRelic->stop_transaction();
			return( array('status' => STATUS_NO_DATA, 'desc' => 'No data from Wetu') );
		}
		
	}

	private function GenerateXML() {
		$this->newRelic->record_transaction("Wetu -" . __FUNCTION__);
		$language = "en";
		$identifier = !empty($this->Reservation['wetu_id']) ? '<Identifier>'.$this->Reservation['wetu_id'].'</Identifier>' : "";
		$dates = array();
		$guests = array();
		$totalAdults = 0;
		$totalChildren = 0;
		$totalPax = 0;

		// Initialise data arrays for entire reservation
		$resDates = array();
		$properties = array();

		// Build a day-by-day array of data
		foreach ($this->Reservation['Itineraries'] as $itinerary) {
			$period = new DatePeriod(
				new DateTime($itinerary['arrive']),
				new DateInterval('P1D'),
				new DateTime($itinerary['depart'])
			);
			foreach ($period as $key => $value) {
				$day = $value->format('Y-m-d');
				$resDates[$day] = isset($resDates[$day]) ? $resDates[$day] : array();
				$resDates[$day][$itinerary['wetu_prop_id']] = isset($resDates[$day][$itinerary['wetu_prop_id']]) ? $resDates[$day][$itinerary['wetu_prop_id']] : array();
				$resDates[$day][$itinerary['wetu_prop_id']][$itinerary['wetu_accomm_id']] = isset($resDates[$day][$itinerary['wetu_prop_id']][$itinerary['wetu_accomm_id']]) ? $resDates[$day][$itinerary['wetu_prop_id']][$itinerary['wetu_accomm_id']] : array();
				$totalAdults = ($totalAdults < $itinerary['adults']) ? $itinerary['adults'] : $totalAdults;
				$totalChildren = ($totalChildren < $itinerary['children']) ? $itinerary['children'] : $totalChildren;
				$category = isset($itinerary['category']) ? $itinerary['category'] . ": " : "";
				if (trim($itinerary['wetu_prop_id']) == '' && trim($itinerary['wetu_accomm_id']) != '') {
					$accomm = $this->LookupName($itinerary['wetu_accomm_id']);
					if (stripos($accomm['type'], "Mobile") === false) {
						$this->ErrorMissingMapping[] = $category . $itinerary['prop_name'];
					}
				}
				if (trim($itinerary['wetu_prop_id']) != '' && trim($itinerary['wetu_accomm_id']) == '') {
					$accomm = $this->LookupName($itinerary['wetu_prop_id']);
					if (stripos($accomm['type'], "Mobile") === false) {
						$this->ErrorMissingMapping[] = $category . $itinerary['prop_name']." - ".$itinerary['accomm_name'];
					}
				}
				if (trim($itinerary['wetu_prop_id']) == '' && trim($itinerary['wetu_accomm_id']) == '') {
					$this->ErrorMissingMapping[] = $category . $itinerary['prop_name']." - ".$itinerary['accomm_name'];
				}
			}
			$properties[$itinerary['wetu_prop_id']][$itinerary['wetu_accomm_id']] = $itinerary;
			$properties[$itinerary['wetu_prop_id']]['prop_name'] = $itinerary['prop_name'];
		}

		$days = count($resDates);
		reset($resDates);
		$arrivalDate = key($resDates);
		end($resDates);
		$departureDate = date('Y-m-d', strtotime(key($resDates) . ' +1 day'));
		$lastnight = key($resDates);

		$totalPax = $totalAdults + $totalChildren;
		if (count($this->Reservation['Guests']) > $totalPax) {
			$totalPax = count($this->Reservation['Guests']);
		}

		foreach ($resDates as $date => $day) {
			if ( count($resDates[$date] ) > 1) {
				$this->ErrorItineraryOverlaps[] = $date;
			}
		}

		$totalDateRange = new DatePeriod(
			new DateTime($arrivalDate),
			new DateInterval('P1D'),
			new DateTime($lastnight)
		);
		foreach ($totalDateRange as $key => $value) {
			if ( !array_key_exists($value->format('Y-m-d'), $resDates) ) {
				$this->ErrorItineraryGaps[] = $value->format('Y-m-d');
			}
		}

		// These are show-stopping error conditions. Don't bother proceeding.
		if (!empty($this->ErrorItineraryOverlaps) || !empty($this->ErrorItineraryGaps) || $this->ErrorMissingMapping) {
			$this->newRelic->stop_transaction();
			return;
		}
		ksort($resDates);

		// Build neccesary data for each leg of the Wetu itinerary. This will also combine concurrent or overlapping identical prop/accomms
		$prevPropAccomm = '';
		$prevProp = '';
		$prevAccomm = '';
		$legCount = 0;
		$legs = array();
		$dayNumber = 0;
		$travelExtras = array();
		$travelExtraSequence = 1;
		$legDay = 0;
		$resDay = 0;
		$count = 0;

		// Build itinerary legs
		foreach ($resDates as $date => $props) {
			$dayNumber++;
			foreach ($props as $prop_id => $accomms) {
				if ($prevProp != $prop_id) {
					$legCount++;
					$legDay = 1;
					$prevProp = $prop_id;
					$legs[$legCount] = array();
					$legs[$legCount]['prop_id'] = $prop_id;
					$legs[$legCount]['nights'] = 1;
					$legs[$legCount]['rooms'] = array();
					foreach ($accomms as $accomm_id => $accomm) {
						$rooms = array();
						$rooms['number'] = $properties[$prop_id][$accomm_id]['accomm_count'];
						$rooms['name'] = $properties[$prop_id][$accomm_id]['accomm_name'];
						$rooms['accomm_id'] = $accomm_id;
						$legs[$legCount]['rooms'][] = $rooms;
					}
				} else {
					$legs[$legCount]['nights']++;
					$legDay++;
				}
				$resDates[$date]['leg'] = $legCount;
				$resDates[$date]['legDay'] = $legDay;
				$resDates[$date]['resDay'] = $resDay;
				$resDay++;
			}
		}

		foreach ($this->Reservation['TravelExtras'] as $id => $travelExtra) {
            $mappedOrigin = $mappedDestination = false;
            $mappedOriginAirport = $mappedDestinationAirport = false;
            $travelExtras[$count]['Mode'] = "Transfer";

            if ($travelExtra['wetu_origin_id'] != "") {
                $origin = $this->LookupName($travelExtra['wetu_origin_id']);
                if ($origin) {
                    $mappedOrigin = $travelExtra['wetu_origin_id'];
                    if (stripos($origin['category'], 'air') !== false) {
                        $mappedOriginAirport = true;
                    }
                }
            }
            if ($travelExtra['wetu_destination_id'] != "") {
                $destination = $this->LookupName($travelExtra['wetu_destination_id']);
                if ($destination) {
                    $mappedDestination = $travelExtra['wetu_destination_id'];
                    if (stripos($destination['category'], 'air') !== false) {
                        $mappedDestinationAirport = true;
                    }
                }
            }
            if ($mappedOriginAirport && $mappedDestinationAirport) {
                $travelExtras[$count]['Mode'] = "ScheduledFlight";
            }

			if (isset($resDates[$travelExtra['service_date']])) {
				$legNumber = $resDates[$travelExtra['service_date']]['leg'];
				$leg = $legs[$legNumber];
				$travelExtras[$count]['firstContentEntityId'] = $legNumber == 1 ? '0' : $legs[$legNumber-1]['prop_id'];
				$travelExtras[$count]['secondContentEntityId'] = $legs[$legNumber]['prop_id'];
                if ($mappedOrigin) {
                    $travelExtras[$count]['firstContentEntityId'] = $mappedOrigin;
                }
                if ($mappedDestination) {
                    $travelExtras[$count]['secondContentEntityId'] = $mappedDestination;
                }
				$travelExtras[$count]['startLegId'] = $legNumber == 1 ? 0 : ($legNumber-1);
				$travelExtras[$count]['endLegId'] = $legNumber;
				$travelExtras[$count]['firstLocation'] = $legNumber == 1 ? '' : htmlspecialchars(trim($properties[$legs[$legNumber-1]['prop_id']]['prop_name']));
				$travelExtras[$count]['secondLocation'] = htmlspecialchars(trim($properties[$legs[$legNumber]['prop_id']]['prop_name']));
				$travelExtras[$count]['day'] = $resDates[$travelExtra['service_date']]['resDay'];
				$travelExtras[$count]['sequence'] = $resDates[$travelExtra['service_date']]['resDay'];
				$travelExtras[$count]['referenceCodes'] = htmlspecialchars(trim($travelExtra['reference']));
				$travelExtras[$count]['notes'] = htmlspecialchars(trim($travelExtra['description']));
				$travelExtras[$count]['agency'] = htmlspecialchars(trim($travelExtra['supplier']));
				$travelExtras[$count]['vehicle'] = 'N/A';
				$travelExtras[$count]['firstTime'] = htmlspecialchars(trim($travelExtra['time_departure']));
				$travelExtras[$count]['secondTime'] = htmlspecialchars(trim($travelExtra['time_arrival']));
	        }

            // Departure date travel
            if ($travelExtra['service_date'] == $departureDate) {
				$legNumber = count($legs)+1;
				$travelExtras[$count]['firstContentEntityId'] = $legNumber == 1 ? '0' : $legs[$legNumber-1]['prop_id'];
				$travelExtras[$count]['secondContentEntityId'] = '1';
                if ($mappedOrigin) {
                    $travelExtras[$count]['firstContentEntityId'] = $mappedOrigin;
                }
                if ($mappedDestination) {
                    $travelExtras[$count]['secondContentEntityId'] = $mappedDestination;
                }
				$travelExtras[$count]['startLegId'] = $legNumber == 1 ? 0 : ($legNumber-1);
				$travelExtras[$count]['endLegId'] = 0;
				$travelExtras[$count]['firstLocation'] = $legNumber == 1 ? '' : htmlspecialchars(trim($properties[$legs[$legNumber-1]['prop_id']]['prop_name']));
				$travelExtras[$count]['secondLocation'] = '';
				$travelExtras[$count]['day'] = $days+1;
				$travelExtras[$count]['sequence'] = $days+1;
				$travelExtras[$count]['referenceCodes'] = htmlspecialchars(trim($travelExtra['reference']));
				$travelExtras[$count]['notes'] = htmlspecialchars(trim($travelExtra['description']));
				$travelExtras[$count]['agency'] = htmlspecialchars(trim($travelExtra['supplier']));
				$travelExtras[$count]['vehicle'] = 'N/A';
				$travelExtras[$count]['firstTime'] = htmlspecialchars(trim($travelExtra['time_departure']));
				$travelExtras[$count]['secondTime'] = htmlspecialchars(trim($travelExtra['time_arrival']));
			}
			$count++;
		}
		$this->Reservation['TravelExtras'] = $travelExtras;

		$activityExtras = array();
		foreach ($resDates as $date => $day) {
			foreach ($this->Reservation['ActivityExtras'] as $activityExtra) {
				if ($activityExtra['service_date'] == $date) {
					$activityExtras[$day['leg']][$day['legDay']][] = $activityExtra;
				}
			}
		}
		
		$this->Reservation['ActivityExtras'] = $activityExtras;

		$itineraries = array();
		foreach ($this->Reservation['Itineraries'] as $itinerary) {
			$itineraries[$itinerary['id']] = $itinerary;
		}
		$this->Reservation['Itineraries'] = $itineraries;

		// Here starts the magic of building our XML!
		$this->WetuXmlString['Header'] = '<?xml version="1.0" encoding="utf-8"?>
			<soap:Envelope xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema" xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/">
				<soap:Body>
					<AuthenticateAndSaveItinerary xmlns="http://wetu.com/Map/ItineraryServiceV7">';

		$this->WetuXmlString['itinerary'] = '<Type>Personal</Type>
							'.$identifier.'
							<Days>'.$days.'</Days>
							<Name>'.htmlspecialchars($this->Reservation['res_name']).'</Name>
							<Language>'.$language.'</Language>
							<ReferenceNumber>'.$this->ReservationID.'</ReferenceNumber>
							<ClientName>'.htmlspecialchars(trim($this->Reservation['contact_name_first'].' '.$this->Reservation['contact_name_last'])).'</ClientName>
							<ClientEmail>'.htmlspecialchars($this->Reservation['contact_email']).'</ClientEmail>
							<ClientPhone>'.htmlspecialchars($this->Reservation['contact_tel']).'</ClientPhone>
							<StartDate>'.$arrivalDate.'</StartDate>
							<HideRouteInformation>false</HideRouteInformation>
							<Price>'.$this->Reservation['payable'].' '.$this->Reservation['currency'].'</Price>
							<IsDisabled>false</IsDisabled>
							<TravellersAdult>'.$totalPax.'</TravellersAdult>
							<StartTravelDays>0</StartTravelDays>
							<EndTravelDays>0</EndTravelDays>
							<BookingStatus>'.$this->Reservation['status'].'</BookingStatus>
							<NotificationFrequency>FirstView</NotificationFrequency>
							<RouteHandlingMode>AutoGenerateRoutesFromTravelArrangements</RouteHandlingMode>';

		$this->WetuXmlString['Travellers'] = '';
		foreach ($this->Reservation['Guests'] as $guest) {
			$this->WetuXmlString['Travellers'] .= '
								<ItineraryTraveller>
									<Title>'.htmlspecialchars($guest['guest_title']).'</Title>
									<Name>'.htmlspecialchars(trim($guest['guest_name_first'].' '.$guest['guest_name_last'])).'</Name>
								</ItineraryTraveller>';
		}
		$this->WetuXmlString['Travellers'] = '<Travellers>'.$this->WetuXmlString['Travellers'].'</Travellers>';

		$legnumber = 1;
		$this->WetuXmlString['Legs'] = '';

		// Add general and activity extras as daily activities to legs
		foreach ($legs as $legNumber => $leg) {
			$days = '';
			if (!empty($this->Reservation['ActivityExtras'][$legNumber])) {
				$itineraryLegDay = array();
				$dayNumbers = array();
				foreach ($this->Reservation['ActivityExtras'][$legNumber] as $dayNumber => $day) {
					$activities = array();
					$sequence = 1;
					foreach ($day as $activity) {
                        $activityName = htmlspecialchars($activity['description']);
                        $activityId = "";

                        if ($activity['wetu_activity_id'] != "") {
                        	if ($activity['wetu_supplier_id'] != "") {
                        		$name = $this->LookupNameSupplier($activity['wetu_activity_id'], $activity['wetu_supplier_id']);
                        		if ($name) {
	                                $activityName = htmlspecialchars($name['name']);
	                                $activityId = "<ContentEntityId>" . $activity['wetu_supplier_id'] . "</ContentEntityId>
	                                <SimpleItemId>" . $activity['wetu_activity_id'] . "</SimpleItemId>";
	                            }
                        	} else {
	                            $name = $this->LookupName($activity['wetu_activity_id']);
	                            if ($name) {
	                                $activityName = htmlspecialchars($name['name']);
	                                $activityId = "<ContentEntityId>" . $activity['wetu_activity_id'] . "</ContentEntityId>";
	                            }
	                        }
                        }
                        
						$activities[] = '<ItineraryLegDayActivity>
											<Sequence>'.$sequence.'</Sequence>
											<Name>'.$activityName.'</Name>
											<Type>Included</Type>
											<TimeSlot>None</TimeSlot>' . $activityId . '
										</ItineraryLegDayActivity>';
						$sequence++;
					}
					if (!empty($activities)) {
						$itineraryLegDay[$dayNumber] = '<ItineraryLegDay>
												<Day>'.$dayNumber.'</Day>
												<Activities>
													'.implode('',$activities).'
												</Activities>
											</ItineraryLegDay>';
					}
					$dayNumbers[] = $dayNumber;
				}
				// Find and fill in any days missing from our ItineraryLegDay's (days with no activities)
				for ($i = 1; $i <= end($dayNumbers); $i++) {
					if (!in_array($i, $dayNumbers)) {
						$itineraryLegDay[$i] = '<ItineraryLegDay>
												<Day>' . $i . '</Day>
												<Activities></Activities>
											</ItineraryLegDay>';
					}
				}
				if (!empty($itineraryLegDay)) {
					sort($itineraryLegDay);
					$days = '<Days>'.implode('',$itineraryLegDay).'</Days>';
				}


			}
			if ($leg['prop_id'] != '0' && !empty($leg['prop_id'])) {
				$this->WetuXmlString['Legs'] .= '
									<ItineraryLeg xsi:type="ItineraryLegByDay">
										<ItineraryLegId>'.$legNumber.'</ItineraryLegId>
										<Sequence>'.$legNumber.'</Sequence>
										<ContentEntityId>'.$leg['prop_id'].'</ContentEntityId>
										<AlternateAccommodations xsi:nil="true" />
										<Nights>'.$leg['nights'].'</Nights>
										<BookingReference>'.htmlspecialchars($this->ReservationID.' - '.trim($properties[$leg['prop_id']]['prop_name'])).'</BookingReference>';
				if (!empty($leg['rooms'])) {
					$this->WetuXmlString['Legs'] .= '<Rooms>';
					foreach ($leg['rooms'] as $room) {
						$this->WetuXmlString['Legs'] .= '
												<ItineraryLegRoom>
													<Rooms>'.$room['number'].'</Rooms>
													<Name>'.htmlspecialchars(trim($room['name'])).'</Name>
													<SimpleItemId>'.$room['accomm_id'].'</SimpleItemId>
												</ItineraryLegRoom>';
					}
					$this->WetuXmlString['Legs'] .= '</Rooms>';
				} else {
					$this->WetuXmlString['Legs'] .= '
												<ItineraryLegTypeEnum>Mobile</ItineraryLegTypeEnum>
												<Stops>
													<ItineraryStop>
														<Type>Camp</Type>
													</ItineraryStop>
												</Stops>
												';
				}

				$this->WetuXmlString['Legs'] .= '
										<DayRooms xsi:nil="true" />
										<Type>Standard</Type>'.$days.'
									</ItineraryLeg>';
			} else {
				$bookingReference = $this->ReservationID;
				if (
					isset($leg['accomm_id']) &&
					isset($properties[$leg['prop_id']][$leg['accomm_id']]) &&
					isset($properties[$leg['prop_id']][$leg['accomm_id']]['prop_name'])
				) {
					$bookingReference .= ' - ' . trim($properties[$leg['prop_id']][$leg['accomm_id']]['prop_name']);
				} else {
					if (isset($properties[$leg['prop_id']]['rooms'][0]['name'])) {
						$bookingReference .= ' - ' . trim($properties[$leg['prop_id']]['rooms'][0]['name']);
					}
				}
				$this->WetuXmlString['Legs'] .= '
								<ItineraryLeg xsi:type="ItineraryLegByDay">
									<ItineraryLegId>'.$legNumber.'</ItineraryLegId>
									<Sequence>'.$legNumber.'</Sequence>
									<Nights>'.$leg['nights'].'</Nights>
									<BookingReference>' . htmlspecialchars($bookingReference) . '</BookingReference>
									<DayRooms xsi:nil="true" />
									<Type>OwnArrangement</Type>'.$days.'
								</ItineraryLeg>';
			}
			$legnumber++;
		}
		$this->WetuXmlString['Legs'] = '<Legs>'.$this->WetuXmlString['Legs'].'</Legs>';

		// Add Travel extras as Travel Arrangements
		$this->WetuXmlString['TravelArrangements'] = '';
		foreach ($this->Reservation['TravelExtras'] as $travelExtra) {
			if (empty($travelExtra['firstContentEntityId']) || empty($travelExtra['secondContentEntityId'])) {
				continue;
			}
			$this->WetuXmlString['TravelArrangements'] .= '
								<ItineraryTravelArrangement>
									<Mode>' . $travelExtra['Mode'] . '</Mode>
									<FirstContentEntityId>'.$travelExtra['firstContentEntityId'].'</FirstContentEntityId>
									<SecondContentEntityId>'.$travelExtra['secondContentEntityId'].'</SecondContentEntityId>
									<StartLegId>'.$travelExtra['startLegId'].'</StartLegId>
            						<EndLegId>'.$travelExtra['endLegId'].'</EndLegId>
            						<FirstLocation>'.$travelExtra['firstLocation'].'</FirstLocation>
            						<SecondLocation>'.$travelExtra['secondLocation'].'</SecondLocation>
									<Day>'.$travelExtra['day'].'</Day>
									<Sequence>'.$travelExtra['sequence'].'</Sequence>
									<ReferenceCodes>'.$travelExtra['referenceCodes'].'</ReferenceCodes>
									<Notes>'.$travelExtra['notes'].'</Notes>
									<Agency>'.$travelExtra['agency'].'</Agency>
									<Vehicle>'.$travelExtra['vehicle'].'</Vehicle>
									<FirstTime>'.$travelExtra['firstTime'].'</FirstTime>
									<SecondTime>'.$travelExtra['secondTime'].'</SecondTime>
								</ItineraryTravelArrangement>';
		}
		$this->WetuXmlString['TravelArrangements'] = '<TravelArrangements>'.$this->WetuXmlString['TravelArrangements'].'</TravelArrangements>';

		$this->WetuXmlString['Contacts'] = '
								<ItineraryContact>
									<Telephone>'.htmlspecialchars($this->Reservation['Contact']['contact_tel']).'</Telephone>
									<ContactPerson>'.htmlspecialchars(trim($this->Reservation['Contact']['contact_name_first'].' '.$this->Reservation['Contact']['contact_name_last'])).'</ContactPerson>
									<Email>'.htmlspecialchars($this->Reservation['Contact']['contact_email']).'</Email>
								</ItineraryContact>';
		$this->WetuXmlString['Contacts'] = '<Contacts>'.$this->WetuXmlString['Contacts'].'</Contacts>';
							
		$this->WetuXmlString['Footer'] = '
						<username>'.$this->WetuCredentials['username'].'</username>
      					<password>'.$this->WetuCredentials['password'].'</password>
					</AuthenticateAndSaveItinerary>
				</soap:Body>
			</soap:Envelope>';
		$this->WetuXmlString = 
			$this->WetuXmlString['Header'] .
			'<itinerary>' .
			$this->WetuXmlString['itinerary'] .
			$this->WetuXmlString['Travellers'] .
			$this->WetuXmlString['Legs'] .
			$this->WetuXmlString['TravelArrangements'] .
			$this->WetuXmlString['Contacts'] .
			'</itinerary>' .
			$this->WetuXmlString['Footer'];

		if ($this->WetuDebug) {
			print_r($this->WetuXmlString);
		}
		$this->newRelic->stop_transaction();
	}
	
}