<?php

require_once(__DIR__ . '/class.financial.export.php');

/**
 * Export invoices in the format used by the Xero accounting system.
 */
class XeroInvoiceExport extends FinancialExport
{
    /**
     * The configuration settings for the Xero interface.
     *
     * @var array
     */
    protected $options;

    /**
     * The id of the batch the invoices should be pulled from.
     *
     * @var string
     */
    protected $batchId;

    /**
     * Create a new instance of a Xero invoice export.
     *
     * @param MySQLDB $lDB The connection to the database.
     * @param array $options The options for the Xero financial interface.
     * @param string $batchId The id of the batch that invoices belong to.
     */
    public function __construct($lDB, $options, $batchId)
    {
        parent::__construct($lDB);
        $this->options = $options;
        $this->batchId = $batchId;

        $this->init();
    }

    /**
     * Initialise the export.
     *
     * @return void
     */
    protected function init()
    {
        $this->buildHeader();
        $this->buildFields();
        $this->buildTables();
        $this->buildFilters();
        $this->buildDataFields();
    }

    protected function buildHeader()
    {
        $this->setHeader([
            'ContactName',
            'EmailAddress',
            'POAddressLine1',
            'POAddressLine2',
            'POAddressLine3',
            'POAddressLine4',
            'POCity',
            'PORegion',
            'POPostalCode',
            'POCountry',
            'InvoiceNumber',
            'Reference',
            'InvoiceDate',
            'DueDate',
            'Total',
            'InventoryItemCode',
            'Description',
            'Quantity',
            'UnitAmount',
            'Discount',
            'AccountCode',
            'TaxType',
            'TrackingName1',
            'TrackingOption1',
            'TrackingName2',
            'TrackingOption2',
            'Currency',
            'BrandingTheme'
        ]);
    }

    protected function buildFields()
    {
        $this->addFields([
            'fn_tran.fn_tran_link_ind AS link_type',
            'fn_tran.rv_reservation_id AS res_number',
            'rv_reservation.rv_res_name AS res_name',
            'fn_tran.fn_tran_link_id AS inv_number',
            'fn_tran_item.fn_tran_item_amt AS item_amt'
        ]);

        switch ($this->options[FinancialOptions::OPTION_DEBTOR_SELECT]) {
            case FinancialOptions::DEBTOR_SELECT_RR:
                $this->addFields([
                    'fn_invoice.fn_inv_to AS debtor_name',
                    'pr_persona.pr_email AS debtor_email',
                    'fn_invoice.fn_inv_to_addr_line1 AS debtor_po_addr_line1',
                    'fn_invoice.fn_inv_to_addr_line2 AS debtor_po_addr_line2',
                    'fn_invoice.fn_inv_to_addr_line3 AS debtor_po_addr_line3',
                    'fn_invoice.fn_inv_to_city AS debtor_po_city',
                    'fn_invoice.fn_inv_to_post_code AS debtor_po_postal_code',
                    'fn_invoice.fn_inv_to_country AS debtor_po_country'
                ]);
                break;
            case FinancialOptions::DEBTOR_SELECT_RR_CASH_MAPPED:
                $this->addFields([
                    'fn_invoice.fn_inv_to AS debtor_name',
                    'pr_persona.pr_email AS debtor_email',
                    'fn_invoice.fn_inv_to_addr_line1 AS debtor_po_addr_line1',
                    'fn_invoice.fn_inv_to_addr_line2 AS debtor_po_addr_line2',
                    'fn_invoice.fn_inv_to_addr_line3 AS debtor_po_addr_line3',
                    'fn_invoice.fn_inv_to_city AS debtor_po_city',
                    'fn_invoice.fn_inv_to_post_code AS debtor_po_postal_code',
                    'fn_invoice.fn_inv_to_country AS debtor_po_country'
                ]);
                $this->addFields([
                    'cash_persona.pr_name_last AS cash_debtor_name_last',
                    'cash_persona.pr_name_first AS cash_debtor_name_first',
                    'cash_persona.pr_email AS cash_debtor_email',
                    'cash_persona.pr_corr_addr_line1 AS cash_debtor_po_addr_line1',
                    'cash_persona.pr_corr_addr_line2 AS cash_debtor_po_addr_line2',
                    'cash_persona.pr_corr_addr_line3 AS cash_debtor_po_addr_line3',
                    'cash_persona.pr_corr_city AS cash_debtor_po_city',
                    'cash_persona.pr_corr_post_code AS cash_debtor_po_postal_code',
                    'cash_country.rf_country_name AS cash_debtor_po_country'
                ]);
                break;
            case FinancialOptions::DEBTOR_SELECT_MAPPED:
            default:
                $this->addFields([
                    'pr_persona.pr_name_last AS debtor_name',
                    'pr_persona.pr_email AS debtor_email',
                    'pr_persona.pr_corr_addr_line1 AS debtor_po_addr_line1',
                    'pr_persona.pr_corr_addr_line2 AS debtor_po_addr_line2',
                    'pr_persona.pr_corr_addr_line3 AS debtor_po_addr_line3',
                    'pr_persona.pr_corr_city AS debtor_po_city',
                    'pr_persona.pr_corr_post_code AS debtor_po_postal_code',
                    'rf_country.rf_country_name AS debtor_po_country'
                ]);
                break;
        }

        switch ($this->options[FinancialOptions::OPTION_REFERENCE_FORMAT]) {
            case FinancialOptions::REFERENCE_FORMAT_LINE_ITEM:
                $this->addField('fn_tran_item.fn_tran_item_type_ind AS ref_type');
                break;
            case FinancialOptions::REFERENCE_FORMAT_DEBTOR_CODE:
                $this->addField('fn_debtor.fn_debtor_code as debtor_code');
                break;
            default:
                break;
        }

        $this->addField("DATE_FORMAT(fn_tran.fn_tran_date_ledger, '%d/%m/%Y') AS inv_date");

        $memoFields = explode(',', $this->options[FinancialOptions::OPTION_MEMO_FORMAT]);
        foreach ($memoFields as $memoField) {
            switch($memoField) {
                case FinancialOptions::MEMO_FORMAT_EXCHANGE_RATE:
                    $this->addField('fn_tran.fn_tran_exch_rate AS exchange_rate');
                    break;
                default:
                    break;
            }
        }

        $this->addField('1 AS quantity');
        $this->addFields([
            'fn_tran_item.fn_tran_item_amt AS gl_excl_tax',
            'fn_tran_item.fn_tran_item_amt_incl AS gl_incl_tax',
            'fn_tran_item.fn_tran_item_amt_source AS source_excl_tax',
            'fn_tran_item.fn_tran_item_amt_source_incl AS source_incl_tax'
        ]);
        $this->addField('fn_ledger.fn_ledger_code AS ledger_code');

        switch ($this->options[FinancialOptions::OPTION_ACCOUNT_CODE_FORMAT]) {
            case FinancialOptions::ACCOUNT_CODE_FORMAT_LEDGER_CODE:
                break;
            default:
                $this->addField('fn_cost_centre.fn_cost_centre_code AS cost_ctr1_code');
            break;
        }

        $this->addField('tax_ledger.fn_ledger_code AS tax_code');

        $trackingLabels = explode(',', $this->options[FinancialOptions::OPTION_TRACKING_LABELS]);
        $this->addField("'{$trackingLabels[0]}' AS tracking_label1");
        $this->addField("'{$trackingLabels[1]}' AS tracking_label2");

        $trackingValues = explode(',', $this->options[FinancialOptions::OPTION_TRACKING_VALUES]);

        switch ($trackingValues[0]) {
            case FinancialOptions::TRACKING_VALUES_EMPTY:
                $this->addField("'' AS tracking_value1");
                break;
            case FinancialOptions::TRACKING_VALUES_COST_CNTR1:
                $this->addField('fn_cost_centre.fn_cost_centre_code AS tracking_value1');
                break;
            case FinancialOptions::TRACKING_VALUES_RES_NUM:
                $this->addField('fn_tran.rv_reservation_id AS tracking_value1');
                break;
            case FinancialOptions::TRACKING_VALUES_RES_NUM:
                $this->addField('fn_tran.fn_tran_link_id AS tracking_value1');
                break;
            case FinancialOptions::TRACKING_VALUES_DEBTOR_CODE:
                $this->addField('fn_debtor.fn_debtor_code AS tracking_value1');
                break;
            case FinancialOptions::TRACKING_VALUES_COST_CNTR2:
                $this->addField('fn_cost_centre_2.fn_cost_centre_code AS tracking_value1');
                break;
                default:
                $this->addField('fn_cost_centre.fn_cost_centre_code AS tracking_value1');
                break;
        }

        switch ($trackingValues[1]) {
            case FinancialOptions::TRACKING_VALUES_EMPTY:
                $this->addField("'' AS tracking_value2");
                break;
            case FinancialOptions::TRACKING_VALUES_COST_CNTR1:
                $this->addField('fn_cost_centre.fn_cost_centre_code AS tracking_value2');
                break;
            case FinancialOptions::TRACKING_VALUES_RES_NUM:
                $this->addField('fn_tran.rv_reservation_id AS tracking_value2');
                break;
            case FinancialOptions::TRACKING_VALUES_RES_NUM:
                $this->addField('fn_tran.fn_tran_link_id AS tracking_value2');
                break;
            case FinancialOptions::TRACKING_VALUES_DEBTOR_CODE:
                $this->addField('fn_debtor.fn_debtor_code AS tracking_value2');
                break;
            case FinancialOptions::TRACKING_VALUES_COST_CNTR2:
                $this->addField('fn_cost_centre_2.fn_cost_centre_code AS tracking_value2');
                break;
                default:
                $this->addField('fn_cost_centre.fn_cost_centre_code AS tracking_value2');
                break;
        }

        $this->addField('fn_invoice.fn_inv_curr AS currency');
    }

    protected function buildTables()
    {
        $this->setTable('fn_tran');
        $this->addJoins([
            'INNER JOIN fn_tran_item ON fn_tran_item.fn_tran_id = fn_tran.fn_tran_ix',
            'INNER JOIN fn_invoice ON fn_invoice.fn_invoice_ix = fn_tran.fn_tran_link_id',
            'INNER JOIN rv_reservation ON rv_reservation.rv_reservation_ix = fn_tran.rv_reservation_id',
            'INNER JOIN fn_ledger ON fn_ledger.fn_ledger_ix = fn_tran_item.fn_ledger_id',
            'LEFT JOIN fn_cost_centre ON fn_cost_centre.fn_cost_centre_ix = fn_tran_item.fn_tran_cost_ctr1_id',
            'LEFT JOIN fn_cost_centre AS fn_cost_centre_2 ON fn_cost_centre_2.fn_cost_centre_ix = fn_tran_item.fn_tran_cost_ctr2_id',
            'LEFT JOIN rf_tax_inv ON rf_tax_inv.rf_tax_inv_ix = fn_tran_item.rf_tax_inv_id',
            'LEFT JOIN fn_ledger AS tax_ledger ON tax_ledger.fn_ledger_ix = rf_tax_inv.fn_ledger_id',
            'INNER JOIN fn_tran_item as debtor_item ON debtor_item.fn_tran_id = fn_tran_item.fn_tran_id AND debtor_item.fn_tran_item_type_ind = 10',
            'LEFT JOIN fn_debtor ON fn_debtor.fn_debtor_ix = debtor_item.fn_debtor_id',
        ]);

        switch ($this->options[FinancialOptions::OPTION_DEBTOR_SELECT]) {
            case FinancialOptions::DEBTOR_SELECT_RR:
                $this->addJoins([
                    'INNER JOIN fn_folio ON fn_folio.fn_folio_ix = fn_tran.fn_folio_id',
                    'LEFT JOIN pr_persona ON pr_persona.pr_persona_ix = fn_folio.fn_folio_to_id',
                ]);
                break;
            case FinancialOptions::DEBTOR_SELECT_RR_CASH_MAPPED:
                $this->addJoins([
                    'INNER JOIN fn_folio ON fn_folio.fn_folio_ix = fn_tran.fn_folio_id',
                    'INNER JOIN pr_business ON pr_business.pr_business_id = fn_tran.pr_business_id',
                    'LEFT JOIN pr_persona ON pr_persona.pr_persona_ix = fn_folio.fn_folio_to_id',
                    'LEFT JOIN fn_debtor AS cash_debtor ON cash_debtor.fn_debtor_ix = pr_business.fn_debtor_id_cash',
                    'LEFT JOIN pr_persona AS cash_persona ON cash_persona.pr_persona_ix = cash_debtor.pr_persona_id AND pr_business.fn_debtor_id_cash = debtor_item.fn_debtor_id',
                    'INNER JOIN rf_country ON rf_country_ix = pr_persona.pr_corr_country_id',
                    'LEFT JOIN rf_country AS cash_country ON cash_country.rf_country_ix = cash_persona.pr_corr_country_id'
                ]);
                break;
            case FinancialOptions::DEBTOR_SELECT_MAPPED:
                $this->addJoins([
                    'LEFT JOIN pr_persona ON pr_persona.pr_persona_ix = fn_debtor.pr_persona_id',
                    'LEFT JOIN rf_country ON rf_country_ix = pr_persona.pr_corr_country_id'
                ]);
                break;
            default:
                break;
        }

        if ($this->options[FinancialOptions::OPTION_REFERENCE_FORMAT]
            == FinancialOptions::REFERENCE_FORMAT_DEBTOR_CODE) {
            $this->addJoin('LEFT JOIN fn_debtor ON fn_debtor.fn_debtor_ix = fn_tran_item.fn_debtor_id');
        }
    }

    protected function buildFilters()
    {
        // Batch filter
        if (!empty($this->batchId) && $this->batchId != 'all' && $this->batchId != '0') {
            $this->addFilter("
                fn_tran.fn_batch_id = '{$this->batchId}'
            ");
        }

        // Only accept invoices
        $this->addFilter("
            fn_tran.fn_tran_link_ind = '".DB_FN_TRAN_LINK_INVOICE."'
        ");

        // Suppress tax lines
        $this->addFilter("
            fn_tran_item.fn_tran_item_type_ind <> '". DB_FN_TRAN_ITEM_TYPE_TAX ."' 
        ");
    }

    protected function buildDataFields()
    {
        $this->setDataFields([
            'debtor_name',
            'debtor_email',
            'debtor_po_addr_line1',
            'debtor_po_addr_line2',
            'debtor_po_addr_line3',
            '',     // Blank
            'debtor_po_city',
            '',     // Blank
            'debtor_po_postal_code',
            'debtor_po_country',
            'inv_number',
            'reference',
            'inv_date',
            'inv_date',
            '',     // Blank
            '',     // Blank
            'description',
            'quantity',
            'inv_amt',
            '',     // Blank
            'account_code',
            'tax_code',
            'tracking_label1',
            'tracking_value1',
            'tracking_label2',
            'tracking_value2',
            'currency',
            ''
        ]);
    }

    protected function processData(&$data)
    {
        foreach($data as &$row) {
            if (
                $this->options[FinancialOptions::OPTION_DEBTOR_SELECT]
                == FinancialOptions::DEBTOR_SELECT_RR_CASH_MAPPED
            ) {
                if ($row['cash_debtor_name_last']) {
                    if ($row['cash_debtor_name_first']) {
                        $row['debtor_name'] = $row['cash_debtor_name_first'] . ' ' . $row['cash_debtor_name_last'];
                    } else {
                        $row['debtor_name'] = $row['cash_debtor_name_last'];
                    }
                    $row['debtor_email'] = $row['cash_debtor_email'];
                    $row['debtor_po_addr_line1'] = $row['cash_debtor_po_addr_line1'];
                    $row['debtor_po_addr_line2'] = $row['cash_debtor_po_addr_line2'];
                    $row['debtor_po_addr_line3'] = $row['cash_debtor_po_addr_line3'];
                    $row['debtor_po_city'] = $row['cash_debtor_po_city'];
                    $row['debtor_po_postal_code'] = $row['cash_debtor_po_postal_code'];
                    $row['debtor_po_country'] = $row['cash_debtor_po_country'];
                }
            }

            switch ($this->options[FinancialOptions::OPTION_REFERENCE_FORMAT]) {
                case FinancialOptions::REFERENCE_FORMAT_LINE_ITEM:
                    switch ($row['ref_type']) {
                        case '10':
                            $row['reference'] = 'Debtor';
                            break;
                        case '20':
                            $row['reference'] = 'Tax';
                            break;
                        case '30':
                            $row['reference'] = 'Accommodation';
                            break;
                        case '40':
                            $row['reference'] = 'Extra';
                            break;
                        case '50':
                            $row['reference'] = 'Deposit';
                            break;
                        case '60':
                            $row['reference'] = 'Bank';
                            break;
                        default:
                            $row['reference'] = 'Unknown';
                            break;
                    }
                    break;
                case FinancialOptions::REFERENCE_FORMAT_RES_NUM:
                    $row['reference'] = $row['res_number'];
                    break;
                case FinancialOptions::REFERENCE_FORMAT_LINK:
                    $row['reference'] = $row['inv_number'];
                    break;
                case FinancialOptions::REFERENCE_FORMAT_DEBTOR_CODE:
                    $row['reference'] = $row['debtor_code'];
                    break;
                case FinancialOptions::REFERENCE_FORMAT_DOC_NUM:
                    $row['reference'] = $row['inv_number'];
                    break;
                default:
                    $row['reference'] = $row['res_number'];
                    break;
            }

            $description = '';
            $memoFields = explode(',', $this->options[FinancialOptions::OPTION_MEMO_FORMAT]);

            foreach ($memoFields as $memoField) {
                if ($description) {
                    $description .= ', ';
                }

                switch($memoField) {
                    case FinancialOptions::MEMO_FORMAT_TRAN_TYPE:
                        switch($row['link_type']) {
                            case DB_FN_TRAN_LINK_INVOICE:
                                if($row['item_amt'] < 0) {
                                    $type = "CN";
                                } else {
                                    $type = "IN";
                                }
                                break;
                            case DB_FN_TRAN_LINK_PAYMENT:
                                if($row['item_amt'] > 0) {
                                    $type = "RF";
                                } else {
                                    $type = "PM";
                                }
                                break;
                            case DB_FN_TRAN_LINK_DEPOSIT:
                                $type = "DP";
                                break;
                            case DB_FN_TRAN_LINK_REVERSAL:
                                $type = "DR";
                                break;
                            default:
                                $type = "UK";
                                break;
                        }
                        $description .= $type;
                        break;
                    case FinancialOptions::MEMO_FORMAT_SOURCE_DOC_ID:
                        $description .= $row['inv_number'];
                        break;
                    case FinancialOptions::MEMO_FORMAT_RES_NUM:
                        $description .= $row['res_number'];
                        break;
                    case FinancialOptions::MEMO_FORMAT_RES_NAME:
                        $description .= $row['res_name'];
                        break;
                    case FinancialOptions::MEMO_FORMAT_DEBTOR_NAME:
                        $description .= $row['debtor_name'];
                        break;
                    case FinancialOptions::MEMO_FORMAT_SOURCE_CURRENCY:
                        $description .= $row['currency'];
                        break;
                    case FinancialOptions::MEMO_FORMAT_EXCHANGE_RATE:
                        $description .= $row['exchange_rate'];
                        break;
                }
            }

            $row['description'] = $description;

            $accountCode = '';

            switch ($this->options[FinancialOptions::OPTION_ACCOUNT_CODE_FORMAT]) {
                case FinancialOptions::ACCOUNT_CODE_FORMAT_LEDGER_CODE:
                    $accountCode = $row['ledger_code'];
                    break;
                case FinancialOptions::ACCOUNT_CODE_FORMAT_LEDGER_CODE_COST_CNTR:
                    $accountCode = $row['ledger_code'] . $row['cost_ctr1_code'];
                    break;
                default:
                    $accountCode = $row['cost_ctr1_code'] . $row['ledger_code'];
                    break;
            }

            $row['account_code'] = $accountCode;

            switch ($this->options[FinancialOptions::OPTION_INV_AMT_TYPE]) {
                case FinancialOptions::INV_AMT_TYPE_SOURCE_CURR_INCL_TAX:
                    $row['inv_amt'] = $row['source_incl_tax'];
                    break;
                case FinancialOptions::INV_AMT_TYPE_SOURCE_CURR_EXCL_TAX:
                    $row['inv_amt'] = $row['source_excl_tax'];
                    break;
                case FinancialOptions::INV_AMT_TYPE_GL_CURR_INCL_TAX:
                    $row['inv_amt'] = $row['gl_incl_tax'];
                    break;
                case FinancialOptions::INV_AMT_TYPE_GL_CURR_EXCL_TAX:
                    $row['inv_amt'] = $row['gl_excl_tax'];
                    break;
                default:
                    $row['inv_amt'] = $row['source_incl_tax'];
                    break;
            }

            $row['inv_amt'] *= -1;
            fixAmtDisplay($row['inv_amt'], 2);
        }
    }
}